# coding=utf-8
# ------------------------------------
# Copyright (c) Microsoft Corporation.
# Licensed under the MIT License.
# ------------------------------------
import base64
import hashlib
import re
import time
import json
from typing import Any, List, Dict, MutableMapping, IO, Optional, Union
from urllib.parse import urlparse
from azure.core.exceptions import ServiceRequestError
from azure.core.pipeline import PipelineRequest

JSON = MutableMapping[str, Any]

BEARER = "Bearer"
AUTHENTICATION_CHALLENGE_PARAMS_PATTERN = re.compile('(?:(\\w+)="([^""]*)")+')
SUPPORTED_API_VERSIONS = [
    "2019-08-15-preview",
    "2021-07-01"
]
DEFAULT_CHUNK_SIZE = 4 * 1024 * 1024 # 4MB
MAX_MANIFEST_SIZE = 4 * 1024 * 1024

# The default audience used for all clouds when audience is not set
DEFAULT_AUDIENCE = "https://containerregistry.azure.net"

# Known manifest media types
OCI_IMAGE_MANIFEST = "application/vnd.oci.image.manifest.v1+json"
DOCKER_MANIFEST = "application/vnd.docker.distribution.manifest.v2+json"

SUPPORTED_MANIFEST_MEDIA_TYPES = ", ".join(
    [
        "*/*",
        OCI_IMAGE_MANIFEST,
        DOCKER_MANIFEST,
        "application/vnd.oci.image.index.v1+json",
        "application/vnd.docker.distribution.manifest.list.v2+json",
        "application/vnd.cncf.oras.artifact.manifest.v1+json"
    ]
)

def _is_tag(tag_or_digest: str) -> bool:
    tag = tag_or_digest.split(":")
    return not (len(tag) == 2 and tag[0].startswith("sha"))

def _clean(matches: List[str]) -> None:
    """This method removes empty strings and commas from the regex matching of the Challenge header.

    :param list[str] matches: The regex list to clean.
    :return: None
    """
    while True:
        try:
            matches.remove("")
        except ValueError:
            break

    while True:
        try:
            matches.remove(",")
        except ValueError:
            return

def _parse_challenge(header: str) -> Dict[str, str]:
    """Parse challenge header into service and scope

    :param str header: The challenge header to parse.
    :return: A service and scope dict parsed from challenge header.
    :rtype: dict[str, str]
    """
    ret: Dict[str, str] = {}
    if header.startswith(BEARER):
        challenge_params = header[len(BEARER) + 1 :]

        matches = re.split(AUTHENTICATION_CHALLENGE_PARAMS_PATTERN, challenge_params)
        _clean(matches)
        for i in range(0, len(matches), 2):
            ret[matches[i]] = matches[i + 1]

    return ret

def _parse_next_link(link_string: str) -> Optional[str]:
    """Parse the next link in the list operations response URL

    Per the Docker v2 HTTP API spec, the Link header is an RFC5988
    compliant rel='next' with URL to next result set, if available.
    See: https://docs.docker.com/registry/spec/api/

    The URI reference can be obtained from link-value as follows:
    Link       = "Link" ":" #link-value
    link-value = "<" URI-Reference ">" * (";" link-param )
    See: https://tools.ietf.org/html/rfc5988#section-5

    :param str link_string: The Link header in HTTP response.
    :return: The URI reference of next link.
    :rtype: str or None
    """
    if not link_string:
        return None
    return link_string[1 : link_string.find(">")]

def _enforce_https(request: PipelineRequest) -> None:
    """Raise ServiceRequestError if the request URL is non-HTTPS and the sender did not specify enforce_https=False

    :param ~azure.core.pipeline.PipelineRequest request: The pipeline request object.
    :return: None
    """

    # move 'enforce_https' from options to context so it persists
    # across retries but isn't passed to a transport implementation
    option = request.context.options.pop("enforce_https", None)

    # True is the default setting; we needn't preserve an explicit opt in to the default behavior
    if option is False:
        request.context["enforce_https"] = option

    enforce_https = request.context.get("enforce_https", True)
    if enforce_https and not request.http_request.url.lower().startswith("https"):
        raise ServiceRequestError(
            "Bearer token authentication is not permitted for non-TLS protected (non-https) URLs."
        )

def _host_only(url: str) -> str:
    return urlparse(url).netloc

def _strip_alg(digest):
    if len(digest.split(":")) == 2:
        return digest.split(":")[1]
    return digest

def _parse_exp_time(raw_token: str) -> float:
    raw_token_list = raw_token.split(".")
    if len(raw_token_list) > 2:
        value = raw_token_list[1]
        padding = len(value) % 4
        if padding > 0:
            value += "=" * padding
        byte_value = base64.b64decode(value).decode("utf-8")
        web_token = json.loads(byte_value)
        return web_token.get("exp", time.time())

    return time.time()

def _compute_digest(data: Union[IO[bytes], bytes]) -> str:
    if isinstance(data, bytes):
        value = data
    else:
        data.seek(0)
        value = data.read()
        data.seek(0)
    return "sha256:" + hashlib.sha256(value).hexdigest()

def _validate_digest(data: IO[bytes], expected_digest: str) -> bool:
    return _compute_digest(data) == expected_digest

def _get_blob_size(headers: Dict[str, str]) -> int:
    if not headers["Content-Range"]:
        raise ValueError("Missing content-range header in response.")
    blob_size = int(headers["Content-Range"].split("/")[1])
    if blob_size <= 0:
        raise ValueError(f"Invalid content-range header in response: {blob_size}")
    return blob_size

def _get_manifest_size(headers: Dict[str, str]) -> int:
    if not headers["Content-Length"]:
        raise ValueError("Missing content-length header in response.")
    manifest_size = int(headers["Content-Length"])
    if manifest_size <= 0:
        raise ValueError(f"Invalid content-length header in response: {manifest_size}")
    return manifest_size
