/*
 * brass - Braille and speech server
 *
 * Copyright (C) 2001-2003 by Roger Butenuth, All rights reserved.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * $Id: english.c,v 1.6 2003/06/04 20:11:28 butenuth Exp $
 */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include "language.h"
#include "english.h"
#include "lookup.h"

static int l_close (lang_t * l);

static int l_change_synth (struct lang_struct *l, synth_t * new_synth);

static int l_speak_string (struct lang_struct *l,
                           const unsigned char *buffer);
static int l_get_param (struct lang_struct *l, lang_par_t par, int *value);

static int l_set_param (struct lang_struct *l, lang_par_t par, int value);

typedef struct lang_state
{
  int param[L_MAX];
  int initialized;
} lang_state_t;

static lang_state_t l_state = {
  {1},
  0
};

static lang_t english = {
  &l_state,
  &languages[LANG_ENGLISH],
  NULL,
  NULL,
  0,
  0,
  l_close,
  l_change_synth,
  l_speak_string,
  l_get_param,
  l_set_param
};

/*
 * Open module for English. The synthesizer is search with the help
 * of context and lookup function.
 */
lang_t *english_open (void *context, lookup_string_t fun)
{
  synth_t *s;

  assert (!l_state.initialized);

  s = open_synthesizer (context, fun);
  if (s == NULL)
    return NULL;

  english.synth = s;
  init_synth_buffer (&english);
  l_state.initialized = 1;

  return &english;
}

/*
 * Close synthesizer and language module.
 */
static int l_close (lang_t * l)
{
  assert (l->state->initialized);

  (*english.synth->close) (english.synth);
  english.synth = NULL;
  deinit_synth_buffer (&english);
  l->state->initialized = 0;

  return 0;
}

/*
 * Change the synthesizer, keep language module.
 */
static int l_change_synth (struct lang_struct *l, synth_t * new_synth)
{
  assert (l->state->initialized);

  english.synth = new_synth;

  return 0;
}

/*
 * Handle a "normal" character: Lookup its representation in the table
 * and send it to synthesizer.
 */
static void handle_other (struct lang_struct *l,
                          const unsigned char *buffer, int *i)
{
  char *str;

  str = english_table[buffer[*i]];
  if (str != NULL)
    add_to_synth_buffer (&english, str);
  (*i)++;
}

/*
 * Handle one of the punctuation characters. 
 */
static void handle_punctuation (struct lang_struct *l,
                                const unsigned char *buffer,
                                int *pi, int last)
{
  char my_str[2];

  /* 
   * Send the unmodified character to the synth, when the last character
   * was a letter. Reason: In this case the dot or question mark can
   * help to find the correct sentence melody. Avoid it after other
   * letters because some synthesizers (e.g. IBM Viavoice) speak them,
   * which can lead to double spoken punctuation characters.
   */
  if (isalpha (last))
   {
/*        printf("isalpha(%c), sending %c\n", last, buffer[*pi]);*/
     my_str[0] = buffer[*pi];
     my_str[1] = 0;
     add_to_synth_buffer (&english, my_str);
   }
/*  else printf("!isalpha(%c), not sending %c\n", last, buffer[*pi]);*/

  /* 
   * When punctuation is on, send the long form, otherwise skip.
   */
  if (l->state->param[L_SPEAK_PUNCTUATION])
    handle_other (l, buffer, pi);
  else
    (*pi)++;
}

/*
 * Speak a zero terminated string. Flush synthesizer.
 */
static int l_speak_string (struct lang_struct *l, const unsigned char *buffer)
{
  int i;

  assert (l->state->initialized);

  i = 0;
  while (buffer[i])
   {
     if (strchr (ENG_PUNCTUATION, buffer[i]))
      {
        handle_punctuation (l, buffer, &i, i > 0 ? buffer[i - 1] : 0);
      }
     else
      {
        handle_other (l, buffer, &i);
      }
   }

  return flush_synth_buffer (&english);
}

/*
 * Read on of the synthesizer parameters.
 */
static int l_get_param (struct lang_struct *l, lang_par_t par, int *value)
{
  if (par >= 0 && par < L_MAX)
   {
     *value = l->state->param[par];
     return 0;
   }
  else
    return 1;
}

/*
 * Set one of the synthesizer parameters.
 */
static int l_set_param (struct lang_struct *l, lang_par_t par, int value)
{
  if (par >= 0 && par < L_MAX)
   {
     l->state->param[par] = value;
     return 0;
   }
  else
    return 1;
}
