
/*
 * BRLTTY - Access software for Unix for a blind person
 *          using a soft Braille terminal
 *
 * Copyright (C) 1995-2000 by The BRLTTY Team, All rights reserved.
 *
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 * Programmers at Brailcom o.p.s <technik@brailcom.cz>
 *
 * BRLTTY comes with ABSOLUTELY NO WARRANTY.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 */

/* MiniBraille/brl.c - Braille display library
 * the following Tieman B.V. braille terminals are supported
 *
 * - MiniBraille v 1.5 (20 braille cells + 2 status)
 *   (probably other versions too)
 *
 * Brailcom o.p.s. <technik@brailcom.cz>
 *
 * Thanks to Tieman B.V., which gives me protocol information. Author.
 */

#define __EXTENSIONS__          /* for termios.h */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
// #include <sys/termios.h>
#include "../sbllog.h"
#include <string.h>
#include <termios.h>

#include "../brl.h"
#include "../misc.h"
#include "brlconf.h"
#include "../brl_driver.h"
/* #include "../message.h" */
#include "minibraille.h"

/* types */
enum mode_t
{ NORMAL_MODE, F1_MODE, F2_MODE, MANAGE_MODE, CLOCK_MODE };

/* global variables */
static int brl_fd = -1;

static struct termios oldtermios;

static unsigned char xtbl[256];

unsigned char xlated[20];       // translated content of display

unsigned char status[2];        // xlated content of status area

enum mode_t mode = NORMAL_MODE;

/* #defines */
// delay this time after command - better: use CTS, DTR, CTR or find other signal which means "ready for next command"
#define AFTER_CMD_DELAY_TIME 10000
#define AFTER_CMD_DELAY do { usleep(AFTER_CMD_DELAY_TIME); } while (0)

static void beep (void)
{
  write (brl_fd, "\eB\r", 3);
};

/* interface */
struct brlinfo identbrl (const char *name, const char *dev)
{
  struct brlinfo retinfo;

  retinfo.cols = 0;
  if (!strcmp (minidevs[0].name, name))
    retinfo = minidevs[0];
  return retinfo;
}

static void initbrl (brldim * brl, const char *brldev)
{
  __label__ __errexit;

  struct termios newtermios;
  unsigned char standard[8] = { 0, 1, 2, 3, 4, 5, 6, 7 };
  unsigned char Tieman[8] = { 0, 7, 1, 6, 2, 5, 3, 4 };
  int n, i;                     /* counters */

  sbl_log ("init minibrl\n");
  // init geometry
  brl->x = 20;
  brl->y = 1;
  brl->disp = malloc (brl->x * brl->y + 1);
  if (brl->disp == NULL)
    goto __errexit;

  // open device
  if (brldev == NULL)
    sbl_log ("error: brldev=NULL\n");
  brl_fd = open (brldev, O_RDWR | O_SYNC | O_NOCTTY);
  if (brl_fd == -1)
    sbl_log ("error: cannot open brldev\n");
  if (!isatty (brl_fd))
    sbl_log ("error: isatty\n");
  tcgetattr (brl_fd, &oldtermios);      // save old terminal setting
/*	tcgetattr(brl_fd, &newtermios); */
  /* cfmakeraw(&newtermios); */
  newtermios.c_cflag = B38400 | CRTSCTS | CS8 | CLOCAL | CREAD;
  newtermios.c_iflag = IGNPAR;
  newtermios.c_oflag = 0;       /* raw output */
  newtermios.c_lflag = 0;       /* don't echo or generate signals */
  newtermios.c_cc[VMIN] = 0;    // non-blocking read
  newtermios.c_cc[VTIME] = 0;
/*	cfsetspeed(&newtermios, B9600); */
  tcflush (brl_fd, TCIFLUSH);
  tcsetattr (brl_fd, TCSANOW, &newtermios);
  /* hm, how to switch to 38400 ? write(brl_fd, "\eV\r", 3); tcflush(brl_fd,
     TCIFLUSH); cfsetspeed(&newtermios, B38400); tcsetattr(brl_fd,TCSANOW,
     &newtermios); */

  memset (xtbl, 0, 256);        // fill xlatetable
  for (n = 0; n < 256; n++)
    for (i = 0; i < 8; i++)
      if (n & 1 << standard[i])
        xtbl[n] |= 1 << Tieman[i];
  sleep (2);
  beep ();
  sbl_log ("init ready\n");
  return;

__errexit:
  LogAndStderr (LOG_ERR, "Cannot initialize MiniBraille");
  if (brl->disp)
    free (brl->disp);
  return;
}

static void closebrl (brldim * brl)
{
  free (brl->disp);             // handles NULL
  tcsetattr (brl_fd, TCSANOW, &oldtermios);
  close (brl_fd);
}

static void refresh (void)
{
  unsigned char datab[] = { 27, 'Z', '1' };
  unsigned char datae[] = { 13 };
  write (brl_fd, datab, sizeof datab);
  write (brl_fd, status, sizeof status);
  write (brl_fd, xlated, sizeof xlated);
  write (brl_fd, datae, sizeof datae);
};

static void writebrl (brldim * brl)
{
  int i;

  for (i = 0; i < 20; i++)
    xlated[i] = xtbl[brl->disp[i]];
  refresh ();
}

static int readbrl (int *type)
{
  unsigned char znak;

  int rv;

  rv = read (brl_fd, &znak, 1);

  sbl_log ("read brlkey\n");
  if (rv < 1)
   {
     sbl_log ("no key\n");
     *type = 0;
     return EOF;
   }

  sbl_log ("key=%i ", znak);
  return EOF;

}                               /* readbrl */

static void setbrlstat (const unsigned char *s)
{
  sbl_log ("update status\n");
  return;
  status[0] = xtbl[s[0]];
  status[1] = xtbl[s[1]];
  // refresh();
}
