/**********************************************************************
 * $pow example -- C source code using TF/ACC PLI routines
 *
 * C source to calculate the result of a number to the power of an
 * exponent.  The value is returned to simulation as a system function.
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *
 * Usage: result = $pow(<base>,<exponent>);
 *
 * Routine definitions for a veriusertfs array:
 *  /* routine prototypes -/
 *   extern int PLIbook_pow_sizetf(), PLIbook_pow_checktf(),
 *              PLIbook_pow_calltf(), PLIbook_pow_misctf();
 *  /* table entries -/
 *   {userfunction,                 /* type of PLI routine -/
 *     0,                           /* user_data value -/
 *     PLIbook_pow_checktf,         /* checktf routine -/
 *     PLIbook_pow_sizetf,          /* sizetf routine -/
 *     PLIbook_pow_calltf,          /* calltf routine -/
 *     PLIbook_pow_misctf,          /* misctf routine -/
 *     "$pow",                      /* system task/function name -/
 *     1                            /* forward reference = true -/
 *   },
 *********************************************************************/

#include "veriuser.h"         /* IEEE 1364 PLI TF  routine library */
#include "acc_user.h"         /* IEEE 1364 PLI ACC routine library */

/**********************************************************************
 * Sizetf application
 *********************************************************************/
int PLIbook_pow_sizetf()
{
  return(32);   /* $pow returns 32-bit values */
}

/**********************************************************************
 * checktf routine
 *********************************************************************/
int PLIbook_pow_checktf()
{
  static int valid_types[4] = {accReg, accIntegerVar, accConstant, 0};
  handle arg_handle;
  
  if (tf_nump() != 2)
    tf_error("$pow must have 2 arguments.\n");
  else if (tf_typep(1) == tf_nullparam)
    tf_error("$pow arg 1 cannot be null.\n");
  else if (tf_typep(2) == tf_nullparam)
    tf_error("$pow arg 2 cannot be null.\n");
  else {
    arg_handle = acc_handle_tfarg(1);
    if (!(acc_object_in_typelist(arg_handle, valid_types)) )
      tf_error("$pow arg1 must be number, variable or net.\n");
    arg_handle = acc_handle_tfarg(2);
    if (!(acc_object_in_typelist(arg_handle, valid_types)) )
      tf_error("$pow arg2 must be number, variable or net.\n");
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
#include <math.h>
int PLIbook_pow_calltf()
{
  int base, exp, result;

  base   = tf_getp(1);       /* read base value from tfarg 1 */
  exp    = tf_getp(2);       /* read exponent value from tfarg 2 */
  result = (int)pow( (double)base, (double)exp );
  tf_putp(0,result);         /* return result */
  return(0);
}

/**********************************************************************
 * misctf routine
 *********************************************************************/
int PLIbook_pow_misctf(int user_data, int reason)
{
  if (reason == reason_endofcompile)
    io_printf("\n$pow PLI application is being used.\n\n");
  return(0);
}
/*********************************************************************/

