/*
 *	Routines for VLAN interface handling
 *
 *	Copyright (C) 2009, 2010  Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Olaf Kirch <okir@suse.de>
 *	         Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <sutils.h>
#include <logging.h>
#include <handle.h>
#include <vlan.h>


nc_vlan_t *
nc_vlan_new(const char *ifname, unsigned int tag)
{
	nc_vlan_t *vlan;

	vlan = calloc(1, sizeof(*vlan));
	if(vlan) {
		vlan->tag = tag;
		if(nc_string_dup(&vlan->interface_name, ifname) == 0)
			return vlan;
	}
	return NULL;
}

int
nc_vlan_bind(nc_interface_t *ifp, nc_handle_t *nh, nc_var_array_t *errs)
{
	nc_stringbuf_t  err = NC_STRINGBUF_INIT;
	nc_interface_t *iface;

	if(!ifp || !ifp->name || !nh || !errs)
		return -1;
	if(!ifp->vlan)
		return -1;
	if(ifp->type != NC_IFTYPE_VLAN &&
	   ifp->type != NC_IFTYPE_INFINIBAND)
		return -1;

	if(!nc_string_len(ifp->vlan->interface_name)) {
		ifp->invalid = 1;
		nc_var_array_set(errs, ifp->name,
			"base interface not specified");
		return 1;
	}

	/* just to verify report when it does not exists ... */
	iface = nc_interface_by_name(nh, ifp->vlan->interface_name);
	if(iface == NULL) {
		nc_trace("no base interface %s for vlan %s -- missed config?",
			ifp->vlan->interface_name, ifp->name);
	} else if(iface->invalid) {
		ifp->invalid = 1;
		nc_stringbuf_printf(&err,
			"Base interface %s is invalid - marking %s invalid",
			iface->name, (ifp->type == NC_IFTYPE_VLAN ? "vlan" : "ib child"));
		nc_var_array_set(errs, ifp->name, err.string);
		nc_stringbuf_destroy(&err);
		return 1;
	}

	/* We do not mark the child as being owned by the VLAN.
	 * In fact, there can be many VLANs per eth device, and
	 * the eth device can be configured in its own right */
	return 0;
}

nc_vlan_t *
nc_vlan_clone(nc_vlan_t *vlan)
{
	assert(vlan);
	return nc_vlan_new(vlan->interface_name, vlan->tag);
}

void
nc_vlan_free(nc_vlan_t *vlan)
{
	if(vlan) {
		nc_string_free(&vlan->interface_name);
		free(vlan);
	}
}

int
nc_vlan_get_settings(const char *vlan, char **real, unsigned int *tag)
{
	char buf[256] = {'\0'}, *p, *tok;
	size_t header = 2;
	unsigned int _tag;
	FILE *f;

	if(!vlan || !real || !tag)
		return -1;

	*tag = 0;
	nc_string_free(real);

	/*
	** private -- only admin is allowed to get
	**            the real device or vlan id.
	*/
	f = fopen("/proc/net/vlan/config", "re");
	if(!f)
		return 1;

	while(fgets(buf, sizeof(buf)-1, f) != NULL) {
		size_t len = strlen(buf);
		if(!(len > 0 && buf[len - 1] == '\n'))
			goto failure;
		buf[len - 1] = '\0';
		if(header) {
			header--;
			continue;
		}

		p = NULL;
		tok = strtok_r(buf, " \t|", &p);
		if(!tok || !*tok)
			goto failure;

		if(!nc_string_eq(tok, vlan))
			continue;

		tok = strtok_r(NULL, " \t|", &p);
		if(!tok || !*tok || nc_parse_uint(tok, &_tag, 10) < 0)
			goto failure;

		tok = strtok_r(NULL, " \t|", &p);
		if(!tok || !*tok || nc_string_dup(real, tok) < 0)
			goto failure;

		*tag = _tag;
		break;
	}
	fclose(f);

	return 0;
failure:
	if(f)
		fclose(f);
	return -1;
}

int
nc_vlan_get_vlan_ifaces(const char *real, nc_string_array_t *vlans)
{
	char buf[256] = {'\0'}, *p, *tok, *vif;
	size_t header = 2;
	unsigned int tag;
	FILE *f;

	if(!real || !vlans)
		return -1;

	nc_string_array_destroy(vlans);

	/*
	** private -- only admin is allowed to get
	**            the real device or vlan id.
	*/
	f = fopen("/proc/net/vlan/config", "re");
	if(!f)
		return 1;

	tag = 0;
	while(fgets(buf, sizeof(buf)-1, f) != NULL) {
		size_t len = strlen(buf);
		if(!(len > 0 && buf[len - 1] == '\n'))
			goto failure;
		buf[len - 1] = '\0';
		if(header) {
			header--;
			continue;
		}

		p = NULL;
		tok = strtok_r(buf, " \t|", &p);
		if(!tok || !*tok)
			goto failure;

		vif = tok;

		tok = strtok_r(NULL, " \t|", &p);
		if(!tok || !*tok || nc_parse_uint(tok, &tag, 10) < 0)
			goto failure;

		tok = strtok_r(NULL, " \t|", &p);
		if(!tok || !*tok)
			goto failure;

		if(nc_string_eq(real, tok)) {
			if(nc_string_array_append(vlans, vif) < 0)
				goto failure;
		}
		break;
	}
	fclose(f);

	return 0;
failure:
	if(f)
		fclose(f);
	return -1;
}

