/*
 * Copyright 2010-2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package com.amazonaws.services.dynamodbv2.datamodeling;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import org.junit.Test;

import com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBAttribute;
import com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBAutoGeneratedKey;
import com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBHashKey;
import com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBIgnore;
import com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBRangeKey;
import com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBTable;
import com.amazonaws.services.dynamodbv2.datamodeling.DynamoDBVersionAttribute;

/**
 * Unit tests for {@code DynamoDBMappingsRegistry}.
 */
public class DynamoDBMappingsRegistryTest {

    @Test
    public void testFindRelevantGettersWithBridgeMethod() {
        final DynamoDBMappingsRegistry.Mappings mappings = DynamoDBMappingsRegistry.instance().mappingsOf(SubClass.class);
        assertEquals("only one getter should be returned", 1, mappings.getMappings().size());

        // To make sure the bridge method is ruled out.
        final DynamoDBMappingsRegistry.Mapping m = mappings.getMappings().iterator().next();
        assertEquals("return type should be Integer rather than Object", Integer.class, m.getter().getReturnType());
    }

    @DynamoDBTable(tableName = "")
    private static abstract class SuperGenericClass<T> {
        public abstract T getT();
        public abstract void setT(T t);
    }

    @DynamoDBTable(tableName = "GenericString")
    private static class SubClass extends SuperGenericClass<Integer> {
        private Integer t;
        @Override
        public Integer getT() { return t; }
        @Override
        public void setT(Integer t) { this.t = t; }
    }

    @Test
    public void testMappings() {
        final DynamoDBMappingsRegistry.Mappings mappings = DynamoDBMappingsRegistry.instance().mappingsOf(TestObject.class);

        assertEquals(TestObject.class, mappings.getObjectType());
        assertEquals(2, mappings.getPrimaryKeys().size());
        assertEquals(8, mappings.getMappings().size());
        assertNotNull(mappings.getTableName());
    }

    @Test
    public void testHashKey() {
        final DynamoDBMappingsRegistry.Mappings mappings = DynamoDBMappingsRegistry.instance().mappingsOf(TestObject.class);

        assertNotNull(mappings.getHashKey());
        assertTrue(mappings.getPrimaryKeys().contains(mappings.getHashKey()));

        for (final DynamoDBMappingsRegistry.Mapping m : mappings.getMappings()) {
            assertEquals((mappings.getHashKey() == m), m.isHashKey());
        }
    }

    @Test
    public void testRangeKey() {
        final DynamoDBMappingsRegistry.Mappings mappings = DynamoDBMappingsRegistry.instance().mappingsOf(TestObject.class);

        assertNotNull(mappings.getRangeKey());
        assertTrue(mappings.hasRangeKey());
        assertTrue(mappings.getPrimaryKeys().contains(mappings.getHashKey()));

        for (final DynamoDBMappingsRegistry.Mapping m : mappings.getMappings()) {
            assertEquals((mappings.getRangeKey() == m), m.isRangeKey());
        }
    }

    @Test
    public void testNonMappedInheritedProperties() {
        final DynamoDBMappingsRegistry.Mappings mappings = DynamoDBMappingsRegistry.instance().mappingsOf(NonMappedInheritedProperties.class);
        assertEquals(1, mappings.getMappings().size());
        assertEquals("doUse", mappings.getMappings().iterator().next().getAttributeName());
    }

    public abstract class AbstractNonMappedInheritedProperties {
        private String doNotUse;
        public String getDoNotUse() {
            return this.doNotUse;
        }
        public void setDoNotUse(final String doNotUse) {
            this.doNotUse = doNotUse;
        }
    }

    @DynamoDBTable(tableName="aws-java-sdk-test")
    public class NonMappedInheritedProperties {
        private String doUse;
        public String getDoUse() {
            return this.doUse;
        }
        public void setDoUse(final String doUse) {
            this.doUse = doUse;
        }
    }

    /**
     * Simple value object with hash and range key.
     */
    @DynamoDBTable(tableName = "aws-java-sdk-range-test")
    public static class TestObject {
        public String key;
        public String rangeKey;
        public Calendar effectiveFrom;
        public Calendar effectiveTo;
        public Set<String> values;
        public Date createdDate;
        public Date lastUpdatedDate;
        public Long versionNumber;

        @DynamoDBHashKey
        public String getKey() {
            return key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        @DynamoDBRangeKey
        @DynamoDBAutoGeneratedKey
        public String getRangeKey() {
            return rangeKey;
        }

        public void setRangeKey(String rangeKey) {
            this.rangeKey = rangeKey;
        }

        public Calendar getEffectiveFrom() {
            return effectiveFrom;
        }

        public void setEffectiveFrom(final Calendar effectiveFrom) {
            this.effectiveFrom = effectiveFrom;
        }

        public Calendar getEffectiveTo() {
            return effectiveTo;
        }

        public void setEffectiveTo(final Calendar effectiveTo) {
            this.effectiveTo = effectiveTo;
        }

        public Set<String> getValues() {
            return values;
        }

        public void setValues(final Set<String> values) {
            this.values = values;
        }

        @DynamoDBIgnore
        public boolean isValue(final String value) {
            return (this.values != null && this.values.contains(value));
        }

        @DynamoDBIgnore
        public void addValue(final String value) {
            if (this.values == null) {
                this.values = new HashSet<String>(2);
            }
            this.values.add(value);
        }

        @DynamoDBAttribute(attributeName = "RCD")
        public Date getCreatedDate() {
            return createdDate;
        }

        public void setCreatedDate(final Date createdDate) {
            this.createdDate = createdDate;
        }

        @DynamoDBAttribute(attributeName = "RUD")
        public Date getLastUpdatedDate() {
            return lastUpdatedDate;
        }

        public void setLastUpdatedDate(final Date lastUpdatedDate) {
            this.lastUpdatedDate = lastUpdatedDate;
        }

        @DynamoDBVersionAttribute(attributeName = "RVN")
        public Long getVersionNumber() {
            return versionNumber;
        }

        public void setVersionNumber(final Long versionNumber) {
            this.versionNumber = versionNumber;
        }
    }

}
