/*
 *	wicked firmware -- utils to firmware config discovery
 *
 *	Copyright (C) 2023 SUSE LLC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Marius Tomaschewski
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <wicked/util.h>
#include <wicked/logging.h>
#include <wicked/xml.h>

#include "appconfig.h"
#include "firmware.h"
#include "process.h"
#include "buffer.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/stat.h>


/*
 * wicked --root-directory <path> option argument in main
 */
extern char *					opt_global_rootdir;

#define NI_NETIF_FIRMWARE_DISCOVERY_CONF	"config"
#define NI_NETIF_FIRMWARE_DISCOVERY_NODE	"netif-firmware-discovery"
#define NI_NETIF_FIRMWARE_DISCOVERY_FILE	"client-firmware.xml"


static ni_bool_t
ni_netif_firmware_discovery_name_match(const ni_string_array_t *names, const char *name)
{
	/* all names are acceptable */
	if (!names || !names->count)
		return TRUE;

	/* whitelist: name in names */
	return ni_string_array_index(names, name) != -1;
}

static const char *
ni_netif_firmware_discovery_config_file(char **config_file)
{
	const char *config_dir = ni_get_global_config_dir();

	if (ni_string_empty(config_dir) || !config_file)
		return NULL;

	if (!ni_string_printf(config_file, "%s/%s", config_dir,
				NI_NETIF_FIRMWARE_DISCOVERY_FILE))
		return NULL;

	return *config_file;
}

static FILE *
ni_netif_firmware_discovery_config_open(const char *filename, char **tempname)
{
	FILE *fp;
	int fd, err;

	if (!ni_string_printf(tempname, "%s.XXXXXX", filename))
		return NULL;

	if ((fd = mkstemp(*tempname)) < 0) {
		err = errno;
		ni_string_free(tempname);
		errno = err;
		return NULL;
	}

	if (fchmod(fd, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0 ||
	    !(fp = fdopen(fd, "we"))) {
		err = errno;
		close(fd);
		unlink(*tempname);
		ni_string_free(tempname);
		errno = err;
		return NULL;
	}
	return fp;
}

static ni_bool_t
ni_netif_firmware_discovery_config_dump(FILE *out, const xml_node_t *config)
{
	if (!out || !config)
		return FALSE;

	if (fputs("<!-- config file generated by "
		  "`wicked firmware <enable|disable> <name>` -->\n", out) <= 0)
		return FALSE;

	return xml_node_print(config, out) == 0;
}

static int
ni_netif_firmware_discovery_config_write(xml_node_t *config)
{
	char *config_file = NULL;
	char *config_temp = NULL;
	FILE *temp;
	int err = NI_WICKED_RC_ERROR;

	if (!config)
		return err;

	if (!ni_netif_firmware_discovery_config_file(&config_file)) {
		ni_error("Unable to construct %s '%s' config file name",
				NI_NETIF_FIRMWARE_DISCOVERY_NODE,
				NI_NETIF_FIRMWARE_DISCOVERY_FILE);
		return err;
	}

	temp = ni_netif_firmware_discovery_config_open(config_file, &config_temp);
	if (!temp) {
		if (errno == EACCES || errno == EPERM)
			err = NI_LSB_RC_NOT_ALLOWED;
		ni_error("Unable to create '%s' config temp file: %m", config_file);
		ni_string_free(&config_file);
		return err;
	}

	if (!ni_netif_firmware_discovery_config_dump(temp, config)) {
		ni_error("Unable to write '%s' config temp file: %m", config_file);
		fclose(temp);
		unlink(config_temp);
		ni_string_free(&config_temp);
		ni_string_free(&config_file);
		return err;
	}

	fclose(temp);
	if (rename(config_temp, config_file) < 0) {
		if (errno == EACCES || errno == EPERM)
			err = NI_LSB_RC_NOT_ALLOWED;
		ni_error("Unable to create '%s' config file: %m", config_file);
		unlink(config_temp);
		ni_string_free(&config_temp);
		ni_string_free(&config_file);
		return err;
	} else {
		ni_debug_application("Updated '%s' config file", config_file);
		ni_string_free(&config_temp);
		ni_string_free(&config_file);
		return NI_WICKED_RC_SUCCESS;
	}
}

static int
ni_netif_firmware_discovery_config_migrate(xml_node_t **result, xml_node_t *root)
{
	const char *fdname = NI_NETIF_FIRMWARE_DISCOVERY_NODE;
	xml_node_t *fdnode = NULL;
	xml_node_t *script = NULL;
	xml_node_t *config = NULL;
	xml_node_t *node = NULL;
	const char *name, *attr;
	ni_bool_t enabled;

	if (!result || *result || !root)
		return NI_WICKED_RC_ERROR;

	if (!(config = xml_node_get_child(root, NI_NETIF_FIRMWARE_DISCOVERY_CONF)))
		return NI_WICKED_RC_ERROR;

	if (!(*result = xml_node_new(config->name, NULL)))
		return NI_WICKED_RC_ERROR;

	while ((fdnode = xml_node_get_next_child(config, fdname, fdnode))) {
		name = xml_node_get_attr(fdnode, "name");
		if (!ni_string_empty(name)) {
			/*
			 * Current firmware discovery extension _override_ definition
			 * in /etc/wicked/client-firmware.xml file contains only the
			 * firmware name and enabled attributes, e.g.:
			 *
			 *  <netif-firmware-discovery name="ibft" enabled="false" />
			 *
			 * We keep it as is and just move to the result node.
			 */
			xml_node_reparent(*result, fdnode);
			fdnode = NULL;
		} else while ((script = xml_node_get_next_child(fdnode, "script", script))) {
			/*
			 * Legacy firmware discovery extension _override_ definition
			 * is a single/merged netif-firmware-discovery node with the
			 * firmware type name in the script name, the enabled flag
			 * and (unfortunately) also the script command:
			 *
			 *  <netif-firmware-discovery>
			 *    <script name="ibft" command="/path/to/script" enabled="false"/>
			 *    […]
			 *  </netif-firmware-discovery>
			 *
			 * We rewrite it to current notation and remove the command as
			 * repeating it causes update issues: overrides command defined
			 * in client.xml, thus we can't change it in new client.xml aka
			 * move the extension scripts to another location.
			 */
			if (!(name = xml_node_get_attr(script, "name")))
				continue; /* invalid: ignore */

			node = xml_node_new(NI_NETIF_FIRMWARE_DISCOVERY_NODE, *result);
			if (node) {
				xml_node_add_attr(node, "name", name);
				attr = xml_node_get_attr(script, "enabled");
				if (ni_parse_boolean(attr, &enabled) == 0 && !enabled) {
					xml_node_add_attr(node, "enabled",
							ni_format_boolean(enabled));
				}
			} else {
				xml_node_free(*result);
				*result = NULL;
				return NI_WICKED_RC_ERROR;
			}
		}
	}

	return NI_WICKED_RC_SUCCESS;
}

static int
ni_netif_firmware_discovery_config_read(xml_node_t **config)
{
	int err = NI_WICKED_RC_ERROR;
	char *config_file = NULL;
	xml_document_t *doc;

	if (!config || *config)
		return err;

	if (!ni_netif_firmware_discovery_config_file(&config_file)) {
		ni_error("Unable to construct %s '%s' config file name",
				NI_NETIF_FIRMWARE_DISCOVERY_NODE,
				NI_NETIF_FIRMWARE_DISCOVERY_FILE);
		return err;
	}

	if (!ni_file_exists(config_file)) {
		/* To avoid xml_document_read error */
		ni_info("Can't read config '%s': %m", config_file);
		err = NI_WICKED_RC_NOT_CONFIGURED;
		ni_string_free(&config_file);
		return err;
	}
	if (!(doc = xml_document_read(config_file))) {
		/* xml_document_read logs errors */
		ni_string_free(&config_file);
		return err;
	}
	ni_string_free(&config_file);

	err = ni_netif_firmware_discovery_config_migrate(config, doc->root);
	xml_document_free(doc);
	return err;
}

static inline ni_bool_t
ni_netif_firmware_discovery_config_revert_xml(xml_node_t *config, xml_node_t **removed,
		const char *fwname)
{
	const char *fdname = NI_NETIF_FIRMWARE_DISCOVERY_NODE;
	xml_node_t *fdnode = NULL;
	ni_bool_t modified = FALSE;
	const char *attr;

	while ((fdnode = xml_node_get_next_child(config, fdname, fdnode))) {
		attr = xml_node_get_attr(fdnode, "name");
		if (!ni_string_eq(attr, fwname))
			continue;

		xml_node_reparent(*removed, fdnode);
		modified = TRUE;
		fdnode = NULL;
	}

	return modified;
}

static int
ni_netif_firmware_discovery_config_revert(xml_node_t *config, xml_node_t **removed,
		const ni_string_array_t *names)
{
	int status = NI_WICKED_RC_NOT_CONFIGURED;
	const char *name;
	unsigned int i;

	if (!config || !removed || !names)
		return NI_WICKED_RC_ERROR;

	if (!*removed && !(*removed = xml_node_new(config->name, NULL)))
		return NI_WICKED_RC_ERROR;

	for (i = 0; i < names->count; ++i) {
		name = names->data[i];

		if (ni_netif_firmware_discovery_config_revert_xml(config, removed, name))
			status = NI_WICKED_RC_SUCCESS;
	}

	return status;
}

static inline ni_bool_t
ni_netif_firmware_discovery_config_modify_xml(xml_node_t *config, xml_node_t *modified,
		const char *fwname, ni_bool_t enable)
{
	const char *fdname = NI_NETIF_FIRMWARE_DISCOVERY_NODE;
	xml_node_t *fdnode = NULL;
	ni_bool_t result = FALSE;
	const char *estr = ni_format_boolean(enable);
	const char *attr;

	/* modify enabled flag in existing override node … */
	while ((fdnode = xml_node_get_next_child(config, fdname, fdnode))) {
		attr = xml_node_get_attr(fdnode, "name");
		if (!ni_string_eq(attr, fwname))
			continue;
		if (enable)
			xml_node_del_attr(fdnode, "enabled");
		else
			xml_node_add_attr(fdnode, "enabled", estr);
		xml_node_clone(fdnode, modified);
		result = TRUE;
	}

	/* … or add override node with desired enabled flag */
	if (!result && (fdnode = xml_node_new(fdname, config))) {
		xml_node_add_attr(fdnode, "name", fwname);
		if (!enable)
			xml_node_add_attr(fdnode, "enabled", estr);
		xml_node_clone(fdnode, modified);
		result = TRUE;
	}

	return result;
}

static int
ni_netif_firmware_discovery_config_modify(xml_node_t *config, xml_node_t **modified,
		const ni_string_array_t *names, ni_bool_t enable)
{
	int status = NI_WICKED_RC_NOT_CONFIGURED;
	ni_extension_t *ex;

	if (!config || !names)
		return NI_WICKED_RC_ERROR;

	if (!*modified && !(*modified = xml_node_new(config->name, NULL)))
		return NI_WICKED_RC_ERROR;

	for (ex = ni_global.config->fw_extensions; ex; ex = ex->next) {
		if (ni_string_empty(ex->name))
			continue;

		if (!ni_netif_firmware_discovery_name_match(names, ex->name))
			continue;

		if (ex->enabled == enable)
			continue;

		if (ni_netif_firmware_discovery_config_modify_xml(config,
				*modified, ex->name, enable)) {
			ex->enabled = enable;
			status = NI_WICKED_RC_SUCCESS;
		}
	}

	return status;
}

static ni_bool_t
ni_netif_firmware_discovery_env_to_xml(xml_node_t *parent, const ni_var_array_t *vars)
{
	const ni_var_t *var;
	xml_node_t *node;
	unsigned int i;

	if (!parent || !vars)
		return FALSE;

	for (i = 0; i < vars->count; ++i) {
		var = &vars->data[i];

		if (ni_string_empty(var->name))
			continue;

		if (!(node = xml_node_new("putenv", parent)))
			return FALSE;

		xml_node_add_attr(node, "name", var->name);
		xml_node_add_attr(node, "value", var->value);
	}

	return TRUE;
}

static ni_bool_t
ni_netif_firmware_discovery_script_to_xml(xml_node_t *parent, const ni_script_action_t *script)
{
	ni_var_array_t vars = NI_VAR_ARRAY_INIT;
	xml_node_t *node;

	if (!parent || !script || !script->process)
		return FALSE;

	if (!(node = xml_node_new("script", parent)))
		return FALSE;

	xml_node_add_attr(node, "name", script->name);
	xml_node_add_attr(node, "command", script->process->command);
	if (!script->enabled)
		xml_node_add_attr(node, "enabled", ni_format_boolean(script->enabled));

	if (!ni_shellcmd_getenv_vars(script->process, &vars) ||
	    !ni_netif_firmware_discovery_env_to_xml(node, &vars)) {
		ni_var_array_destroy(&vars);
		return FALSE;
	}

	ni_var_array_destroy(&vars);
	return TRUE;
}

static ni_bool_t
ni_netif_firmware_discovery_to_xml(xml_node_t *parent,
		const ni_string_array_t *names, ni_bool_t expand)
{
	const ni_script_action_t *script;
	const ni_extension_t *ex;
	xml_node_t *node;

	if (!parent)
		return FALSE;

	for (ex = ni_global.config->fw_extensions; ex; ex = ex->next) {
		if (ni_string_empty(ex->name))
			continue;

		if (!ni_netif_firmware_discovery_name_match(names, ex->name))
			continue;

		node = xml_node_new(NI_NETIF_FIRMWARE_DISCOVERY_NODE, parent);
		if (!node)
			return FALSE;

		xml_node_add_attr(node, "name", ex->name);
		if (!ex->enabled)
			xml_node_add_attr(node, "enabled", ni_format_boolean(ex->enabled));

		if (!expand)
			continue;

		/* builtins are not supported in netif-firmware-discovery */

		for (script = ex->actions; script; script = script->next) {
			if (ni_string_empty(script->name) || !script->process)
				continue;
			if (ni_string_empty(script->process->command))
				continue;

			if (!ni_netif_firmware_discovery_script_to_xml(node, script))
				return FALSE;
		}

		if (!ni_netif_firmware_discovery_env_to_xml(node, &ex->environment))
			return FALSE;
	}

	return TRUE;
}

static int
ni_netif_firmware_discovery_show_xml_config(const ni_string_array_t *names, ni_bool_t expand)
{
	int status = NI_WICKED_RC_SUCCESS;
	xml_node_t *config;

	config = xml_node_new(NI_NETIF_FIRMWARE_DISCOVERY_CONF, NULL);
	if (!config || !ni_netif_firmware_discovery_to_xml(config, names, expand))
		status = NI_WICKED_RC_ERROR;
	else if (xml_node_print(config, stdout) != 0)
		status = NI_WICKED_RC_ERROR;

	xml_node_free(config);
	return status;
}

static int
ni_netif_firmware_discovery_show_txt_modified(const xml_node_t *config)
{
	const char *fdname = NI_NETIF_FIRMWARE_DISCOVERY_NODE;
	const xml_node_t *fdnode = NULL;

	while ((fdnode = xml_node_get_next_child(config, fdname, fdnode))) {
		ni_bool_t enabled = TRUE;
		const char *name, *attr;

		name = xml_node_get_attr(fdnode, "name");
		if (ni_string_empty(name))
			continue;

		if ((attr = xml_node_get_attr(fdnode, "enabled")))
			ni_parse_boolean(attr, &enabled);

		printf("%-15s %s\n", name, enabled ?
				"enabled" : "disabled");
	}

	return NI_WICKED_RC_SUCCESS;
}

static int
ni_netif_firmware_discovery_show_txt_config(const ni_string_array_t *names)
{
	const ni_extension_t *ex;

	for (ex = ni_global.config->fw_extensions; ex; ex = ex->next) {
		if (ni_string_empty(ex->name))
			continue;

		if (!ni_netif_firmware_discovery_name_match(names, ex->name))
			continue;

		printf("%-15s %s\n", ex->name, ex->enabled ?
				"enabled" : "disabled");
	}

	return NI_WICKED_RC_SUCCESS;
}

static int
ni_netif_firmware_discovery_show_xml_ifnames(const ni_netif_firmware_ifnames_t *list,
		const ni_string_array_t *names)
{
	const ni_netif_firmware_ifnames_t *item;
	int status = NI_WICKED_RC_SUCCESS;
	xml_node_t *root, *fwnode;
	const char *ifname;
	unsigned int i;

	if (!(root = xml_node_new("interfaces", NULL)))
		return NI_WICKED_RC_ERROR;

	for (item = list; item; item = item->next) {
		if (ni_string_empty(item->fwname) || !item->ifnames.count)
			continue;

		if (!ni_netif_firmware_discovery_name_match(names, item->fwname))
			continue;

		if (!(fwnode = xml_node_new("firmware", NULL))) {
			status = NI_WICKED_RC_ERROR;
			continue;
		}
		xml_node_add_attr(fwnode, "name", item->fwname);

		for (i = 0; i < item->ifnames.count; ++i) {
			ifname = item->ifnames.data[i];
			if (ni_string_empty(ifname))
				continue;

			xml_node_new_element("interface", fwnode, ifname);
		}

		if (xml_node_is_empty(fwnode)) {
			xml_node_free(fwnode);
			status = NI_WICKED_RC_ERROR;
		} else {
			xml_node_add_child(root, fwnode);
		}
	}

	if (status == NI_WICKED_RC_SUCCESS && xml_node_print(root, stdout))
		status = NI_WICKED_RC_ERROR;

	xml_node_free(root);
	return status;
}


static int
ni_netif_firmware_discovery_show_txt_ifnames(const ni_netif_firmware_ifnames_t *list,
		const ni_string_array_t *names)
{
	const ni_netif_firmware_ifnames_t *item;

	for (item = list; item; item = item->next) {
		ni_stringbuf_t ifnames = NI_STRINGBUF_INIT_DYNAMIC;

		if (ni_string_empty(item->fwname) || !item->ifnames.count)
			continue;

		if (!ni_netif_firmware_discovery_name_match(names, item->fwname))
			continue;

		ni_stringbuf_join(&ifnames, &item->ifnames, " ");
		if (ifnames.len)
			printf("%-15s %s\n", item->fwname, ifnames.string);
		ni_stringbuf_destroy(&ifnames);
	}

	return NI_WICKED_RC_SUCCESS;
}


/*
 * output format options
 */
enum {
	OPTION_FORMAT_NONE = 0U,
	OPTION_FORMAT_TXT,
	OPTION_FORMAT_XML,
};

static const ni_intmap_t	option_fmt_map[] = {
	{ "txt",		OPTION_FORMAT_TXT	},
	{ "xml",		OPTION_FORMAT_XML	},
	{ NULL }
};


/*
 * wicked firmware <actions>
 */
static int
ni_wicked_firmware_config_revert(const char *caller, int argc, char **argv)
{
	enum {
		OPTION_HELP	= 'h',
		OPTION_FORMAT	= 'F',
		OPTION_SHOW	= 'S',
	};
	static const struct option	options[] = {
		{ "help",		no_argument,		NULL,	OPTION_HELP	},
		{ "format",		required_argument,	NULL,	OPTION_FORMAT	},
		{ "show",		no_argument,		NULL,	OPTION_SHOW	},
		{ NULL }
	};
	int opt, status = NI_WICKED_RC_USAGE;
	char *argv0, *command = NULL;
	ni_string_array_t names = NI_STRING_ARRAY_INIT;
	unsigned int format = OPTION_FORMAT_NONE;
	ni_bool_t show_only = FALSE;
	xml_node_t *config = NULL;
	xml_node_t *removed = NULL;

	ni_string_printf(&command, "%s %s", caller, argv[0]);
	argv0 = argv[0];
	argv[0] = command;
	optind = 1;
	while ((opt = getopt_long(argc, argv, "+hF:S", options, NULL))  != -1) {
		switch (opt) {
		case OPTION_FORMAT:
			if (ni_parse_uint_mapped(optarg, option_fmt_map, &format))
				goto usage;
			break;

		case OPTION_SHOW:
			/* consistently use txt by default */
			format = format ?: OPTION_FORMAT_TXT;
			show_only = TRUE;
			break;

		case OPTION_HELP:
			status = NI_WICKED_RC_SUCCESS;
			/* fall through */
		default:
		usage:
			fprintf(stderr,
				"\nUsage:\n"
				"  %s [options] <firmware name…|all>\n"
				"\n"
				"Options:\n"
				"  --help, -h			show this help text and exit.\n"
				"  --show, -S			show modified config on stdout and exit\n"
				"  --format, -F <txt|xml>	show modified config using specified format\n"
				"\n", command);
			goto cleanup;
		}
	}

	if (optind >= argc) {
		fprintf(stderr, "%s: Missing firmware name argument\n", command);
		goto usage;
	}

	while (optind < argc) {
		const char *name = argv[optind++];

		if (ni_string_eq(name, "all")) {
			ni_string_array_destroy(&names);
			break;
		}

		if (ni_string_array_index(&names, name) == -1)
			ni_string_array_append(&names, name);
	}

	status = ni_netif_firmware_discovery_config_read(&config);
	if (status == NI_WICKED_RC_SUCCESS)
		status = ni_netif_firmware_discovery_config_revert(config, &removed, &names);

	if (status == NI_WICKED_RC_NOT_CONFIGURED) {
		ni_note("No configuration change needed");
		status = NI_WICKED_RC_SUCCESS;
		goto cleanup;
	} else if (status != NI_WICKED_RC_SUCCESS) {
		ni_error("Unable to modify %s config", NI_NETIF_FIRMWARE_DISCOVERY_NODE);
		goto cleanup;
	}

	switch (format) {
	case OPTION_FORMAT_TXT:
		ni_netif_firmware_discovery_show_txt_modified(removed);
		break;

	case OPTION_FORMAT_XML:
		xml_node_print(removed, stdout);
		break;

	default:
		break;
	}

	if (!show_only && status == NI_WICKED_RC_SUCCESS)
		status = ni_netif_firmware_discovery_config_write(config);

cleanup:
	argv[0] = argv0;
	xml_node_free(config);
	xml_node_free(removed);
	ni_string_free(&command);
	ni_string_array_destroy(&names);
	return status;
}

static int
ni_wicked_firmware_config_modify(const char *caller, int argc, char **argv)
{
	enum {
		OPTION_HELP	= 'h',
		OPTION_FORMAT	= 'F',
		OPTION_SHOW	= 'S',
		OPTION_STDOUT	= 1000, /* backward compat option to 0.6.72 */
	};
	static const struct option	options[] = {
		{ "help",		no_argument,		NULL,	OPTION_HELP	},
		{ "format",		required_argument,	NULL,	OPTION_FORMAT	},
		{ "show",		no_argument,		NULL,	OPTION_SHOW	},
		{ "stdout",		no_argument,		NULL,	OPTION_STDOUT	},
		{ NULL }
	};
	int opt, status = NI_WICKED_RC_USAGE;
	char *argv0, *command = NULL;
	ni_string_array_t names = NI_STRING_ARRAY_INIT;
	unsigned int format = OPTION_FORMAT_NONE;
	ni_bool_t show_only = FALSE;
	xml_node_t *config = NULL;
	xml_node_t *modified = NULL;
	ni_bool_t enable;

	if (ni_string_eq(argv[0], "enable"))
		enable = TRUE;
	else
	if (ni_string_eq(argv[0], "disable"))
		enable = FALSE;
	else {
		/* we don't implement other actions here */
		fprintf(stderr, "%s: unsupported action '%s'\n", caller, argv[0]);
		return NI_WICKED_RC_NOT_IMPLEMENTED;
	}

	ni_string_printf(&command, "%s %s", caller, argv[0]);
	argv0 = argv[0];
	argv[0] = command;
	optind = 1;
	while ((opt = getopt_long(argc, argv, "+hF:S", options, NULL))  != -1) {
		switch (opt) {
		case OPTION_FORMAT:
			if (ni_parse_uint_mapped(optarg, option_fmt_map, &format))
				goto usage;
			break;

		case OPTION_SHOW:
			/* consistently use txt by default */
			format = format ?: OPTION_FORMAT_TXT;
			show_only = TRUE;
			break;

		case OPTION_STDOUT:
			/* backward compat option to 0.6.72 */
			format = format ?: OPTION_FORMAT_XML;
			show_only = TRUE;
			break;

		case OPTION_HELP:
			status = NI_WICKED_RC_SUCCESS;
			/* fall through */
		default:
		usage:
			fprintf(stderr,
				"\nUsage:\n"
				"  %s [options] <firmware name…|all>\n"
				"\n"
				"Options:\n"
				"  --help, -h			show this help text and exit\n"
				"  --show, -S			show modified config on stdout and exit\n"
				"  --format, -F <txt|xml>	show modified config using specified format\n"
				"\n", command);
			goto cleanup;
		}
	}

	if (optind >= argc) {
		fprintf(stderr, "%s: Missing firmware name argument\n", command);
		goto usage;
	}

	while (optind < argc) {
		const char *name = argv[optind++];

		if (ni_string_eq(name, "all")) {
			ni_string_array_destroy(&names);
			break;
		}

		if (ni_string_array_index(&names, name) < 0)
			ni_string_array_append(&names, name);
	}

	if (ni_netif_firmware_discovery_config_read(&config) != NI_WICKED_RC_SUCCESS)
		config = xml_node_new(NI_NETIF_FIRMWARE_DISCOVERY_CONF, NULL);

	status = ni_netif_firmware_discovery_config_modify(config, &modified, &names, enable);
	if (status == NI_WICKED_RC_NOT_CONFIGURED) {
		ni_note("No configuration change needed");
		status = NI_WICKED_RC_SUCCESS;
		goto cleanup;
	} else if (status != NI_WICKED_RC_SUCCESS) {
		ni_error("Unable to modify %s config", NI_NETIF_FIRMWARE_DISCOVERY_NODE);
		goto cleanup;
	}

	switch (format) {
	case OPTION_FORMAT_TXT:
		ni_netif_firmware_discovery_show_txt_modified(modified);
		break;

	case OPTION_FORMAT_XML:
		xml_node_print(modified, stdout);
		break;

	default:
		break;
	}

	if (!show_only && status == NI_WICKED_RC_SUCCESS)
		status = ni_netif_firmware_discovery_config_write(config);

cleanup:
	argv[0] = argv0;
	xml_node_free(config);
	xml_node_free(modified);
	ni_string_free(&command);
	ni_string_array_destroy(&names);
	return status;
}

static int
ni_wicked_firmware_extensions(const char *caller, int argc, char **argv)
{
	enum {
		OPTION_HELP	= 'h',
		OPTION_FORMAT	= 'F',
		OPTION_EXPAND	= 'E',
	};
	static const struct option	options[] = {
		{ "help",		no_argument,		NULL,	OPTION_HELP	},
		{ "format",		required_argument,	NULL,	OPTION_FORMAT	},
		{ "expand",		no_argument,		NULL,	OPTION_EXPAND	},
		{ NULL }
	};
	int opt, status = NI_WICKED_RC_USAGE;
	char *argv0, *command = NULL;
	ni_string_array_t names = NI_STRING_ARRAY_INIT;
	unsigned int format = OPTION_FORMAT_TXT;
	ni_bool_t expand = FALSE;

	ni_string_printf(&command, "%s %s", caller, argv[0]);
	argv0 = argv[0];
	argv[0] = command;
	optind = 1;
	while ((opt = getopt_long(argc, argv, "+hF:E", options, NULL))  != -1) {
		switch (opt) {
		case OPTION_FORMAT:
			if (ni_parse_uint_mapped(optarg, option_fmt_map, &format))
				goto usage;
			break;

		case OPTION_EXPAND:
			expand = TRUE;
			break;

		case OPTION_HELP:
			status = NI_WICKED_RC_SUCCESS;
			/* fall through */
		default:
		usage:
			fprintf(stderr,
				"\nUsage:\n"
				"  %s [options] [firmware name…|all]\n"
				"\n"
				"Options:\n"
				"  --help, -h			show this help text and exit\n"
				"  --format, -F <txt|xml>	show firmware extensions as xml (default: txt)\n"
				"  --expand, -E			expand xml to complete extension definition\n"
				"\n", command);
			goto cleanup;
		}
	}

	while (optind < argc) {
		const char *name = argv[optind++];

		if (ni_string_eq(name, "all")) {
			ni_string_array_destroy(&names);
			break;
		}

		if (ni_string_array_index(&names, name) == -1)
			ni_string_array_append(&names, name);
	}

	if (!ni_global.config) {
		fprintf(stderr, "%s: application config is not initialized\n", command);
		status = NI_WICKED_RC_ERROR;
		goto cleanup;
	}

	switch (format) {
	case OPTION_FORMAT_TXT:
		status = ni_netif_firmware_discovery_show_txt_config(&names);
		break;

	case OPTION_FORMAT_XML:
		status = ni_netif_firmware_discovery_show_xml_config(&names, expand);
		break;

	default:
		goto usage;
		break;
	}

cleanup:
	argv[0] = argv0;
	ni_string_free(&command);
	ni_string_array_destroy(&names);
	return status;
}

static int
ni_wicked_firmware_interfaces(const char *caller, int argc, char **argv)
{
	enum {
		OPTION_HELP	= 'h',
		OPTION_FORMAT	= 'F',
	};
	static const struct option	options[] = {
		{ "help",		no_argument,		NULL,	OPTION_HELP	},
		{ "format",		required_argument,	NULL,	OPTION_FORMAT	},
		{ NULL }
	};
	int opt, status = NI_WICKED_RC_USAGE;
	char *argv0, *command = NULL;
	ni_string_array_t names = NI_STRING_ARRAY_INIT;
	ni_netif_firmware_ifnames_t *list = NULL;
	unsigned int format = OPTION_FORMAT_TXT;

	ni_string_printf(&command, "%s %s", caller, argv[0]);
	argv0 = argv[0];
	argv[0] = command;
	optind = 1;
	while ((opt = getopt_long(argc, argv, "+hF:", options, NULL))  != -1) {
		switch (opt) {
		case OPTION_FORMAT:
			if (ni_parse_uint_mapped(optarg, option_fmt_map, &format))
				goto usage;
			break;

		case OPTION_HELP:
			status = NI_WICKED_RC_SUCCESS;
			/* fall through */
		default:
		usage:
			fprintf(stderr,
				"\nUsage:\n"
				"  %s [options] [firmware name…|all]\n"
				"\n"
				"Options:\n"
				"  --help, -h			show this help text and exit\n"
				"  --format, -F <txt|xml>	show firmware interfaces as xml (default: txt)\n"
				"\n", command);
			goto cleanup;
		}
	}

	while (optind < argc) {
		const char *name = argv[optind++];

		if (ni_string_eq(name, "all")) {
			ni_string_array_destroy(&names);
			break;
		}

		if (ni_string_array_index(&names, name) == -1)
			ni_string_array_append(&names, name);
	}

	if (!ni_netif_firmware_discover_ifnames(&list, NULL, opt_global_rootdir, NULL)) {
		status = NI_WICKED_RC_ERROR;
	} else {
		switch (format) {
		case OPTION_FORMAT_TXT:
			status = ni_netif_firmware_discovery_show_txt_ifnames(list, &names);
			break;

		case OPTION_FORMAT_XML:
			status = ni_netif_firmware_discovery_show_xml_ifnames(list, &names);
			break;

		default:
			break;
		}
	}

cleanup:
	argv[0] = argv0;
	ni_string_free(&command);
	ni_string_array_destroy(&names);
	ni_netif_firmware_ifnames_list_destroy(&list);
	return status;
}

int
ni_wicked_firmware(const char *caller, int argc, char **argv)
{
	enum {
		OPTION_HELP	= 'h',
	};
	enum {
		ACTION_EXTENSIONS,
		ACTION_INTERFACES,
		ACTION_DISABLE,
		ACTION_ENABLE,
		ACTION_REVERT,
	};
	static const struct option	options[] = {
		{ "help",		no_argument,		NULL,	OPTION_HELP	},
		{ NULL }
	};
	static const ni_intmap_t	actions[] = {
		{ "interfaces",		ACTION_INTERFACES	},
		{ "extensions",		ACTION_EXTENSIONS	},
		{ "disable",		ACTION_DISABLE		},
		{ "enable",		ACTION_ENABLE		},
		{ "revert",		ACTION_REVERT		},
		{ NULL,			-1U			}
	};
	int opt, status = NI_WICKED_RC_USAGE;
	char *argv0, *command = NULL;
	unsigned int action = -1U;

	ni_string_printf(&command, "%s %s", caller, argv[0]);
	argv0 = argv[0];
	argv[0] = command;
	optind = 1;
	while ((opt = getopt_long(argc, argv, "+h", options, NULL))  != -1) {
		switch (opt) {
		case OPTION_HELP:
			status = NI_WICKED_RC_SUCCESS;
			/* fall through */
		default:
		usage:
			fprintf(stderr,
				"\nUsage:\n"
				"  %s [options] <action> …\n"
				"\n"
				"Options:\n"
				"  --help, -h			show this help text and exit\n"
				"\n"
				"Actions:\n"
				"  interfaces [name…]		list firmware and interface names it configures\n"
				"  extensions [name…]		list firmware config extensions and status\n"
				"\n"
				"  disable <name…>		disable specified client-firmware extension\n"
				"  enable  <name…>		enable specified client-firmware extension\n"
				"  revert  <name…>		revert client-firmware extension overrides\n"
				"\n", command);
			goto cleanup;
		}
	}

	if (optind >= argc || ni_string_empty(argv[optind]) ||
	    ni_parse_uint_mapped(argv[optind], actions, &action)) {
		fprintf(stderr, "%s: please specify an action\n", command);
		goto usage;
	}

	/* execute actions that do not need decoding */
	switch (action) {
		case ACTION_INTERFACES:
			status = ni_wicked_firmware_interfaces(command,
					argc - optind, argv + optind);
			goto cleanup;

		case ACTION_EXTENSIONS:
			status = ni_wicked_firmware_extensions(command,
					argc - optind, argv + optind);
			goto cleanup;

		case ACTION_DISABLE:
		case ACTION_ENABLE:
			status = ni_wicked_firmware_config_modify(command,
					argc - optind, argv + optind);
			goto cleanup;

		case ACTION_REVERT:
			status = ni_wicked_firmware_config_revert(command,
					argc - optind, argv + optind);
			goto cleanup;

		default:
			break;
	}

cleanup:
	argv[0] = argv0;
	ni_string_free(&command);
	return status;
}
