/*
 *	Internal test utility to call netcf functions
 *
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#ifdef WITH_MTRACE
#include <mcheck.h>
#endif

#include <netcontrol/netcf.h>

extern void             nc_logger_open_stderr(void);

static const char *	get_opt_name(int opt);
static const char *	get_flags_descr(unsigned int flags);

static struct netcf_if *do_get_netcf_if(struct netcf *ncf, const char *ifname);
static int		do_handle_netcf_request(struct netcf *netcf, int opt,
						unsigned int flags, int nnames,
						const char *ifname,
						const char *macstr,
						const char *xfile);


enum {
	OPT_IFACE_COUNT		= 1,
	OPT_IFACE_LIST,
	OPT_IFACE_BY_NAME,
	OPT_IFACE_BY_MAC,
	OPT_IFACE_DUMPXML,
	OPT_IFACE_DEFINE,
	OPT_IFACE_UNDEFINE,
	OPT_IFACE_IFUP,
	OPT_IFACE_IFDOWN,
	OPT_IFACE_IFSTATUS,
	OPT_FLAGS,
	OPT_COUNT,
	OPT_ROOT,
};

static struct option    options[] = {
	/* -- options  -- */
	{ "root",		required_argument,	NULL,	OPT_ROOT		},
	{ "flags",		required_argument,	NULL,	OPT_FLAGS		},
	{ "count",		required_argument,	NULL,	OPT_COUNT		},

	/* -- commands -- */
	{ "iface-count",	no_argument,		NULL,	OPT_IFACE_COUNT		},
	{ "iface-list",		no_argument,		NULL,	OPT_IFACE_LIST		},

	{ "iface-by-name",	required_argument,	NULL,	OPT_IFACE_BY_NAME	},
	{ "iface-by-mac",	required_argument,	NULL,	OPT_IFACE_BY_MAC	},
	{ "iface-dumpxml",	required_argument,	NULL,	OPT_IFACE_DUMPXML	},

	{ "iface-define",	required_argument,	NULL,	OPT_IFACE_DEFINE	},
	{ "iface-undefine",	required_argument,	NULL,	OPT_IFACE_UNDEFINE	},

	{ "iface-ifup",		required_argument,	NULL,	OPT_IFACE_IFUP		},
	{ "iface-ifdown",	required_argument,	NULL,	OPT_IFACE_IFDOWN	},
	{ "iface-ifstatus",	required_argument,	NULL,	OPT_IFACE_IFSTATUS	},
	{ NULL, 0, NULL, 0 }
};

static const char *
get_opt_name(int opt)
{
	struct option *o;
	for(o = options; o && o->name; o++) {
		if(opt == o->val)
			return o->name;
	}
	return NULL;
}

static const char *
get_flags_descr(unsigned int flags)
{
	switch(flags) {
	case 0:
		return "none";
	break;
	case 1:
		return "inactive";
	break;
	case 2:
		return "active";
	break;
	case 3:
		return "active+inactive";
	break;
	default:
		return "unknown";
	}
}

/*
 * here we go
 */
int main(int argc, char *argv[])
{
        struct netcf *netcf;
	int ret = 1;
	int c;
	int cmd = 0;
	unsigned int flags = 0;
	int count = 0;
	const char *ifname = NULL;
	const char *macstr = NULL;
	const char *xfile = NULL;
	const char *root = NULL;

#ifdef WITH_MTRACE
	mtrace();
#endif
	nc_logger_open_stderr();

	while ((c = getopt_long(argc, argv, "+", options, NULL)) != EOF) {
		switch(c) {
			default:
			usage:
			/* -- commands -- */
				fprintf(stderr,
					"Usage: %s <command [arg]> [opt ..]\n"
					"commands:\n"
					"--iface-count     [--flags <flags>]\n"
					"--iface-list      --count <num> [--flags <flags>]\n"
					"--iface-by-name   <ifname>\n"
					"--iface-by-mac    <mac>\n"
					"--iface-dumpxml   <ifname> [--flags <flags>]\n"
					"--iface-define    <xml-file> [--flags <flags>]\n"
					"--iface-undefine  <ifname>\n"
					"--iface-ifup      <ifname> [--flags <flags>]\n"
					"--iface-ifdown    <ifname> [--flags <flags>]\n"
					"--iface-ifstatus  <ifname>\n"
					"\n", argv[0]);
				return ret;
			break;

			case OPT_ROOT:
				if(optarg)
					root = optarg;
			break;
			case OPT_FLAGS:
				if(optarg)
					flags = atoi(optarg);
			break;
			case OPT_COUNT:
				if(optarg)
					count = atoi(optarg);
			break;

			/* -- commands -- */
			case OPT_IFACE_BY_MAC:
				macstr = optarg;
				cmd = c;
			break;
			case OPT_IFACE_DEFINE:
				xfile = optarg;
				cmd = c;
			break;

			case OPT_IFACE_UNDEFINE:
			case OPT_IFACE_IFUP:
			case OPT_IFACE_IFDOWN:
			case OPT_IFACE_IFSTATUS:
			case OPT_IFACE_BY_NAME:
			case OPT_IFACE_DUMPXML:
				ifname = optarg;
				cmd = c;
			break;
			case OPT_IFACE_COUNT:
			case OPT_IFACE_LIST:
				cmd = c;
			break;
		}
	}

	fprintf(stderr, "command '%s'> flags=%u, count=%d, ifname=%s, macstr=%s, xml-file=%s\n",
		get_opt_name(cmd), flags, count, ifname, macstr, xfile);

	switch(cmd) {
		case OPT_IFACE_COUNT:	/* [flags] */
			if(count != 0 || flags > 3 || ifname || macstr || xfile) {
				ret = 1; goto usage;
			}
		break;
		case OPT_IFACE_LIST:	/* [count] [flags] */
			if(count < 0  || flags > 3 || ifname || macstr || xfile) {
				ret = 1; goto usage;
			}
		break;
		case OPT_IFACE_BY_NAME:	/* <ifname> */
			if(count != 0 || flags != 0             || !ifname || macstr || xfile) {
				ret = 1; goto usage;
			}
		break;
		case OPT_IFACE_BY_MAC:	/* <macstr> */
			if(count != 0 || flags != 0             || ifname || !macstr || xfile) {
				ret = 1; goto usage;
			}
		break;
		case OPT_IFACE_DEFINE:	/* <xfile> [flags]*/
			if(count != 0 || flags > 3 || ifname || macstr || !xfile) {
				ret = 1; goto usage;
			}
		break;
		case OPT_IFACE_UNDEFINE:/* <ifinfo->ifname> */
			if(count != 0 || flags != 0             || !ifname || macstr || xfile) {
				ret = 1; goto usage;
			}
		break;
		case OPT_IFACE_DUMPXML:	/* <ifinfo->ifname> [flags] */
			if(count != 0 || flags > 3 || !ifname || macstr || xfile) {
				ret = 1; goto usage;
			}
		break;
		case OPT_IFACE_IFUP:	/* <ifinfo->ifname> [flags] */
		case OPT_IFACE_IFDOWN:	/* <ifinfo->ifname> [flags] */
			if(count != 0 || flags > 3 || !ifname || macstr || xfile) {
				ret = 1; goto usage;
			}
		break;
		case OPT_IFACE_IFSTATUS:/* <ifinfo (ifname)> */
			if(count != 0 || flags != 0             || !ifname || macstr || xfile) {
				ret = 1; goto usage;
			}
		break;
		default:
			ret = 1;
			goto usage;
		break;
	}

        if( ncf_init(&netcf, root) != 0) {
            fprintf(stderr, "can't initialize netcf\n");
            return 1;
        }

        ret = do_handle_netcf_request(netcf, cmd, flags, count, ifname, macstr, xfile);

        if(ncf_close(netcf) != 0) {
		fprintf(stderr, "can't close netcf\n");
		return 1;
	}

        return ret < 0 ? 1 : 0;
}

static int
do_handle_netcf_request(struct netcf *ncf, int opt, unsigned int flags, int nnames,
			const char *ifname, const char *macstr, const char *xfile)
{
        int  ret, num, i;
        char **names, *xmlstr;
	struct netcf_if *nif;

	ret = -1;
	switch(opt) {
        /* ================================================= */
	case OPT_IFACE_COUNT:	/* [flags] */
	{
            num = ncf_num_of_interfaces(ncf, flags);

            printf("%s ifaces: %d\n", get_flags_descr(flags), num);

            ret = num < 0 ? 1 : 0;
	}
	break;

        /* ================================================= */
	case OPT_IFACE_LIST:	/* <count> [flags] */
	{
	    if (nnames <= 0 && (nnames = ncf_num_of_interfaces(ncf, flags)) <= 0) {
		    fprintf(stderr, "can't get number of %s interfaces\n",
				    get_flags_descr(flags));
		    return -1;
	    }

	    names = calloc(nnames + 1, sizeof(char *));
            if(!names) {
                fprintf(stderr, "can't allocate names array\n");
                return -1;
            }

            num = ncf_list_interfaces(ncf, nnames, names, flags);

            printf("%s ifaces: %d\n", get_flags_descr(flags), num);
            for(i = 0; i < num && names[i]; i++) {
                printf("iface[%d]: '%s'\n", i, names[i]);
                free(names[i]);
            }
            free(names);

            ret = num < 0 ? 1 : 0;
	}
	break;

        /* ================================================= */
	case OPT_IFACE_BY_NAME:	/* <ifname> */
	{
		nif = do_get_netcf_if(ncf, ifname);
		if(!nif) {
			fprintf(stderr, "can't fetch interface by name %s\n", ifname);
			ret = -1;
		} else {
			printf("ifname %s => ifname: %s, macstr: %s\n", ifname,
				ncf_if_name(nif), ncf_if_mac_string(nif));
			fprintf(stderr, "free(nif[\"%s\"])\n", ncf_if_name(nif));
			ncf_if_free(nif);
		}
	}
	break;

        /* ================================================= */
	case OPT_IFACE_BY_MAC:	/* <macstr> */
	{
		const int max_nifs = 42;
		struct netcf_if **nifs;
		int n;

		nifs = calloc(max_nifs, sizeof(*nifs));
		if(!nifs)
			return -1;

		num = ncf_lookup_by_mac_string(ncf, macstr, max_nifs, nifs);
		if(num < 0) {
			ret = -1;
			fprintf(stderr, "failed to fetch interface by MAC %s\n", macstr);
		} else
		if (num == 0) {
			ret = 0;
			fprintf(stderr, "couldn't find interface by MAC %s\n", macstr);
		} else
		if (num > 1) {
			ret = -1;
			fprintf(stderr, "multiple interfaces match MAC %s:\n", macstr);
			for(n = 0; n < max_nifs; ++n) {
				if(nifs[n]) {
					fprintf(stderr, "\tmac %s => ifname: %s, macstr: %s\n",
						macstr,
						ncf_if_name(nifs[n]),
						ncf_if_mac_string(nifs[n]));
				}
			}
		} else {
			ret = 0;
			printf("mac %s => ifname: %s, macstr: %s\n", macstr,
				ncf_if_name(nifs[0]), ncf_if_mac_string(nifs[0]));
		}

		for(n = 0; n < max_nifs; ++n) {
			if(nifs[n]) {
				fprintf(stderr, "free(nif[\"%s\"])\n", ncf_if_name(nifs[n]));
				ncf_if_free(nifs[n]);
			}
		}
		free(nifs);
	}
	break;

        /* ================================================= */
	case OPT_IFACE_DUMPXML:	/* <ifinfo->ifname> [flags] */
	{
		nif = do_get_netcf_if(ncf, ifname);
		if(!nif) {
			fprintf(stderr, "can't fetch interface by name %s\n", ifname);
			ret = -1;
		} else {
			const char *type;
			if(flags & 1) { /* inactive */
				type   = "inactive";
				xmlstr = ncf_if_xml_desc(nif);
			} else
			if(flags & 2) { /* active */
				type   = "active";
				xmlstr = ncf_if_xml_state(nif);
			} else {
				type   = get_flags_descr(flags);
				xmlstr = NULL;
			}
			printf("xmldump for %s iface name=%s:\n%s\n",
				type, ifname, (xmlstr ? xmlstr : "NULL"));
			if(xmlstr)
				free(xmlstr);
			fprintf(stderr, "free(nif[\"%s\"])\n", ncf_if_name(nif));
			ncf_if_free(nif);
		}
	}
	break;

        /* ================================================= */
	case OPT_IFACE_DEFINE:	/* <xfile> ([flags])*/
	{
		FILE  *m, *f;
		size_t size = 0;
		char  *data = NULL;
		char   buff[BUFSIZ];

		f = fopen(xfile, "re");
		if(!f) {
			ret = -1;
			break;
		}
		m = open_memstream(&data, &size);
		if(!m) {
			fclose(f);
			ret = -1;
			break;
		}

		ret = 0;
		memset(buff, 0, sizeof(buff));
		while(fgets(buff, sizeof(buff), f)) {
			fputs(buff, stderr);
			if(fputs(buff, m) == EOF) {
				ret = -1;
				break;
			}
			fflush(m);
			memset(buff, 0, sizeof(buff));
		}
		fclose(f);
		fclose(m);
#if 0
		fprintf(stderr, "<DUMP siz=%zu, len=%zu>\n%s\n</DUMP>\n",
			size, strlen(data), data);
#endif
		if(ret != -1) {
			nif = ncf_define(ncf, data);
			if(nif) {
				printf("successfully created interface config for %s\n",
					ncf_if_name(nif));
				fprintf(stderr, "free(nif[\"%s\"])\n", ncf_if_name(nif));
				ncf_if_free(nif);
			} else {
				const char *msg;
				int         err;

				err = ncf_error(ncf, &msg, NULL);
				fprintf(stderr, "ncf_error[%d]: %s\n", err, msg);
				ret = -1;
			}
		}
		if(data) {
			free(data);
		}
	}
	break;

        /* ================================================= */
	case OPT_IFACE_UNDEFINE:/* <ifinfo->ifname> */
		nif = do_get_netcf_if(ncf, ifname);
		if(!nif) {
			fprintf(stderr, "can't fetch interface config by name %s\n", ifname);
			ret = -1;
		} else {
			ret = ncf_if_undefine(nif);
			if (ret < 0) {
				const char *errmsg = NULL, *details = NULL;
				int errcode = ncf_error(ncf, &errmsg, &details);
				fprintf(stderr, "ncf_if_undefine failed with error=%d: %s [%s]\n",
					errcode, errmsg, (details ? details : "no details"));
			} else {
				printf("configuration for interface %s removed\n", ifname);
			}
			fprintf(stderr, "free(nif[\"%s\"])\n", ncf_if_name(nif));
			ncf_if_free(nif);
		}
	break;

        /* ================================================= */
	case OPT_IFACE_IFUP:	/* <ifinfo->ifname> [flags] */
		nif = do_get_netcf_if(ncf, ifname);
		if(!nif) {
			fprintf(stderr, "can't fetch interface by name %s\n", ifname);
			ret = -1;
		} else {
			ret = ncf_if_up(nif);
			if (ret < 0) {
				const char *errmsg = NULL, *details = NULL;
				int errcode = ncf_error(ncf, &errmsg, &details);
				fprintf(stderr, "ncf_if_up failed with error=%d: %s [%s]\n",
					errcode, errmsg, (details ? details : "no details"));
			} else {
				printf("started interface %s\n", ifname);
			}
			fprintf(stderr, "free(nif[\"%s\"])\n", ncf_if_name(nif));
			ncf_if_free(nif);
		}
	break;

	case OPT_IFACE_IFDOWN:	/* <ifinfo->ifname> [flags] */
		nif = do_get_netcf_if(ncf, ifname);
		if(!nif) {
			fprintf(stderr, "can't fetch interface by name %s\n", ifname);
			ret = -1;
		} else {
			ret = ncf_if_down(nif);
			if (ret < 0) {
				const char *errmsg = NULL, *details = NULL;
				int errcode = ncf_error(ncf, &errmsg, &details);
				fprintf(stderr, "ncf_if_down failed with error=%d: %s [%s]\n",
					errcode, errmsg, (details ? details : "no details"));
			} else {
				printf("stopped interface %s\n", ifname);
			}
			fprintf(stderr, "free(nif[\"%s\"])\n", ncf_if_name(nif));
			ncf_if_free(nif);
		}
	break;
	case OPT_IFACE_IFSTATUS:/* <ifinfo (ifname)> */
		nif = do_get_netcf_if(ncf, ifname);
		if(!nif) {
			fprintf(stderr, "can't fetch interface by name %s\n", ifname);
			ret = -1;
		} else {
			ret = ncf_if_status(nif, &flags);
			if (ret < 0) {
				const char *errmsg = NULL, *details = NULL;
				int errcode = ncf_error(ncf, &errmsg, &details);
				fprintf(stderr, "ncf_if_status failed with error=%d: %s [%s]\n",
					errcode, errmsg, (details ? details : "no details"));
			} else {
				printf("status of interface %s is: %s [%d]\n",
					ifname, get_flags_descr(flags), flags);
			}
			fprintf(stderr, "free(nif[\"%s\"])\n", ncf_if_name(nif));
			ncf_if_free(nif);
		}
	break;

	default:
		fprintf(stderr, "unknown action");
		ret = -1;
	break;
	}
	return ret;
}

static struct netcf_if *
do_get_netcf_if(struct netcf *ncf, const char *ifname)
{
        struct netcf_if *nif = ncf_lookup_by_name(ncf, ifname);
        if(!nif) {
                const char *errmsg, *details;
                int errcode = ncf_error(ncf, &errmsg, &details);
                if(errcode != NETCF_NOERROR) {
                        fprintf(stderr,"couldn't find interface named '%s' (netcf: %s - %s)\n",
                                ifname, errmsg, details ? details : "");
                } else {
                        fprintf(stderr,"couldn't find interface named '%s'\n", ifname);
                }
        } else {
		fprintf(stderr, "lookup by name for %s => nif[\"%s\"]\n",
				ifname, ncf_if_name(nif));
	}
        return nif;
}

