/*
* Test case inputs for: __uint128_t __fixunstfti (long double)
* Conversion from long double (IBM double-double) to 128 bit integer.
*/

#define INFINITY __builtin_inf()
#define QNAN __builtin_nan("")
#define INIT_U128(HI, LO) (((__uint128_t) (HI) << 64) | (LO))

struct testCase {
  double hiInput;
  double loInput;
  __uint128_t result128;
};

// This test case tests long doubles generated using the following approach:
// For each possible exponent in double precision, we generate a random number
// in [1,2) interval as a mantissa for the high double. Given that the low
// double must be less than half an ULP of the high double, we choose the
// exponent of the low double to be less than the exponent of the high double
// minus 52. For the given exponent in the low double, we generate a random
// number in [1,2) interval as a mantissa. From this generated long double,
// we generate four long doubles by setting the sign of the high and low
// doubles to + or -.
struct testCase testList[] = {
  { 0x0p+0, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { -0x0p+0, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { -0x0p+0, -0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x0p+0, -0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { -0x1p+0, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x1p+0, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000001 ) },
  { -INFINITY, 0x0p+0, ((__uint128_t)0x0000000000000000 << 64) | 0x0000000000000000 },
  { INFINITY, 0x0p+0, ((__uint128_t)0xffffffffffffffff << 64) | 0xffffffffffffffff },
  { QNAN, 0x0p+0, ((__uint128_t)0x7ff8000000000000 << 64) | 0x0000000000000000 },
  { -QNAN, 0x0p+0, ((__uint128_t)0x7ff8000000000000 << 64) | 0x0000000000000000 },
  { -0x1p+127, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x1p+127, -0x1p+0, INIT_U128( 0x7fffffffffffffff, 0xffffffffffffffff ) },
  { 0x1p+0, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000001 ) },
  { 0x1p+60, 0x0p+0, INIT_U128( 0x0000000000000000, 0x1000000000000000 ) },
  { 0x1p+64, -0x1p+0, INIT_U128( 0x0000000000000000, 0xffffffffffffffff ) },
  { 0x1p+63, -0x1p+0, INIT_U128( 0x0000000000000000, 0x7fffffffffffffff ) },
  { 0x1p+64, 0x0p+0, INIT_U128( 0x0000000000000001, 0x0000000000000000 ) },
  { 0x1p+64, 0x1p+0, INIT_U128( 0x0000000000000001, 0x0000000000000001 ) },
  { 0x1.8p+64, -0x1p+0, INIT_U128( 0x0000000000000001, 0x7fffffffffffffff ) },
  { 0x1.1p+64, 0x0p+0, INIT_U128( 0x0000000000000001, 0x1000000000000000 ) },
  { 0x1p+65, -0x1p+0, INIT_U128( 0x0000000000000001, 0xffffffffffffffff ) },
  { 0x1p+127, -0x1p+64, INIT_U128( 0x7fffffffffffffff, 0x0000000000000000 ) },
  { 0x1p+127, -0x1.ep+64, INIT_U128( 0x7ffffffffffffffe, 0x2000000000000000 ) },
  { 0x1p+127, -0x1p+63, INIT_U128( 0x7fffffffffffffff, 0x8000000000000000 ) },
  { 0x1p+124, 0x0p+0, INIT_U128( 0x1000000000000000, 0x0000000000000000 ) },
  { 0x1p+124, 0x1p+0, INIT_U128( 0x1000000000000000, 0x0000000000000001 ) },
  { 0x1p+124, 0x1p+63, INIT_U128( 0x1000000000000000, 0x8000000000000000 ) },
  { 0x1p+124, 0x1p+64, INIT_U128( 0x1000000000000001, 0x0000000000000000 ) },
  { -0x1p+64, 0x0p+0, INIT_U128( 0x00000000000000000, 0x0000000000000000 ) },
  { 0x1.84p+70, 0x1.84p+6, INIT_U128( 0x0000000000000061, 0x0000000000000061 ) },
  { 0x1.5cp+6, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000057 ) },
  { 0x1p+64, -0x1.88p+6, INIT_U128( 0x0000000000000000, 0xffffffffffffff9e ) },
  { 0x1.88p+6, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000062 ) },
  { 0x1.00cp+10, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000403 ) },
  { 0x1.fffffffffffffp+63, 0x1.fep+9, INIT_U128( 0x0000000000000000, 0xfffffffffffffbfc ) },
  { 0x1.028p+10, 0x0p+0, INIT_U128( 0x0000000000000000, 0x000000000000040a ) },
  { 0x1.44p+10, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000510 ) },
  { 0x1.fffffffffffffp+63, 0x1.738p+9, INIT_U128( 0x0000000000000000, 0xfffffffffffffae7 ) },
  { 0x1.808p+10, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000602 ) },
  { 0x1.fffffffffffffp+63, 0x1.fdp+8, INIT_U128( 0x0000000000000000, 0xfffffffffffff9fd ) },
  { 0x1.048p+13, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000002090 ) },
  { 0x1.ffffffffffffbp+63, 0x1.ed8p+9, INIT_U128( 0x0000000000000000, 0xffffffffffffdbdb ) },
  { 0x1.0101p+17, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000020202 ) },
  { 0x1.fffffffffffbep+63, -0x1.09p+8, INIT_U128( 0x0000000000000000, 0xfffffffffffdeef7 ) },
  { 0x1.9002p+17, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000032004 ) },
  { 0x1.fffffffffff9cp+63, -0x1.4p+2, INIT_U128( 0x0000000000000000, 0xfffffffffffcdffb ) },
  { 0x1.902p+17, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000032040 ) },
  { 0x1.ffffffffff7fcp+63, -0x1.14p+6, INIT_U128( 0x0000000000000000, 0xffffffffffbfdfbb ) },
  { 0x1.00822p+22, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000402088 ) },
  { 0x1.0010011p+31, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000080080088 ) },
  { 0x1.0a000001p+35, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000850000008 ) },
  { 0x1.000000224p+37, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000002000000448 ) },
  { 0x1.ffffffbffefb8p+63, -0x1p+0, INIT_U128( 0x0000000000000000, 0xffffffdfff7dbfff ) },
  { 0x1.00080044p+102, 0x1.00080044p+38, INIT_U128( 0x0000004002001100, 0x0000004002001100 ) },
  { 0x1.00400000018p+107, 0x1.00400000018p+43, INIT_U128( 0x000008020000000c, 0x000008020000000c ) },
  { 0x1.ffffeffcp+63, -0x1.ap+3, INIT_U128( 0x0000000000000000, 0xfffff7fdfffffff3 ) },
  { 0x1.000000001048p+47, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000800000000824 ) },
  { 0x1.fffbffffffp+63, -0x1.808p+9, INIT_U128( 0x0000000000000000, 0xfffdffffff7ffcff ) },
  { 0x1.000810004p+62, 0x0p+0, INIT_U128( 0x0000000000000000, 0x4002040010000000 ) },
  { 0x1.7ffbf7ffep+63, -0x1p+0, INIT_U128( 0x0000000000000000, 0xbffdfbffefffffff ) },
  { 0x1p+63, 0x0p+0, INIT_U128( 0x0000000000000000, 0x8000000000000000 ) },
  { 0x1.ffffffffe8319p+63, -0x1.1f8p+9, INIT_U128( 0x0000000000000000, 0xfffffffff418c5c1 ) },
  { 0x1p+68, -0x1p+0, INIT_U128( 0x000000000000000f, 0xffffffffffffffff ) },
  { 0x1p+72, -0x1p+0, INIT_U128( 0x00000000000000ff, 0xffffffffffffffff ) },
  { 0x1p+76, -0x1p+0, INIT_U128( 0x0000000000000fff, 0xffffffffffffffff ) },
  { 0x1p+80, -0x1p+0, INIT_U128( 0x000000000000ffff, 0xffffffffffffffff ) },
  { 0x1p+84, -0x1p+0, INIT_U128( 0x00000000000fffff, 0xffffffffffffffff ) },
  { 0x1p+88, -0x1p+0, INIT_U128( 0x0000000000ffffff, 0xffffffffffffffff ) },
  { 0x1p+92, -0x1p+0, INIT_U128( 0x000000000fffffff, 0xffffffffffffffff ) },
  { 0x1p+96, -0x1p+0, INIT_U128( 0x00000000ffffffff, 0xffffffffffffffff ) },
  { 0x1p+100, -0x1p+0, INIT_U128( 0x0000000fffffffff, 0xffffffffffffffff ) },
  { 0x1p+104, -0x1p+0, INIT_U128( 0x000000ffffffffff, 0xffffffffffffffff ) },
  { 0x1p+108, -0x1p+0, INIT_U128( 0x00000fffffffffff, 0xffffffffffffffff ) },
  { 0x1p+112, -0x1p+0, INIT_U128( 0x0000ffffffffffff, 0xffffffffffffffff ) },
  { 0x1p+116, -0x1p+0, INIT_U128( 0x000fffffffffffff, 0xffffffffffffffff ) },
  { 0x1p+120, -0x1p+0, INIT_U128( 0x00ffffffffffffff, 0xffffffffffffffff ) },
  { 0x1p+124, -0x1p+0, INIT_U128( 0x0fffffffffffffff, 0xffffffffffffffff ) },
  { 0x1p+124, 0x0p+0, INIT_U128( 0x1000000000000000, 0x0000000000000000 ) },
  { 0x1p+124, 0x1.1p+4, INIT_U128( 0x1000000000000000, 0x0000000000000011 ) },
  { 0x1p+124, 0x1.11p+8, INIT_U128( 0x1000000000000000, 0x0000000000000111 ) },
  { 0x1p+124, 0x1.111p+12, INIT_U128( 0x1000000000000000, 0x0000000000001111 ) },
  { 0x1p+124, 0x1.1111p+16, INIT_U128( 0x1000000000000000, 0x0000000000011111 ) },
  { 0x1p+124, 0x1.11111p+20, INIT_U128( 0x1000000000000000, 0x0000000000111111 ) },
  { 0x1p+124, 0x1.111111p+24, INIT_U128( 0x1000000000000000, 0x0000000001111111 ) },
  { 0x1p+124, 0x1.1111111p+28, INIT_U128( 0x1000000000000000, 0x0000000011111111 ) },
  { 0x1p+124, 0x1.11111111p+32, INIT_U128( 0x1000000000000000, 0x0000000111111111 ) },
  { 0x1p+124, 0x1.111111111p+36, INIT_U128( 0x1000000000000000, 0x0000001111111111 ) },
  { 0x1p+124, 0x1.1111111111p+40, INIT_U128( 0x1000000000000000, 0x0000011111111111 ) },
  { 0x1p+124, 0x1.11111111111p+44, INIT_U128( 0x1000000000000000, 0x0000111111111111 ) },
  { 0x1p+124, 0x1.111111111111p+48, INIT_U128( 0x1000000000000000, 0x0001111111111111 ) },
  { 0x1p+124, 0x1.1111111111111p+52, INIT_U128( 0x1000000000000000, 0x0011111111111111 ) },
  { 0x1p+124, 0x1.11111111111111p+56, INIT_U128( 0x1000000000000000, 0x0111111111111110 ) },
  { 0x1p+124, 0x1.111111111111111p+60, INIT_U128( 0x1000000000000000, 0x1111111111111100 ) },
  { 0x1.6ffffffefp+63, -0x1p+0, INIT_U128( 0x0000000000000000, 0xb7ffffff77ffffff ) },
  { 0x1p+106, 0x0p+0, INIT_U128( 0x0000040000000000, 0x0000000000000000 ) },
  { 0x1.ff8p+29, 0x0p+0, INIT_U128( 0x0000000000000000, 0x000000003ff00000 ) },
  { 0x1.6ff7ffffffffcp+63, -0x1p+0, INIT_U128( 0x0000000000000000, 0xb7fbffffffffdfff ) },
  { 0x1.2400000000004p+62, 0x0p+0, INIT_U128( 0x0000000000000000, 0x4900000000001000 ) },
  { 0x1.24000000001p+126, 0x1.24000000001p+62, INIT_U128( 0x4900000000040000, 0x4900000000040000 ) },
  { 0x1.2400001p+126, 0x1.2400001p+62, INIT_U128( 0x4900000400000000, 0x4900000400000000 ) },
  { 0x1.240001p+126, 0x1.240001p+62, INIT_U128( 0x4900004000000000, 0x4900004000000000 ) },
  { 0x1.24001p+126, 0x1.24001p+62, INIT_U128( 0x4900040000000000, 0x4900040000000000 ) },
  { 0x1.24008p+126, 0x1.24008p+62, INIT_U128( 0x4900200000000000, 0x4900200000000000 ) },
  { 0x1.2404p+126, 0x1.2404p+62, INIT_U128( 0x4901000000000000, 0x4901000000000000 ) },
  { 0x1.244p+126, 0x1.244p+62, INIT_U128( 0x4910000000000000, 0x4910000000000000 ) },
  { 0x1.26p+126, 0x1.26p+62, INIT_U128( 0x4980000000000000, 0x4980000000000000 ) },
  { 0x1.3p+126, 0x1.3p+62, INIT_U128( 0x4c00000000000000, 0x4c00000000000000 ) },
  { 0x1.800000000001p+126, 0x1.6000000000004p+64, INIT_U128( 0x6000000000004001, 0x6000000000004000 ) },
  { 0x1.cp+126, 0x1.00ep+71, INIT_U128( 0x7000000000000080, 0x7000000000000000 ) },
  { 0x1.c000000000008p+126, 0x1.c000000000008p+62, INIT_U128( 0x7000000000002000, 0x7000000000002000 ) },
  { 0x1.c00000000004p+126, 0x1.c00000000004p+62, INIT_U128( 0x7000000000010000, 0x7000000000010000 ) },
  { 0x1.c0000000002p+126, 0x1.c0000000002p+62, INIT_U128( 0x7000000000080000, 0x7000000000080000 ) },
  { 0x1.c000000002p+126, 0x1.c000000002p+62, INIT_U128( 0x7000000000800000, 0x7000000000800000 ) },
  { 0x1.c00000002p+126, 0x1.c00000002p+62, INIT_U128( 0x7000000008000000, 0x7000000008000000 ) },
  { 0x1.c0000008p+126, 0x1.c0000008p+62, INIT_U128( 0x7000000200000000, 0x7000000200000000 ) },
  { 0x1.c000002p+126, 0x1.c000002p+62, INIT_U128( 0x7000000800000000, 0x7000000800000000 ) },
  { 0x1.c00004p+126, 0x1.c00004p+62, INIT_U128( 0x7000010000000000, 0x7000010000000000 ) },
  { 0x1.c0002p+126, 0x1.c0002p+62, INIT_U128( 0x7000080000000000, 0x7000080000000000 ) },
  { 0x1.c008p+126, 0x1.c008p+62, INIT_U128( 0x7002000000000000, 0x7002000000000000 ) },
  { 0x1.c02p+126, 0x1.c02p+62, INIT_U128( 0x7008000000000000, 0x7008000000000000 ) },
  { 0x1.c2p+126, 0x1.c2p+62, INIT_U128( 0x7080000000000000, 0x7080000000000000 ) },
  { 0x1.dp+126, 0x1.dp+62, INIT_U128( 0x7400000000000000, 0x7400000000000000 ) },
  { 0x1.00014f3089001p+64, 0x1.e133333333333p+6, INIT_U128( 0x0000000000000001, 0x00014f3089001078 ) },
  { 0x1.fffffffffffffp+127, 0x1.fffffffffffffp+74, INIT_U128( 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFC00000 ) },
  { 0x1.f066666666666p+6, -0x1.f066666666667p-768, INIT_U128( 0x0000000000000000, 0x000000000000007C ) },
  { 0x1.f066666666666p+6, 0x1.f066666666667p-768, INIT_U128( 0x0000000000000000, 0x000000000000007C ) },
  { 0x1.1111111111111p+124, 0x1.1p+68, INIT_U128( 0x1111111111111111, 0x0000000000000000 ) },
  { 0x0.0000000000001p-1022, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x0.0000001160e9fp-1022, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x0.0000001a26f3cp-1022, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { -0x0.0000001134f35p-1022, -0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x0.00003f9eec3ep-1022, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x0.0a40bec0e4818p-1022, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x0.0d4e1fcc5a9c4p-1022, 0x0p+0, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x1.acf5f5ef59ebfp-1007, 0x0.00000000000a8p-1022, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x1.1f45384e3e8a7p-1007, 0x0.00000000000b4p-1022, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x1.8474983f08e93p-1007, 0x0.00000000000d9p-1022, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x1.47ad6cf88f5aep-1007, 0x0.00000000000bcp-1022, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x1.2da4480a5b489p-1, 0x1.9e20ea6f3c41dp-618, INIT_U128( 0x0000000000000000, 0x0000000000000000 ) },
  { 0x1.438fba0a871f8p+0, 0x1.086e1f6e10dc4p-992, INIT_U128( 0x0000000000000000, 0x0000000000000001 ) },
  { 0x1.51c874cca390ep+0, 0x1.5dd30576bba6p-992, INIT_U128( 0x0000000000000000, 0x0000000000000001 ) },
  { 0x1.3691a8086d235p+1, 0x1.3670ebb66ce1ep-281, INIT_U128( 0x0000000000000000, 0x0000000000000002 ) },
  { 0x1.b800c73570019p+1, 0x1.1bf9094c37f21p-281, INIT_U128( 0x0000000000000000, 0x0000000000000003 ) },
  { 0x1.c1e89f3783d14p+1, 0x1.69999266d3332p-281, INIT_U128( 0x0000000000000000, 0x0000000000000003 ) },
  { 0x1.390b19fa72163p+1, 0x1.25bf7f6c4b7fp-281, INIT_U128( 0x0000000000000000, 0x0000000000000002 ) },
  { 0x1.4beee12897ddcp+2, 0x1.549d853aa93bp-815, INIT_U128( 0x0000000000000000, 0x0000000000000005 ) },
  { 0x1.e37742bfc6ee9p+2, 0x1.081a6dda1034ep-815, INIT_U128( 0x0000000000000000, 0x0000000000000007 ) },
  { 0x1.e6c0b50fcd816p+2, 0x1.c3d7967b87af3p-815, INIT_U128( 0x0000000000000000, 0x0000000000000007 ) },
  { 0x1.590290feb2052p+2, 0x1.07756f600eeaep-815, INIT_U128( 0x0000000000000000, 0x0000000000000005 ) },
  { 0x1.66154348cc2a8p+3, 0x1.bb93d1a97727ap-619, INIT_U128( 0x0000000000000000, 0x000000000000000b ) },
  { 0x1.69cf9e9ed39f4p+3, 0x1.ce842e739d086p-619, INIT_U128( 0x0000000000000000, 0x000000000000000b ) },
  { 0x1.e884574bd108bp+3, 0x1.9c6561f338cacp-619, INIT_U128( 0x0000000000000000, 0x000000000000000f ) },
  { 0x1.6263a430c4c74p+4, 0x1.6b327792d664fp-117, INIT_U128( 0x0000000000000000, 0x0000000000000016 ) },
  { 0x1.43637d1286c7p+4, 0x1.546df60aa8dbfp-117, INIT_U128( 0x0000000000000000, 0x0000000000000014 ) },
  { 0x1.faa21259f5443p+4, 0x1.b6708ecf6ce12p-117, INIT_U128( 0x0000000000000000, 0x000000000000001f ) },
  { 0x1.fd3105d1fa62p+4, 0x1.370912046e122p-117, INIT_U128( 0x0000000000000000, 0x000000000000001f ) },
  { 0x1.cf952d399f2a5p+5, 0x1.0d283ebe1a508p-735, INIT_U128( 0x0000000000000000, 0x0000000000000039 ) },
  { 0x1.6ac8f880d591fp+5, 0x1.ec284da5d8509p-735, INIT_U128( 0x0000000000000000, 0x000000000000002d ) },
  { 0x1.919c82bf2339p+5, 0x1.a58971fd4b12ep-735, INIT_U128( 0x0000000000000000, 0x0000000000000032 ) },
  { 0x1.789a89ccf1351p+5, 0x1.ebd48283d7a91p-735, INIT_U128( 0x0000000000000000, 0x000000000000002f ) },
  { 0x1.8db2068d1b641p+6, 0x1.056174ca0ac2ep-802, INIT_U128( 0x0000000000000000, 0x0000000000000063 ) },
  { 0x1.8d618e6f1ac32p+6, 0x1.0643da660c87cp-802, INIT_U128( 0x0000000000000000, 0x0000000000000063 ) },
  { 0x1.f10fc45fe21f8p+6, 0x1.d71f2d6bae3e5p-802, INIT_U128( 0x0000000000000000, 0x000000000000007c ) },
  { 0x1.1cafcb76395fap+8, 0x1.026dfe9a04dcp-78, INIT_U128( 0x0000000000000000, 0x000000000000011c ) },
  { 0x1.8b9a3b5917348p+8, 0x1.f64ef081ec9dep-78, INIT_U128( 0x0000000000000000, 0x000000000000018b ) },
  { 0x1.347ea22c68fd4p+8, 0x1.a86c77e750d8fp-78, INIT_U128( 0x0000000000000000, 0x0000000000000134 ) },
  { 0x1.c498c82389319p+8, 0x1.0d4d96061a9b3p-78, INIT_U128( 0x0000000000000000, 0x00000000000001c4 ) },
  { 0x1.3e4fa8b47c9f5p+9, 0x1.d0718139a0e3p-293, INIT_U128( 0x0000000000000000, 0x000000000000027c ) },
  { 0x1.5ffd3878bffa7p+9, 0x1.1828a29c30514p-293, INIT_U128( 0x0000000000000000, 0x00000000000002bf ) },
  { 0x1.bbbb3b8577768p+9, 0x1.ba79b21b74f36p-293, INIT_U128( 0x0000000000000000, 0x0000000000000377 ) },
  { 0x1.8777cb810eefap+9, 0x1.bb3afdc57676p-293, INIT_U128( 0x0000000000000000, 0x000000000000030e ) },
  { 0x1.8e7992a11cf32p+10, 0x1.a18e4033431c8p-789, INIT_U128( 0x0000000000000000, 0x0000000000000639 ) },
  { 0x1.7684faeced0ap+10, 0x1.5f11706abe22ep-789, INIT_U128( 0x0000000000000000, 0x00000000000005da ) },
  { 0x1.7d7ac11afaf58p+10, 0x1.64553beec8aa8p-789, INIT_U128( 0x0000000000000000, 0x00000000000005f5 ) },
  { 0x1.a540fa454a81fp+10, 0x1.329efbd6653ep-789, INIT_U128( 0x0000000000000000, 0x0000000000000695 ) },
  { 0x1.92ed812325dbp+11, 0x0.000012eafc716p-1022, INIT_U128( 0x0000000000000000, 0x0000000000000c97 ) },
  { 0x1.b10abb4562158p+11, 0x0.00001e4765564p-1022, INIT_U128( 0x0000000000000000, 0x0000000000000d88 ) },
  { 0x1.f6751879ecea3p+11, 0x0.000014435b4b9p-1022, INIT_U128( 0x0000000000000000, 0x0000000000000fb3 ) },
  { 0x1.32e6b0a465cd6p+11, 0x0.00001729d5c09p-1022, INIT_U128( 0x0000000000000000, 0x0000000000000997 ) },
  { 0x1.11f336c623e67p+12, 0x1.9d9af61f3b35fp-489, INIT_U128( 0x0000000000000000, 0x000000000000111f ) },
  { 0x1.18ebab5631d76p+12, 0x1.47d8de9e8fb1cp-489, INIT_U128( 0x0000000000000000, 0x000000000000118e ) },
  { 0x1.c25a29f984b45p+12, 0x1.c65b51e78cb6ap-489, INIT_U128( 0x0000000000000000, 0x0000000000001c25 ) },
  { 0x1.cf37f3299e6ffp+12, 0x1.38601ed270c04p-489, INIT_U128( 0x0000000000000000, 0x0000000000001cf3 ) },
  { 0x1.00db141c01b62p+13, 0x1.a4ea801149d5p-963, INIT_U128( 0x0000000000000000, 0x000000000000201b ) },
  { 0x1.b81f3643703e7p+13, 0x1.849a11a509342p-963, INIT_U128( 0x0000000000000000, 0x0000000000003703 ) },
  { 0x1.6dbbb6a4db777p+13, 0x1.362f46546c5e9p-963, INIT_U128( 0x0000000000000000, 0x0000000000002db7 ) },
  { 0x1.eeb645abdd6c8p+13, 0x1.1edee6683dbddp-963, INIT_U128( 0x0000000000000000, 0x0000000000003dd6 ) },
  { 0x1.dc7ac6d3b8f59p+14, 0x1.3a4d2846749a5p-96, INIT_U128( 0x0000000000000000, 0x000000000000771e ) },
  { 0x1.51b38df4a3672p+14, 0x1.ef5f5533debeap-96, INIT_U128( 0x0000000000000000, 0x000000000000546c ) },
  { 0x1.5cdcee10b9b9ep+14, 0x1.dd44c049ba898p-96, INIT_U128( 0x0000000000000000, 0x0000000000005737 ) },
  { 0x1.f5266d45ea4cdp+14, 0x1.3970615e72e0cp-96, INIT_U128( 0x0000000000000000, 0x0000000000007d49 ) },
  { 0x1.2ab9252a55724p+15, 0x1.a85ba3d950b74p-690, INIT_U128( 0x0000000000000000, 0x000000000000955c ) },
  { 0x1.707a4edee0f4ap+15, 0x1.00541f2800a84p-690, INIT_U128( 0x0000000000000000, 0x000000000000b83d ) },
  { 0x1.594c6252b298cp+15, 0x1.837b6c0706f6ep-690, INIT_U128( 0x0000000000000000, 0x000000000000aca6 ) },
  { 0x1.41fad8e683f5bp+15, 0x1.7cb3a2bef9674p-690, INIT_U128( 0x0000000000000000, 0x000000000000a0fd ) },
  { 0x1.f150e9b3e2a1dp+16, 0x1.155a86a62ab51p-762, INIT_U128( 0x0000000000000000, 0x000000000001f150 ) },
  { 0x1.0a3ca44214794p+16, 0x1.5c85e07eb90bcp-762, INIT_U128( 0x0000000000000000, 0x0000000000010a3c ) },
  { 0x1.0ef4814e1de9p+16, 0x1.2cd0510659a0ap-762, INIT_U128( 0x0000000000000000, 0x0000000000010ef4 ) },
  { 0x1.99373a97326e7p+16, 0x1.98d239d731a47p-762, INIT_U128( 0x0000000000000000, 0x0000000000019937 ) },
  { 0x1.0062da5400c5cp+17, 0x1.12b3124825662p-940, INIT_U128( 0x0000000000000000, 0x00000000000200c5 ) },
  { 0x1.8ac0f0251581ep+17, 0x1.60b254d0c164ap-940, INIT_U128( 0x0000000000000000, 0x0000000000031581 ) },
  { 0x1.51c0eb94a381ep+17, 0x1.ce1da4059c3b4p-940, INIT_U128( 0x0000000000000000, 0x000000000002a381 ) },
  { 0x1.5e83e6a4bd07dp+17, 0x1.adcff7815b9ffp-940, INIT_U128( 0x0000000000000000, 0x000000000002bd07 ) },
  { 0x1.242e55bc485cap+18, 0x0.000000000001fp-1022, INIT_U128( 0x0000000000000000, 0x00000000000490b9 ) },
  { 0x1.7d59ac2cfab36p+18, 0x0.000000000001p-1022, INIT_U128( 0x0000000000000000, 0x000000000005f566 ) },
  { 0x1.d19e6101a33ccp+18, 0x0.0000000000014p-1022, INIT_U128( 0x0000000000000000, 0x0000000000074679 ) },
  { 0x1.34c9981869933p+18, 0x0.0000000000011p-1022, INIT_U128( 0x0000000000000000, 0x000000000004d326 ) },
  { 0x1.c066e34f80cddp+19, 0x1.e3f363b5c7e6dp-881, INIT_U128( 0x0000000000000000, 0x00000000000e0337 ) },
  { 0x1.df32bc2fbe658p+19, 0x1.df947163bf28ep-881, INIT_U128( 0x0000000000000000, 0x00000000000ef995 ) },
  { 0x1.3bbc859e7779p+19, 0x1.0773506c0ee6ap-881, INIT_U128( 0x0000000000000000, 0x000000000009dde4 ) },
  { 0x1.3b65fdae76ccp+19, 0x1.36924fde6d24ap-881, INIT_U128( 0x0000000000000000, 0x000000000009db2f ) },
  { 0x1.52c7a810a58f5p+20, 0x1.a11a939f42352p-561, INIT_U128( 0x0000000000000000, 0x0000000000152c7a ) },
  { 0x1.9546ee252a8dep+20, 0x1.eeb4a28ddd695p-561, INIT_U128( 0x0000000000000000, 0x000000000019546e ) },
  { 0x1.f50465bdea08cp+20, 0x1.7288f4c2e511ep-561, INIT_U128( 0x0000000000000000, 0x00000000001f5046 ) },
  { 0x1.b8199d2770334p+20, 0x1.a2d4ddfb45a9cp-561, INIT_U128( 0x0000000000000000, 0x00000000001b8199 ) },
  { 0x1.efe67d0fdfccfp+21, 0x1.05ac37920b587p-121, INIT_U128( 0x0000000000000000, 0x00000000003dfccf ) },
  { 0x1.a1c8a86343915p+21, 0x1.ff7f144bfefe2p-121, INIT_U128( 0x0000000000000000, 0x0000000000343915 ) },
  { 0x1.0b3b76001676fp+21, 0x1.742fba58e85f8p-121, INIT_U128( 0x0000000000000000, 0x000000000021676e ) },
  { 0x1.cb12f6579625fp+21, 0x1.5e77e020bcefcp-121, INIT_U128( 0x0000000000000000, 0x000000000039625e ) },
  { 0x1.bd380f437a702p+22, 0x1.491fe5e8923fcp-995, INIT_U128( 0x0000000000000000, 0x00000000006f4e03 ) },
  { 0x1.46fbb89c8df77p+22, 0x1.a09fc8f7413f9p-995, INIT_U128( 0x0000000000000000, 0x000000000051beee ) },
  { 0x1.17e871f42fd0ep+22, 0x1.a11fc1a7423f8p-995, INIT_U128( 0x0000000000000000, 0x000000000045fa1c ) },
  { 0x1.277e999a4efd3p+22, 0x1.4bd3e11097a7cp-995, INIT_U128( 0x0000000000000000, 0x000000000049dfa6 ) },
  { 0x1.6e4d3250dc9a6p+23, 0x1.edf09145dbe12p-447, INIT_U128( 0x0000000000000000, 0x0000000000b72699 ) },
  { 0x1.eb0413bfd6083p+23, 0x1.29c840b053908p-447, INIT_U128( 0x0000000000000000, 0x0000000000f58209 ) },
  { 0x1.283f1d32507e4p+23, 0x1.06daa0fe0db54p-447, INIT_U128( 0x0000000000000000, 0x0000000000941f8e ) },
  { 0x1.cf7790bd9eef2p+23, 0x1.22ebe99845d7dp-447, INIT_U128( 0x0000000000000000, 0x0000000000e7bbc8 ) },
  { 0x1.39fb1e1473f64p+24, 0x1.baafa729755f5p-263, INIT_U128( 0x0000000000000000, 0x000000000139fb1e ) },
  { 0x1.553510f0aa6a2p+24, 0x1.806a3d5d00d48p-263, INIT_U128( 0x0000000000000000, 0x0000000001553510 ) },
  { 0x1.876715390ece3p+24, 0x1.cdf668119becdp-263, INIT_U128( 0x0000000000000000, 0x0000000001876715 ) },
  { 0x1.11816ac62302ep+24, 0x1.5e8451ecbd08ap-263, INIT_U128( 0x0000000000000000, 0x000000000111816a ) },
  { 0x1.3e7e811e7cfdp+25, 0x1.2dc6a5685b8d4p-155, INIT_U128( 0x0000000000000000, 0x00000000027cfd02 ) },
  { 0x1.6ebc2c8cdd786p+25, 0x1.22f38cd645e72p-155, INIT_U128( 0x0000000000000000, 0x0000000002dd7859 ) },
  { 0x1.421ccf068439ap+25, 0x1.bef8b6f57df17p-155, INIT_U128( 0x0000000000000000, 0x000000000284399e ) },
  { 0x1.e10ee555c21dcp+25, 0x1.033cdd7a0679cp-155, INIT_U128( 0x0000000000000000, 0x0000000003c21dca ) },
  { 0x1.f0b6e7ede16ddp+26, 0x1.bc0a81517815p-634, INIT_U128( 0x0000000000000000, 0x0000000007c2db9f ) },
  { 0x1.f7b66dc5ef6cdp+26, 0x1.2bd3184a57a63p-634, INIT_U128( 0x0000000000000000, 0x0000000007ded9b7 ) },
  { 0x1.259706244b2e1p+26, 0x1.8b7dd07716fbap-634, INIT_U128( 0x0000000000000000, 0x0000000004965c18 ) },
  { 0x1.fdf5519ffbeaap+26, 0x1.19d4925433a92p-634, INIT_U128( 0x0000000000000000, 0x0000000007f7d546 ) },
  { 0x1.bb13b1d776276p+27, 0x1.e509cb23ca13ap-395, INIT_U128( 0x0000000000000000, 0x000000000dd89d8e ) },
  { 0x1.20754b2a40eaap+27, 0x1.bcc8cd6b7991ap-395, INIT_U128( 0x0000000000000000, 0x000000000903aa59 ) },
  { 0x1.dc036999b806dp+27, 0x1.62b438eec5687p-395, INIT_U128( 0x0000000000000000, 0x000000000ee01b4c ) },
  { 0x1.92b76e3d256eep+27, 0x1.33f6413067ec8p-395, INIT_U128( 0x0000000000000000, 0x000000000c95bb71 ) },
  { 0x1.7a6c4408f4d88p+28, 0x1.0f6d05b41edap-716, INIT_U128( 0x0000000000000000, 0x0000000017a6c440 ) },
  { 0x1.25d2c9ae4ba59p+28, 0x1.005f5a6a00becp-716, INIT_U128( 0x0000000000000000, 0x00000000125d2c9a ) },
  { 0x1.c8949d0b91293p+28, 0x1.6e7ca504dcf94p-716, INIT_U128( 0x0000000000000000, 0x000000001c8949d0 ) },
  { 0x1.21af1f7a435e4p+28, 0x1.98e05ef731c0cp-716, INIT_U128( 0x0000000000000000, 0x00000000121af1f7 ) },
  { 0x1.307b63f060f6cp+29, 0x1.17d4949a2fa92p-112, INIT_U128( 0x0000000000000000, 0x00000000260f6c7e ) },
  { 0x1.053d62740a7acp+29, 0x1.a8881e6551104p-112, INIT_U128( 0x0000000000000000, 0x0000000020a7ac4e ) },
  { 0x1.9b60d35b36c1ap+29, 0x1.005bb09600b76p-112, INIT_U128( 0x0000000000000000, 0x00000000336c1a6b ) },
  { 0x1.3ba73afe774e8p+29, 0x1.0efca74e1df95p-112, INIT_U128( 0x0000000000000000, 0x000000002774e75f ) },
  { 0x1.dd796675baf2dp+30, 0x1.aaba00775574p-990, INIT_U128( 0x0000000000000000, 0x00000000775e599d ) },
  { 0x1.8b282c8d16506p+30, 0x1.4e04f8449c09fp-990, INIT_U128( 0x0000000000000000, 0x0000000062ca0b23 ) },
  { 0x1.7d6ab930fad57p+30, 0x1.764a6e3cec94ep-990, INIT_U128( 0x0000000000000000, 0x000000005f5aae4c ) },
  { 0x1.6906027cd20cp+30, 0x1.d5b4ef4dab69ep-990, INIT_U128( 0x0000000000000000, 0x000000005a41809f ) },
  { 0x1.04a9da360953cp+31, 0x1.6c11a2e4d8234p-857, INIT_U128( 0x0000000000000000, 0x000000008254ed1b ) },
  { 0x1.fbba6e93f774ep+31, 0x1.0e9a6e5c1d34ep-857, INIT_U128( 0x0000000000000000, 0x00000000fddd3749 ) },
  { 0x1.380b108470162p+31, 0x1.65c0e5f6cb81cp-857, INIT_U128( 0x0000000000000000, 0x000000009c058842 ) },
  { 0x1.050dfc000a1cp+31, 0x1.1df2fc803be6p-857, INIT_U128( 0x0000000000000000, 0x000000008286fe00 ) },
  { 0x1.88bf9e1d117f4p+32, 0x1.8dd4aa1b1ba95p-126, INIT_U128( 0x0000000000000000, 0x0000000188bf9e1d ) },
  { 0x1.8b35d7ff166bbp+32, 0x1.91ca210923944p-126, INIT_U128( 0x0000000000000000, 0x000000018b35d7ff ) },
  { 0x1.286a366250d47p+32, 0x1.012c86ac02591p-126, INIT_U128( 0x0000000000000000, 0x00000001286a3662 ) },
  { 0x1.ef233cd5de467p+32, 0x1.2292a62645255p-126, INIT_U128( 0x0000000000000000, 0x00000001ef233cd5 ) },
  { 0x1.917a959722f53p+33, 0x1.85a7d93f0b4fbp-478, INIT_U128( 0x0000000000000000, 0x0000000322f52b2e ) },
  { 0x1.f630053bec6p+33, 0x1.f79a5227ef34bp-478, INIT_U128( 0x0000000000000000, 0x00000003ec600a77 ) },
  { 0x1.1a062b14340c6p+33, 0x1.21179acc422f4p-478, INIT_U128( 0x0000000000000000, 0x00000002340c5628 ) },
  { 0x1.d62acf15ac55ap+33, 0x1.bf1cd2697e39ap-478, INIT_U128( 0x0000000000000000, 0x00000003ac559e2b ) },
  { 0x1.823ddfaf047bcp+34, 0x1.e2d35df1c5a6cp-649, INIT_U128( 0x0000000000000000, 0x0000000608f77ebc ) },
  { 0x1.996ef6e332ddfp+34, 0x1.7af28278f5e5p-649, INIT_U128( 0x0000000000000000, 0x0000000665bbdb8c ) },
  { 0x1.81a2bfc703458p+34, 0x1.acc15cd15982cp-649, INIT_U128( 0x0000000000000000, 0x00000006068aff1c ) },
  { 0x1.4517e98e8a2fdp+34, 0x1.a3233fc546468p-649, INIT_U128( 0x0000000000000000, 0x00000005145fa63a ) },
  { 0x1.09f551a013eaap+35, 0x0.0000000000006p-1022, INIT_U128( 0x0000000000000000, 0x000000084faa8d00 ) },
  { 0x1.a2911b3d45224p+35, 0x0.0000000000005p-1022, INIT_U128( 0x0000000000000000, 0x0000000d1488d9ea ) },
  { 0x1.77a301deef46p+35, 0x0.0000000000008p-1022, INIT_U128( 0x0000000000000000, 0x0000000bbd180ef7 ) },
  { 0x1.f60ea85fec1d5p+35, 0x0.0000000000006p-1022, INIT_U128( 0x0000000000000000, 0x0000000fb07542ff ) },
  { 0x1.75c28ed8eb852p+36, 0x1.64300234c86p-938, INIT_U128( 0x0000000000000000, 0x000000175c28ed8e ) },
  { 0x1.394f8bae729f2p+36, 0x1.2119204042324p-938, INIT_U128( 0x0000000000000000, 0x0000001394f8bae7 ) },
  { 0x1.94d6bc7929ad8p+36, 0x1.4a8aa2aa95154p-938, INIT_U128( 0x0000000000000000, 0x000000194d6bc792 ) },
  { 0x1.1f519dcc3ea34p+36, 0x1.b4fbc9c969f79p-938, INIT_U128( 0x0000000000000000, 0x00000011f519dcc3 ) },
  { 0x1.bb2fb46f765f6p+37, 0x1.cd4047139a809p-718, INIT_U128( 0x0000000000000000, 0x0000003765f68dee ) },
  { 0x1.f80989d1f0131p+37, 0x1.c0546f4180a8ep-718, INIT_U128( 0x0000000000000000, 0x0000003f01313a3e ) },
  { 0x1.90be9171217d2p+37, 0x1.bcc6089d798c1p-718, INIT_U128( 0x0000000000000000, 0x0000003217d22e24 ) },
  { 0x1.37469d226e8d4p+37, 0x1.06aff8000d5ffp-718, INIT_U128( 0x0000000000000000, 0x00000026e8d3a44d ) },
  { 0x1.2e0114905c022p+38, 0x1.298bde145317cp-115, INIT_U128( 0x0000000000000000, 0x0000004b80452417 ) },
  { 0x1.77354c42ee6aap+38, 0x1.d9a75513b34eap-115, INIT_U128( 0x0000000000000000, 0x0000005dcd5310bb ) },
  { 0x1.0881a80c11035p+38, 0x1.44e5e01889cbcp-115, INIT_U128( 0x0000000000000000, 0x00000042206a0304 ) },
  { 0x1.a32b590f4656bp+38, 0x1.760fbd2eec1f8p-115, INIT_U128( 0x0000000000000000, 0x00000068cad643d1 ) },
  { 0x1.e7bc6861cf78dp+39, 0x1.72addca8e55bcp-767, INIT_U128( 0x0000000000000000, 0x000000f3de3430e7 ) },
  { 0x1.e1e11b1bc3c24p+39, 0x1.713729dae26e5p-767, INIT_U128( 0x0000000000000000, 0x000000f0f08d8de1 ) },
  { 0x1.8bbb377f17767p+39, 0x1.c296d213852dap-767, INIT_U128( 0x0000000000000000, 0x000000c5dd9bbf8b ) },
  { 0x1.7cb13d18f9628p+39, 0x1.8dcf87351b9f1p-767, INIT_U128( 0x0000000000000000, 0x000000be589e8c7c ) },
  { 0x1.1ddbaf383bb76p+40, 0x1.20c8d9d04191bp-497, INIT_U128( 0x0000000000000000, 0x0000011ddbaf383b ) },
  { 0x1.fde474e3fbc8ep+40, 0x1.7e6de35afcdbcp-497, INIT_U128( 0x0000000000000000, 0x000001fde474e3fb ) },
  { 0x1.02a202b00544p+40, 0x1.311b40f462368p-497, INIT_U128( 0x0000000000000000, 0x00000102a202b005 ) },
  { 0x1.ec0b6577d816cp+40, 0x1.8e49e85d1c93dp-497, INIT_U128( 0x0000000000000000, 0x000001ec0b6577d8 ) },
  { 0x1.95fa4b912bf4ap+41, 0x1.295d953652bb2p-872, INIT_U128( 0x0000000000000000, 0x0000032bf4972257 ) },
  { 0x1.fd243093fa486p+41, 0x1.91e0a45723c14p-872, INIT_U128( 0x0000000000000000, 0x000003fa486127f4 ) },
  { 0x1.d0beeb21a17dep+41, 0x1.97f98e272ff32p-872, INIT_U128( 0x0000000000000000, 0x000003a17dd64342 ) },
  { 0x1.750735faea0e6p+41, 0x1.97a4c8c92f499p-872, INIT_U128( 0x0000000000000000, 0x000002ea0e6bf5d4 ) },
  { 0x1.ab409c8f56814p+42, 0x1.1e820fc23d042p-84, INIT_U128( 0x0000000000000000, 0x000006ad02723d5a ) },
  { 0x1.85e79a910bcf3p+42, 0x1.75d44930eba89p-84, INIT_U128( 0x0000000000000000, 0x000006179e6a442f ) },
  { 0x1.e2fa47dfc5f49p+42, 0x1.f91c1067f2382p-84, INIT_U128( 0x0000000000000000, 0x0000078be91f7f17 ) },
  { 0x1.ecaf7567d95eep+42, 0x1.dd787be3baf1p-84, INIT_U128( 0x0000000000000000, 0x000007b2bdd59f65 ) },
  { 0x1.4fd770a89faeep+43, 0x1.883956a11072bp-669, INIT_U128( 0x0000000000000000, 0x00000a7ebb8544fd ) },
  { 0x1.b2b2aa2d65655p+43, 0x1.0c2516f2184a3p-669, INIT_U128( 0x0000000000000000, 0x00000d9595516b2b ) },
  { 0x1.5848b5b4b0916p+43, 0x1.d9a0d8cfb341bp-669, INIT_U128( 0x0000000000000000, 0x00000ac245ada584 ) },
  { 0x1.be7daa1f7cfb5p+43, 0x1.f0c9d223e193bp-669, INIT_U128( 0x0000000000000000, 0x00000df3ed50fbe7 ) },
  { 0x1.3f6d46f07eda9p+44, 0x1.3d2fa1b27a5f4p-539, INIT_U128( 0x0000000000000000, 0x000013f6d46f07ed ) },
  { 0x1.5eb8dcaebd71cp+44, 0x1.8e170e7b1c2e2p-539, INIT_U128( 0x0000000000000000, 0x000015eb8dcaebd7 ) },
  { 0x1.893b63631276cp+44, 0x1.9447ca53288f9p-539, INIT_U128( 0x0000000000000000, 0x00001893b6363127 ) },
  { 0x1.4e089b389c114p+44, 0x1.d191b053a3236p-539, INIT_U128( 0x0000000000000000, 0x000014e089b389c1 ) },
  { 0x1.2e36d90e5c6dbp+45, 0x1.314d394c629a7p-889, INIT_U128( 0x0000000000000000, 0x000025c6db21cb8d ) },
  { 0x1.7f784de4fef0ap+45, 0x1.5413e986a827dp-889, INIT_U128( 0x0000000000000000, 0x00002fef09bc9fde ) },
  { 0x1.74448388e889p+45, 0x1.752c3c2cea588p-889, INIT_U128( 0x0000000000000000, 0x00002e8890711d11 ) },
  { 0x1.ebb20c51d7641p+45, 0x1.50a52f7ca14a6p-889, INIT_U128( 0x0000000000000000, 0x00003d76418a3aec ) },
  { 0x1.4b51066896a21p+46, 0x1.d43cc973a8799p-550, INIT_U128( 0x0000000000000000, 0x000052d4419a25a8 ) },
  { 0x1.070f0d280e1e2p+46, 0x1.c25aa6dd84b55p-550, INIT_U128( 0x0000000000000000, 0x000041c3c34a0387 ) },
  { 0x1.7f735ca4fee6cp+46, 0x1.a92889d752511p-550, INIT_U128( 0x0000000000000000, 0x00005fdcd7293fb9 ) },
  { 0x1.72ed987ae5db3p+46, 0x1.fae14a03f5c29p-550, INIT_U128( 0x0000000000000000, 0x00005cbb661eb976 ) },
  { 0x1.7352fdf0e6a6p+47, 0x1.1a64275034c85p-857, INIT_U128( 0x0000000000000000, 0x0000b9a97ef87353 ) },
  { 0x1.9f4b98f33e973p+47, 0x1.657b10c0caf62p-857, INIT_U128( 0x0000000000000000, 0x0000cfa5cc799f4b ) },
  { 0x1.b12eb79b625d7p+47, 0x1.4826ca96904dap-857, INIT_U128( 0x0000000000000000, 0x0000d8975bcdb12e ) },
  { 0x1.6148cbcac291ap+47, 0x1.8672a1ad0ce54p-857, INIT_U128( 0x0000000000000000, 0x0000b0a465e56148 ) },
  { 0x1.1df8159e3bf02p+48, 0x1.7c6dbd68f8db8p-407, INIT_U128( 0x0000000000000000, 0x00011df8159e3bf0 ) },
  { 0x1.d8e17545b1c2ep+48, 0x1.959fcc5f2b3fap-407, INIT_U128( 0x0000000000000000, 0x0001d8e17545b1c2 ) },
  { 0x1.ea57e075d4afcp+48, 0x1.56705400ace0ap-407, INIT_U128( 0x0000000000000000, 0x0001ea57e075d4af ) },
  { 0x1.d2aa9e99a5554p+48, 0x1.9ebcab993d796p-407, INIT_U128( 0x0000000000000000, 0x0001d2aa9e99a555 ) },
  { 0x1.4fe075a49fc0ep+49, 0x1.e950f533d2a1ep-694, INIT_U128( 0x0000000000000000, 0x00029fc0eb493f81 ) },
  { 0x1.a936db51526dcp+49, 0x1.8a1397df14273p-694, INIT_U128( 0x0000000000000000, 0x0003526db6a2a4db ) },
  { 0x1.138ce7682719dp+49, 0x1.7f879c76ff0f4p-694, INIT_U128( 0x0000000000000000, 0x00022719ced04e33 ) },
  { 0x1.21e981b043d3p+49, 0x1.eaefe1f3d5dfcp-694, INIT_U128( 0x0000000000000000, 0x000243d3036087a6 ) },
  { 0x1.793e9a3cf27d4p+50, 0x1.40169b90802d4p-934, INIT_U128( 0x0000000000000000, 0x0005e4fa68f3c9f5 ) },
  { 0x1.7e1caadcfc396p+50, 0x1.371020466e204p-934, INIT_U128( 0x0000000000000000, 0x0005f872ab73f0e5 ) },
  { 0x1.33a019d667403p+50, 0x1.9c77f7cb38effp-934, INIT_U128( 0x0000000000000000, 0x0004ce8067599d00 ) },
  { 0x1.b4966ecb692cep+50, 0x1.c4dd0f8989ba2p-934, INIT_U128( 0x0000000000000000, 0x0006d259bb2da4b3 ) },
  { 0x1.822ad9630455bp+51, 0x1.3fade6a07f5bdp-561, INIT_U128( 0x0000000000000000, 0x000c1156cb1822ad ) },
  { 0x1.3a77dd3c74efcp+51, 0x1.55dd3018abba6p-561, INIT_U128( 0x0000000000000000, 0x0009d3bee9e3a77e ) },
  { 0x1.d375773da6eafp+51, 0x1.ad40b9955a817p-561, INIT_U128( 0x0000000000000000, 0x000e9babb9ed3757 ) },
  { 0x1.16059bfe2c0b4p+51, 0x1.ff801c63ff003p-561, INIT_U128( 0x0000000000000000, 0x0008b02cdff1605a ) },
  { 0x1.af8f1e955f1e4p+52, 0x1.2c4cc4fa58998p-237, INIT_U128( 0x0000000000000000, 0x001af8f1e955f1e4 ) },
  { 0x1.579e5322af3cap+52, 0x1.e9974457d32e8p-237, INIT_U128( 0x0000000000000000, 0x001579e5322af3ca ) },
  { 0x1.2b9d921a573b2p+52, 0x1.d8798265b0f3p-237, INIT_U128( 0x0000000000000000, 0x0012b9d921a573b2 ) },
  { 0x1.b746d5596e8dbp+52, 0x1.a75bfc954eb8p-237, INIT_U128( 0x0000000000000000, 0x001b746d5596e8db ) },
  { 0x1.497ec4f092fd8p+53, 0x1.5c597ab2b8b3p-364, INIT_U128( 0x0000000000000000, 0x00292fd89e125fb0 ) },
  { 0x1.8a65536914caap+53, 0x1.958565492b0adp-364, INIT_U128( 0x0000000000000000, 0x00314caa6d229954 ) },
  { 0x1.11b7146c236e2p+53, 0x1.3154f5b662a9ep-364, INIT_U128( 0x0000000000000000, 0x002236e28d846dc4 ) },
  { 0x1.f71b5e7bee36cp+53, 0x1.efc7aa5ddf8f6p-364, INIT_U128( 0x0000000000000000, 0x003ee36bcf7dc6d8 ) },
  { 0x1.5a71157ab4e22p+54, 0x1.b4fd1c6b69fa4p-39, INIT_U128( 0x0000000000000000, 0x00569c455ead3888 ) },
  { 0x1.4ab52e26956a6p+54, 0x1.204833ee40906p-39, INIT_U128( 0x0000000000000000, 0x0052ad4b89a55a98 ) },
  { 0x1.7b9298b4f7253p+54, 0x1.084ca6f410995p-39, INIT_U128( 0x0000000000000000, 0x005ee4a62d3dc94c ) },
  { 0x1.8be06c0317c0ep+54, 0x1.9677f6df2ceffp-39, INIT_U128( 0x0000000000000000, 0x0062f81b00c5f038 ) },
  { 0x1.53534daca6a6ap+55, 0x1.af26be6f5e4d8p-905, INIT_U128( 0x0000000000000000, 0x00a9a9a6d6535350 ) },
  { 0x1.ee7424c1dce84p+55, 0x1.38a375387146ep-905, INIT_U128( 0x0000000000000000, 0x00f73a1260ee7420 ) },
  { 0x1.4275718484eaep+55, 0x1.693f342ed27e6p-905, INIT_U128( 0x0000000000000000, 0x00a13ab8c2427570 ) },
  { 0x1.4e48bc049c918p+55, 0x1.30d3b39661a76p-905, INIT_U128( 0x0000000000000000, 0x00a7245e024e48c0 ) },
  { 0x1.6fcb01f0df96p+56, 0x1.f3322f93e6646p-339, INIT_U128( 0x0000000000000000, 0x016fcb01f0df9600 ) },
  { 0x1.6f16f2e4de2dep+56, 0x1.b50cb2d16a196p-339, INIT_U128( 0x0000000000000000, 0x016f16f2e4de2de0 ) },
  { 0x1.6fcb3cb2df968p+56, 0x1.f7623e45eec48p-339, INIT_U128( 0x0000000000000000, 0x016fcb3cb2df9680 ) },
  { 0x1.a41a78314834fp+56, 0x1.ee812a93dd026p-339, INIT_U128( 0x0000000000000000, 0x01a41a78314834f0 ) },
  { 0x1.73544f7ce6a8ap+57, 0x1.fbf6a069f7ed4p-786, INIT_U128( 0x0000000000000000, 0x02e6a89ef9cd5140 ) },
  { 0x1.8d4beb3f1a97ep+57, 0x1.6f6e15a0dedc2p-786, INIT_U128( 0x0000000000000000, 0x031a97d67e352fc0 ) },
  { 0x1.70dfc328e1bf8p+57, 0x1.56963a34ad2c8p-786, INIT_U128( 0x0000000000000000, 0x02e1bf8651c37f00 ) },
  { 0x1.6e5e39acdcbc7p+57, 0x1.62dfb7d4c5bf7p-786, INIT_U128( 0x0000000000000000, 0x02dcbc7359b978e0 ) },
  { 0x1.10a375142146ep+58, 0x1.dde963f1bbd2cp-687, INIT_U128( 0x0000000000000000, 0x04428dd450851b80 ) },
  { 0x1.7eacb1acfd596p+58, 0x1.e59952a9cb32bp-687, INIT_U128( 0x0000000000000000, 0x05fab2c6b3f56580 ) },
  { 0x1.3f2bac4a7e576p+58, 0x1.d21ee367a43ddp-687, INIT_U128( 0x0000000000000000, 0x04fcaeb129f95d80 ) },
  { 0x1.be738acb7ce71p+58, 0x1.d4b6334fa96c7p-687, INIT_U128( 0x0000000000000000, 0x06f9ce2b2df39c40 ) },
  { 0x1.b322eff56645ep+59, 0x0.00000014b8158p-1022, INIT_U128( 0x0000000000000000, 0x0d99177fab322f00 ) },
  { 0x1.b8dfbdbd71bf8p+59, 0x0.00000010ac2d6p-1022, INIT_U128( 0x0000000000000000, 0x0dc6fdedeb8dfc00 ) },
  { 0x1.e45f6d33c8bedp+59, 0x0.0000001c79003p-1022, INIT_U128( 0x0000000000000000, 0x0f22fb699e45f680 ) },
  { 0x1.10c7106e218e2p+59, 0x0.0000001ea2457p-1022, INIT_U128( 0x0000000000000000, 0x08863883710c7100 ) },
  { 0x1.c48c230989185p+60, 0x1.a60d3fb34c1a8p-116, INIT_U128( 0x0000000000000000, 0x1c48c23098918500 ) },
  { 0x1.5e9345fabd268p+60, 0x1.4898e6d49131dp-116, INIT_U128( 0x0000000000000000, 0x15e9345fabd26800 ) },
  { 0x1.b56942576ad28p+60, 0x1.aff4a0655fe94p-116, INIT_U128( 0x0000000000000000, 0x1b56942576ad2800 ) },
  { 0x1.7f865930ff0cbp+60, 0x1.13a0876e27411p-116, INIT_U128( 0x0000000000000000, 0x17f865930ff0cb00 ) },
  { 0x1.ef482c31de906p+61, 0x1.43e655d887ccap-501, INIT_U128( 0x0000000000000000, 0x3de905863bd20c00 ) },
  { 0x1.9fa15a7d3f42bp+61, 0x1.b00fcc55601fap-501, INIT_U128( 0x0000000000000000, 0x33f42b4fa7e85600 ) },
  { 0x1.d2c465fda588dp+61, 0x1.98c2f6e73185fp-501, INIT_U128( 0x0000000000000000, 0x3a588cbfb4b11a00 ) },
  { 0x1.f038608de070cp+61, 0x1.7b4fa8a0f69f5p-501, INIT_U128( 0x0000000000000000, 0x3e070c11bc0e1800 ) },
  { 0x1.adfb2db35bf66p+62, 0x1.38efaf6271df6p+8, INIT_U128( 0x0000000000000000, 0x6b7ecb6cd6fd9938 ) },
  { 0x1.1679474c2cf29p+62, 0x1.ae04d7f95c09bp+8, INIT_U128( 0x0000000000000000, 0x459e51d30b3ca5ae ) },
  { 0x1.890c63b91218cp+62, 0x1.133030ac26606p+8, INIT_U128( 0x0000000000000000, 0x624318ee44863113 ) },
  { 0x1.08fc576811f8bp+62, 0x1.521d194ea43a3p+8, INIT_U128( 0x0000000000000000, 0x423f15da047e2d52 ) },
  { 0x1.5c2e1ea2b85c4p+63, 0x1.bbf1e79d77e3dp-836, INIT_U128( 0x0000000000000000, 0xae170f515c2e2000 ) },
  { 0x1.3a1d0742743a1p+63, 0x1.849ecbad093dap-836, INIT_U128( 0x0000000000000000, 0x9d0e83a13a1d0800 ) },
  { 0x1.ac698c2758d32p+63, 0x1.7a316edaf462ep-836, INIT_U128( 0x0000000000000000, 0xd634c613ac699000 ) },
  { 0x1.8542412f0a848p+63, 0x1.a53fa9cd4a7f5p-836, INIT_U128( 0x0000000000000000, 0xc2a1209785424000 ) },
  { 0x1.f526fb77ea4ep+64, 0x1.170327882e065p-848, INIT_U128( 0x0000000000000001, 0xf526fb77ea4e0000 ) },
  { 0x1.acca54155994ap+64, 0x1.4c44fdb4988ap-848, INIT_U128( 0x0000000000000001, 0xacca54155994a000 ) },
  { 0x1.b47ed77768fdbp+64, 0x1.e6883245cd107p-848, INIT_U128( 0x0000000000000001, 0xb47ed77768fdb000 ) },
  { 0x1.bf32165b7e643p+64, 0x1.7da93100fb526p-848, INIT_U128( 0x0000000000000001, 0xbf32165b7e643000 ) },
  { 0x1.c6aa72a58d54fp+65, 0x1.700d04ece01ap-810, INIT_U128( 0x0000000000000003, 0x8d54e54b1aa9e000 ) },
  { 0x1.651ffffcca4p+65, 0x1.b6e3b8e56dc77p-810, INIT_U128( 0x0000000000000002, 0xca3ffff994800000 ) },
  { 0x1.f59076c9eb20fp+65, 0x1.41622b1082c46p-810, INIT_U128( 0x0000000000000003, 0xeb20ed93d641e000 ) },
  { 0x1.2362224a46c44p+65, 0x1.0fe4f0321fc9ep-810, INIT_U128( 0x0000000000000002, 0x46c444948d888000 ) },
  { 0x1.96643d852cc88p+66, 0x1.5aadaff0b55b6p-820, INIT_U128( 0x0000000000000006, 0x5990f614b3220000 ) },
  { 0x1.38a95f0e7152cp+66, 0x1.8432d89b0865bp-820, INIT_U128( 0x0000000000000004, 0xe2a57c39c54b0000 ) },
  { 0x1.b674a85b6ce95p+66, 0x1.3adbee1a75b7ep-820, INIT_U128( 0x0000000000000006, 0xd9d2a16db3a54000 ) },
  { 0x1.81b2bc3303658p+66, 0x1.0e771c4e1cee4p-820, INIT_U128( 0x0000000000000006, 0x06caf0cc0d960000 ) },
  { 0x1.017e066002fc1p+67, 0x1.69eb9d80d3d74p-860, INIT_U128( 0x0000000000000008, 0x0bf0330017e08000 ) },
  { 0x1.b75b9b136eb74p+67, 0x1.ddf2ec69bbe5ep-860, INIT_U128( 0x000000000000000d, 0xbadcd89b75ba0000 ) },
  { 0x1.71432fe4e2866p+67, 0x1.cbea0a3797d41p-860, INIT_U128( 0x000000000000000b, 0x8a197f2714330000 ) },
  { 0x1.65e3ce88cbc7ap+67, 0x1.dd466e4dba8cep-860, INIT_U128( 0x000000000000000b, 0x2f1e74465e3d0000 ) },
  { 0x1.d76842dfaed09p+68, 0x1.d4739f6ba8e74p-740, INIT_U128( 0x000000000000001d, 0x76842dfaed090000 ) },
  { 0x1.9180cb312301ap+68, 0x1.5961b442b2c36p-740, INIT_U128( 0x0000000000000019, 0x180cb312301a0000 ) },
  { 0x1.5ea7abd8bd4f6p+68, 0x1.0afd825415fbp-740, INIT_U128( 0x0000000000000015, 0xea7abd8bd4f60000 ) },
  { 0x1.bcf6493f79ec9p+68, 0x1.39f6643a73eccp-740, INIT_U128( 0x000000000000001b, 0xcf6493f79ec90000 ) },
  { 0x1.6c264bbad84cap+69, 0x1.3d2b92de7a572p-358, INIT_U128( 0x000000000000002d, 0x84c9775b09940000 ) },
  { 0x1.13b3e09a2767cp+69, 0x1.a3ead92f47d5bp-358, INIT_U128( 0x0000000000000022, 0x767c1344ecf80000 ) },
  { 0x1.8518219d0a304p+69, 0x1.c9a99edf93534p-358, INIT_U128( 0x0000000000000030, 0xa30433a146080000 ) },
  { 0x1.afa032e75f406p+69, 0x1.76f3e70cede7dp-358, INIT_U128( 0x0000000000000035, 0xf4065cebe80c0000 ) },
  { 0x1.1aa2f5343545ep+70, 0x1.cd612ccd9ac25p-491, INIT_U128( 0x0000000000000046, 0xa8bd4d0d51780000 ) },
  { 0x1.2c8c2e1a59186p+70, 0x1.53ac1260a7582p-491, INIT_U128( 0x000000000000004b, 0x230b869646180000 ) },
  { 0x1.b92d16ef725a3p+70, 0x1.05faddde0bf5cp-491, INIT_U128( 0x000000000000006e, 0x4b45bbdc968c0000 ) },
  { 0x1.9fc802a33f9p+70, 0x1.203a627a4074cp-491, INIT_U128( 0x0000000000000067, 0xf200a8cfe4000000 ) },
  { 0x1.240746b6480e9p+71, 0x1.78c39518f1872p-676, INIT_U128( 0x0000000000000092, 0x03a35b2407480000 ) },
  { 0x1.863a24750c744p+71, 0x1.96d2b31d2da56p-676, INIT_U128( 0x00000000000000c3, 0x1d123a863a200000 ) },
  { 0x1.597fbe8ab2ff8p+71, 0x1.93afb023275f6p-676, INIT_U128( 0x00000000000000ac, 0xbfdf45597fc00000 ) },
  { 0x1.e1080a67c2102p+71, 0x1.b5c9f2a36b93ep-676, INIT_U128( 0x00000000000000f0, 0x840533e108100000 ) },
  { 0x1.5c1897a6b8313p+72, 0x1.e08b1a6fc1164p-272, INIT_U128( 0x000000000000015c, 0x1897a6b831300000 ) },
  { 0x1.9ba232cf37446p+72, 0x1.5f66bf90becd8p-272, INIT_U128( 0x000000000000019b, 0xa232cf3744600000 ) },
  { 0x1.595f744cb2beep+72, 0x1.f7a95a67ef52cp-272, INIT_U128( 0x0000000000000159, 0x5f744cb2bee00000 ) },
  { 0x1.a763ae594ec76p+72, 0x1.c295524f852aap-272, INIT_U128( 0x00000000000001a7, 0x63ae594ec7600000 ) },
  { 0x1.06eca6c40dd95p+73, 0x1.f918431ff2309p-572, INIT_U128( 0x000000000000020d, 0xd94d881bb2a00000 ) },
  { 0x1.4f9fc82a9f3f9p+73, 0x1.257089f24ae11p-572, INIT_U128( 0x000000000000029f, 0x3f90553e7f200000 ) },
  { 0x1.0fa3bdc41f478p+73, 0x1.1ca9162039523p-572, INIT_U128( 0x000000000000021f, 0x477b883e8f000000 ) },
  { 0x1.be3be7ef7c77dp+73, 0x1.ae73d50d5ce7bp-572, INIT_U128( 0x000000000000037c, 0x77cfdef8efa00000 ) },
  { 0x1.da6d4389b4da8p+74, 0x1.1806570a300cbp-230, INIT_U128( 0x0000000000000769, 0xb50e26d36a000000 ) },
  { 0x1.55276624aa4edp+74, 0x1.004fb390009f6p-230, INIT_U128( 0x0000000000000554, 0x9d9892a93b400000 ) },
  { 0x1.aeab3c995d568p+74, 0x1.08d9156011b22p-230, INIT_U128( 0x00000000000006ba, 0xacf265755a000000 ) },
  { 0x1.a281549f4502ap+74, 0x1.cb98cbdf9731ap-230, INIT_U128( 0x000000000000068a, 0x05527d140a800000 ) },
  { 0x1.35dae4746bb5cp+75, 0x1.492edd3c925dcp-684, INIT_U128( 0x00000000000009ae, 0xd723a35dae000000 ) },
  { 0x1.e6b8db83cd71cp+75, 0x1.e8282f8fd0506p-684, INIT_U128( 0x0000000000000f35, 0xc6dc1e6b8e000000 ) },
  { 0x1.17587f082eb1p+75, 0x1.45ebb9f28bd77p-684, INIT_U128( 0x00000000000008ba, 0xc3f8417588000000 ) },
  { 0x1.957ac7292af59p+75, 0x1.35b408566b681p-684, INIT_U128( 0x0000000000000cab, 0xd6394957ac800000 ) },
  { 0x1.6e0e0850dc1c1p+76, 0x1.9e623e393cc48p-1002, INIT_U128( 0x00000000000016e0, 0xe0850dc1c1000000 ) },
  { 0x1.90fee6ff21fddp+76, 0x1.7ceca2caf9d94p-1002, INIT_U128( 0x000000000000190f, 0xee6ff21fdd000000 ) },
  { 0x1.5798708eaf30ep+76, 0x1.d3322f4fa6646p-1002, INIT_U128( 0x0000000000001579, 0x8708eaf30e000000 ) },
  { 0x1.c2ef5f4185decp+76, 0x1.96ad4d692d5aap-1002, INIT_U128( 0x0000000000001c2e, 0xf5f4185dec000000 ) },
  { 0x1.ae14b81f5c297p+77, 0x1.062f208c0c5e4p-169, INIT_U128( 0x00000000000035c2, 0x9703eb852e000000 ) },
  { 0x1.1f2ef58a3e5dep+77, 0x1.97a029192f405p-169, INIT_U128( 0x00000000000023e5, 0xdeb147cbbc000000 ) },
  { 0x1.74861d64e90c4p+77, 0x1.fb289c69f6513p-169, INIT_U128( 0x0000000000002e90, 0xc3ac9d2188000000 ) },
  { 0x1.11782bc422f06p+77, 0x1.fe294db5fc529p-169, INIT_U128( 0x000000000000222f, 0x0578845e0c000000 ) },
  { 0x1.3af34bd275e6ap+78, 0x1.ba66054574cc1p-910, INIT_U128( 0x0000000000004ebc, 0xd2f49d79a8000000 ) },
  { 0x1.0708e3fc0e11cp+78, 0x1.09ae142c135c2p-910, INIT_U128( 0x00000000000041c2, 0x38ff038470000000 ) },
  { 0x1.c313a69786275p+78, 0x1.fc165a27f82ccp-910, INIT_U128( 0x00000000000070c4, 0xe9a5e189d4000000 ) },
  { 0x1.990c9ad532193p+78, 0x1.072499060e493p-910, INIT_U128( 0x0000000000006643, 0x26b54c864c000000 ) },
  { 0x1.dd2363c1ba46cp+79, 0x1.d163c99ba2c79p-815, INIT_U128( 0x000000000000ee91, 0xb1e0dd2360000000 ) },
  { 0x1.b0ae4ad1615c9p+79, 0x1.8f2f90f91e5f2p-815, INIT_U128( 0x000000000000d857, 0x2568b0ae48000000 ) },
  { 0x1.9a26bbb7344d8p+79, 0x1.ed90d6d9db21bp-815, INIT_U128( 0x000000000000cd13, 0x5ddb9a26c0000000 ) },
  { 0x1.71ec17ace3d83p+79, 0x1.ec2b79cfd856fp-815, INIT_U128( 0x000000000000b8f6, 0x0bd671ec18000000 ) },
  { 0x1.689f9cb2d13f4p+80, 0x1.adfbd9175bf7bp-935, INIT_U128( 0x000000000001689f, 0x9cb2d13f40000000 ) },
  { 0x1.cde2888d9bc51p+80, 0x1.0d3598f01a6b3p-935, INIT_U128( 0x000000000001cde2, 0x888d9bc510000000 ) },
  { 0x1.6866c948d0cd9p+80, 0x1.0d9da3cc1b3b4p-935, INIT_U128( 0x0000000000016866, 0xc948d0cd90000000 ) },
  { 0x1.eee79cbdddcf3p+80, 0x1.a60bf9374c17fp-935, INIT_U128( 0x000000000001eee7, 0x9cbdddcf30000000 ) },
  { 0x1.3a27b29c744f6p+81, 0x1.1270039224ep-231, INIT_U128( 0x000000000002744f, 0x6538e89ec0000000 ) },
  { 0x1.608a6c38c114ep+81, 0x1.a6ff1b154dfe4p-231, INIT_U128( 0x000000000002c114, 0xd8718229c0000000 ) },
  { 0x1.b8ddf2c971bbep+81, 0x1.7a6c452cf4d88p-231, INIT_U128( 0x00000000000371bb, 0xe592e377c0000000 ) },
  { 0x1.34056f04680aep+81, 0x1.3aa39ba075474p-231, INIT_U128( 0x000000000002680a, 0xde08d015c0000000 ) },
  { 0x1.df36567dbe6cbp+82, 0x1.948aa54b29155p-104, INIT_U128( 0x0000000000077cd9, 0x59f6f9b2c0000000 ) },
  { 0x1.00c5bf20018b8p+82, 0x1.0354f44e06a9ep-104, INIT_U128( 0x0000000000040316, 0xfc80062e00000000 ) },
  { 0x1.5a6d471ab4da9p+82, 0x1.ea755ca5d4eabp-104, INIT_U128( 0x00000000000569b5, 0x1c6ad36a40000000 ) },
  { 0x1.58acff6eb15ap+82, 0x1.f6c3b1b9ed876p-104, INIT_U128( 0x00000000000562b3, 0xfdbac56800000000 ) },
  { 0x1.9288c20b25118p+83, 0x1.477be5208ef7cp-445, INIT_U128( 0x00000000000c9446, 0x1059288c00000000 ) },
  { 0x1.3556fa5c6aaep+83, 0x1.f200a591e4014p-445, INIT_U128( 0x000000000009aab7, 0xd2e3557000000000 ) },
  { 0x1.88dec0dd11bd8p+83, 0x1.a1ceac19439d6p-445, INIT_U128( 0x00000000000c46f6, 0x06e88dec00000000 ) },
  { 0x1.603498e4c0693p+83, 0x1.94ccf0d52999ep-445, INIT_U128( 0x00000000000b01a4, 0xc726034980000000 ) },
  { 0x1.1dfbb7a43bf77p+84, 0x1.d7dd8bdbafbb2p-926, INIT_U128( 0x000000000011dfbb, 0x7a43bf7700000000 ) },
  { 0x1.5f5d18b8beba3p+84, 0x1.ac1b923558372p-926, INIT_U128( 0x000000000015f5d1, 0x8b8beba300000000 ) },
  { 0x1.8b32b85d16657p+84, 0x1.37ae11cc6f5c2p-926, INIT_U128( 0x000000000018b32b, 0x85d1665700000000 ) },
  { 0x1.506f56eca0debp+84, 0x1.185445da30a88p-926, INIT_U128( 0x00000000001506f5, 0x6eca0deb00000000 ) },
  { 0x1.506cf3dea0d9ep+85, 0x1.2e68e2945cd1cp-635, INIT_U128( 0x00000000002a0d9e, 0x7bd41b3c00000000 ) },
  { 0x1.99ef268733de5p+85, 0x1.2ce0960e59c13p-635, INIT_U128( 0x0000000000333de4, 0xd0e67bca00000000 ) },
  { 0x1.d46cd273a8d9ap+85, 0x1.fb0bae61f6176p-635, INIT_U128( 0x00000000003a8d9a, 0x4e751b3400000000 ) },
  { 0x1.31deaa8263bd6p+85, 0x1.8a752b4514ea6p-635, INIT_U128( 0x0000000000263bd5, 0x504c77ac00000000 ) },
  { 0x1.95956f032b2aep+86, 0x1.ddd18753bba31p-535, INIT_U128( 0x000000000065655b, 0xc0cacab800000000 ) },
  { 0x1.db4b6705b696dp+86, 0x1.fc438061f887p-535, INIT_U128( 0x000000000076d2d9, 0xc16da5b400000000 ) },
  { 0x1.42c9320885926p+86, 0x1.9258f3ab24b1ep-535, INIT_U128( 0x000000000050b24c, 0x8221649800000000 ) },
  { 0x1.08ca2a0e11946p+86, 0x1.1c860974390c1p-535, INIT_U128( 0x000000000042328a, 0x8384651800000000 ) },
  { 0x1.ddf27f51bbe5p+87, 0x1.a776e8c94eeddp-479, INIT_U128( 0x0000000000eef93f, 0xa8ddf28000000000 ) },
  { 0x1.3f5b6af47eb6ep+87, 0x1.cf47dd9d9e8fcp-479, INIT_U128( 0x00000000009fadb5, 0x7a3f5b7000000000 ) },
  { 0x1.4b9d6480973acp+87, 0x1.c249100d84922p-479, INIT_U128( 0x0000000000a5ceb2, 0x404b9d6000000000 ) },
  { 0x1.eea053a5dd40bp+87, 0x1.26e6f8d24dcdfp-479, INIT_U128( 0x0000000000f75029, 0xd2eea05800000000 ) },
  { 0x1.4e0a84329c15p+88, 0x1.b969043772d2p-931, INIT_U128( 0x00000000014e0a84, 0x329c150000000000 ) },
  { 0x1.20fcf75c41f9fp+88, 0x1.a487f5bf490ffp-931, INIT_U128( 0x000000000120fcf7, 0x5c41f9f000000000 ) },
  { 0x1.0937a468126f4p+88, 0x1.dd15685bba2adp-931, INIT_U128( 0x00000000010937a4, 0x68126f4000000000 ) },
  { 0x1.3a1b94d674372p+88, 0x1.0e98a7441d315p-931, INIT_U128( 0x00000000013a1b94, 0xd674372000000000 ) },
  { 0x1.b9b702f1736ep+89, 0x1.afdf8f8b5fbf2p-758, INIT_U128( 0x0000000003736e05, 0xe2e6dc0000000000 ) },
  { 0x1.f93973b7f272fp+89, 0x1.77696130eed2cp-758, INIT_U128( 0x0000000003f272e7, 0x6fe4e5e000000000 ) },
  { 0x1.85595b3b0ab2cp+89, 0x1.4c1c459298388p-758, INIT_U128( 0x00000000030ab2b6, 0x7615658000000000 ) },
  { 0x1.89e4ed0b13c9ep+89, 0x1.4c88ddd89911cp-758, INIT_U128( 0x000000000313c9da, 0x162793c000000000 ) },
  { 0x1.6d0dfe6ada1cp+90, 0x1.0873b16610e76p+20, INIT_U128( 0x0000000005b437f9, 0xab6870000010873b ) },
  { 0x1.403e37e4807c7p+90, 0x1.630feeecc61fep+20, INIT_U128( 0x000000000500f8df, 0x9201f1c0001630fe ) },
  { 0x1.1d3cdade3a79cp+90, 0x1.ecceb30bd99d7p+20, INIT_U128( 0x000000000474f36b, 0x78e9e700001ecceb ) },
  { 0x1.f50698adea0d3p+90, 0x1.4ed5f3749dabep+20, INIT_U128( 0x0000000007d41a62, 0xb7a834c00014ed5f ) },
  { 0x1.d869b87fb0d37p+91, 0x1.53d108a0a7a21p-525, INIT_U128( 0x000000000ec34dc3, 0xfd869b8000000000 ) },
  { 0x1.cd078fb39a0f2p+91, 0x1.228b56084516bp-525, INIT_U128( 0x000000000e683c7d, 0x9cd0790000000000 ) },
  { 0x1.c67dbd798cfb7p+91, 0x1.65e1ed28cbc3ep-525, INIT_U128( 0x000000000e33edeb, 0xcc67db8000000000 ) },
  { 0x1.c41a2ed388346p+91, 0x1.4799717a8f32ep-525, INIT_U128( 0x000000000e20d176, 0x9c41a30000000000 ) },
  { 0x1.db416739b682dp+92, 0x1.fea650affd4cap-706, INIT_U128( 0x000000001db41673, 0x9b682d0000000000 ) },
  { 0x1.872189f10e431p+92, 0x1.e1de445fc3bc8p-706, INIT_U128( 0x000000001872189f, 0x10e4310000000000 ) },
  { 0x1.d2ad0fd1a55a2p+92, 0x1.0a06ac7e140d6p-706, INIT_U128( 0x000000001d2ad0fd, 0x1a55a20000000000 ) },
  { 0x1.2535c3a84a6b8p+92, 0x1.65205312ca40ap-706, INIT_U128( 0x0000000012535c3a, 0x84a6b80000000000 ) },
  { 0x1.dd518ce9baa31p+93, 0x1.b123211362464p-663, INIT_U128( 0x000000003baa319d, 0x3754620000000000 ) },
  { 0x1.9a89da733513bp+93, 0x1.8c0a0b2b18142p-663, INIT_U128( 0x0000000033513b4e, 0x66a2760000000000 ) },
  { 0x1.01fc693203f8dp+93, 0x1.81040a6d02081p-663, INIT_U128( 0x00000000203f8d26, 0x407f1a0000000000 ) },
  { 0x1.1d9ae72e3b35dp+93, 0x1.fb816ce9f702dp-663, INIT_U128( 0x0000000023b35ce5, 0xc766ba0000000000 ) },
  { 0x1.1b8311c037062p+94, 0x1.3b413a6c76828p-946, INIT_U128( 0x0000000046e0c470, 0x0dc1880000000000 ) },
  { 0x1.e7c1239fcf825p+94, 0x1.e89cbe61d1398p-946, INIT_U128( 0x0000000079f048e7, 0xf3e0940000000000 ) },
  { 0x1.3ba2c92677459p+94, 0x1.4c19bb4098338p-946, INIT_U128( 0x000000004ee8b249, 0x9dd1640000000000 ) },
  { 0x1.e81787a3d02f1p+94, 0x1.21981a6a43304p-946, INIT_U128( 0x000000007a05e1e8, 0xf40bc40000000000 ) },
  { 0x1.e18ca9a3c3195p+95, 0x1.d2ce473ba59c9p-353, INIT_U128( 0x00000000f0c654d1, 0xe18ca80000000000 ) },
  { 0x1.ec1a5457d834ap+95, 0x1.673af5f4ce75ep-353, INIT_U128( 0x00000000f60d2a2b, 0xec1a500000000000 ) },
  { 0x1.3d4fcc727a9fap+95, 0x1.d577142daaee2p-353, INIT_U128( 0x000000009ea7e639, 0x3d4fd00000000000 ) },
  { 0x1.6b318358d663p+95, 0x1.d96b9445b2d72p-353, INIT_U128( 0x00000000b598c1ac, 0x6b31800000000000 ) },
  { 0x1.f8fedfa1f1fdcp+96, 0x1.6e54dd28dca9cp-828, INIT_U128( 0x00000001f8fedfa1, 0xf1fdc00000000000 ) },
  { 0x1.4b5ec85896bd9p+96, 0x1.e4251105c84a2p-828, INIT_U128( 0x000000014b5ec858, 0x96bd900000000000 ) },
  { 0x1.bc8f0397791ep+96, 0x1.b514998d6a293p-828, INIT_U128( 0x00000001bc8f0397, 0x791e000000000000 ) },
  { 0x1.056a2dfe0ad46p+96, 0x1.c98d1a15931a3p-828, INIT_U128( 0x00000001056a2dfe, 0x0ad4600000000000 ) },
  { 0x1.21192da842326p+97, 0x1.64dbdea4c9b7cp-314, INIT_U128( 0x0000000242325b50, 0x8464c00000000000 ) },
  { 0x1.95c269f12b84dp+97, 0x1.31bfb62e637f7p-314, INIT_U128( 0x000000032b84d3e2, 0x5709a00000000000 ) },
  { 0x1.f4a37e69e947p+97, 0x1.2c5cc8d658b99p-314, INIT_U128( 0x00000003e946fcd3, 0xd28e000000000000 ) },
  { 0x1.73c765e6e78ecp+97, 0x1.34b3a3f469674p-314, INIT_U128( 0x00000002e78ecbcd, 0xcf1d800000000000 ) },
  { 0x1.cafa773f95f4fp+98, 0x1.28d3338851a66p-207, INIT_U128( 0x000000072be9dcfe, 0x57d3c00000000000 ) },
  { 0x1.7b7b6aa4f6f6ep+98, 0x1.2c99252259324p-207, INIT_U128( 0x00000005ededaa93, 0xdbdb800000000000 ) },
  { 0x1.8c39a15918734p+98, 0x1.cdcd60b79b9acp-207, INIT_U128( 0x0000000630e68564, 0x61cd000000000000 ) },
  { 0x1.da220eb5b4442p+98, 0x1.cd0b52b19a16bp-207, INIT_U128( 0x0000000768883ad6, 0xd110800000000000 ) },
  { 0x1.0f34e6fa1e69dp+99, 0x1.596be5d4b2d7cp-575, INIT_U128( 0x0000000879a737d0, 0xf34e800000000000 ) },
  { 0x1.5318af1aa6316p+99, 0x1.b2d7a70f65af5p-575, INIT_U128( 0x0000000a98c578d5, 0x318b000000000000 ) },
  { 0x1.2f3f14005e7e2p+99, 0x1.bf1a11977e342p-575, INIT_U128( 0x0000000979f8a002, 0xf3f1000000000000 ) },
  { 0x1.c95da71792bb5p+99, 0x1.b673d5896ce7bp-575, INIT_U128( 0x0000000e4aed38bc, 0x95da800000000000 ) },
  { 0x1.fdead1dffbd5ap+100, 0x1.5a750c74b4ea2p-189, INIT_U128( 0x0000001fdead1dff, 0xbd5a000000000000 ) },
  { 0x1.fe8116bbfd023p+100, 0x1.c0314d4580629p-189, INIT_U128( 0x0000001fe8116bbf, 0xd023000000000000 ) },
  { 0x1.9177917922ef2p+100, 0x1.b5d94c0f6bb2ap-189, INIT_U128( 0x0000001917791792, 0x2ef2000000000000 ) },
  { 0x1.39ab1bf273564p+100, 0x1.4de96f889bd2ep-189, INIT_U128( 0x000000139ab1bf27, 0x3564000000000000 ) },
  { 0x1.52e21a8aa5c44p+101, 0x0.0000075bfbb25p-1022, INIT_U128( 0x0000002a5c435154, 0xb888000000000000 ) },
  { 0x1.c362c16186c58p+101, 0x0.00000417370ap-1022, INIT_U128( 0x000000386c582c30, 0xd8b0000000000000 ) },
  { 0x1.dcbb50a5b976ap+101, 0x0.000005dcb4b01p-1022, INIT_U128( 0x0000003b976a14b7, 0x2ed4000000000000 ) },
  { 0x1.fae77febf5cfp+101, 0x0.00000567e1492p-1022, INIT_U128( 0x0000003f5ceffd7e, 0xb9e0000000000000 ) },
  { 0x1.b94feaf3729fdp+102, 0x1.eb191b6bd6324p-32, INIT_U128( 0x0000006e53fabcdc, 0xa7f4000000000000 ) },
  { 0x1.ffcd483fff9a9p+102, 0x1.f6c951e5ed92ap-32, INIT_U128( 0x0000007ff3520fff, 0xe6a4000000000000 ) },
  { 0x1.f4a75991e94ebp+102, 0x1.5d81e52ebb03cp-32, INIT_U128( 0x0000007d29d6647a, 0x53ac000000000000 ) },
  { 0x1.d9014a81b202ap+102, 0x1.1d1a5c4c3a34cp-32, INIT_U128( 0x000000764052a06c, 0x80a8000000000000 ) },
  { 0x1.d33430cda6686p+103, 0x1.58855d10b10acp-383, INIT_U128( 0x000000e99a1866d3, 0x3430000000000000 ) },
  { 0x1.5e801048bd002p+103, 0x1.54a52a72a94a6p-383, INIT_U128( 0x000000af4008245e, 0x8010000000000000 ) },
  { 0x1.590a254ab2144p+103, 0x1.0307881c060f1p-383, INIT_U128( 0x000000ac8512a559, 0x0a20000000000000 ) },
  { 0x1.6dfaecf0dbf5ep+103, 0x1.54a4b130a9496p-383, INIT_U128( 0x000000b6fd76786d, 0xfaf0000000000000 ) },
  { 0x1.011eb1b0023d6p+104, 0x1.39df71da73beep-20, INIT_U128( 0x000001011eb1b002, 0x3d60000000000000 ) },
  { 0x1.5ee2142ebdc42p+104, 0x1.43553d9086aa8p-20, INIT_U128( 0x0000015ee2142ebd, 0xc420000000000000 ) },
  { 0x1.100502e4200ap+104, 0x1.534d9720a69b3p-20, INIT_U128( 0x000001100502e420, 0x0a00000000000000 ) },
  { 0x1.2aefc5ac55df8p+104, 0x1.f553d14beaa7ap-20, INIT_U128( 0x0000012aefc5ac55, 0xdf80000000000000 ) },
  { 0x1.d0559891a0ab3p+105, 0x1.87b2239d0f644p-316, INIT_U128( 0x000003a0ab312341, 0x5660000000000000 ) },
  { 0x1.0929084e12521p+105, 0x1.e345fe55c68cp-316, INIT_U128( 0x0000021252109c24, 0xa420000000000000 ) },
  { 0x1.70c26be4e184ep+105, 0x1.7a6011c6f4c02p-316, INIT_U128( 0x000002e184d7c9c3, 0x09c0000000000000 ) },
  { 0x1.8c345d471868cp+105, 0x1.b4a5d903694bbp-316, INIT_U128( 0x0000031868ba8e30, 0xd180000000000000 ) },
  { 0x1.66bf48cecd7e9p+106, 0x1.d926e3a9b24dcp-461, INIT_U128( 0x0000059afd233b35, 0xfa40000000000000 ) },
  { 0x1.caf9648595f2cp+106, 0x1.3edd8e587dbb2p-461, INIT_U128( 0x0000072be5921657, 0xcb00000000000000 ) },
  { 0x1.0ff9c5341ff38p+106, 0x1.5ecf4954bd9e9p-461, INIT_U128( 0x0000043fe714d07f, 0xce00000000000000 ) },
  { 0x1.010bbf3602178p+106, 0x1.90183dfb20308p-461, INIT_U128( 0x000004042efcd808, 0x5e00000000000000 ) },
  { 0x1.90f1198321e23p+107, 0x1.18ec849e31d9p-508, INIT_U128( 0x00000c8788cc190f, 0x1180000000000000 ) },
  { 0x1.88876447110ecp+107, 0x1.3dbc4a967b78ap-508, INIT_U128( 0x00000c443b223888, 0x7600000000000000 ) },
  { 0x1.e955d221d2abap+107, 0x1.1f1f0f983e3e2p-508, INIT_U128( 0x00000f4aae910e95, 0x5d00000000000000 ) },
  { 0x1.79931aacf3264p+107, 0x1.5c6d47c2b8da9p-508, INIT_U128( 0x00000bcc98d56799, 0x3200000000000000 ) },
  { 0x1.49a28e4c93452p+108, 0x1.0819b1c610336p-949, INIT_U128( 0x0000149a28e4c934, 0x5200000000000000 ) },
  { 0x1.b61387f56c271p+108, 0x1.fda021a7fb404p-949, INIT_U128( 0x00001b61387f56c2, 0x7100000000000000 ) },
  { 0x1.8364b71506c97p+108, 0x1.972f04852e5ep-949, INIT_U128( 0x000018364b71506c, 0x9700000000000000 ) },
  { 0x1.a902b10952056p+108, 0x1.aee9cfcf5dd3ap-949, INIT_U128( 0x00001a902b109520, 0x5600000000000000 ) },
  { 0x1.d96f1c29b2de4p+109, 0x1.8415bcdf082b8p-695, INIT_U128( 0x00003b2de385365b, 0xc800000000000000 ) },
  { 0x1.e8dfe3f9d1bfcp+109, 0x1.c31bb2ed86377p-695, INIT_U128( 0x00003d1bfc7f3a37, 0xf800000000000000 ) },
  { 0x1.3a0cc61474199p+109, 0x1.06f2795a0de4fp-695, INIT_U128( 0x0000274198c28e83, 0x3200000000000000 ) },
  { 0x1.1f3e5f4a3e7ccp+109, 0x1.d7cc8b85af992p-695, INIT_U128( 0x000023e7cbe947cf, 0x9800000000000000 ) },
  { 0x1.dc6af865b8d5fp+110, 0x1.b5d225136ba45p-22, INIT_U128( 0x0000771abe196e35, 0x7c00000000000000 ) },
  { 0x1.4ca9fcc69954p+110, 0x1.eaa024f7d5404p-22, INIT_U128( 0x0000532a7f31a655, 0x0000000000000000 ) },
  { 0x1.414c72988298ep+110, 0x1.a488753b4910fp-22, INIT_U128( 0x000050531ca620a6, 0x3800000000000000 ) },
  { 0x1.dc6b39dbb8d67p+110, 0x1.068d04580d1ap-22, INIT_U128( 0x0000771ace76ee35, 0x9c00000000000000 ) },
  { 0x1.74e50e90e9ca2p+111, 0x1.3f289f007e514p-487, INIT_U128( 0x0000ba72874874e5, 0x1000000000000000 ) },
  { 0x1.5f0a2632be145p+111, 0x1.f790c8d9ef219p-487, INIT_U128( 0x0000af8513195f0a, 0x2800000000000000 ) },
  { 0x1.d422bf03a8458p+111, 0x1.fbacd695f759bp-487, INIT_U128( 0x0000ea115f81d422, 0xc000000000000000 ) },
  { 0x1.d9b5cefdb36bap+111, 0x1.c968397b92d07p-487, INIT_U128( 0x0000ecdae77ed9b5, 0xd000000000000000 ) },
  { 0x1.f3e78001e7cfp+112, 0x1.4b4970649692ep-744, INIT_U128( 0x0001f3e78001e7cf, 0x0000000000000000 ) },
  { 0x1.17c200042f84p+112, 0x1.8d8e8a711b1d1p-744, INIT_U128( 0x000117c200042f84, 0x0000000000000000 ) },
  { 0x1.0fe4fde81fcap+112, 0x1.256616f84acc3p-744, INIT_U128( 0x00010fe4fde81fca, 0x0000000000000000 ) },
  { 0x1.f2773487e4ee6p+112, 0x1.075dd87a0ebbbp-744, INIT_U128( 0x0001f2773487e4ee, 0x6000000000000000 ) },
  { 0x1.7f2646f4fe4c9p+113, 0x1.bde0f7d57bc1fp-552, INIT_U128( 0x0002fe4c8de9fc99, 0x2000000000000000 ) },
  { 0x1.552a71c4aa54ep+113, 0x1.5f75cb84beebap-552, INIT_U128( 0x0002aa54e38954a9, 0xc000000000000000 ) },
  { 0x1.016211fe02c42p+113, 0x1.755a3ebeeab48p-552, INIT_U128( 0x000202c423fc0588, 0x4000000000000000 ) },
  { 0x1.2da4a1405b494p+113, 0x1.c7b5a5b78f6b5p-552, INIT_U128( 0x00025b494280b692, 0x8000000000000000 ) },
  { 0x1.454dcac88a9bap+114, 0x1.39e2f64073c5fp-541, INIT_U128( 0x000515372b222a6e, 0x8000000000000000 ) },
  { 0x1.f5769ad1eaed4p+114, 0x1.273e2b1a4e7c6p-541, INIT_U128( 0x0007d5da6b47abb5, 0x0000000000000000 ) },
  { 0x1.52bf04cca57ep+114, 0x1.2d92613e5b24cp-541, INIT_U128( 0x00054afc133295f8, 0x0000000000000000 ) },
  { 0x1.1b84a7fc37095p+114, 0x1.8b3360311666cp-541, INIT_U128( 0x00046e129ff0dc25, 0x4000000000000000 ) },
  { 0x1.82ef082b05de1p+115, 0x1.5205f44aa40bep-62, INIT_U128( 0x000c177841582ef0, 0x8000000000000000 ) },
  { 0x1.bafcbdf175f98p+115, 0x1.a9ea203153d44p-62, INIT_U128( 0x000dd7e5ef8bafcc, 0x0000000000000000 ) },
  { 0x1.64b56bd8c96aep+115, 0x1.42c3a3da85874p-62, INIT_U128( 0x000b25ab5ec64b57, 0x0000000000000000 ) },
  { 0x1.55c9b3b6ab936p+115, 0x1.48f9d11a91f3ap-62, INIT_U128( 0x000aae4d9db55c9b, 0x0000000000000000 ) },
  { 0x1.5321473ca6429p+116, 0x1.181db1c6303b6p-92, INIT_U128( 0x0015321473ca6429, 0x0000000000000000 ) },
  { 0x1.cdc20a3f9b841p+116, 0x1.50077ebca00fp-92, INIT_U128( 0x001cdc20a3f9b841, 0x0000000000000000 ) },
  { 0x1.92fee4f925fdcp+116, 0x1.028f8f16051f2p-92, INIT_U128( 0x00192fee4f925fdc, 0x0000000000000000 ) },
  { 0x1.1bab55e43756ap+116, 0x1.40855e9c810acp-92, INIT_U128( 0x0011bab55e43756a, 0x0000000000000000 ) },
  { 0x1.db43e365b687dp+117, 0x1.41baf8828375fp-732, INIT_U128( 0x003b687c6cb6d0fa, 0x0000000000000000 ) },
  { 0x1.a552a1674aa54p+117, 0x1.baa689f9754d1p-732, INIT_U128( 0x0034aa542ce954a8, 0x0000000000000000 ) },
  { 0x1.fe56ea4dfcadep+117, 0x1.5684197ead083p-732, INIT_U128( 0x003fcadd49bf95bc, 0x0000000000000000 ) },
  { 0x1.6c9b7a08d937p+117, 0x1.2337f65a466ffp-732, INIT_U128( 0x002d936f411b26e0, 0x0000000000000000 ) },
  { 0x1.076705de0ecep+118, 0x1.98bce7cb3179dp-844, INIT_U128( 0x0041d9c17783b380, 0x0000000000000000 ) },
  { 0x1.9b22f3533645ep+118, 0x1.74f31c32e9e64p-844, INIT_U128( 0x0066c8bcd4cd9178, 0x0000000000000000 ) },
  { 0x1.c1d136e583a27p+118, 0x1.767b76ceecf6fp-844, INIT_U128( 0x0070744db960e89c, 0x0000000000000000 ) },
  { 0x1.c885934b910b3p+118, 0x1.cd65a24d9acb4p-844, INIT_U128( 0x00722164d2e442cc, 0x0000000000000000 ) },
  { 0x1.fb2c8085f659p+119, 0x1.4485cbfc890bap-269, INIT_U128( 0x00fd964042fb2c80, 0x0000000000000000 ) },
  { 0x1.78eebbd0f1dd8p+119, 0x1.373c5ea66e78cp-269, INIT_U128( 0x00bc775de878eec0, 0x0000000000000000 ) },
  { 0x1.1cbf9dfc397f4p+119, 0x1.3a787bfa74f1p-269, INIT_U128( 0x008e5fcefe1cbfa0, 0x0000000000000000 ) },
  { 0x1.0b37d4fc166fap+119, 0x1.3ec2f1ca7d85ep-269, INIT_U128( 0x00859bea7e0b37d0, 0x0000000000000000 ) },
  { 0x1.f7d41d9befa83p+119, 0x1.737e360ee6fc7p+45, INIT_U128( 0x00fbea0ecdf7d418, 0x00002e6fc6c1dcdf ) },
  { 0x1.5f9adea4bf35cp+119, 0x1.f2a34d73e5469p+45, INIT_U128( 0x00afcd6f525f9ae0, 0x00003e5469ae7ca8 ) },
  { 0x1.1625a2842c4b4p+119, 0x1.dffa968fbff53p+45, INIT_U128( 0x008b12d1421625a0, 0x00003bff52d1f7fe ) },
  { 0x1.77b9ea5cef73ep+119, 0x1.03d8c57207b18p+45, INIT_U128( 0x00bbdcf52e77b9f0, 0x0000207b18ae40f6 ) },
  { 0x1.80c91d3d01924p+119, 0x1.dc2e84abb85dp+45, INIT_U128( 0x00c0648e9e80c920, 0x00003b85d095770b ) },
  { 0x1.a13e2abb427c5p+119, 0x1.be5baad37cb75p+45, INIT_U128( 0x00d09f155da13e28, 0x000037cb755a6f96 ) },
  { 0x1.3d0a39607a147p+119, 0x1.5d047beeba09p+45, INIT_U128( 0x009e851cb03d0a38, 0x00002ba08f7dd741 ) },
  { 0x1.e914f2edd229fp+119, 0x1.a87d6b5150faep+45, INIT_U128( 0x00f48a7976e914f8, 0x0000350fad6a2a1f ) },
  { 0x1.1bfba16037f74p+119, 0x1.f1667f23e2cdp+45, INIT_U128( 0x008dfdd0b01bfba0, 0x00003e2ccfe47c59 ) },
  { 0x1.e409cc97c8139p+119, 0x1.760ec180ec1d8p+45, INIT_U128( 0x00f204e64be409c8, 0x00002ec1d8301d83 ) },
  { 0x1.56095810ac12bp+119, 0x1.fd72c88bfae59p+45, INIT_U128( 0x00ab04ac08560958, 0x00003fae59117f5c ) },
  { 0x1.014b8dfe02972p+119, 0x1.5051ec7ca0a3ep+45, INIT_U128( 0x0080a5c6ff014b90, 0x00002a0a3d8f9414 ) },
  { 0x1.2c8071285900ep+119, 0x1.2df331a65be66p+45, INIT_U128( 0x00964038942c8070, 0x000025be6634cb7c ) },
  { 0x1.ec4edc7bd89dbp+119, 0x1.5689910cad132p+45, INIT_U128( 0x00f6276e3dec4ed8, 0x00002ad1322195a2 ) },
  { 0x1.acab22af59564p+120, 0x1.4302395486047p-789, INIT_U128( 0x01acab22af595640, 0x0000000000000000 ) },
  { 0x1.6d777264daeeep+120, 0x1.a94793a7528f2p-789, INIT_U128( 0x016d777264daeee0, 0x0000000000000000 ) },
  { 0x1.179807f22f301p+120, 0x1.f73f61cbee7ecp-789, INIT_U128( 0x01179807f22f3010, 0x0000000000000000 ) },
  { 0x1.2306ea1a460dep+120, 0x1.7f322c14fe646p-789, INIT_U128( 0x012306ea1a460de0, 0x0000000000000000 ) },
  { 0x1.4b89313897126p+121, 0x1.24f0cf5649e1ap-97, INIT_U128( 0x02971262712e24c0, 0x0000000000000000 ) },
  { 0x1.977d3ba32efa8p+121, 0x1.8d5ae3291ab5cp-97, INIT_U128( 0x032efa77465df500, 0x0000000000000000 ) },
  { 0x1.4951cf1a92a3ap+121, 0x1.ff3e4eaffe7cap-97, INIT_U128( 0x0292a39e35254740, 0x0000000000000000 ) },
  { 0x1.6487918ac90f2p+121, 0x1.8e5a3b671cb48p-97, INIT_U128( 0x02c90f2315921e40, 0x0000000000000000 ) },
  { 0x1.a71b395f4e367p+122, 0x1.4851ea7e90a3ep-824, INIT_U128( 0x069c6ce57d38d9c0, 0x0000000000000000 ) },
  { 0x1.58097738b012fp+122, 0x1.a4e92d4749d26p-824, INIT_U128( 0x056025dce2c04bc0, 0x0000000000000000 ) },
  { 0x1.431d2f4a863a6p+122, 0x1.ff84386fff087p-824, INIT_U128( 0x050c74bd2a18e980, 0x0000000000000000 ) },
  { 0x1.9f1393ad3e272p+122, 0x1.d7c0a6ddaf815p-824, INIT_U128( 0x067c4e4eb4f89c80, 0x0000000000000000 ) },
  { 0x1.8983ab7913076p+123, 0x1.fa01bdc9f4037p-825, INIT_U128( 0x0c4c1d5bc8983b00, 0x0000000000000000 ) },
  { 0x1.f3551e67e6aa4p+123, 0x1.029c51f00538ap-825, INIT_U128( 0x0f9aa8f33f355200, 0x0000000000000000 ) },
  { 0x1.a9da0d1d53b42p+123, 0x1.b5c718d36b8e3p-825, INIT_U128( 0x0d4ed068ea9da100, 0x0000000000000000 ) },
  { 0x1.95c79e2b2b8f4p+123, 0x1.384e4c32709cap-825, INIT_U128( 0x0cae3cf1595c7a00, 0x0000000000000000 ) },
  { 0x1.73af4650e75e9p+124, 0x1.ec4b7265d896fp-92, INIT_U128( 0x173af4650e75e900, 0x0000000000000000 ) },
  { 0x1.5e15a3a0bc2b4p+124, 0x1.2ff859825ff0bp-92, INIT_U128( 0x15e15a3a0bc2b400, 0x0000000000000000 ) },
  { 0x1.4dbe24a69b7c4p+124, 0x1.f5b685e5eb6d1p-92, INIT_U128( 0x14dbe24a69b7c400, 0x0000000000000000 ) },
  { 0x1.cd65ac439acb6p+124, 0x1.164d910a2c9b2p-92, INIT_U128( 0x1cd65ac439acb600, 0x0000000000000000 ) },
  { 0x1.f5bbadd5eb775p+125, 0x1.b16aeddb62d5ep-83, INIT_U128( 0x3eb775babd6eea00, 0x0000000000000000 ) },
  { 0x1.b66841396cd08p+125, 0x1.f9957d69f32afp-83, INIT_U128( 0x36cd08272d9a1000, 0x0000000000000000 ) },
  { 0x1.05be50760b7cap+125, 0x1.f9ebf8ddf3d7fp-83, INIT_U128( 0x20b7ca0ec16f9400, 0x0000000000000000 ) },
  { 0x1.51312262a2624p+125, 0x1.11c69084238d2p-83, INIT_U128( 0x2a26244c544c4800, 0x0000000000000000 ) },
  { 0x1.30eab12c61d56p+126, 0x1.f1e0e851e3c1dp-773, INIT_U128( 0x4c3aac4b18755800, 0x0000000000000000 ) },
  { 0x1.ac34bcf158698p+126, 0x1.a55b126f4ab62p-773, INIT_U128( 0x6b0d2f3c561a6000, 0x0000000000000000 ) },
  { 0x1.02246a040448ep+126, 0x1.61a83652c3507p-773, INIT_U128( 0x40891a8101123800, 0x0000000000000000 ) },
  { 0x1.ae881f955d104p+126, 0x1.c694aaa18d296p-773, INIT_U128( 0x6ba207e557441000, 0x0000000000000000 ) },
  { 0x1.08016b641002ep+127, 0x1.23a890d047512p-406, INIT_U128( 0x8400b5b208017000, 0x0000000000000000 ) },
  { 0x1.3d90e7327b21dp+127, 0x1.0aa664c0154ccp-406, INIT_U128( 0x9ec873993d90e800, 0x0000000000000000 ) },
  { 0x1.ff931617ff263p+127, 0x1.47d5e2a08fabcp-406, INIT_U128( 0xffc98b0bff931800, 0x0000000000000000 ) },
  { 0x1.f565d1dfeacbap+127, 0x1.827bc0c304f78p-406, INIT_U128( 0xfab2e8eff565d000, 0x0000000000000000 ) },
  { 0x1.fe1985fbfc33p+128, 0x1.a6e724cd4dce4p-1021, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.54709ed6a8e14p+128, 0x1.74a32d1ce9466p-1021, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.a8407dcf5081p+128, 0x1.ffb065f3ff60cp-1021, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.7a1cfcf2f43ap+130, 0x1.84bb5ea30976cp-548, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.6bddaae8d7bb6p+194, 0x1.de64c609bcc99p-754, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.5c6c29d4b8d85p+146, 0x1.0644f6d60c89fp-345, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.c6ce8bcf8d9d2p+158, 0x1.443b2e7088766p-26, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.747f6aa0e8feep+168, 0x1.fe663ae9fccc8p-143, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.26ee550a4ddcap+173, 0x1.951f708f2a3eep-45, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.3ecfe0ea7d9fcp+189, 0x1.9cb53593396a7p-619, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.9d394a313a729p+190, 0x1.9ecfff853dap-150, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.244a568a4894bp+241, 0x1.e23f42d7c47e9p-956, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.b564bb276ac98p+398, 0x1.7af8d1ccf5f1ap+215, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.ee06b389dc0d7p+468, 0x1.b54fa74f6a9f5p+322, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.ac0045bd58009p+588, 0x1.ba99775b7532fp-365, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.692078cad240fp+661, 0x1.bc4a88a978951p+356, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.be1e09dd7c3c1p+765, 0x1.428bc5dc85178p-848, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.5f5554c4beaaap+853, 0x1.a9e23bdd53c48p-807, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.748e96ace91d3p+993, 0x1.ab54ab4b56a96p-22, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { 0x1.0629e7380c53dp+1023, 0x1.c0b4b15581696p+323, INIT_U128( 0xffffffffffffffff, 0xffffffffffffffff ) },
  { INFINITY, 0x1.21bff4bc437fep-333, ((__uint128_t) 0xffffffffffffffff << 64) | 0xffffffffffffffff },
  { INFINITY, 0x1.47e9a0228fd34p-333, ((__uint128_t) 0xffffffffffffffff << 64) | 0xffffffffffffffff },
  { 0x1.b88939bd71127p+0, -0x1.58af6e36b15eep-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.f17b9de9e2f74p+0, -0x1.b03dc0f5607b8p-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.86c3f1ed0d87ep+0, -0x1.e41d5187c83aap-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.0180f91c0301fp+0, -0x1.f92c6b8bf258ep-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.8b3f17cd167e3p+0, -0x1.72dddf20e5bbcp-53, INIT_U128( 0x0, 0x1 ) },
  { 0x1.07f13dc20fe28p+1, -0x1.4af7122695ee2p-52, INIT_U128( 0x0, 0x2 ) },
  { 0x1.5296871aa52d1p+1, -0x1.0301d39e0603ap-52, INIT_U128( 0x0, 0x2 ) },
  { 0x1.c5390b7f8a722p+1, -0x1.ebbca5dfd7795p-52, INIT_U128( 0x0, 0x3 ) },
  { 0x1.446c141688d82p+1, -0x1.648f94aac91f2p-52, INIT_U128( 0x0, 0x2 ) },
  { 0x1.14cd55a0299aap+1, -0x1.4f479c509e8f4p-52, INIT_U128( 0x0, 0x2 ) },
  { 0x1.3364edd666c9ep+2, -0x1.723247eee4649p-51, INIT_U128( 0x0, 0x4 ) },
  { 0x1.6b5eb37ad6bd6p+2, -0x1.536fde56a6dfcp-51, INIT_U128( 0x0, 0x5 ) },
  { 0x1.c64dbd798c9b7p+2, -0x1.8e4392db1c872p-51, INIT_U128( 0x0, 0x7 ) },
  { 0x1.d93aa501b2754p+2, -0x1.9f5d97f33ebb3p-51, INIT_U128( 0x0, 0x7 ) },
  { 0x1.a25f6b9344beep+2, -0x1.8bd0d71f17a1bp-51, INIT_U128( 0x0, 0x6 ) },
  { 0x1.778d7cfeef1b0p+3, -0x1.8d1c08d11a381p-50, INIT_U128( 0x0, 0xb ) },
  { 0x1.f1ab9d7de3573p+3, -0x1.2aaf71b2555eep-50, INIT_U128( 0x0, 0xf ) },
  { 0x1.79f70e04f3ee2p+3, -0x1.391c62da7238cp-50, INIT_U128( 0x0, 0xb ) },
  { 0x1.ab4eaacb569d5p+3, -0x1.555629c6aaac5p-50, INIT_U128( 0x0, 0xd ) },
  { 0x1.a8c6081f518c1p+3, -0x1.2b722c7056e46p-50, INIT_U128( 0x0, 0xd ) },
  { 0x1.b9b5bf4b736b8p+4, -0x1.9ded29933bda5p-49, INIT_U128( 0x0, 0x1b ) },
  { 0x1.3d4ac1867a958p+4, -0x1.b426d34d684dap-49, INIT_U128( 0x0, 0x13 ) },
  { 0x1.3ff7aa6c7fef6p+4, -0x1.7cca4048f9948p-49, INIT_U128( 0x0, 0x13 ) },
  { 0x1.09e6491a13cc9p+4, -0x1.2c9d607e593acp-49, INIT_U128( 0x0, 0x10 ) },
  { 0x1.574cd232ae99ap+4, -0x1.655ffb9acac00p-49, INIT_U128( 0x0, 0x15 ) },
  { 0x1.9e7c5e773cf8cp+5, -0x1.a4151463482a2p-48, INIT_U128( 0x0, 0x33 ) },
  { 0x1.f96b3e2bf2d68p+5, -0x1.60e1bee6c1c38p-48, INIT_U128( 0x0, 0x3f ) },
  { 0x1.8c53eb5918a7ep+5, -0x1.6dc39116db872p-48, INIT_U128( 0x0, 0x31 ) },
  { 0x1.0bb0baaa17618p+5, -0x1.4e97ccbc9d2fap-48, INIT_U128( 0x0, 0x21 ) },
  { 0x1.02bea008057d4p+5, -0x1.97bb9f5f2f774p-48, INIT_U128( 0x0, 0x20 ) },
  { 0x1.1485dfa8290bcp+6, -0x1.6fe84d7cdfd0ap-47, INIT_U128( 0x0, 0x45 ) },
  { 0x1.9c85adfd390b6p+6, -0x1.6dba2a66db746p-47, INIT_U128( 0x0, 0x67 ) },
  { 0x1.d31a2603a6345p+6, -0x1.1c08e9103811dp-47, INIT_U128( 0x0, 0x74 ) },
  { 0x1.c01dfd75803bfp+6, -0x1.43b63c7c876c8p-47, INIT_U128( 0x0, 0x70 ) },
  { 0x1.a497fb6549300p+6, -0x1.1bd9f54637b3ep-47, INIT_U128( 0x0, 0x69 ) },
  { 0x1.899d865f133b1p+7, -0x1.6cff8dced9ff2p-46, INIT_U128( 0x0, 0xc4 ) },
  { 0x1.bb975a8d772ebp+7, -0x1.4fefa0549fdf4p-46, INIT_U128( 0x0, 0xdd ) },
  { 0x1.e7ab3fb5cf568p+7, -0x1.06ef668a0ddedp-46, INIT_U128( 0x0, 0xf3 ) },
  { 0x1.76d7b408edaf6p+7, -0x1.fb47b6f1f68f7p-46, INIT_U128( 0x0, 0xbb ) },
  { 0x1.6901e158d203cp+7, -0x1.349b577c6936bp-46, INIT_U128( 0x0, 0xb4 ) },
  { 0x1.50a4408ea1488p+8, -0x1.cc46429f988c9p-45, INIT_U128( 0x0, 0x150 ) },
  { 0x1.0ffc7f1a1ff90p+8, -0x1.69141998d2283p-45, INIT_U128( 0x0, 0x10f ) },
  { 0x1.84ee0ee109dc2p+8, -0x1.2605fb8e4c0c0p-45, INIT_U128( 0x0, 0x184 ) },
  { 0x1.9305895f260b1p+8, -0x1.5c75f2a0b8ebep-45, INIT_U128( 0x0, 0x193 ) },
  { 0x1.ac6592f758cb2p+8, -0x1.8223938704472p-45, INIT_U128( 0x0, 0x1ac ) },
  { 0x1.8912fa131225fp+9, -0x1.0dff23901bfe4p-44, INIT_U128( 0x0, 0x312 ) },
  { 0x1.0946da82128dcp+9, -0x1.aadf3a2d55be7p-44, INIT_U128( 0x0, 0x212 ) },
  { 0x1.df7a91abbef52p+9, -0x1.1a77f25a34efep-44, INIT_U128( 0x0, 0x3be ) },
  { 0x1.e6be34f9cd7c6p+9, -0x1.5c174f14b82eap-44, INIT_U128( 0x0, 0x3cd ) },
  { 0x1.c52b68af8a56dp+9, -0x1.0b2c89e416591p-44, INIT_U128( 0x0, 0x38a ) },
  { 0x1.3132857462650p+10, -0x1.90cf8fd5219f2p-43, INIT_U128( 0x0, 0x4c4 ) },
  { 0x1.ed458277da8b1p+10, -0x1.fc7ee319f8fddp-43, INIT_U128( 0x0, 0x7b5 ) },
  { 0x1.7ea276a0fd44fp+10, -0x1.1fc7d1443f8fap-43, INIT_U128( 0x0, 0x5fa ) },
  { 0x1.2dc6f4d65b8dep+10, -0x1.bdfdda277bfbbp-43, INIT_U128( 0x0, 0x4b7 ) },
  { 0x1.68dd1454d1ba2p+10, -0x1.1ae82ef435d06p-43, INIT_U128( 0x0, 0x5a3 ) },
  { 0x1.3560214e6ac04p+11, -0x1.57bb4e74af76ap-42, INIT_U128( 0x0, 0x9ab ) },
  { 0x1.b3d29d8b67a54p+11, -0x1.b957887b72af1p-42, INIT_U128( 0x0, 0xd9e ) },
  { 0x1.c484125f89082p+11, -0x1.be2838f77c507p-42, INIT_U128( 0x0, 0xe24 ) },
  { 0x1.4ef7c8cc9def9p+11, -0x1.1bbc8a9a37792p-42, INIT_U128( 0x0, 0xa77 ) },
  { 0x1.183b6db43076ep+11, -0x1.ccc3996b99873p-42, INIT_U128( 0x0, 0x8c1 ) },
  { 0x1.ec8b6ab1d916ep+12, -0x1.03a448d007489p-41, INIT_U128( 0x0, 0x1ec8 ) },
  { 0x1.c1a222f983445p+12, -0x1.a63373db4c66ep-41, INIT_U128( 0x0, 0x1c1a ) },
  { 0x1.c11ba2cf82375p+12, -0x1.fab6ba1ff56d8p-41, INIT_U128( 0x0, 0x1c11 ) },
  { 0x1.52720894a4e41p+12, -0x1.eb4d5589d69aap-41, INIT_U128( 0x0, 0x1527 ) },
  { 0x1.a4a4de054949cp+12, -0x1.257324c84ae64p-41, INIT_U128( 0x0, 0x1a4a ) },
  { 0x1.e39b5dc1c736cp+13, -0x1.425a2a2684b46p-40, INIT_U128( 0x0, 0x3c73 ) },
  { 0x1.5e16b4c2bc2d6p+13, -0x1.18ce2ee2319c6p-40, INIT_U128( 0x0, 0x2bc2 ) },
  { 0x1.e591c84fcb239p+13, -0x1.44b8d2868971ap-40, INIT_U128( 0x0, 0x3cb2 ) },
  { 0x1.485d1b5890ba4p+13, -0x1.abc35bd75786cp-40, INIT_U128( 0x0, 0x290b ) },
  { 0x1.51f6cfb2a3edap+13, -0x1.3dcfc1707b9f8p-40, INIT_U128( 0x0, 0x2a3e ) },
  { 0x1.38867b90710d0p+14, -0x1.a75552f14eaaap-39, INIT_U128( 0x0, 0x4e21 ) },
  { 0x1.92a2a87b25455p+14, -0x1.1c28510c3850ap-39, INIT_U128( 0x0, 0x64a8 ) },
  { 0x1.419b75d68336ep+14, -0x1.107fd6a620ffbp-39, INIT_U128( 0x0, 0x5066 ) },
  { 0x1.52da00cea5b40p+14, -0x1.ff4fff1ffea00p-39, INIT_U128( 0x0, 0x54b6 ) },
  { 0x1.181e2e02303c6p+14, -0x1.c69a6bf98d34ep-39, INIT_U128( 0x0, 0x4607 ) },
  { 0x1.0b533e8016a68p+15, -0x1.3472ded668e5cp-38, INIT_U128( 0x0, 0x85a9 ) },
  { 0x1.e0363aafc06c8p+15, -0x1.bc3cbbe178798p-38, INIT_U128( 0x0, 0xf01b ) },
  { 0x1.fa0c7d53f418fp+15, -0x1.922e0dd1245c2p-38, INIT_U128( 0x0, 0xfd06 ) },
  { 0x1.398360c27306cp+15, -0x1.4ecdb3889d9b6p-38, INIT_U128( 0x0, 0x9cc1 ) },
  { 0x1.6af60494d5ec0p+15, -0x1.4ec8a7569d915p-38, INIT_U128( 0x0, 0xb57b ) },
  { 0x1.5f487dfebe910p+16, -0x1.a3de06a747bc1p-37, INIT_U128( 0x0, 0x15f48 ) },
  { 0x1.a32e05c7465c1p+16, -0x1.2992482a53249p-37, INIT_U128( 0x0, 0x1a32e ) },
  { 0x1.f02cb423e0596p+16, -0x1.8a560c6914ac2p-37, INIT_U128( 0x0, 0x1f02c ) },
  { 0x1.bec8eb417d91ep+16, -0x1.88c953751192ap-37, INIT_U128( 0x0, 0x1bec8 ) },
  { 0x1.45060f568a0c2p+16, -0x1.acdfe4af59bfcp-37, INIT_U128( 0x0, 0x14506 ) },
  { 0x1.3f0df2807e1bep+17, -0x1.166bad3a2cd76p-36, INIT_U128( 0x0, 0x27e1b ) },
  { 0x1.653960c2ca72cp+17, -0x1.2f31029e5e620p-36, INIT_U128( 0x0, 0x2ca72 ) },
  { 0x1.7df9a9f6fbf35p+17, -0x1.4824ec449049ep-36, INIT_U128( 0x0, 0x2fbf3 ) },
  { 0x1.ec02f2edd805fp+17, -0x1.4e892dca9d126p-36, INIT_U128( 0x0, 0x3d805 ) },
  { 0x1.781b75f6f036ep+17, -0x1.f277db23e4efcp-36, INIT_U128( 0x0, 0x2f036 ) },
  { 0x1.7ccde78cf99bdp+18, -0x1.0379569c06f2bp-35, INIT_U128( 0x0, 0x5f337 ) },
  { 0x1.8c42aa1318855p+18, -0x1.b2f8693f65f0dp-35, INIT_U128( 0x0, 0x6310a ) },
  { 0x1.5fd84deebfb0ap+18, -0x1.abc2291757845p-35, INIT_U128( 0x0, 0x57f61 ) },
  { 0x1.c22dd651845bbp+18, -0x1.8200cba50401ap-35, INIT_U128( 0x0, 0x708b7 ) },
  { 0x1.b131ce456263ap+18, -0x1.5c2ef038b85dep-35, INIT_U128( 0x0, 0x6c4c7 ) },
  { 0x1.a453dc7f48a7cp+19, -0x1.74577c36e8af0p-34, INIT_U128( 0x0, 0xd229e ) },
  { 0x1.c2e0aa0385c15p+19, -0x1.9bd246f537a49p-34, INIT_U128( 0x0, 0xe1705 ) },
  { 0x1.8e4a22191c944p+19, -0x1.6a04f49ed409ep-34, INIT_U128( 0x0, 0xc7251 ) },
  { 0x1.6d7e4b3cdafcap+19, -0x1.070c66d40e18dp-34, INIT_U128( 0x0, 0xb6bf2 ) },
  { 0x1.ba04d0157409ap+19, -0x1.2144d5384289ap-34, INIT_U128( 0x0, 0xdd026 ) },
  { 0x1.cd06fe519a0e0p+20, -0x1.db64f4cfb6c9ep-33, INIT_U128( 0x0, 0x1cd06f ) },
  { 0x1.5d65c074bacb8p+20, -0x1.d03af229a075ep-33, INIT_U128( 0x0, 0x15d65c ) },
  { 0x1.2c69e26258d3cp+20, -0x1.ae32a8295c655p-33, INIT_U128( 0x0, 0x12c69e ) },
  { 0x1.228a245e45144p+20, -0x1.4a30a72894615p-33, INIT_U128( 0x0, 0x1228a2 ) },
  { 0x1.e602ee1fcc05ep+20, -0x1.809c56710138bp-33, INIT_U128( 0x0, 0x1e602e ) },
  { 0x1.eaa14b11d542ap+21, -0x1.c3ab387187567p-32, INIT_U128( 0x0, 0x3d5429 ) },
  { 0x1.cab79f47956f4p+21, -0x1.c613f6598c27fp-32, INIT_U128( 0x0, 0x3956f3 ) },
  { 0x1.e758cb93ceb1ap+21, -0x1.1cea3a5a39d48p-32, INIT_U128( 0x0, 0x3ceb19 ) },
  { 0x1.f84f2f2bf09e6p+21, -0x1.575cebd2aeb9ep-32, INIT_U128( 0x0, 0x3f09e5 ) },
  { 0x1.2797ddfc4f2fcp+21, -0x1.fa30aa3ff4616p-32, INIT_U128( 0x0, 0x24f2fb ) },
  { 0x1.a863526950c6ap+22, -0x1.a20f8553441f1p-31, INIT_U128( 0x0, 0x6a18d4 ) },
  { 0x1.ec402c7bd8805p+22, -0x1.85d40d930ba82p-31, INIT_U128( 0x0, 0x7b100b ) },
  { 0x1.067064020ce0cp+22, -0x1.d0bf3389a17e6p-31, INIT_U128( 0x0, 0x419c19 ) },
  { 0x1.946a6bfb28d4ep+22, -0x1.c5f544b58bea8p-31, INIT_U128( 0x0, 0x651a9a ) },
  { 0x1.8c093b0d18128p+22, -0x1.9b11d1033623ap-31, INIT_U128( 0x0, 0x63024e ) },
  { 0x1.b7fc0b5f6ff82p+23, -0x1.827807fd04f01p-30, INIT_U128( 0x0, 0xdbfe05 ) },
  { 0x1.d9d4f3d1b3a9ep+23, -0x1.6b566f0ad6acep-30, INIT_U128( 0x0, 0xecea79 ) },
  { 0x1.6587194acb0e3p+23, -0x1.6a4ca886d4995p-30, INIT_U128( 0x0, 0xb2c38c ) },
  { 0x1.0c5c2dd818b86p+23, -0x1.7a2e1170f45c2p-30, INIT_U128( 0x0, 0x862e16 ) },
  { 0x1.00021f0400044p+23, -0x1.893027a912605p-30, INIT_U128( 0x0, 0x80010f ) },
  { 0x1.59ef5b32b3decp+24, -0x1.116f9fc422df4p-29, INIT_U128( 0x0, 0x159ef5b ) },
  { 0x1.d764e479aec9cp+24, -0x1.40b57780816afp-29, INIT_U128( 0x0, 0x1d764e4 ) },
  { 0x1.dbc74885b78e9p+24, -0x1.315a4d0c62b4ap-29, INIT_U128( 0x0, 0x1dbc748 ) },
  { 0x1.b3d2550b67a4bp+24, -0x1.fe91f6adfd23fp-29, INIT_U128( 0x0, 0x1b3d255 ) },
  { 0x1.69e865b8d3d0cp+24, -0x1.4a0eb898941d7p-29, INIT_U128( 0x0, 0x169e865 ) },
  { 0x1.b45b612168b6cp+25, -0x1.e482acdbc9055p-28, INIT_U128( 0x0, 0x368b6c2 ) },
  { 0x1.5eb0be4cbd618p+25, -0x1.b631cd396c63ap-28, INIT_U128( 0x0, 0x2bd617c ) },
  { 0x1.3f61f4e27ec3ep+25, -0x1.84335e410866cp-28, INIT_U128( 0x0, 0x27ec3e9 ) },
  { 0x1.95a2fe072b460p+25, -0x1.5ca99278b9532p-28, INIT_U128( 0x0, 0x32b45fc ) },
  { 0x1.9d5f09fb3abe1p+25, -0x1.d526e9b3aa4ddp-28, INIT_U128( 0x0, 0x33abe13 ) },
  { 0x1.068136c00d027p+26, -0x1.a74628314e8c5p-27, INIT_U128( 0x0, 0x41a04db ) },
  { 0x1.a5d81edf4bb04p+26, -0x1.50f834d4a1f06p-27, INIT_U128( 0x0, 0x697607b ) },
  { 0x1.9b06ed6f360dep+26, -0x1.d70eef59ae1dep-27, INIT_U128( 0x0, 0x66c1bb5 ) },
  { 0x1.bb919c3577234p+26, -0x1.bc89ee5b7913ep-27, INIT_U128( 0x0, 0x6ee4670 ) },
  { 0x1.72bbbb92e5778p+26, -0x1.c96dc6dd92db9p-27, INIT_U128( 0x0, 0x5caeeee ) },
  { 0x1.4da1251e9b424p+27, -0x1.de1a9831bc353p-26, INIT_U128( 0x0, 0xa6d0928 ) },
  { 0x1.3ac42aaa75886p+27, -0x1.d31a17d1a6343p-26, INIT_U128( 0x0, 0x9d62155 ) },
  { 0x1.a841018550820p+27, -0x1.20c207ae41841p-26, INIT_U128( 0x0, 0xd42080c ) },
  { 0x1.c8082fe190106p+27, -0x1.5e332d90bc666p-26, INIT_U128( 0x0, 0xe40417f ) },
  { 0x1.af2b62675e56cp+27, -0x1.b11098c162213p-26, INIT_U128( 0x0, 0xd795b13 ) },
  { 0x1.881f8819103f1p+28, -0x1.5cd0d6fab9a1bp-25, INIT_U128( 0x0, 0x1881f881 ) },
  { 0x1.195ac55632b58p+28, -0x1.a9ae7bf3535d0p-25, INIT_U128( 0x0, 0x1195ac55 ) },
  { 0x1.731c42d0e6388p+28, -0x1.d6e8b6a7add17p-25, INIT_U128( 0x0, 0x1731c42d ) },
  { 0x1.b5034ed96a06ap+28, -0x1.89a331ef13466p-25, INIT_U128( 0x0, 0x1b5034ed ) },
  { 0x1.5cbd49beb97a9p+28, -0x1.f44d90bde89b2p-25, INIT_U128( 0x0, 0x15cbd49b ) },
  { 0x1.150f986c2a1f3p+29, -0x1.1934355632686p-24, INIT_U128( 0x0, 0x22a1f30d ) },
  { 0x1.776762e8eececp+29, -0x1.1cd647a039ac9p-24, INIT_U128( 0x0, 0x2eecec5d ) },
  { 0x1.cde6e0619bcdcp+29, -0x1.c898ea6d9131dp-24, INIT_U128( 0x0, 0x39bcdc0c ) },
  { 0x1.2569c1684ad38p+29, -0x1.0a59b38e14b36p-24, INIT_U128( 0x0, 0x24ad382d ) },
  { 0x1.b4082f8368106p+29, -0x1.f7ab0b9def562p-24, INIT_U128( 0x0, 0x368105f0 ) },
  { 0x1.ce93c2459d278p+30, -0x1.090f390c121e7p-23, INIT_U128( 0x0, 0x73a4f091 ) },
  { 0x1.92b26afb2564dp+30, -0x1.8be3655917c6dp-23, INIT_U128( 0x0, 0x64ac9abe ) },
  { 0x1.8030f6170061fp+30, -0x1.3b4938d276927p-23, INIT_U128( 0x0, 0x600c3d85 ) },
  { 0x1.547d6e2aa8faep+30, -0x1.a82e838d505d0p-23, INIT_U128( 0x0, 0x551f5b8a ) },
  { 0x1.c504cb2d8a09ap+30, -0x1.bd506de37aa0ep-23, INIT_U128( 0x0, 0x714132cb ) },
  { 0x1.85e958510bd2bp+31, -0x1.633cab8ac6796p-22, INIT_U128( 0x0, 0xc2f4ac28 ) },
  { 0x1.6d965f18db2ccp+31, -0x1.82a9a0d705534p-22, INIT_U128( 0x0, 0xb6cb2f8c ) },
  { 0x1.7a54ac72f4a96p+31, -0x1.b4f2a82569e55p-22, INIT_U128( 0x0, 0xbd2a5639 ) },
  { 0x1.be01e1317c03cp+31, -0x1.30a4db346149cp-22, INIT_U128( 0x0, 0xdf00f098 ) },
  { 0x1.bc88408f79108p+31, -0x1.3fde0a7c7fbc2p-22, INIT_U128( 0x0, 0xde442047 ) },
  { 0x1.7b27434ef64e8p+32, -0x1.349200ea69240p-21, INIT_U128( 0x0, 0x17b27434e ) },
  { 0x1.dcc08477b9810p+32, -0x1.44521bfc88a44p-21, INIT_U128( 0x0, 0x1dcc08477 ) },
  { 0x1.01a289e003451p+32, -0x1.8e0118f31c023p-21, INIT_U128( 0x0, 0x101a289e0 ) },
  { 0x1.f13dc47de27b8p+32, -0x1.1da30f403b462p-21, INIT_U128( 0x0, 0x1f13dc47d ) },
  { 0x1.a04cb01940996p+32, -0x1.1cf69c6439ed4p-21, INIT_U128( 0x0, 0x1a04cb019 ) },
  { 0x1.d1e9448ba3d28p+33, -0x1.c935b325926b7p-20, INIT_U128( 0x0, 0x3a3d28917 ) },
  { 0x1.f9da8e5bf3b52p+33, -0x1.30094b1c6012ap-20, INIT_U128( 0x0, 0x3f3b51cb7 ) },
  { 0x1.6fe7559adfceap+33, -0x1.20c372d44186ep-20, INIT_U128( 0x0, 0x2dfceab35 ) },
  { 0x1.555573beaaaaep+33, -0x1.c515cab98a2bap-20, INIT_U128( 0x0, 0x2aaaae77d ) },
  { 0x1.0967e6b612cfdp+33, -0x1.563cccf4ac79ap-20, INIT_U128( 0x0, 0x212cfcd6c ) },
  { 0x1.e43de5e7c87bdp+34, -0x1.088c431211188p-19, INIT_U128( 0x0, 0x790f7979f ) },
  { 0x1.262f52544c5eap+34, -0x1.c5754eb58aeaap-19, INIT_U128( 0x0, 0x498bd4951 ) },
  { 0x1.e26beb1bc4d7ep+34, -0x1.b4c2803369850p-19, INIT_U128( 0x0, 0x789afac6f ) },
  { 0x1.7df83358fbf06p+34, -0x1.44a72a40894e6p-19, INIT_U128( 0x0, 0x5f7e0cd63 ) },
  { 0x1.74b0755ee960ep+34, -0x1.24dfc74e49bf9p-19, INIT_U128( 0x0, 0x5d2c1d57b ) },
  { 0x1.45a33a228b468p+35, -0x1.2151962c42a33p-18, INIT_U128( 0x0, 0xa2d19d114 ) },
  { 0x1.e66760e7cccecp+35, -0x1.8a59ae4b14b36p-18, INIT_U128( 0x0, 0xf333b073e ) },
  { 0x1.e0967b41c12d0p+35, -0x1.a0e9edf941d3ep-18, INIT_U128( 0x0, 0xf04b3da0e ) },
  { 0x1.a976310d52ec6p+35, -0x1.8d700b011ae02p-18, INIT_U128( 0x0, 0xd4bb1886a ) },
  { 0x1.52e559b6a5cabp+35, -0x1.ce9fd98d9d3fbp-18, INIT_U128( 0x0, 0xa972acdb5 ) },
  { 0x1.836259bf06c4bp+36, -0x1.773ee4ceee7dcp-17, INIT_U128( 0x0, 0x1836259bf0 ) },
  { 0x1.bbe60f1177cc2p+36, -0x1.2de6c9dc5bcd9p-17, INIT_U128( 0x0, 0x1bbe60f117 ) },
  { 0x1.52c6912ca58d2p+36, -0x1.a762df6b4ec5cp-17, INIT_U128( 0x0, 0x152c6912ca ) },
  { 0x1.c8b475999168fp+36, -0x1.e4d4b351c9a97p-17, INIT_U128( 0x0, 0x1c8b475999 ) },
  { 0x1.6f2e619cde5ccp+36, -0x1.00711f5000e24p-17, INIT_U128( 0x0, 0x16f2e619cd ) },
  { 0x1.a15b0d4b42b62p+37, -0x1.99c4d6573389bp-16, INIT_U128( 0x0, 0x342b61a968 ) },
  { 0x1.2734e1564e69cp+37, -0x1.aee949975dd29p-16, INIT_U128( 0x0, 0x24e69c2ac9 ) },
  { 0x1.7d892322fb124p+37, -0x1.b098478f61309p-16, INIT_U128( 0x0, 0x2fb124645f ) },
  { 0x1.27d72c5e4fae6p+37, -0x1.9911f45b3223ep-16, INIT_U128( 0x0, 0x24fae58bc9 ) },
  { 0x1.c6f8be398df18p+37, -0x1.7b7358c8f6e6bp-16, INIT_U128( 0x0, 0x38df17c731 ) },
  { 0x1.22d6383445ac7p+38, -0x1.29820f4c53042p-15, INIT_U128( 0x0, 0x48b58e0d11 ) },
  { 0x1.0e56c29e1cad8p+38, -0x1.e58b93edcb172p-15, INIT_U128( 0x0, 0x4395b0a787 ) },
  { 0x1.e5fcb33dcbf97p+38, -0x1.3991a26e73234p-15, INIT_U128( 0x0, 0x797f2ccf72 ) },
  { 0x1.d35678c5a6acfp+38, -0x1.e1e33f23c3c68p-15, INIT_U128( 0x0, 0x74d59e3169 ) },
  { 0x1.90cc88bb21991p+38, -0x1.ec9fc09fd93f8p-15, INIT_U128( 0x0, 0x6433222ec8 ) },
  { 0x1.9d38082f3a701p+39, -0x1.2411150e48222p-14, INIT_U128( 0x0, 0xce9c04179d ) },
  { 0x1.bd23096b7a461p+39, -0x1.118732be230e6p-14, INIT_U128( 0x0, 0xde9184b5bd ) },
  { 0x1.8cfa8b8919f52p+39, -0x1.bd202fdf7a406p-14, INIT_U128( 0x0, 0xc67d45c48c ) },
  { 0x1.e6a23f2bcd448p+39, -0x1.500431c2a0086p-14, INIT_U128( 0x0, 0xf3511f95e6 ) },
  { 0x1.358fc5b06b1f8p+39, -0x1.3601f5da6c03ep-14, INIT_U128( 0x0, 0x9ac7e2d835 ) },
  { 0x1.6f93984adf273p+40, -0x1.3fc827227f905p-13, INIT_U128( 0x0, 0x16f93984adf ) },
  { 0x1.21ab66464356dp+40, -0x1.750d3858ea1a7p-13, INIT_U128( 0x0, 0x121ab664643 ) },
  { 0x1.61acf8f8c359fp+40, -0x1.b21b005b64360p-13, INIT_U128( 0x0, 0x161acf8f8c3 ) },
  { 0x1.d62c156bac582p+40, -0x1.d00ec3a5a01d8p-13, INIT_U128( 0x0, 0x1d62c156bac ) },
  { 0x1.e195f665c32bfp+40, -0x1.6cc0a5b2d9814p-13, INIT_U128( 0x0, 0x1e195f665c3 ) },
  { 0x1.29e0d59253c1ap+41, -0x1.b545fd876a8c0p-12, INIT_U128( 0x0, 0x253c1ab24a7 ) },
  { 0x1.054a2f4a0a946p+41, -0x1.b9bcb73b73797p-12, INIT_U128( 0x0, 0x20a945e9415 ) },
  { 0x1.efbead17df7d5p+41, -0x1.77fa3c90eff48p-12, INIT_U128( 0x0, 0x3df7d5a2fbe ) },
  { 0x1.b7c263b56f84cp+41, -0x1.f94aa763f2955p-12, INIT_U128( 0x0, 0x36f84c76adf ) },
  { 0x1.6e57dfbcdcafcp+41, -0x1.dfb1c1a1bf638p-12, INIT_U128( 0x0, 0x2dcafbf79b9 ) },
  { 0x1.91ccc1fd23998p+42, -0x1.4cb353e29966ap-11, INIT_U128( 0x0, 0x6473307f48e ) },
  { 0x1.1cc2178639843p+42, -0x1.92de0f9325bc2p-11, INIT_U128( 0x0, 0x473085e18e6 ) },
  { 0x1.ff9eb547ff3d6p+42, -0x1.12b2dcb82565cp-11, INIT_U128( 0x0, 0x7fe7ad51ffc ) },
  { 0x1.c7f8da418ff1bp+42, -0x1.8b49d5891693bp-11, INIT_U128( 0x0, 0x71fe369063f ) },
  { 0x1.45068cc48a0d2p+42, -0x1.3f7eb6227efd7p-11, INIT_U128( 0x0, 0x5141a331228 ) },
  { 0x1.b778f2b56ef1ep+43, -0x1.5e836c72bd06ep-10, INIT_U128( 0x0, 0xdbbc795ab77 ) },
  { 0x1.57ce13d4af9c2p+43, -0x1.3653d5b06ca7ap-10, INIT_U128( 0x0, 0xabe709ea57c ) },
  { 0x1.160597522c0b3p+43, -0x1.d48ed96ba91dbp-10, INIT_U128( 0x0, 0x8b02cba9160 ) },
  { 0x1.21422d4c42846p+43, -0x1.6514a626ca295p-10, INIT_U128( 0x0, 0x90a116a6214 ) },
  { 0x1.f7f2b5bbefe56p+43, -0x1.0b1326f816265p-10, INIT_U128( 0x0, 0xfbf95addf7f ) },
  { 0x1.1dc603b23b8c0p+44, -0x1.a944e29f5289cp-9, INIT_U128( 0x0, 0x11dc603b23b8 ) },
  { 0x1.348eb828691d7p+44, -0x1.5b139576b6272p-9, INIT_U128( 0x0, 0x1348eb828691 ) },
  { 0x1.9a924b9b3524ap+44, -0x1.bb1ee607763ddp-9, INIT_U128( 0x0, 0x19a924b9b352 ) },
  { 0x1.19a2527a3344ap+44, -0x1.4271fe0684e40p-9, INIT_U128( 0x0, 0x119a2527a334 ) },
  { 0x1.78c5d3b8f18bap+44, -0x1.09efb52613df6p-9, INIT_U128( 0x0, 0x178c5d3b8f18 ) },
  { 0x1.de7e8e93bcfd2p+45, -0x1.cdc1262d9b825p-8, INIT_U128( 0x0, 0x3bcfd1d2779f ) },
  { 0x1.32d5cc5265abap+45, -0x1.4668f5168cd1ep-8, INIT_U128( 0x0, 0x265ab98a4cb5 ) },
  { 0x1.517c1c0ea2f84p+45, -0x1.7a215378f442ap-8, INIT_U128( 0x0, 0x2a2f8381d45f ) },
  { 0x1.2366361846cc7p+45, -0x1.436ae2ee86d5cp-8, INIT_U128( 0x0, 0x246cc6c308d9 ) },
  { 0x1.9e7e78673cfcfp+45, -0x1.0802e5361005cp-8, INIT_U128( 0x0, 0x33cfcf0ce79f ) },
  { 0x1.c321cb3f8643ap+46, -0x1.7fe4b1f8ffc96p-7, INIT_U128( 0x0, 0x70c872cfe190 ) },
  { 0x1.c256b00184ad6p+46, -0x1.4c62971298c53p-7, INIT_U128( 0x0, 0x7095ac00612b ) },
  { 0x1.56524c3aaca4ap+46, -0x1.85caddef0b95cp-7, INIT_U128( 0x0, 0x5594930eab29 ) },
  { 0x1.538c041aa7180p+46, -0x1.0085b322010b6p-7, INIT_U128( 0x0, 0x54e30106a9c5 ) },
  { 0x1.46eea83a8ddd5p+46, -0x1.72569766e4ad3p-7, INIT_U128( 0x0, 0x51bbaa0ea377 ) },
  { 0x1.ec77eae1d8efep+47, -0x1.d3843f91a7088p-6, INIT_U128( 0x0, 0xf63bf570ec77 ) },
  { 0x1.16ef22802dde4p+47, -0x1.5cade1c4b95bcp-6, INIT_U128( 0x0, 0x8b77914016ef ) },
  { 0x1.64b0c6fac9619p+47, -0x1.395d426a72ba8p-6, INIT_U128( 0x0, 0xb258637d64b0 ) },
  { 0x1.b941f4657283ep+47, -0x1.799a6142f334cp-6, INIT_U128( 0x0, 0xdca0fa32b941 ) },
  { 0x1.c5fcf8578bf9fp+47, -0x1.52cb33fea5966p-6, INIT_U128( 0x0, 0xe2fe7c2bc5fc ) },
  { 0x1.d89611efb12c2p+48, -0x1.809d16e9013a3p-5, INIT_U128( 0x0, 0x1d89611efb12c ) },
  { 0x1.8122ca0d02459p+48, -0x1.85b1b6110b637p-5, INIT_U128( 0x0, 0x18122ca0d0245 ) },
  { 0x1.f8084d79f0109p+48, -0x1.63ae38f8c75c7p-5, INIT_U128( 0x0, 0x1f8084d79f010 ) },
  { 0x1.3732788a6e64fp+48, -0x1.68291aa0d0524p-5, INIT_U128( 0x0, 0x13732788a6e64 ) },
  { 0x1.c4d6fc6789ae0p+48, -0x1.5c31688eb862dp-5, INIT_U128( 0x0, 0x1c4d6fc6789ad ) },
  { 0x1.958eaafb2b1d5p+49, -0x1.f20d94e1e41b2p-4, INIT_U128( 0x0, 0x32b1d55f6563a ) },
  { 0x1.6ba7f624d74ffp+49, -0x1.55f10592abe20p-4, INIT_U128( 0x0, 0x2d74fec49ae9f ) },
  { 0x1.a8bc399351787p+49, -0x1.811e6bed023cep-4, INIT_U128( 0x0, 0x351787326a2f0 ) },
  { 0x1.01bb82d603770p+49, -0x1.9f1452713e28ap-4, INIT_U128( 0x0, 0x2037705ac06ed ) },
  { 0x1.24c60882498c1p+49, -0x1.5ff84806bff09p-4, INIT_U128( 0x0, 0x2498c11049318 ) },
  { 0x1.c3a68101874d0p+50, -0x1.ab6bf5c156d7fp-3, INIT_U128( 0x0, 0x70e9a04061d33 ) },
  { 0x1.13951062272a2p+50, -0x1.f5ace7a9eb59dp-3, INIT_U128( 0x0, 0x44e5441889ca8 ) },
  { 0x1.51de5038a3bcap+50, -0x1.4c89ca8c9913ap-3, INIT_U128( 0x0, 0x5477940e28ef2 ) },
  { 0x1.3e975f6e7d2ecp+50, -0x1.534e25caa69c4p-3, INIT_U128( 0x0, 0x4fa5d7db9f4ba ) },
  { 0x1.a93f2b5f527e6p+50, -0x1.00a6de60014dcp-3, INIT_U128( 0x0, 0x6a4fcad7d49f9 ) },
  { 0x1.81c54a4f038a9p+51, -0x1.29d8771e53b0fp-2, INIT_U128( 0x0, 0xc0e2a52781c54 ) },
  { 0x1.c8ecc99191d99p+51, -0x1.80c2120101842p-2, INIT_U128( 0x0, 0xe47664c8c8ecc ) },
  { 0x1.e0ba5a09c174bp+51, -0x1.514ed938a29dbp-2, INIT_U128( 0x0, 0xf05d2d04e0ba5 ) },
  { 0x1.fcbacefbf975ap+51, -0x1.0bcfb664179f7p-2, INIT_U128( 0x0, 0xfe5d677dfcbac ) },
  { 0x1.017c9e0e02f94p+51, -0x1.e8346931d068dp-2, INIT_U128( 0x0, 0x80be4f07017c9 ) },
  { 0x1.34be33be697c6p+52, -0x1.eaafbc57d55f7p-1, INIT_U128( 0x0, 0x134be33be697c5 ) },
  { 0x1.3dfde2227bfbcp+52, -0x1.71058416e20b0p-1, INIT_U128( 0x0, 0x13dfde2227bfbb ) },
  { 0x1.71ac6278e358cp+52, -0x1.f9792cd5f2f25p-1, INIT_U128( 0x0, 0x171ac6278e358b ) },
  { 0x1.2351a3f446a34p+52, -0x1.0c8622e8190c4p-1, INIT_U128( 0x0, 0x12351a3f446a33 ) },
  { 0x1.8d4834e91a906p+52, -0x1.7d5193f8faa32p-1, INIT_U128( 0x0, 0x18d4834e91a905 ) },
  { 0x1.cea06d339d40dp+53, -0x1.e47267a9c8e4dp+0, INIT_U128( 0x0, 0x39d40da673a818 ) },
  { 0x1.f0421e0fe0844p+53, -0x1.b19d3e95633a8p+0, INIT_U128( 0x0, 0x3e0843c1fc1086 ) },
  { 0x1.99d1a7c733a35p+53, -0x1.c46707b788ce1p+0, INIT_U128( 0x0, 0x333a34f8e67468 ) },
  { 0x1.af16c40f5e2d8p+53, -0x1.7fe193e8ffc32p+0, INIT_U128( 0x0, 0x35e2d881ebc5ae ) },
  { 0x1.f0e71801e1ce3p+53, -0x1.5f7a5778bef4bp+0, INIT_U128( 0x0, 0x3e1ce3003c39c4 ) },
  { 0x1.58f3844cb1e70p+54, -0x1.0d8a262a1b145p+1, INIT_U128( 0x0, 0x563ce1132c79bd ) },
  { 0x1.1a10491234209p+54, -0x1.ccf7a13399ef4p+1, INIT_U128( 0x0, 0x468412448d0820 ) },
  { 0x1.4a49352c94926p+54, -0x1.4274d56284e9ap+1, INIT_U128( 0x0, 0x52924d4b252495 ) },
  { 0x1.26e73d0c4dce8p+54, -0x1.27be76ee4f7cfp+1, INIT_U128( 0x0, 0x49b9cf4313739d ) },
  { 0x1.d95bb585b2b77p+54, -0x1.bfeffa1b7fdffp+1, INIT_U128( 0x0, 0x7656ed616cadd8 ) },
  { 0x1.845701d308ae0p+55, -0x1.4e01c1a69c038p+2, INIT_U128( 0x0, 0xc22b80e98456fa ) },
  { 0x1.cde355919bc6bp+55, -0x1.83a2a78707455p+2, INIT_U128( 0x0, 0xe6f1aac8cde351 ) },
  { 0x1.e31cfeffc63a0p+55, -0x1.f4d41fcbe9a84p+2, INIT_U128( 0x0, 0xf18e7f7fe31cf8 ) },
  { 0x1.025e45f404bc8p+55, -0x1.77630f52eec62p+2, INIT_U128( 0x0, 0x812f22fa025e3a ) },
  { 0x1.445ab08688b56p+55, -0x1.20d9415841b28p+2, INIT_U128( 0x0, 0xa22d5843445aab ) },
  { 0x1.634e16bcc69c3p+56, -0x1.d92219f9b2443p+3, INIT_U128( 0x0, 0x1634e16bcc69c21 ) },
  { 0x1.00e090e601c12p+56, -0x1.8bd266a517a4dp+3, INIT_U128( 0x0, 0x100e090e601c113 ) },
  { 0x1.4bc260c09784cp+56, -0x1.aa46c623548d9p+3, INIT_U128( 0x0, 0x14bc260c09784b2 ) },
  { 0x1.f84887cbf0911p+56, -0x1.8f26dc8b1e4dcp+3, INIT_U128( 0x0, 0x1f84887cbf09103 ) },
  { 0x1.12c9841025930p+56, -0x1.2409948a48132p+3, INIT_U128( 0x0, 0x112c984102592f6 ) },
  { 0x1.18ddb45631bb6p+57, -0x1.a65432fd4ca86p+4, INIT_U128( 0x0, 0x231bb68ac6376a5 ) },
  { 0x1.1db742503b6e8p+57, -0x1.ea4cb1f1d4996p+4, INIT_U128( 0x0, 0x23b6e84a076dce1 ) },
  { 0x1.0b25f34a164bep+57, -0x1.fb71965ff6e33p+4, INIT_U128( 0x0, 0x2164be6942c97a0 ) },
  { 0x1.d493b057a9276p+57, -0x1.984360a73086cp+4, INIT_U128( 0x0, 0x3a92760af524ea6 ) },
  { 0x1.2415c74c482b9p+57, -0x1.9a080cb934102p+4, INIT_U128( 0x0, 0x2482b8e98905706 ) },
  { 0x1.444ed2dc889dap+58, -0x1.fd1a2b65fa346p+5, INIT_U128( 0x0, 0x5113b4b72227640 ) },
  { 0x1.8c4107f118821p+58, -0x1.4378b31a86f16p+5, INIT_U128( 0x0, 0x631041fc4620817 ) },
  { 0x1.678247a6cf049p+58, -0x1.87b982f30f730p+5, INIT_U128( 0x0, 0x59e091e9b3c120f ) },
  { 0x1.2e0db5f05c1b6p+58, -0x1.c6e815758dd02p+5, INIT_U128( 0x0, 0x4b836d7c1706d47 ) },
  { 0x1.b134c76562699p+58, -0x1.d3467fa1a68d0p+5, INIT_U128( 0x0, 0x6c4d31d9589a605 ) },
  { 0x1.ceb816019d703p+59, -0x1.9e8cc3ad3d198p+6, INIT_U128( 0x0, 0xe75c0b00ceb8118 ) },
  { 0x1.36d024526da04p+59, -0x1.2378228a46f04p+6, INIT_U128( 0x0, 0x9b68122936d01b7 ) },
  { 0x1.bd802f437b006p+59, -0x1.28708f4450e12p+6, INIT_U128( 0x0, 0xdec017a1bd802b5 ) },
  { 0x1.258abaaa4b158p+59, -0x1.0aaf9e52155f4p+6, INIT_U128( 0x0, 0x92c55d55258abbd ) },
  { 0x1.4e28516e9c50ap+59, -0x1.412994cc82532p+6, INIT_U128( 0x0, 0xa71428b74e284af ) },
  { 0x1.c8e317bf91c63p+60, -0x1.e5c75db5cb8ebp+7, INIT_U128( 0x0, 0x1c8e317bf91c620d ) },
  { 0x1.acaf7329595eep+60, -0x1.d1a68b5fa34d2p+7, INIT_U128( 0x0, 0x1acaf7329595ed17 ) },
  { 0x1.a4ed0b0149da2p+60, -0x1.f08cb111e1196p+7, INIT_U128( 0x0, 0x1a4ed0b0149da107 ) },
  { 0x1.8f1974bd1e32ep+60, -0x1.450db9f88a1b7p+7, INIT_U128( 0x0, 0x18f1974bd1e32d5d ) },
  { 0x1.6885dc84d10bcp+60, -0x1.8fb0eef31f61ep+7, INIT_U128( 0x0, 0x16885dc84d10bb38 ) },
  { 0x1.b8217e3970430p+61, -0x1.eb8d9863d71b3p+8, INIT_U128( 0x0, 0x37042fc72e085e14 ) },
  { 0x1.f63d36b7ec7a7p+61, -0x1.663b6ab2cc76ep+8, INIT_U128( 0x0, 0x3ec7a6d6fd8f4c99 ) },
  { 0x1.a764ff7f4eca0p+61, -0x1.e04e4a23c09c9p+8, INIT_U128( 0x0, 0x34ec9fefe9d93e1f ) },
  { 0x1.c615a7d78c2b5p+61, -0x1.301ab64e60357p+8, INIT_U128( 0x0, 0x38c2b4faf18568cf ) },
  { 0x1.01c141ae03828p+61, -0x1.a9eeaae553dd5p+8, INIT_U128( 0x0, 0x20382835c0704e56 ) },
  { 0x1.9a7b5bf534f6cp+62, -0x1.eae84ad3d5d0ap+9, INIT_U128( 0x0, 0x669ed6fd4d3dac2a ) },
  { 0x1.2ff604cc5fec0p+62, -0x1.3144550c6288ap+9, INIT_U128( 0x0, 0x4bfd813317fafd9d ) },
  { 0x1.c0f5440181ea8p+62, -0x1.104dee78209bep+9, INIT_U128( 0x0, 0x703d5100607a9ddf ) },
  { 0x1.fbdb86dbf7b71p+62, -0x1.e656fb05ccae0p+9, INIT_U128( 0x0, 0x7ef6e1b6fdedc033 ) },
  { 0x1.4c9265b69924cp+62, -0x1.e63af6a3cc75fp+9, INIT_U128( 0x0, 0x5324996da6492c33 ) },
  { 0x1.c68940c78d128p+63, -0x1.308e66f4611cdp+10, INIT_U128( 0x0, 0xe344a063c6893b3d ) },
  { 0x1.60a91d44c1524p+63, -0x1.cb78616996f0cp+10, INIT_U128( 0x0, 0xb0548ea260a918d2 ) },
  { 0x1.75670c4ceace2p+63, -0x1.d00d5933a01abp+10, INIT_U128( 0x0, 0xbab38626756708bf ) },
  { 0x1.baf5a40575eb4p+63, -0x1.236c195446d83p+10, INIT_U128( 0x0, 0xdd7ad202baf59b72 ) },
  { 0x1.54b06e62a960ep+63, -0x1.28f5a2bc51eb4p+10, INIT_U128( 0x0, 0xaa58373154b06b5c ) },
  { 0x1.3943913672872p+64, -0x1.41e9773e83d2fp+11, INIT_U128( 0x1, 0x39439136728715f0 ) },
  { 0x1.284072445080ep+64, -0x1.e57f6fbfcafeep+11, INIT_U128( 0x1, 0x284072445080d0d4 ) },
  { 0x1.cbba66639774dp+64, -0x1.5db61b5ebb6c4p+11, INIT_U128( 0x1, 0xcbba66639774c512 ) },
  { 0x1.8e4482531c890p+64, -0x1.cd09b9ad9a137p+11, INIT_U128( 0x1, 0x8e4482531c88f197 ) },
  { 0x1.98821b1531044p+64, -0x1.1e46936a3c8d2p+11, INIT_U128( 0x1, 0x98821b153104370d ) },
  { 0x1.ee53ec9ddca7dp+65, -0x1.5a70aa30b4e16p+12, INIT_U128( 0x3, 0xdca7d93bb94f8a58 ) },
  { 0x1.7ddcc386fbb98p+65, -0x1.5742c89eae859p+12, INIT_U128( 0x2, 0xfbb9870df772ea8b ) },
  { 0x1.80386b5d0070ep+65, -0x1.179cc39a2f398p+12, INIT_U128( 0x3, 0x70d6ba00e1ae86 ) },
  { 0x1.59863adab30c8p+65, -0x1.76ef58eceddebp+12, INIT_U128( 0x2, 0xb30c75b56618e891 ) },
  { 0x1.5c6ec8acb8dd9p+65, -0x1.15d2ea4c2ba5ep+12, INIT_U128( 0x2, 0xb8dd915971bb0ea2 ) },
  { 0x1.738905ace7120p+66, -0x1.5f63c252bec78p+13, INIT_U128( 0x5, 0xce2416b39c47d413 ) },
  { 0x1.2c6d7bfe58db0p+66, -0x1.d94f6605b29edp+13, INIT_U128( 0x4, 0xb1b5eff9636bc4d6 ) },
  { 0x1.f795f971ef2bfp+66, -0x1.4213262684265p+13, INIT_U128( 0x7, 0xde57e5c7bcaf97bd ) },
  { 0x1.9c83d0573907ap+66, -0x1.f0a44e21e148ap+13, INIT_U128( 0x6, 0x720f415ce41e41eb ) },
  { 0x1.47e716c08fce3p+66, -0x1.2368b02846d16p+13, INIT_U128( 0x5, 0x1f9c5b023f389b92 ) },
  { 0x1.3b0573c4760aep+67, -0x1.d520aa17aa415p+14, INIT_U128( 0x9, 0xd82b9e23b0568ab7 ) },
  { 0x1.4c10030298200p+67, -0x1.7036935ce06d2p+14, INIT_U128( 0xa, 0x60801814c0ffa3f2 ) },
  { 0x1.86097dab0c130p+67, -0x1.0cebe0e019d7cp+14, INIT_U128( 0xc, 0x304bed586097bcc5 ) },
  { 0x1.6c4fa332d89f4p+67, -0x1.8b14d0fd1629ap+14, INIT_U128( 0xb, 0x627d1996c4f99d3a ) },
  { 0x1.64643702c8c87p+67, -0x1.ec1b32f3d8367p+14, INIT_U128( 0xb, 0x2321b816464304f9 ) },
  { 0x1.2e2e59185c5cbp+68, -0x1.308574c2610aep+15, INIT_U128( 0x12, 0xe2e59185c5ca67bd ) },
  { 0x1.2129c2e442538p+68, -0x1.d1178d37a22f1p+15, INIT_U128( 0x12, 0x129c2e4425371774 ) },
  { 0x1.f4803d61e9007p+68, -0x1.d2008319a4011p+15, INIT_U128( 0x1f, 0x4803d61e900616ff ) },
  { 0x1.a7212d314e426p+68, -0x1.b2cbda7d6597bp+15, INIT_U128( 0x1a, 0x7212d314e425269a ) },
  { 0x1.23026dd84604ep+68, -0x1.da63bb8bb4c78p+15, INIT_U128( 0x12, 0x3026dd84604d12ce ) },
  { 0x1.e74f9c6dce9f3p+69, -0x1.6c57f6a6d8affp+16, INIT_U128( 0x3c, 0xe9f38db9d3e493a8 ) },
  { 0x1.f76cc7a5eed99p+69, -0x1.9dfe7a353bfcfp+16, INIT_U128( 0x3e, 0xed98f4bddb306201 ) },
  { 0x1.8a19ad2914336p+69, -0x1.4a09151e94122p+16, INIT_U128( 0x31, 0x4335a522866ab5f6 ) },
  { 0x1.7a8e89e2f51d1p+69, -0x1.f9af45a3f35e8p+16, INIT_U128( 0x2f, 0x51d13c5ea3a00650 ) },
  { 0x1.cac6d2d9958dbp+69, -0x1.680ced04d019ep+16, INIT_U128( 0x39, 0x58da5b32b1b497f3 ) },
  { 0x1.3a0d701e741aep+70, -0x1.35f982386bf30p+17, INIT_U128( 0x4e, 0x835c079d06b5940c ) },
  { 0x1.e8c55cb5d18abp+70, -0x1.b2de212b65bc4p+17, INIT_U128( 0x7a, 0x31572d7462a89a43 ) },
  { 0x1.8d41dcb71a83cp+70, -0x1.700fc96ae01f9p+17, INIT_U128( 0x63, 0x50772dc6a0ed1fe0 ) },
  { 0x1.dc67c013b8cf8p+70, -0x1.577224f0aee44p+17, INIT_U128( 0x77, 0x19f004ee33dd511b ) },
  { 0x1.f5709f27eae14p+70, -0x1.b48d1799691a3p+17, INIT_U128( 0x7d, 0x5c27c9fab84c96e5 ) },
  { 0x1.fe962cb9fd2c5p+71, -0x1.e770b583cee16p+18, INIT_U128( 0xff, 0x4b165cfe9620623d ) },
  { 0x1.e11ffb25c2400p+71, -0x1.bacc824975990p+18, INIT_U128( 0xf0, 0x8ffd92e11ff914cd ) },
  { 0x1.22d96f4e45b2ep+71, -0x1.19a629d4334c5p+18, INIT_U128( 0x91, 0x6cb7a722d96b9967 ) },
  { 0x1.4f9fac0e9f3f6p+71, -0x1.59fa9f9cb3f54p+18, INIT_U128( 0xa7, 0xcfd6074f9faa9815 ) },
  { 0x1.0cd8c0c619b18p+71, -0x1.12754a4424eaap+18, INIT_U128( 0x86, 0x6c60630cd8bbb62a ) },
  { 0x1.828513b1050a2p+72, -0x1.f9008969f2011p+19, INIT_U128( 0x182, 0x8513b1050a1037fb ) },
  { 0x1.5072ae5aa0e56p+72, -0x1.d6dccf67adb9ap+19, INIT_U128( 0x150, 0x72ae5aa0e5514919 ) },
  { 0x1.8b69e70116d3dp+72, -0x1.6266e1e2c4cdcp+19, INIT_U128( 0x18b, 0x69e70116d3c4ecc8 ) },
  { 0x1.49d7976693af3p+72, -0x1.fd3fec69fa7fdp+19, INIT_U128( 0x149, 0xd7976693af201600 ) },
  { 0x1.b80c6eb17018ep+72, -0x1.1801cdba3003ap+19, INIT_U128( 0x1b8, 0xc6eb17018d73ff1 ) },
  { 0x1.31a7f790634ffp+73, -0x1.420c3ec684188p+20, INIT_U128( 0x263, 0x4fef20c69fcbdf3c ) },
  { 0x1.9c06de65380dcp+73, -0x1.75e23a2eebc48p+20, INIT_U128( 0x338, 0xdbcca701b68a1dc ) },
  { 0x1.8857847510af0p+73, -0x1.fdd97fbbfbb30p+20, INIT_U128( 0x310, 0xaf08ea215de02268 ) },
  { 0x1.f6da0925edb41p+73, -0x1.51a6534ea34cap+20, INIT_U128( 0x3ed, 0xb4124bdb680ae59a ) },
  { 0x1.288322b651064p+73, -0x1.9c8ac76d39159p+20, INIT_U128( 0x251, 0x6456ca20c663753 ) },
  { 0x1.fef1a951fde35p+74, -0x1.529d9ed8a53b4p+21, INIT_U128( 0x7fb, 0xc6a547f78d15ac4c ) },
  { 0x1.4fdd8b429fbb2p+74, -0x1.4111e7ca8223dp+21, INIT_U128( 0x53f, 0x762d0a7eec57ddc3 ) },
  { 0x1.5913b586b2276p+74, -0x1.4a9f33d2953e6p+21, INIT_U128( 0x564, 0x4ed61ac89d56ac19 ) },
  { 0x1.8cab729b1956ep+74, -0x1.1a9c956435392p+21, INIT_U128( 0x632, 0xadca6c655b5cac6d ) },
  { 0x1.907ecf9520fdap+74, -0x1.c5affd1f8b5ffp+21, INIT_U128( 0x641, 0xfb3e5483f6474a00 ) },
  { 0x1.c3897cdb8712fp+75, -0x1.148c2cd829186p+22, INIT_U128( 0xe1c, 0x4be6dc38973adcf4 ) },
  { 0x1.66328026cc650p+75, -0x1.8d7c58211af8bp+22, INIT_U128( 0xb31, 0x94013663279ca0e9 ) },
  { 0x1.2a071f8e540e4p+75, -0x1.8ee042851dc08p+22, INIT_U128( 0x950, 0x38fc72a0719c47ef ) },
  { 0x1.e17de15fc2fbcp+75, -0x1.808fca0b011f9p+22, INIT_U128( 0xf0b, 0xef0afe17dd9fdc0d ) },
  { 0x1.c1a1b1d783436p+75, -0x1.e582dce1cb05bp+22, INIT_U128( 0xe0d, 0xd8ebc1a1a869f48 ) },
  { 0x1.302210b260442p+76, -0x1.1c153e8c382a8p+23, INIT_U128( 0x1302, 0x210b26044171f560 ) },
  { 0x1.36b1d3f26d63ap+76, -0x1.b174653162e8dp+23, INIT_U128( 0x136b, 0x1d3f26d6392745cd ) },
  { 0x1.77246250ee48cp+76, -0x1.b711c47b6e238p+23, INIT_U128( 0x1772, 0x46250ee48b24771d ) },
  { 0x1.cb9df155973bep+76, -0x1.03b9e3c80773cp+23, INIT_U128( 0x1cb9, 0xdf155973bd7e230e ) },
  { 0x1.91363beb226c8p+76, -0x1.f6c1a2aded835p+23, INIT_U128( 0x1913, 0x63beb226c7049f2e ) },
  { 0x1.85a1e5330b43dp+77, -0x1.0328c0f206518p+24, INIT_U128( 0x30b4, 0x3ca6616878fcd73f ) },
  { 0x1.83b88aff07711p+77, -0x1.55f82df8abf06p+24, INIT_U128( 0x3077, 0x115fe0ee20aa07d2 ) },
  { 0x1.3b7b0ad876f62p+77, -0x1.d1c099cda3813p+24, INIT_U128( 0x276f, 0x615b0edec22e3f66 ) },
  { 0x1.edd5d85bdbabbp+77, -0x1.0580b8aa0b017p+24, INIT_U128( 0x3dba, 0xbb0b7b7574fa7f47 ) },
  { 0x1.b6f51ddd6dea4p+77, -0x1.0dea732a1bd4ep+24, INIT_U128( 0x36de, 0xa3bbadbd46f2158c ) },
  { 0x1.c4fc37a789f87p+78, -0x1.cbbf476f977e9p+25, INIT_U128( 0x713f, 0xde9e27e18688171 ) },
  { 0x1.cf792b399ef26p+78, -0x1.5dd8c484bbb18p+25, INIT_U128( 0x73de, 0x4ace67bc95444e76 ) },
  { 0x1.549a6732a934dp+78, -0x1.82f0aed505e16p+25, INIT_U128( 0x5526, 0x99ccaa4d30fa1ea2 ) },
  { 0x1.86196fc90c32ep+78, -0x1.0022e6fe0045dp+25, INIT_U128( 0x6186, 0x5bf2430cb5ffba32 ) },
  { 0x1.561b14f8ac362p+78, -0x1.d9d7462fb3ae9p+25, INIT_U128( 0x5586, 0xc53e2b0d844c5173 ) },
  { 0x1.ab97dffd572fcp+79, -0x1.af3bdbd55e77cp+26, INIT_U128( 0xd5cb, 0xeffeab97d9431090 ) },
  { 0x1.b4bc948169792p+79, -0x1.259dda3c4b3bcp+26, INIT_U128( 0xda5e, 0x4a40b4bc8b698897 ) },
  { 0x1.33884d686710ap+79, -0x1.a9adb483535b6p+26, INIT_U128( 0x99c4, 0x26b433884959492d ) },
  { 0x1.756cfbf2eada0p+79, -0x1.029291e605252p+26, INIT_U128( 0xbab6, 0x7df9756cfbf5b5b8 ) },
  { 0x1.606b566cc0d6bp+79, -0x1.4b89488897129p+26, INIT_U128( 0xb035, 0xab36606b52d1dadd ) },
  { 0x1.ce79f75f9cf3fp+80, -0x1.c18250f58304ap+27, INIT_U128( 0x1ce79, 0xf75f9cf3e1f3ed78 ) },
  { 0x1.c1a537f5834a7p+80, -0x1.d09e1cffa13c3p+27, INIT_U128( 0x1c1a5, 0x37f5834a617b0f18 ) },
  { 0x1.aa75632d54eacp+80, -0x1.ad9199935b233p+27, INIT_U128( 0x1aa75, 0x632d54eab2937333 ) },
  { 0x1.5ccd7568b99aep+80, -0x1.a689a8214d135p+27, INIT_U128( 0x15ccd, 0x7568b99ad2cbb2be ) },
  { 0x1.cc27825f984f0p+80, -0x1.ad70602d5ae0cp+27, INIT_U128( 0x1cc27, 0x825f984ef2947cfe ) },
  { 0x1.571e14aeae3c2p+81, -0x1.1706eed62e0dep+28, INIT_U128( 0x2ae3c, 0x295d5c782e8f9112 ) },
  { 0x1.199980be33330p+81, -0x1.815302eb02a60p+28, INIT_U128( 0x23333, 0x17c6665e7eacfd1 ) },
  { 0x1.ccdc4b7399b8ap+81, -0x1.b706c6236e0d9p+28, INIT_U128( 0x399b8, 0x96e73371248f939d ) },
  { 0x1.78891716f1123p+81, -0x1.be4c9ed17c994p+28, INIT_U128( 0x2f112, 0x2e2de224441b3612 ) },
  { 0x1.8eeabbd11dd58p+81, -0x1.4e4f0bee9c9e2p+28, INIT_U128( 0x31dd5, 0x77a23baaeb1b0f41 ) },
  { 0x1.fbe0a583f7c14p+82, -0x1.fb4508bbf68a1p+29, INIT_U128( 0x7ef82, 0x960fdf04c0975ee8 ) },
  { 0x1.a1ceb0dd439d6p+82, -0x1.7ee85dacfdd0cp+29, INIT_U128( 0x6873a, 0xc3750e755022f44a ) },
  { 0x1.2c58bdda58b18p+82, -0x1.e773e123cee7cp+29, INIT_U128( 0x4b162, 0xf76962c5c31183db ) },
  { 0x1.fe7acffbfcf5ap+82, -0x1.65da57e2cbb4bp+29, INIT_U128( 0x7f9eb, 0x3feff3d65344b503 ) },
  { 0x1.e72d5acfce5acp+82, -0x1.61731afcc2e64p+29, INIT_U128( 0x79cb5, 0x6b3f396ad3d19ca0 ) },
  { 0x1.1879e12030f3cp+83, -0x1.c123bcb382477p+30, INIT_U128( 0x8c3cf, 0x901879d8fb710d3 ) },
  { 0x1.7f705b84fee0cp+83, -0x1.3f4993107e932p+30, INIT_U128( 0xbfb82, 0xdc27f705b02d9b3b ) },
  { 0x1.8d989f011b314p+83, -0x1.1d1fe3403a3fcp+30, INIT_U128( 0xc6cc4, 0xf808d989b8b8072f ) },
  { 0x1.1864ebfc30c9ep+83, -0x1.d728987fae513p+30, INIT_U128( 0x8c327, 0x5fe1864e8a35d9e0 ) },
  { 0x1.5610f62cac21fp+83, -0x1.4c5f844898bf0p+30, INIT_U128( 0xab087, 0xb165610f2ce81eed ) },
  { 0x1.33d3656c67a6cp+84, -0x1.95cfd3212b9fap+31, INIT_U128( 0x133d36, 0x56c67a6b3518166f ) },
  { 0x1.fbaa2b05f7546p+84, -0x1.55e269c0abc4dp+31, INIT_U128( 0x1fbaa2, 0xb05f7545550ecb1f ) },
  { 0x1.b75584bf6eab0p+84, -0x1.dc7bae0bb8f76p+31, INIT_U128( 0x1b7558, 0x4bf6eaaf11c228fa ) },
  { 0x1.9d9f6abf3b3edp+84, -0x1.e641152dcc822p+31, INIT_U128( 0x19d9f6, 0xabf3b3ec0cdf7569 ) },
  { 0x1.258aa83e4b155p+84, -0x1.2c03a55e58074p+31, INIT_U128( 0x1258aa, 0x83e4b15469fe2d50 ) },
  { 0x1.4923889e92471p+85, -0x1.732613eee64c2p+32, INIT_U128( 0x292471, 0x13d248e08cd9ec11 ) },
  { 0x1.4a4eac6c949d6p+85, -0x1.1ec3eb343d87ep+32, INIT_U128( 0x2949d5, 0x8d9293aae13c14cb ) },
  { 0x1.6b236f7ed646ep+85, -0x1.43e5e19087cbcp+32, INIT_U128( 0x2d646d, 0xefdac8dabc1a1e6f ) },
  { 0x1.d9b5b4b1b36b6p+85, -0x1.28a9dd345153cp+32, INIT_U128( 0x3b36b6, 0x96366d6ad75622cb ) },
  { 0x1.7e8c46f6fd189p+85, -0x1.1ba7907a374f2p+32, INIT_U128( 0x2fd188, 0xdedfa310e4586f85 ) },
  { 0x1.f8233e87f0468p+86, -0x1.4f1891a09e312p+33, INIT_U128( 0x7e08cf, 0xa1fc119d61cedcbe ) },
  { 0x1.3559a6ce6ab35p+86, -0x1.66838faecd072p+33, INIT_U128( 0x4d5669, 0xb39aacd132f8e0a2 ) },
  { 0x1.9287350d250e7p+86, -0x1.de3d4d25bc7a9p+33, INIT_U128( 0x64a1cd, 0x43494398438565b4 ) },
  { 0x1.51636112a2c6cp+86, -0x1.4aa4221095484p+33, INIT_U128( 0x5458d8, 0x44a8b1ad6ab7bbde ) },
  { 0x1.69680d46d2d02p+86, -0x1.58ef43c2b1de8p+33, INIT_U128( 0x5a5a03, 0x51b4b4054e21787a ) },
  { 0x1.9cd5255339aa5p+87, -0x1.5edb06bebdb61p+34, INIT_U128( 0xce6a92, 0xa99cd5228493e505 ) },
  { 0x1.8784e3f50f09cp+87, -0x1.44c75c08898ecp+34, INIT_U128( 0xc3c271, 0xfa8784daece28fdd ) },
  { 0x1.606eec84c0ddep+87, -0x1.04221f4608444p+34, INIT_U128( 0xb03776, 0x42606eebef7782e7 ) },
  { 0x1.533ab0e8a6756p+87, -0x1.41a4283883485p+34, INIT_U128( 0xa99d58, 0x74533aaaf96f5f1d ) },
  { 0x1.a827b7e7504f7p+87, -0x1.a12743df424e8p+34, INIT_U128( 0xd413db, 0xf3a827b17b62f082 ) },
  { 0x1.7f649104fec92p+88, -0x1.f1803d15e3007p+35, INIT_U128( 0x17f6491, 0x4fec91073fe1750 ) },
  { 0x1.3c245f267848cp+88, -0x1.538cdceea719cp+35, INIT_U128( 0x13c245f, 0x267848b56399188a ) },
  { 0x1.ac7c20b158f84p+88, -0x1.a88bafb751176p+35, INIT_U128( 0x1ac7c20, 0xb158f832bba28245 ) },
  { 0x1.0766b6760ecd7p+88, -0x1.ed9a3ae7db348p+35, INIT_U128( 0x10766b6, 0x760ecd60932e28c1 ) },
  { 0x1.326196f064c33p+88, -0x1.0bc6d9c8178dbp+35, INIT_U128( 0x1326196, 0xf064c327a1c931bf ) },
  { 0x1.0fe8f9101fd1fp+89, -0x1.d0efbbfda1df8p+36, INIT_U128( 0x21fd1f2, 0x203fa3c2f1044025 ) },
  { 0x1.1293e4362527cp+89, -0x1.d9b782d9b36f1p+36, INIT_U128( 0x22527c8, 0x6c4a4f626487d264 ) },
  { 0x1.7adec6b8f5bd9p+89, -0x1.cf30a87b9e615p+36, INIT_U128( 0x2f5bd8d, 0x71eb7b030cf57846 ) },
  { 0x1.c0b0e58f8161dp+89, -0x1.0bc433b817886p+36, INIT_U128( 0x38161cb, 0x1f02c38f43bcc47e ) },
  { 0x1.5f5110a6bea22p+89, -0x1.378549726f0a9p+36, INIT_U128( 0x2bea221, 0x4d7d442c87ab68d9 ) },
  { 0x1.e010f92bc021fp+90, -0x1.1c799a5838f34p+37, INIT_U128( 0x78043e4, 0xaf00879c70ccb4f8 ) },
  { 0x1.0a13d53e1427ap+90, -0x1.14fa976a29f53p+37, INIT_U128( 0x4284f54, 0xf8509e5d60ad12ba ) },
  { 0x1.20c171344182ep+90, -0x1.0b600f6a16c02p+37, INIT_U128( 0x48305c4, 0xd1060b5e93fe12bd ) },
  { 0x1.dc4fcb69b89fap+90, -0x1.baa89e2d75514p+37, INIT_U128( 0x7713f2d, 0xa6e27e48aaec3a51 ) },
  { 0x1.9460210728c04p+90, -0x1.7f8615f4ff0c2p+37, INIT_U128( 0x6518084, 0x1ca300d00f3d4160 ) },
  { 0x1.4eb6be6e9d6d8p+91, -0x1.9f09d4d53e13ap+38, INIT_U128( 0xa75b5f3, 0x74eb6b983d8acab0 ) },
  { 0x1.aace088d559c1p+91, -0x1.d48d8957a91b1p+38, INIT_U128( 0xd567044, 0x6aace00adc9daa15 ) },
  { 0x1.0c12d2ca1825ap+91, -0x1.ae9e99db5d3d3p+38, INIT_U128( 0x8609696, 0x50c12c9458598928 ) },
  { 0x1.cb18343596306p+91, -0x1.6b0ac1eed6158p+38, INIT_U128( 0xe58c1a1, 0xacb182a53d4f844a ) },
  { 0x1.8005595b000abp+91, -0x1.49fec5a893fd8p+38, INIT_U128( 0xc002aca, 0xd800552d804e95db ) },
  { 0x1.555ed514aabdap+92, -0x1.e0e7654dc1cecp+39, INIT_U128( 0x1555ed51, 0x4aabd90f8c4d591f ) },
  { 0x1.9b90037b37200p+92, -0x1.653f4916ca7e9p+39, INIT_U128( 0x19b90037, 0xb371ff4d605b749a ) },
  { 0x1.e4c55f0dc98acp+92, -0x1.ca319be194634p+39, INIT_U128( 0x1e4c55f0, 0xdc98ab1ae7320f35 ) },
  { 0x1.693b70b4d276ep+92, -0x1.48b14bc29162ap+39, INIT_U128( 0x1693b70b, 0x4d276d5ba75a1eb7 ) },
  { 0x1.1d3ed51a3a7dap+92, -0x1.4f20aa329e416p+39, INIT_U128( 0x11d3ed51, 0xa3a7d9586faae6b0 ) },
  { 0x1.fdbf440ffb7e8p+93, -0x1.3414dd086829cp+40, INIT_U128( 0x3fb7e881, 0xff6fcecbeb22f797 ) },
  { 0x1.9f1f9ef93e3f4p+93, -0x1.e910ef75d221ep+40, INIT_U128( 0x33e3f3df, 0x27c7e616ef108a2d ) },
  { 0x1.330fb51e661f6p+93, -0x1.afa4f75f5f49fp+40, INIT_U128( 0x2661f6a3, 0xccc3ea505b08a0a0 ) },
  { 0x1.908c5cad2118cp+93, -0x1.915cf7bd22b9fp+40, INIT_U128( 0x32118b95, 0xa423166ea30842dd ) },
  { 0x1.f69c40d5ed388p+93, -0x1.daaf7ad5b55f0p+40, INIT_U128( 0x3ed3881a, 0xbda70e2550852a4a ) },
  { 0x1.a4e116b749c23p+94, -0x1.9b3b90ed36772p+41, INIT_U128( 0x693845ad, 0xd27088c988de2593 ) },
  { 0x1.e3ecdcb1c7d9bp+94, -0x1.51c17edca3830p+41, INIT_U128( 0x78fb372c, 0x71f6695c7d0246b8 ) },
  { 0x1.a0e2291141c45p+94, -0x1.404e932a809d2p+41, INIT_U128( 0x68388a44, 0x5071117f62d9aafe ) },
  { 0x1.74637034e8c6ep+94, -0x1.3ff2cd8c7fe5ap+41, INIT_U128( 0x5d18dc0d, 0x3a31b5801a64e700 ) },
  { 0x1.2903bd0252078p+94, -0x1.057c671e0af8dp+41, INIT_U128( 0x4a40ef40, 0x9481ddf50731c3ea ) },
  { 0x1.b5215e7f6a42cp+95, -0x1.f293ca6be527ap+42, INIT_U128( 0xda90af3f, 0xb5215835b0d6506b ) },
  { 0x1.83f0c22907e18p+95, -0x1.9e9f270b3d3e5p+42, INIT_U128( 0xc1f86114, 0x83f0b9858363d30b ) },
  { 0x1.794ea1e0f29d4p+95, -0x1.cdd6a0019bad4p+42, INIT_U128( 0xbca750f0, 0x794e98c8a57ff991 ) },
  { 0x1.691230eed2246p+95, -0x1.64cc6042c998cp+42, INIT_U128( 0xb4891877, 0x69122a6cce7ef4d9 ) },
  { 0x1.b68ddf1f6d1bcp+95, -0x1.4e7240b69ce48p+42, INIT_U128( 0xdb46ef8f, 0xb68ddac636fd258c ) },
  { 0x1.8ec0d3e11d81ap+96, -0x1.78a8191ef1503p+43, INIT_U128( 0x18ec0d3e1, 0x1d81943abf370875 ) },
  { 0x1.b89ae6ad7135dp+96, -0x1.5d8a5ab6bb14cp+43, INIT_U128( 0x1b89ae6ad, 0x7135c513ad2a4a27 ) },
  { 0x1.6306c1d6c60d8p+96, -0x1.494f5aca929ecp+43, INIT_U128( 0x16306c1d6, 0xc60d75b58529ab6b ) },
  { 0x1.3be3253677c64p+96, -0x1.2234122844682p+43, INIT_U128( 0x13be32536, 0x77c636ee5f6ebddc ) },
  { 0x1.c0d3393581a67p+96, -0x1.cf9ff6dd9f3ffp+43, INIT_U128( 0x1c0d33935, 0x81a6618300491306 ) },
  { 0x1.9d7696df3aed3p+97, -0x1.904f8d43209f2p+44, INIT_U128( 0x33aed2dbe, 0x75da46fb072bcdf6 ) },
  { 0x1.f51bed87ea37dp+97, -0x1.849dd3b5093bap+44, INIT_U128( 0x3ea37db0f, 0xd46f87b622c4af6c ) },
  { 0x1.d310146da6202p+97, -0x1.a1b294dd43652p+44, INIT_U128( 0x3a62028db, 0x4c4025e4d6b22bc9 ) },
  { 0x1.1a5aadfa34b56p+97, -0x1.f92b4a29f256ap+44, INIT_U128( 0x234b55bf4, 0x696aa06d4b5d60da ) },
  { 0x1.56b5a4e0ad6b4p+97, -0x1.7b9c7faaf7390p+44, INIT_U128( 0x2ad6b49c1, 0x5ad668463805508c ) },
  { 0x1.c4ebda7789d7bp+98, -0x1.f2464e2be48cap+45, INIT_U128( 0x713af69de, 0x275e81b7363a836e ) },
  { 0x1.147a605228f4cp+98, -0x1.2fba62965f74cp+45, INIT_U128( 0x451e98148, 0xa3d2da08b3ad3411 ) },
  { 0x1.ff5a5975feb4bp+98, -0x1.fb03d16ff607ap+45, INIT_U128( 0x7fd6965d7, 0xfad2809f85d2013f ) },
  { 0x1.8b535eb316a6cp+98, -0x1.67c1084ecf821p+45, INIT_U128( 0x62d4d7acc, 0x5a9ad307def6260f ) },
  { 0x1.ec5edc05d8bdcp+98, -0x1.f6e18d93edc31p+45, INIT_U128( 0x7b17b7017, 0x62f6c123ce4d8247 ) },
  { 0x1.9894226f31284p+99, -0x1.2f32edb85e65ep+46, INIT_U128( 0xcc4a11379, 0x8941b4334491e866 ) },
  { 0x1.285e37e250bc7p+99, -0x1.863505270c6a1p+46, INIT_U128( 0x942f1bf12, 0x85e31e72beb63ce5 ) },
  { 0x1.01d184d203a30p+99, -0x1.3a789d6c74f14p+46, INIT_U128( 0x80e8c2690, 0x1d17b161d8a4e2c3 ) },
  { 0x1.2cbeeb98597dep+99, -0x1.0ab4ca901569ap+46, INIT_U128( 0x965f75cc2, 0xcbeebd52cd5bfaa5 ) },
  { 0x1.3a6f2d2874de6p+99, -0x1.882f4607105e9p+46, INIT_U128( 0x9d3796943, 0xa6f29df42e7e3be8 ) },
  { 0x1.8333177706663p+100, -0x1.25f65a144beccp+47, INIT_U128( 0x1833317770, 0x66626d04d2f5da09 ) },
  { 0x1.8db762651b6ecp+100, -0x1.79f42f14f3e86p+47, INIT_U128( 0x18db762651, 0xb6eb4305e875860b ) },
  { 0x1.70d5bca8e1ab8p+100, -0x1.bb0edb49761dcp+47, INIT_U128( 0x170d5bca8e, 0x1ab72278925b44f1 ) },
  { 0x1.ea41c903d4839p+100, -0x1.5ac507a0b58a1p+47, INIT_U128( 0x1ea41c903d, 0x4838529d7c2fa53a ) },
  { 0x1.e0fa0cc9c1f41p+100, -0x1.423d1498847a2p+47, INIT_U128( 0x1e0fa0cc9c, 0x1f405ee175b3bdc2 ) },
  { 0x1.7cb5b206f96b6p+101, -0x1.ca84212d95084p+48, INIT_U128( 0x2f96b640df, 0x2d6a357bded26af7 ) },
  { 0x1.d538ebbfaa71ep+101, -0x1.71154fe0e22aap+48, INIT_U128( 0x3aa71d77f5, 0x4e3a8eeab01f1dd5 ) },
  { 0x1.f94495e9f2892p+101, -0x1.0134cf020269ap+48, INIT_U128( 0x3f2892bd3e, 0x5122fecb30fdfd96 ) },
  { 0x1.272b29184e565p+101, -0x1.200c167a40183p+48, INIT_U128( 0x24e5652309, 0xcac8dff3e985bfe7 ) },
  { 0x1.9a00459134009p+101, -0x1.19f3e5f233e7cp+48, INIT_U128( 0x334008b226, 0x8010e60c1a0dcc18 ) },
  { 0x1.d502c13daa058p+102, -0x1.687df8cad0fbfp+49, INIT_U128( 0x7540b04f6a, 0x815d2f040e6a5e08 ) },
  { 0x1.c343006b86860p+102, -0x1.d09890d7a1312p+49, INIT_U128( 0x70d0c01ae1, 0xa17c5ecede50bd9d ) },
  { 0x1.12d5a57025ab4p+102, -0x1.639868d2c730dp+49, INIT_U128( 0x44b5695c09, 0x6acd38cf2e5a719e ) },
  { 0x1.2e1c8a005c392p+102, -0x1.30918fb861232p+49, INIT_U128( 0x4b87228017, 0xe459edce08f3db9 ) },
  { 0x1.a1a6df9b434dcp+102, -0x1.23addea8475bcp+49, INIT_U128( 0x6869b7e6d0, 0xd36db8a442af7148 ) },
  { 0x1.24e2adac49c56p+103, -0x1.5c6f2e30b8de6p+50, INIT_U128( 0x927156d624, 0xe2aa8e43473d1c86 ) },
  { 0x1.7c7b44aaf8f68p+103, -0x1.a261467144c29p+50, INIT_U128( 0xbe3da2557c, 0x7b39767ae63aecf5 ) },
  { 0x1.bc552c6578aa6p+103, -0x1.2fe92c325fd26p+50, INIT_U128( 0xde2a9632bc, 0x552b405b4f3680b6 ) },
  { 0x1.986724fd30ce4p+103, -0x1.35916dc86b22ep+50, INIT_U128( 0xcc33927e98, 0x671b29ba48de5374 ) },
  { 0x1.0629fea00c540p+103, -0x1.2281030845020p+50, INIT_U128( 0x8314ff5006, 0x29fb75fbf3deebf8 ) },
  { 0x1.86196bdd0c32ep+104, -0x1.86efd1710ddfap+51, INIT_U128( 0x186196bdd0c, 0x32d3c88174779103 ) },
  { 0x1.b781612b6f02cp+104, -0x1.e18af0a9c315ep+51, INIT_U128( 0x1b781612b6f, 0x2b0f3a87ab1e751 ) },
  { 0x1.0538cf500a71ap+104, -0x1.ca88fa479511fp+51, INIT_U128( 0x10538cf500a, 0x7191abb82dc35770 ) },
  { 0x1.26f828784df05p+104, -0x1.0b76d92e16edbp+51, INIT_U128( 0x126f828784d, 0xf047a449368f4892 ) },
  { 0x1.add81fa15bb04p+104, -0x1.7719b60cee337p+51, INIT_U128( 0x1add81fa15b, 0xb03447324f988e64 ) },
  { 0x1.a702e23f4e05cp+105, -0x1.db4083d9b6810p+52, INIT_U128( 0x34e05c47e9c, 0xb624bf7c26497f0 ) },
  { 0x1.10d1f1a221a3ep+105, -0x1.4c21b18c98436p+52, INIT_U128( 0x221a3e34443, 0x47ab3de4e7367bca ) },
  { 0x1.6ea2b496dd456p+105, -0x1.9afb5f4735f6cp+52, INIT_U128( 0x2dd45692dba, 0x8aa6504a0b8ca094 ) },
  { 0x1.21eb30b643d66p+105, -0x1.d8868fe7b10d2p+52, INIT_U128( 0x243d6616c87, 0xaca277970184ef2e ) },
  { 0x1.ba11808f74230p+105, -0x1.607d81f2c0fb0p+52, INIT_U128( 0x37423011ee8, 0x45e9f827e0d3f050 ) },
  { 0x1.2b067c3a560d0p+106, -0x1.26a888f24d511p+53, INIT_U128( 0x4ac19f0e958, 0x33db2aeee1b655de ) },
  { 0x1.321f6220643ecp+106, -0x1.5f8f20a6bf1e4p+53, INIT_U128( 0x4c87d888190, 0xfad40e1beb281c38 ) },
  { 0x1.d6a8d6b3ad51bp+106, -0x1.af61093b5ec21p+53, INIT_U128( 0x75aa35aceb5, 0x468a13ded89427be ) },
  { 0x1.8aa18d1315432p+106, -0x1.45aa07728b541p+53, INIT_U128( 0x62a86344c55, 0xc574abf11ae957e ) },
  { 0x1.91cbb90123977p+106, -0x1.5829ecd8b053ep+53, INIT_U128( 0x6472ee4048e, 0x5d94fac264e9f584 ) },
  { 0x1.f3254c21e64a9p+107, -0x1.e26b0355c4d61p+54, INIT_U128( 0xf992a610f32, 0x5407653f2a8eca7c ) },
  { 0x1.baf1933b75e32p+107, -0x1.9e1ca4613c394p+54, INIT_U128( 0xdd78c99dbaf, 0x189878d6e7b0f1b0 ) },
  { 0x1.fe9a2653fd345p+107, -0x1.2111e1644223cp+54, INIT_U128( 0xff4d1329fe9, 0xa237bb87a6ef7710 ) },
  { 0x1.47ba6a568f74ep+107, -0x1.1db3308e3b666p+54, INIT_U128( 0xa3dd352b47b, 0xa6b89333dc712668 ) },
  { 0x1.7d425134fa84ap+107, -0x1.7111591ae222bp+54, INIT_U128( 0xbea1289a7d4, 0x24a3bba9b9477754 ) },
  { 0x1.2072625640e4cp+108, -0x1.a3f61c0747ec4p+55, INIT_U128( 0x12072625640e, 0x4b2e04f1fc5c09e0 ) },
  { 0x1.e9a0237bd3405p+108, -0x1.8eaca10f1d594p+55, INIT_U128( 0x1e9a0237bd34, 0x438a9af78715360 ) },
  { 0x1.e6d68de9cdad2p+108, -0x1.f8f6dd1df1edbp+55, INIT_U128( 0x1e6d68de9cda, 0xd103849171070928 ) },
  { 0x1.db61e073b6c3cp+108, -0x1.7104d820e209bp+55, INIT_U128( 0x1db61e073b6c, 0x3b477d93ef8efb28 ) },
  { 0x1.0f217c801e430p+108, -0x1.dd89659fbb12dp+55, INIT_U128( 0x10f217c801e4, 0x2f113b4d30227698 ) },
  { 0x1.fe9b4705fd369p+109, -0x1.3fa3eb287f47ep+56, INIT_U128( 0x3fd368e0bfa6, 0xd0c05c14d780b820 ) },
  { 0x1.5d571bb6baae4p+109, -0x1.b2fb8ca765f72p+56, INIT_U128( 0x2baae376d755, 0xc64d0473589a08e0 ) },
  { 0x1.240ce5c24819cp+109, -0x1.1a00cc123401ap+56, INIT_U128( 0x24819cb84903, 0x36e5ff33edcbfe60 ) },
  { 0x1.bdf6e81b7beddp+109, -0x1.001b75b40036ep+56, INIT_U128( 0x37bedd036f7d, 0xb8ffe48a4bffc920 ) },
  { 0x1.8ec816c51d903p+109, -0x1.361aa5a66c354p+56, INIT_U128( 0x31d902d8a3b2, 0x4c9e55a5993cac0 ) },
  { 0x1.2f1182c45e230p+110, -0x1.9edd32253dba6p+57, INIT_U128( 0x4bc460b11788, 0xbcc2459bb5848b40 ) },
  { 0x1.0fe20a461fc42p+110, -0x1.3ea6d59e7d4dap+57, INIT_U128( 0x43f8829187f1, 0x582b254c30564c0 ) },
  { 0x1.1c303b3e38608p+110, -0x1.ac1d5e63583acp+57, INIT_U128( 0x470c0ecf8e18, 0x1ca7c543394f8a80 ) },
  { 0x1.ebc1498bd7829p+110, -0x1.641b6f02c836ep+57, INIT_U128( 0x7af05262f5e0, 0xa137c921fa6f9240 ) },
  { 0x1.1716fbaa2e2e0p+110, -0x1.2150b17642a16p+57, INIT_U128( 0x45c5beea8b8b, 0x7dbd5e9d137abd40 ) },
  { 0x1.3b517d8a76a30p+111, -0x1.52366708a46cdp+58, INIT_U128( 0x9da8bec53b51, 0x7ab72663dd6e4cc0 ) },
  { 0x1.5251dcbea4a3cp+111, -0x1.6f366110de6ccp+58, INIT_U128( 0xa928ee5f5251, 0xda43267bbc864d00 ) },
  { 0x1.a55106b94aa21p+111, -0x1.c3e2594f87c4bp+58, INIT_U128( 0xd2a8835ca551, 0xf0769ac1e0ed40 ) },
  { 0x1.62bf8b74c57f2p+111, -0x1.a65316f14ca63p+58, INIT_U128( 0xb15fc5ba62bf, 0x8966b3a43acd6740 ) },
  { 0x1.e4f9ec8fc9f3dp+111, -0x1.7f9cc39eff398p+58, INIT_U128( 0xf27cf647e4f9, 0xe2018cf184031a00 ) },
  { 0x1.2bba220257744p+112, -0x1.be6446237cc89p+59, INIT_U128( 0x12bba22025774, 0x320cddcee419bb80 ) },
  { 0x1.227fb52644ff6p+112, -0x1.7d16ddb2fa2dcp+59, INIT_U128( 0x1227fb52644ff, 0x54174912682e9200 ) },
  { 0x1.9e678f2b3ccf2p+112, -0x1.8445a305088b4p+59, INIT_U128( 0x19e678f2b3ccf, 0x13ddd2e7d7bba600 ) },
  { 0x1.d67c0a0facf81p+112, -0x1.f45dbbcfe8bb8p+59, INIT_U128( 0x1d67c0a0facf8, 0x5d122180ba2400 ) },
  { 0x1.63941cdec7284p+112, -0x1.73e0e39ae7c1cp+59, INIT_U128( 0x163941cdec728, 0x3460f8e328c1f200 ) },
  { 0x1.37c33cea6f868p+113, -0x1.60a0728ac140ep+60, INIT_U128( 0x26f8679d4df0c, 0xe9f5f8d753ebf200 ) },
  { 0x1.06f3897a0de71p+113, -0x1.bc1d99cf783b3p+60, INIT_U128( 0x20de712f41bce, 0x43e2663087c4d00 ) },
  { 0x1.3bba5d6e7774cp+113, -0x1.9ec97aa13d92fp+60, INIT_U128( 0x27774badceee9, 0x66136855ec26d100 ) },
  { 0x1.5d2dc0c4ba5b8p+113, -0x1.090d47ec121a9p+60, INIT_U128( 0x2ba5b818974b6, 0xef6f2b813ede5700 ) },
  { 0x1.862425a10c485p+113, -0x1.df83bbd5bf078p+60, INIT_U128( 0x30c484b421890, 0x8207c442a40f8800 ) },
  { 0x1.63c95ed8c792cp+114, -0x1.2dfeb9045bfd7p+61, INIT_U128( 0x58f257b631e4a, 0xda4028df74805200 ) },
  { 0x1.225c74d244b8ep+114, -0x1.d14633e5a28c6p+61, INIT_U128( 0x48971d34912e3, 0x45d739834bae7400 ) },
  { 0x1.452717808a4e3p+114, -0x1.6cfb2a0cd9f66p+61, INIT_U128( 0x5149c5e022938, 0x92609abe64c13400 ) },
  { 0x1.cd9b9c979b373p+114, -0x1.6c1d6264d83acp+61, INIT_U128( 0x7366e725e6cdc, 0x927c53b364f8a800 ) },
  { 0x1.283afc3450760p+114, -0x1.24917ae649230p+61, INIT_U128( 0x4a0ebf0d141d7, 0xdb6dd0a336dba000 ) },
  { 0x1.c35af58986b5fp+115, -0x1.ad9439fb5b287p+62, INIT_U128( 0xe1ad7ac4c35af, 0x149af1812935e400 ) },
  { 0x1.b6a181e96d430p+115, -0x1.4a59c41894b38p+62, INIT_U128( 0xdb50c0f4b6a17, 0xad698ef9dad32000 ) },
  { 0x1.29dd7fee53bb0p+115, -0x1.597a52fab2f4ap+62, INIT_U128( 0x94eebff729dd7, 0xa9a16b415342d800 ) },
  { 0x1.87790c010ef22p+115, -0x1.5f961ddebf2c4p+62, INIT_U128( 0xc3bc860087790, 0xa81a78885034f000 ) },
  { 0x1.30dc51c261b8ap+115, -0x1.2aff976855ff3p+62, INIT_U128( 0x986e28e130dc4, 0xb5401a25ea803400 ) },
  { 0x1.97fac1772ff58p+116, -0x1.be9bc7d57d379p+63, INIT_U128( 0x197fac1772ff57, 0x20b21c1541643800 ) },
  { 0x1.2ab92a3a55726p+116, -0x1.0a5ada7c14b5cp+63, INIT_U128( 0x12ab92a3a55725, 0x7ad292c1f5a52000 ) },
  { 0x1.2eec55645dd8ap+116, -0x1.faa22043f5444p+63, INIT_U128( 0x12eec55645dd89, 0x2aeefde055de000 ) },
  { 0x1.a8365a3d506cbp+116, -0x1.8253592d04a6bp+63, INIT_U128( 0x1a8365a3d506ca, 0x3ed653697daca800 ) },
  { 0x1.ccad1d47995a3p+116, -0x1.b3062d77660c6p+63, INIT_U128( 0x1ccad1d47995a2, 0x267ce9444cf9d000 ) },
  { 0x1.0c80807219010p+117, -0x1.fceee00bf9ddcp+64, INIT_U128( 0x2190100e43201e, 0x3111ff406224000 ) },
  { 0x1.5c84fde8b90a0p+117, -0x1.4c1dc3b0983b8p+64, INIT_U128( 0x2b909fbd17213e, 0xb3e23c4f67c48000 ) },
  { 0x1.771d5b1aee3acp+117, -0x1.6d951fd0db2a4p+64, INIT_U128( 0x2ee3ab635dc756, 0x926ae02f24d5c000 ) },
  { 0x1.2c30e7a65861dp+117, -0x1.f65276a3eca4fp+64, INIT_U128( 0x25861cf4cb0c38, 0x9ad895c135b1000 ) },
  { 0x1.00ad5120015aap+117, -0x1.aed3fec35da80p+64, INIT_U128( 0x2015aa24002b52, 0x512c013ca2580000 ) },
  { 0x1.a9761f0752ec4p+118, -0x1.60c7b866c18f7p+65, INIT_U128( 0x6a5d87c1d4bb0d, 0x3e708f327ce12000 ) },
  { 0x1.e31b1195c6362p+118, -0x1.34b0f2ae6961ep+65, INIT_U128( 0x78c6c465718d85, 0x969e1aa32d3c4000 ) },
  { 0x1.e7b72027cf6e4p+118, -0x1.4c21d91c9843bp+65, INIT_U128( 0x79edc809f3db8d, 0x67bc4dc6cf78a000 ) },
  { 0x1.4910b92892217p+118, -0x1.2a84481055089p+65, INIT_U128( 0x52442e4a248859, 0xaaf76fdf55eee000 ) },
  { 0x1.76a20bc0ed442p+118, -0x1.c6003a578c007p+65, INIT_U128( 0x5da882f03b5104, 0x73ff8b50e7ff2000 ) },
  { 0x1.33955a28672acp+119, -0x1.365bbfb26cb78p+66, INIT_U128( 0x99caad1433955b, 0x269101364d220000 ) },
  { 0x1.2cae3656595c7p+119, -0x1.7185fe0ee30c0p+66, INIT_U128( 0x96571b2b2cae32, 0x39e807c473d00000 ) },
  { 0x1.2059fcd240b40p+119, -0x1.608e8c06c11d2p+66, INIT_U128( 0x902cfe692059fa, 0x7dc5cfe4fb8b8000 ) },
  { 0x1.c156446f82ac8p+119, -0x1.7ed61f72fdac4p+66, INIT_U128( 0xe0ab2237c1563a, 0x4a78234094f0000 ) },
  { 0x1.b3871223670e2p+119, -0x1.cc00ea8b9801ep+66, INIT_U128( 0xd9c38911b38708, 0xcffc55d19ff88000 ) },
  { 0x1.1822c3a830458p+120, -0x1.fa8c50f1f518ap+67, INIT_U128( 0x11822c3a8304570, 0x2b9d7870573b0000 ) },
  { 0x1.251099024a213p+120, -0x1.17a6fff02f4e0p+67, INIT_U128( 0x1251099024a2127, 0x42c8007e85900000 ) },
  { 0x1.dda73a49bb4e7p+120, -0x1.39c03aa673808p+67, INIT_U128( 0x1dda73a49bb4e66, 0x31fe2acc63fc0000 ) },
  { 0x1.701bfa5ae0380p+120, -0x1.86498f110c932p+67, INIT_U128( 0x1701bfa5ae037f3, 0xcdb387779b670000 ) },
  { 0x1.f7cf8d21ef9f1p+120, -0x1.27b329f64f665p+67, INIT_U128( 0x1f7cf8d21ef9f06, 0xc266b04d84cd8000 ) },
  { 0x1.8841b5fd10837p+121, -0x1.271ae62a4e35dp+68, INIT_U128( 0x310836bfa2106cd, 0x8e519d5b1ca30000 ) },
  { 0x1.a5f1059d4be21p+121, -0x1.fd763aeffaec8p+68, INIT_U128( 0x34be20b3a97c400, 0x289c510051380000 ) },
  { 0x1.c977257b92ee4p+121, -0x1.4242d5508485ap+68, INIT_U128( 0x392ee4af725dc6b, 0xdbd2aaf7b7a60000 ) },
  { 0x1.3ccf2642799e5p+121, -0x1.842c1cef08584p+68, INIT_U128( 0x2799e4c84f33c87, 0xbd3e310f7a7c0000 ) },
  { 0x1.9bd31ce137a64p+121, -0x1.499699de932d3p+68, INIT_U128( 0x337a639c26f4c6b, 0x66966216cd2d0000 ) },
  { 0x1.8356d48506adap+122, -0x1.87046fad0e08ep+69, INIT_U128( 0x60d5b52141ab64f, 0x1f720a5e3ee40000 ) },
  { 0x1.0d4dfebc1a9c0p+122, -0x1.fbb9c1dff7738p+69, INIT_U128( 0x43537faf06a6fc0, 0x88c7c40111900000 ) },
  { 0x1.236cebd446d9ep+122, -0x1.5a62b7beb4c57p+69, INIT_U128( 0x48db3af511b6754, 0xb3a9082967520000 ) },
  { 0x1.817d9deb02fb4p+122, -0x1.ff40bba3fe818p+69, INIT_U128( 0x605f677ac0becc0, 0x17e88b802fd00000 ) },
  { 0x1.fcb6f693f96dfp+122, -0x1.e34fecb1c69fdp+69, INIT_U128( 0x7f2dbda4fe5b783, 0x960269c72c060000 ) },
  { 0x1.2592c2024b258p+123, -0x1.759d0554eb3a0p+70, INIT_U128( 0x92c961012592ba2, 0x98beaac531800000 ) },
  { 0x1.f9c92243f3925p+123, -0x1.ea956b2fd52aep+70, INIT_U128( 0xfce49121f9c9205, 0x5aa5340ab5480000 ) },
  { 0x1.342c050c68580p+123, -0x1.c1d40b9783a82p+70, INIT_U128( 0x9a160286342bf8f, 0x8afd1a1f15f80000 ) },
  { 0x1.48d87b4291b10p+123, -0x1.99cd81eb339b0p+70, INIT_U128( 0xa46c3da148d8799, 0x8c9f853319400000 ) },
  { 0x1.958743c92b0e8p+123, -0x1.3b32a80876655p+70, INIT_U128( 0xcac3a1e495873b1, 0x3355fde266ac0000 ) },
  { 0x1.95955fc92b2acp+124, -0x1.788b1bb4f1164p+71, INIT_U128( 0x195955fc92b2ab43, 0xba72258774e00000 ) },
  { 0x1.77cbd758ef97bp+124, -0x1.59e96db4b3d2ep+71, INIT_U128( 0x177cbd758ef97a53, 0xb4925a616900000 ) },
  { 0x1.3d395a667a72cp+124, -0x1.bf685c877ed0cp+71, INIT_U128( 0x13d395a667a72b20, 0x4bd1bc4097a00000 ) },
  { 0x1.350561da6a0acp+124, -0x1.e278ef33c4f1ep+71, INIT_U128( 0x1350561da6a0ab0e, 0xc388661d87100000 ) },
  { 0x1.cd0e5a619a1cbp+124, -0x1.896fac3b12df6p+71, INIT_U128( 0x1cd0e5a619a1ca3b, 0x4829e27690500000 ) },
  { 0x1.4b43b7d296877p+125, -0x1.d3815993a702bp+72, INIT_U128( 0x296876fa52d0ec2c, 0x7ea66c58fd500000 ) },
  { 0x1.6d4edb7eda9dcp+125, -0x1.63b2e178c765cp+72, INIT_U128( 0x2da9db6fdb53b69c, 0x4d1e87389a400000 ) },
  { 0x1.9ee589813dcb1p+125, -0x1.4586b2f88b0d6p+72, INIT_U128( 0x33dcb13027b960ba, 0x794d0774f2a00000 ) },
  { 0x1.be11ceaf7c23ap+125, -0x1.206d205c40da4p+72, INIT_U128( 0x37c239d5ef8472df, 0x92dfa3bf25c00000 ) },
  { 0x1.7a568e10f4ad2p+125, -0x1.8a0f883b141f1p+72, INIT_U128( 0x2f4ad1c21e95a275, 0xf077c4ebe0f00000 ) },
  { 0x1.4977e4c292efcp+126, -0x1.e5115df1ca22cp+73, INIT_U128( 0x525df930a4bbec35, 0xdd441c6bba800000 ) },
  { 0x1.c78a4d258f149p+126, -0x1.df8a0993bf141p+73, INIT_U128( 0x71e2934963c52040, 0xebecd881d7e00000 ) },
  { 0x1.68f9b5ced1f36p+126, -0x1.71b68e02e36d2p+73, INIT_U128( 0x5a3e6d73b47cd51c, 0x92e3fa3925c00000 ) },
  { 0x1.4537e7968a6fdp+126, -0x1.4e903ffa9d208p+73, INIT_U128( 0x514df9e5a29bf162, 0xdf800ac5bf000000 ) },
  { 0x1.b243217364864p+126, -0x1.329c4dd06538ap+73, INIT_U128( 0x6c90c85cd9218d9a, 0xc7645f358ec00000 ) },
  { 0x1.782300caf0460p+127, -0x1.71273148e24e6p+74, INIT_U128( 0xbc1180657822fa3b, 0x633adc76c6800000 ) },
  { 0x1.919451a52328ap+127, -0x1.d1317b3fa2630p+74, INIT_U128( 0xc8ca28d2919448bb, 0x3a13017674000000 ) },
  { 0x1.5b41037ab6820p+127, -0x1.1672e9762ce5dp+74, INIT_U128( 0xada081bd5b40fba6, 0x345a274c68c00000 ) },
  { 0x1.fb844769f7089p+127, -0x1.ddd6cd13bbad9p+74, INIT_U128( 0xfdc223b4fb844088, 0xa4cbb11149c00000 ) },
  { 0x1.bd60d6a77ac1bp+127, -0x1.d081f99da103fp+74, INIT_U128( 0xdeb06b53bd60d0bd, 0xf819897bf0400000 ) }
};

static const int numTests = sizeof(testList) / sizeof(struct testCase);

