package org.multiverse.commitbarriers;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.multiverse.TestThread;

import java.util.concurrent.TimeUnit;

import static org.junit.Assert.*;
import static org.multiverse.TestUtils.*;
import static org.multiverse.api.TxnThreadLocal.clearThreadLocalTxn;

public class CountDownCommitBarrier_tryAwaitOpenTest {
    private CountDownCommitBarrier barrier;

    @Before
    public void setUp() {
        clearThreadLocalTxn();
        clearCurrentThreadInterruptedStatus();
    }

    @After
    public void tearDown() {
        clearCurrentThreadInterruptedStatus();
    }

    @Test
    public void whenNullTimeUnit_thenNullPointerException() throws InterruptedException {
        barrier = new CountDownCommitBarrier(1);

        try {
            barrier.tryAwaitOpen(10, null);
            fail();
        } catch (NullPointerException expected) {
        }

        assertTrue(barrier.isClosed());
    }

    @Test
    public void whenNegativeTimeoutAndBufferedOpen() throws InterruptedException {
        barrier = new CountDownCommitBarrier(1);

        boolean result = barrier.tryAwaitOpen(-1, TimeUnit.DAYS);
        assertFalse(result);
        assertTrue(barrier.isClosed());
    }

    @Test
    public void whenAbortedWhileWaiting() throws InterruptedException {
        barrier = new CountDownCommitBarrier(2);

        TestThread t = new TestThread() {
            @Override
            public void doRun() throws Exception {
                boolean success = barrier.tryAwaitOpen(1, TimeUnit.DAYS);
                assertTrue(success);
            }
        };
        t.start();
        sleepMs(500);

        barrier.abort();

        assertEventuallyNotAlive(t);
        assertNothingThrown(t);
        assertTrue(barrier.isAborted());
        assertEquals(0, barrier.getNumberWaiting());
    }

    @Test
    public void whenCommittedWhileWaiting() throws InterruptedException {
        barrier = new CountDownCommitBarrier(1);

        TestThread t = new TestThread() {
            @Override
            public void doRun() throws Exception {
                boolean success = barrier.tryAwaitOpen(1, TimeUnit.DAYS);
                assertTrue(success);
            }
        };
        t.start();
        sleepMs(500);

        barrier.countDown();

        t.join();
        assertNothingThrown(t);
        assertTrue(barrier.isCommitted());
        assertEquals(0, barrier.getNumberWaiting());
    }

    @Test
    public void whenInterruptedWhileWaiting() throws InterruptedException {
        barrier = new CountDownCommitBarrier(2);

        TestThread t = new TestThread() {
            @Override
            public void doRun() throws Exception {
                barrier.tryAwaitOpen(1, TimeUnit.DAYS);
                fail();
            }
        };
        t.setPrintStackTrace(false);
        t.start();
        sleepMs(500);

        t.interrupt();

        t.join();
        t.assertFailedWithException(InterruptedException.class);
        assertTrue(barrier.isClosed());
        assertEquals(0, barrier.getNumberWaiting());
    }

    @Test
    public void whenTimeoutWhileWaiting() throws InterruptedException {
        barrier = new CountDownCommitBarrier(2);

        TestThread t = new TestThread() {
            @Override
            public void doRun() throws Exception {
                boolean success = barrier.tryAwaitOpen(1, TimeUnit.SECONDS);
                assertFalse(success);
            }
        };
        t.start();
        t.join();
        assertTrue(barrier.isClosed());
    }

    @Test
    public void whenAborted() throws InterruptedException {
        barrier = new CountDownCommitBarrier(1);
        barrier.abort();

        boolean result = barrier.tryAwaitOpen(1, TimeUnit.DAYS);
        assertTrue(result);
        assertTrue(barrier.isAborted());
    }

    @Test
    public void whenCommitted() throws InterruptedException {
        barrier = new CountDownCommitBarrier(0);

        boolean result = barrier.tryAwaitOpen(1, TimeUnit.DAYS);
        assertTrue(result);
        assertTrue(barrier.isCommitted());
    }
}
