package aws

import (
	"fmt"
	"log"
	"regexp"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/hashicorp/terraform-plugin-sdk/helper/schema"
)

// setTags is a helper to set the tags for a resource. It expects the
// tags field to be named "tags"
func setTagsSSM(conn *ssm.SSM, d *schema.ResourceData, id, resourceType string) error {
	if d.HasChange("tags") {
		oraw, nraw := d.GetChange("tags")
		o := oraw.(map[string]interface{})
		n := nraw.(map[string]interface{})
		create, remove := diffTagsSSM(tagsFromMapSSM(o), tagsFromMapSSM(n))

		// Set tags
		if len(remove) > 0 {
			log.Printf("[DEBUG] Removing tags: %#v", remove)
			k := make([]*string, len(remove))
			for i, t := range remove {
				k[i] = t.Key
			}

			_, err := conn.RemoveTagsFromResource(&ssm.RemoveTagsFromResourceInput{
				ResourceId:   aws.String(id),
				ResourceType: aws.String(resourceType),
				TagKeys:      k,
			})
			if err != nil {
				return err
			}
		}
		if len(create) > 0 {
			log.Printf("[DEBUG] Creating tags: %#v", create)
			_, err := conn.AddTagsToResource(&ssm.AddTagsToResourceInput{
				ResourceId:   aws.String(id),
				ResourceType: aws.String(resourceType),
				Tags:         create,
			})
			if err != nil {
				return err
			}
		}
	}

	return nil
}

// diffTags takes our tags locally and the ones remotely and returns
// the set of tags that must be created, and the set of tags that must
// be destroyed.
func diffTagsSSM(oldTags, newTags []*ssm.Tag) ([]*ssm.Tag, []*ssm.Tag) {
	// First, we're creating everything we have
	create := make(map[string]interface{})
	for _, t := range newTags {
		create[*t.Key] = *t.Value
	}

	// Build the list of what to remove
	var remove []*ssm.Tag
	for _, t := range oldTags {
		old, ok := create[*t.Key]
		if !ok || old != *t.Value {
			// Delete it!
			remove = append(remove, t)
		}
	}

	return tagsFromMapSSM(create), remove
}

// tagsFromMap returns the tags for the given map of data.
func tagsFromMapSSM(m map[string]interface{}) []*ssm.Tag {
	result := make([]*ssm.Tag, 0, len(m))
	for k, v := range m {
		t := &ssm.Tag{
			Key:   aws.String(k),
			Value: aws.String(v.(string)),
		}
		if !tagIgnoredSSM(t) {
			result = append(result, t)
		}
	}

	return result
}

// tagsToMap turns the list of tags into a map.
func tagsToMapSSM(ts []*ssm.Tag) map[string]string {
	result := make(map[string]string)
	for _, t := range ts {
		if !tagIgnoredSSM(t) {
			result[*t.Key] = *t.Value
		}
	}

	return result
}

// compare a tag against a list of strings and checks if it should
// be ignored or not
func tagIgnoredSSM(t *ssm.Tag) bool {
	filter := []string{"^aws:"}
	for _, v := range filter {
		log.Printf("[DEBUG] Matching %v with %v\n", v, *t.Key)
		r, _ := regexp.MatchString(v, *t.Key)
		if r {
			log.Printf("[DEBUG] Found AWS specific tag %s (val: %s), ignoring.\n", *t.Key, *t.Value)
			return true
		}
	}
	return false
}

func saveTagsSSM(conn *ssm.SSM, d *schema.ResourceData, id, resourceType string) error {
	resp, err := conn.ListTagsForResource(&ssm.ListTagsForResourceInput{
		ResourceId:   aws.String(id),
		ResourceType: aws.String(resourceType),
	})

	if err != nil {
		return fmt.Errorf("Error retrieving tags for SSM resource (%s): %s", id, err)
	}

	var dt []*ssm.Tag
	if len(resp.TagList) > 0 {
		dt = resp.TagList
	}

	return d.Set("tags", tagsToMapSSM(dt))
}
