/*
 * brass - Braille and speech server
 *
 * Copyright (C) 2001-2003 by Roger Butenuth, All rights reserved.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * Speech module for German. This module does some preprocessing and
 * sends the result to a synthesizer driver.
 *
 * $Id: german.c,v 1.6 2003/06/04 20:11:28 butenuth Exp $
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "language.h"
#include "german.h"
#include "lookup.h"

static int l_close (lang_t * l);

static int l_change_synth (struct lang_struct *l, synth_t * new_synth);

static int l_speak_string (struct lang_struct *l,
                           const unsigned char *buffer);
static int l_get_param (struct lang_struct *l, lang_par_t par, int *value);

static int l_set_param (struct lang_struct *l, lang_par_t par, int value);

/*
 * State information for the German preprocessor. Every preprocessor
 * (language module) has its own state. For the outside world it is
 * just a black box.
 */
typedef struct lang_state
{
  int param[L_MAX];
  int initialized;
} lang_state_t;

static lang_state_t l_state = {
  {1},                          /* speak punctuation is on */
  0                             /* not initialized */
};

static lang_t german = {
  &l_state,
  &languages[LANG_GERMAN],
  NULL,
  NULL,
  0,
  0,
  l_close,
  l_change_synth,
  l_speak_string,
  l_get_param,
  l_set_param
};

/* forward declarations of private functions */
static int handle_number (const unsigned char *buffer);

/*
 * Open the preprocessor for German.
 * context: The context for the lookup function.
 * fun: Lookup function, can be used to read parameters.
 */
lang_t *german_open (void *context, lookup_string_t fun)
{
  synth_t *s;

  assert (!l_state.initialized);

  s = open_synthesizer (context, fun);
  if (s == NULL)
    return NULL;

  german.synth = s;
  init_synth_buffer (&german);
  l_state.initialized = 1;

  return &german;
}

/*
 * Close the synthesizer and the preprocessor.
 */
static int l_close (lang_t * l)
{
  assert (l->state->initialized);

  (*german.synth->close) (german.synth);
  german.synth = NULL;
  deinit_synth_buffer (&german);
  l->state->initialized = 0;

  return 0;
}

/*
 * Set a new synthesizer.
 */
static int l_change_synth (struct lang_struct *l, synth_t * new_synth)
{
  assert (l->state->initialized);

  german.synth = new_synth;

  return 0;
}

/*
 * Speak a string and flush the synthesizer buffer. 
 */
static int l_speak_string (struct lang_struct *l, const unsigned char *buffer)
{
  int pos = 0;

  assert (l->state->initialized);
  while (buffer[pos] != 0)
   {
     int c = buffer[pos];

     if (isdigit (c) || ((c == ',' || c == '.') && isdigit (buffer[pos + 1])))
      {
        pos += handle_number (&buffer[pos]);
      }
     else if (strchr (".,:;?! ", c))
      {
        /* 
         * Always write the punctuation character to the synthesizer
         * because some synthesizers use it to change the melody of
         * a sentence.
         */
        char my_str[2];

        my_str[0] = c;
        my_str[1] = 0;
        add_to_synth_buffer (&german, my_str);
        /* 
         * Depending of the "speak punctuation" parameter speak or
         * skip the punctuation character.
         */
        if (l->state->param[L_SPEAK_PUNCTUATION])
         {
           if (german_table[c] != NULL)
             add_to_synth_buffer (&german, german_table[c]);
         }
        pos++;
      }
     else
      {
        if (german_table[c] != NULL)
          add_to_synth_buffer (&german, german_table[c]);
        pos++;
      }
   }                            /* while */

  return flush_synth_buffer (&german);
}

/*
 *
 */
static int l_get_param (struct lang_struct *l, lang_par_t par, int *value)
{
  if (par >= 0 && par < L_MAX)
   {
     *value = l->state->param[par];
     return 0;
   }
  else
    return 1;
}

/*
 *
 */
static int l_set_param (struct lang_struct *l, lang_par_t par, int value)
{
  if (par >= 0 && par < L_MAX)
   {
     l->state->param[par] = value;
     return 0;
   }
  else
    return 1;
}

/* Start of private stuff */

/* Plain: The numbers from 0 to 19 with one exception: "ein", not "eins" */
static char *zero_to_nineteen[] = {
  "null", "ein", "zwei", "drei", "vier",
  "fnf", "sechs", "sieben", "acht", "neun",
  "zehn", "elf", "zwlf", "dreizehn", "vierzehn",
  "fnfzehn", "sechzehn", "siebzehn", "achtzehn", "neunzehn"
};

/* The multiples of 10 */
static char *xties[] = {
  "", "", "zwanzig", "dreiig", "vierzig",
  "fnfzig", "sechzig", "siebzig", "achtzig", "neunzig"
};

/* more forward references */
static int german_number (const unsigned char *buffer);

static int simple_number (const unsigned char *buffer);

static int digit_by_digit_number (const unsigned char *buffer);

static int other_number (const unsigned char *buffer);

static void speak_integer (int value, int prefix);

/*
 * The buffer starts with ",", "." or a digit. Handle the entire number
 * and return the characters used from the input buffer. 
 */
static int handle_number (const unsigned char *buffer)
{
  int count = 0;

  if (buffer[0] == '0')
   {
     count = digit_by_digit_number (buffer);
   }
  else
   {
     /* Nacheinander verschiedene Varianten probieren */
     count = simple_number (buffer);
     if (count == 0)
       count = german_number (buffer);
     if (count == 0)
       count = other_number (buffer);
   }
  return count;
}

/*
 * A German number is a simple number in the style "123,4567" or a
 * formattet number with dots all three digits in front of the ",".
 * Example: "5.123.456,0815". So there ist at most one ",". All dots
 * must be left from the ",". There must be three digits between dots
 * and three digits between the last dot and ",".
 * Speak the part before "," as a complete number (like "hundert") and
 * the part behind "," digit by digit.
 */
static int german_number (const unsigned char *buffer)
{
  int ok = 1;                   /* so far it is a German number... */

  int last_dot_pos = -1;

  int pos = 0;

  int value = 0;                /* value of the number, integer part */

  int digit_count = 0;          /* number of digits before "," */

  int count = 0;

  int start_of_fraction = -1;

  /* 
   * By contract the first character is a digit, "," or ".". In case
   * of "," we simply skip the integer part. "." as first character is
   * unusual but allowed.
   */
  if (buffer[pos] != ',')
   {
     while (buffer[pos] != 0 && ok)
      {
        if (isdigit (buffer[pos]))
         {
           value = 10 * value + (buffer[pos] - '0');
           digit_count++;
         }
        else if (buffer[pos] == '.')
         {
           if (last_dot_pos > 0)
            {
              if (last_dot_pos + 4 != pos)
                ok = 0;
            }
           else
            {
              last_dot_pos = pos;
            }
         }
        else
         {
           break;
         }
        pos++;
      }
     /* 
      * When there are one or more dots, the last dot must be 3 digits
      * before the end of the number (delimiter between thousands).
      * Otherwise it could be a time (e.g. 20.00 Uhr) or anything else.
      */
     if (last_dot_pos >= 0 && last_dot_pos + 4 != pos)
       ok = 0;
   }
  if (ok && buffer[pos] == ',')
   {
     if (last_dot_pos > 0)
      {
        if (last_dot_pos + 4 != pos)
          ok = 0;
      }
     pos++;                     /* skip the "," */
     start_of_fraction = pos;
     while (isdigit (buffer[pos]) && ok)
      {
        pos++;
      }
     if (buffer[pos] == '.' || buffer[pos] == ',')
       ok = 0;
   }
  if (ok)
   {
     /* 
      * An integer part with less than 10 digits is spoken as a number,
      * larger numbers are spoken digit by digit.
      */
     if (digit_count > 0)
      {
        if (digit_count < 10)
         {
           speak_integer (value, 0);    /* integer part */
         }
        else
         {
           int i;

           for (i = 0; i < start_of_fraction - 1; i++)
            {
              /* Ignore the dots in the integer part */
              if (isdigit (buffer[i]))
               {
                 add_to_synth_buffer (&german, german_table[buffer[i]]);
               }
            }
         }
      }

     if (start_of_fraction >= 0)
      {
        int i;

        add_to_synth_buffer (&german, " komma ");
        for (i = start_of_fraction; i < pos; i++)
         {
           add_to_synth_buffer (&german, german_table[buffer[i]]);
         }
      }
     count = pos;
   }
  return count;
}

/*
 * Speak a primitive number (without "," or ".") as number.
 * A simple number is a sequence of digits terminated by a character
 * not equal to "," or ".".
 * Return the number of characters used from the buffer or zero if the
 * buffer does not point to a simple number.
 */
static int simple_number (const unsigned char *buffer)
{
  int count;

  for (count = 0; isdigit (buffer[count]); count++)
    ;
  if (buffer[count] == ',' || buffer[count] == '.')
    count = 0;                  /* not a simple number */
  if (count > 0)
   {                            /* we have at least one digit, let's speak */
     /* 
      * Speak numbers with less than 10 digits as numbers, 
      * larger numbers are spoken digit by digit.
      */
     if (count < 10)
      {
        int value = 0;

        int i;

        for (i = 0; i < count; i++)
          value = 10 * value + (buffer[i] - '0');
        speak_integer (value, 0);
        add_to_synth_buffer (&german, " ");
      }
     else
      {
        digit_by_digit_number (buffer);
      }
   }

  return count;
}

/*
 * Speak a primitive number (without "," or ".") digit by digit.
 * Returns the number of used characters from buffer.
 */
static int digit_by_digit_number (const unsigned char *buffer)
{
  int pos = 0;

  while (isdigit (buffer[pos]))
   {
     add_to_synth_buffer (&german, german_table[buffer[pos]]);
     pos++;
   }

  return pos;
}

/*
 * We have a "chaotic" number. This can be a date (20.1.2002) or any
 * other characters in the set of digits, ",", and ".". Speak the
 * numbers between the separating characters.
 * Numbers starting with 0 or more than 9 digits are spoken digit by
 * digit, others numbers as integers.
 */
static int other_number (const unsigned char *buffer)
{
  int pos = 0;

  while (isdigit (buffer[pos]) || buffer[pos] == '.' || buffer[pos] == ',')
   {
     if (isdigit (buffer[pos]))
      {
        int start = pos;

        int value = 0;

        while (isdigit (buffer[pos]))
         {
           value = 10 * value + (buffer[pos] - '0');
           pos++;
         }
        if (pos - start < 10 && buffer[start] != '0')
          speak_integer (value, 0);
        else
         {
           int i;

           for (i = start; i < pos; i++)
             add_to_synth_buffer (&german, german_table[buffer[i]]);
         }
      }
     else
      {                         /* ',' or '.' */
        add_to_synth_buffer (&german, german_table[buffer[pos]]);
        pos++;
      }
   }
  return pos;
}

/*
 * Speak a positive or negative integer. The synthiser is not flushed,
 * no space is sent to the synthesizer before or after the number.
 * value: The value to be spoken.
 * prefix: if true, speak the number as a prefix. This changes "eins"
 * to "ein".
 */
static void speak_integer (int value, int prefix)
{
  if (value < 0)
   {
     add_to_synth_buffer (&german, "- ");
     value = -value;
   }
  if (value < 20)
   {                            /* 0-19 */
     add_to_synth_buffer (&german, zero_to_nineteen[value]);
     if (value == 1 && !prefix)
       add_to_synth_buffer (&german, "s");
   }
  else if (value < 100)
   {                            /* 20-99 */
     int tens = value / 10;

     int rest = value % 10;

     if (rest != 0)
      {
        add_to_synth_buffer (&german, zero_to_nineteen[rest]);
        add_to_synth_buffer (&german, "und");
      }
     add_to_synth_buffer (&german, xties[tens]);
   }
  else if (value < 1000)
   {                            /* 100-999 */
     int rest = value % 100;

     add_to_synth_buffer (&german, zero_to_nineteen[value / 100]);
     add_to_synth_buffer (&german, "hundert");
     if (rest != 0)
      {
        speak_integer (rest, prefix);
      }
   }
  else if (value < 1000000)
   {                            /* less than a million */
     int thousands = value / 1000;

     int rest = value % 1000;

     speak_integer (thousands, 1);
     add_to_synth_buffer (&german, "tausend");
     if (rest != 0)
      {
        speak_integer (rest, prefix);
      }
   }
  else if (value < 1000000000)
   {                            /* less than a billion */
     int millions = value / 1000000;

     int rest = value % 1000000;

     speak_integer (millions, 1);
     add_to_synth_buffer (&german, "millionen");
     if (rest != 0)
      {
        speak_integer (rest, prefix);
      }
   }
  else
   {                            /* more than a billion, but less than 2^31 on 
                                   32 bit systems. */
     int billions = value / 1000000;

     int rest = value % 1000000;

     speak_integer (billions, 1);
     add_to_synth_buffer (&german, "milliarden");
     if (rest != 0)
      {
        speak_integer (rest, prefix);
      }
   }
}
