package kubernetes

import (
	v1 "k8s.io/api/core/v1"

	"github.com/hashicorp/terraform-plugin-sdk/helper/schema"
)

// Flatteners

func flattenAWSElasticBlockStoreVolumeSource(in *v1.AWSElasticBlockStoreVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["volume_id"] = in.VolumeID
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.Partition != 0 {
		att["partition"] = in.Partition
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenAzureDiskVolumeSource(in *v1.AzureDiskVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["disk_name"] = in.DiskName
	att["data_disk_uri"] = in.DataDiskURI
	att["caching_mode"] = string(*in.CachingMode)
	if in.FSType != nil {
		att["fs_type"] = *in.FSType
	}
	if in.ReadOnly != nil {
		att["read_only"] = *in.ReadOnly
	}
	return []interface{}{att}
}

func flattenAzureFileVolumeSource(in *v1.AzureFileVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["secret_name"] = in.SecretName
	att["share_name"] = in.ShareName
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenAzureFilePersistentVolumeSource(in *v1.AzureFilePersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["secret_name"] = in.SecretName
	att["share_name"] = in.ShareName
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenCephFSVolumeSource(in *v1.CephFSVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["monitors"] = newStringSet(schema.HashString, in.Monitors)
	if in.Path != "" {
		att["path"] = in.Path
	}
	if in.User != "" {
		att["user"] = in.User
	}
	if in.SecretFile != "" {
		att["secret_file"] = in.SecretFile
	}
	if in.SecretRef != nil {
		att["secret_ref"] = flattenLocalObjectReference(in.SecretRef)
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenCephFSPersistentVolumeSource(in *v1.CephFSPersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["monitors"] = newStringSet(schema.HashString, in.Monitors)
	if in.Path != "" {
		att["path"] = in.Path
	}
	if in.User != "" {
		att["user"] = in.User
	}
	if in.SecretFile != "" {
		att["secret_file"] = in.SecretFile
	}
	if in.SecretRef != nil {
		att["secret_ref"] = flattenSecretReference(in.SecretRef)
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenCinderPersistentVolumeSource(in *v1.CinderPersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["volume_id"] = in.VolumeID
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenCinderVolumeSource(in *v1.CinderVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["volume_id"] = in.VolumeID
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenFCVolumeSource(in *v1.FCVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["target_ww_ns"] = newStringSet(schema.HashString, in.TargetWWNs)
	att["lun"] = *in.Lun
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenFlexPersistentVolumeSource(in *v1.FlexPersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["driver"] = in.Driver
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.SecretRef != nil {
		att["secret_ref"] = flattenSecretReference(in.SecretRef)
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	if len(in.Options) > 0 {
		att["options"] = in.Options
	}
	return []interface{}{att}
}

func flattenFlexVolumeSource(in *v1.FlexVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["driver"] = in.Driver
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.SecretRef != nil {
		att["secret_ref"] = flattenLocalObjectReference(in.SecretRef)
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	if len(in.Options) > 0 {
		att["options"] = in.Options
	}
	return []interface{}{att}
}

func flattenFlockerVolumeSource(in *v1.FlockerVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["dataset_name"] = in.DatasetName
	att["dataset_uuid"] = in.DatasetUUID
	return []interface{}{att}
}

func flattenGCEPersistentDiskVolumeSource(in *v1.GCEPersistentDiskVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["pd_name"] = in.PDName
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.Partition != 0 {
		att["partition"] = in.Partition
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenGlusterfsPersistentVolumeSource(in *v1.GlusterfsPersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["endpoints_name"] = in.EndpointsName
	att["path"] = in.Path
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenGlusterfsVolumeSource(in *v1.GlusterfsVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["endpoints_name"] = in.EndpointsName
	att["path"] = in.Path
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenHostPathVolumeSource(in *v1.HostPathVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["path"] = in.Path
	if in.Type != nil {
		att["type"] = string(*in.Type)
	}
	return []interface{}{att}
}

func flattenLocalVolumeSource(in *v1.LocalVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["path"] = in.Path
	return []interface{}{att}
}

func flattenISCSIVolumeSource(in *v1.ISCSIVolumeSource) []interface{} {
	att := make(map[string]interface{})
	if in.TargetPortal != "" {
		att["target_portal"] = in.TargetPortal
	}
	if in.IQN != "" {
		att["iqn"] = in.IQN
	}
	if in.Lun != 0 {
		att["lun"] = in.Lun
	}
	if in.ISCSIInterface != "" {
		att["iscsi_interface"] = in.ISCSIInterface
	}
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenISCSIPersistentVolumeSource(in *v1.ISCSIPersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	if in.TargetPortal != "" {
		att["target_portal"] = in.TargetPortal
	}
	if in.IQN != "" {
		att["iqn"] = in.IQN
	}
	if in.Lun != 0 {
		att["lun"] = in.Lun
	}
	if in.ISCSIInterface != "" {
		att["iscsi_interface"] = in.ISCSIInterface
	}
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenLocalObjectReference(in *v1.LocalObjectReference) []interface{} {
	att := make(map[string]interface{})
	if in.Name != "" {
		att["name"] = in.Name
	}
	return []interface{}{att}
}

func flattenSecretReference(in *v1.SecretReference) []interface{} {
	att := make(map[string]interface{})
	if in.Name != "" {
		att["name"] = in.Name
	}
	if in.Namespace != "" {
		att["namespace"] = in.Namespace
	}
	return []interface{}{att}
}

func flattenNFSVolumeSource(in *v1.NFSVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["server"] = in.Server
	att["path"] = in.Path
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenPersistentVolumeSource(in v1.PersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	if in.GCEPersistentDisk != nil {
		att["gce_persistent_disk"] = flattenGCEPersistentDiskVolumeSource(in.GCEPersistentDisk)
	}
	if in.AWSElasticBlockStore != nil {
		att["aws_elastic_block_store"] = flattenAWSElasticBlockStoreVolumeSource(in.AWSElasticBlockStore)
	}
	if in.HostPath != nil {
		att["host_path"] = flattenHostPathVolumeSource(in.HostPath)
	}
	if in.Local != nil {
		att["local"] = flattenLocalVolumeSource(in.Local)
	}
	if in.Glusterfs != nil {
		att["glusterfs"] = flattenGlusterfsPersistentVolumeSource(in.Glusterfs)
	}
	if in.NFS != nil {
		att["nfs"] = flattenNFSVolumeSource(in.NFS)
	}
	if in.RBD != nil {
		att["rbd"] = flattenRBDPersistentVolumeSource(in.RBD)
	}
	if in.ISCSI != nil {
		att["iscsi"] = flattenISCSIPersistentVolumeSource(in.ISCSI)
	}
	if in.Cinder != nil {
		att["cinder"] = flattenCinderPersistentVolumeSource(in.Cinder)
	}
	if in.CephFS != nil {
		att["ceph_fs"] = flattenCephFSPersistentVolumeSource(in.CephFS)
	}
	if in.FC != nil {
		att["fc"] = flattenFCVolumeSource(in.FC)
	}
	if in.Flocker != nil {
		att["flocker"] = flattenFlockerVolumeSource(in.Flocker)
	}
	if in.FlexVolume != nil {
		att["flex_volume"] = flattenFlexPersistentVolumeSource(in.FlexVolume)
	}
	if in.AzureFile != nil {
		att["azure_file"] = flattenAzureFilePersistentVolumeSource(in.AzureFile)
	}
	if in.VsphereVolume != nil {
		att["vsphere_volume"] = flattenVsphereVirtualDiskVolumeSource(in.VsphereVolume)
	}
	if in.Quobyte != nil {
		att["quobyte"] = flattenQuobyteVolumeSource(in.Quobyte)
	}
	if in.AzureDisk != nil {
		att["azure_disk"] = flattenAzureDiskVolumeSource(in.AzureDisk)
	}
	if in.PhotonPersistentDisk != nil {
		att["photon_persistent_disk"] = flattenPhotonPersistentDiskVolumeSource(in.PhotonPersistentDisk)
	}
	if in.CSI != nil {
		att["csi"] = flattenCSIVolumeSource(in.CSI)
	}
	return []interface{}{att}
}

func flattenPersistentVolumeSpec(in v1.PersistentVolumeSpec) []interface{} {
	att := make(map[string]interface{})
	if len(in.Capacity) > 0 {
		att["capacity"] = flattenResourceList(in.Capacity)
	}

	att["persistent_volume_source"] = flattenPersistentVolumeSource(in.PersistentVolumeSource)
	if len(in.AccessModes) > 0 {
		att["access_modes"] = flattenPersistentVolumeAccessModes(in.AccessModes)
	}
	if in.PersistentVolumeReclaimPolicy != "" {
		att["persistent_volume_reclaim_policy"] = in.PersistentVolumeReclaimPolicy
	}
	if in.StorageClassName != "" {
		att["storage_class_name"] = in.StorageClassName
	}
	if in.NodeAffinity != nil {
		att["node_affinity"] = flattenVolumeNodeAffinity(in.NodeAffinity)
	}
	if in.MountOptions != nil {
		att["mount_options"] = flattenPersistentVolumeMountOptions(in.MountOptions)
	}
	if in.VolumeMode != nil {
		att["volume_mode"] = in.VolumeMode
	}
	return []interface{}{att}
}

func flattenPhotonPersistentDiskVolumeSource(in *v1.PhotonPersistentDiskVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["pd_id"] = in.PdID
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	return []interface{}{att}
}

func flattenCSIVolumeSource(in *v1.CSIPersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["driver"] = in.Driver
	att["volume_handle"] = in.VolumeHandle
	att["read_only"] = in.ReadOnly
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if len(in.VolumeAttributes) > 0 {
		att["volume_attributes"] = in.VolumeAttributes
	}
	if in.ControllerExpandSecretRef != nil {
		att["controller_expand_secret_ref"] = flattenSecretReference(in.ControllerExpandSecretRef)
	}
	if in.ControllerPublishSecretRef != nil {
		att["controller_publish_secret_ref"] = flattenSecretReference(in.ControllerPublishSecretRef)
	}
	if in.NodePublishSecretRef != nil {
		att["node_publish_secret_ref"] = flattenSecretReference(in.NodePublishSecretRef)
	}
	if in.NodeStageSecretRef != nil {
		att["node_stage_secret_ref"] = flattenSecretReference(in.NodeStageSecretRef)
	}
	return []interface{}{att}
}

func flattenQuobyteVolumeSource(in *v1.QuobyteVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["registry"] = in.Registry
	att["volume"] = in.Volume
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	if in.User != "" {
		att["user"] = in.User
	}
	if in.Group != "" {
		att["group"] = in.Group
	}
	return []interface{}{att}
}

func flattenRBDVolumeSource(in *v1.RBDVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["ceph_monitors"] = newStringSet(schema.HashString, in.CephMonitors)
	att["rbd_image"] = in.RBDImage
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.RBDPool != "" {
		att["rbd_pool"] = in.RBDPool
	}
	if in.RadosUser != "" {
		att["rados_user"] = in.RadosUser
	}
	if in.Keyring != "" {
		att["keyring"] = in.Keyring
	}
	if in.SecretRef != nil {
		att["secret_ref"] = flattenLocalObjectReference(in.SecretRef)
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenRBDPersistentVolumeSource(in *v1.RBDPersistentVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["ceph_monitors"] = newStringSet(schema.HashString, in.CephMonitors)
	att["rbd_image"] = in.RBDImage
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	if in.RBDPool != "" {
		att["rbd_pool"] = in.RBDPool
	}
	if in.RadosUser != "" {
		att["rados_user"] = in.RadosUser
	}
	if in.Keyring != "" {
		att["keyring"] = in.Keyring
	}
	if in.SecretRef != nil {
		att["secret_ref"] = flattenSecretReference(in.SecretRef)
	}
	if in.ReadOnly != false {
		att["read_only"] = in.ReadOnly
	}
	return []interface{}{att}
}

func flattenVsphereVirtualDiskVolumeSource(in *v1.VsphereVirtualDiskVolumeSource) []interface{} {
	att := make(map[string]interface{})
	att["volume_path"] = in.VolumePath
	if in.FSType != "" {
		att["fs_type"] = in.FSType
	}
	return []interface{}{att}
}

// Expanders

func expandAWSElasticBlockStoreVolumeSource(l []interface{}) *v1.AWSElasticBlockStoreVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.AWSElasticBlockStoreVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.AWSElasticBlockStoreVolumeSource{
		VolumeID: in["volume_id"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["partition"].(int); ok {
		obj.Partition = int32(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandAzureDiskVolumeSource(l []interface{}) *v1.AzureDiskVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.AzureDiskVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	cachingMode := v1.AzureDataDiskCachingMode(in["caching_mode"].(string))
	obj := &v1.AzureDiskVolumeSource{
		CachingMode: &cachingMode,
		DiskName:    in["disk_name"].(string),
		DataDiskURI: in["data_disk_uri"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = ptrToString(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = ptrToBool(v)
	}
	return obj
}

func expandAzureFileVolumeSource(l []interface{}) *v1.AzureFileVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.AzureFileVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.AzureFileVolumeSource{
		SecretName: in["secret_name"].(string),
		ShareName:  in["share_name"].(string),
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandAzureFilePersistentVolumeSource(l []interface{}) *v1.AzureFilePersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.AzureFilePersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.AzureFilePersistentVolumeSource{
		SecretName: in["secret_name"].(string),
		ShareName:  in["share_name"].(string),
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandCephFSVolumeSource(l []interface{}) *v1.CephFSVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.CephFSVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.CephFSVolumeSource{
		Monitors: sliceOfString(in["monitors"].(*schema.Set).List()),
	}
	if v, ok := in["path"].(string); ok {
		obj.Path = v
	}
	if v, ok := in["user"].(string); ok {
		obj.User = v
	}
	if v, ok := in["secret_file"].(string); ok {
		obj.SecretFile = v
	}
	if v, ok := in["secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.SecretRef = expandLocalObjectReference(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandCephFSPersistentVolumeSource(l []interface{}) *v1.CephFSPersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.CephFSPersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.CephFSPersistentVolumeSource{
		Monitors: sliceOfString(in["monitors"].(*schema.Set).List()),
	}
	if v, ok := in["path"].(string); ok {
		obj.Path = v
	}
	if v, ok := in["user"].(string); ok {
		obj.User = v
	}
	if v, ok := in["secret_file"].(string); ok {
		obj.SecretFile = v
	}
	if v, ok := in["secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.SecretRef = expandSecretReference(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandCinderPersistentVolumeSource(l []interface{}) *v1.CinderPersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.CinderPersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.CinderPersistentVolumeSource{
		VolumeID: in["volume_id"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandCinderVolumeSource(l []interface{}) *v1.CinderVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.CinderVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.CinderVolumeSource{
		VolumeID: in["volume_id"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandFCVolumeSource(l []interface{}) *v1.FCVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.FCVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.FCVolumeSource{
		TargetWWNs: sliceOfString(in["target_ww_ns"].(*schema.Set).List()),
		Lun:        ptrToInt32(int32(in["lun"].(int))),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandFlexPersistentVolumeSource(l []interface{}) *v1.FlexPersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.FlexPersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.FlexPersistentVolumeSource{
		Driver: in["driver"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.SecretRef = expandSecretReference(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	if v, ok := in["options"].(map[string]interface{}); ok && len(v) > 0 {
		obj.Options = expandStringMap(v)
	}
	return obj
}

func expandFlexVolumeSource(l []interface{}) *v1.FlexVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.FlexVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.FlexVolumeSource{
		Driver: in["driver"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.SecretRef = expandLocalObjectReference(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	if v, ok := in["options"].(map[string]interface{}); ok && len(v) > 0 {
		obj.Options = expandStringMap(v)
	}
	return obj
}

func expandFlockerVolumeSource(l []interface{}) *v1.FlockerVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.FlockerVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.FlockerVolumeSource{
		DatasetName: in["dataset_name"].(string),
		DatasetUUID: in["dataset_uuid"].(string),
	}
	return obj
}

func expandGCEPersistentDiskVolumeSource(l []interface{}) *v1.GCEPersistentDiskVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.GCEPersistentDiskVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.GCEPersistentDiskVolumeSource{
		PDName: in["pd_name"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["partition"].(int); ok {
		obj.Partition = int32(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandGlusterfsPersistentVolumeSource(l []interface{}) *v1.GlusterfsPersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.GlusterfsPersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.GlusterfsPersistentVolumeSource{
		EndpointsName: in["endpoints_name"].(string),
		Path:          in["path"].(string),
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandGlusterfsVolumeSource(l []interface{}) *v1.GlusterfsVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.GlusterfsVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.GlusterfsVolumeSource{
		EndpointsName: in["endpoints_name"].(string),
		Path:          in["path"].(string),
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandHostPathVolumeSource(l []interface{}) *v1.HostPathVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.HostPathVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	typ := v1.HostPathType(in["type"].(string))
	obj := &v1.HostPathVolumeSource{
		Path: in["path"].(string),
		Type: &typ,
	}
	return obj
}

func expandLocalVolumeSource(l []interface{}) *v1.LocalVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.LocalVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.LocalVolumeSource{
		Path: in["path"].(string),
	}
	return obj
}

func expandISCSIVolumeSource(l []interface{}) *v1.ISCSIVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.ISCSIVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.ISCSIVolumeSource{
		TargetPortal: in["target_portal"].(string),
		IQN:          in["iqn"].(string),
	}
	if v, ok := in["lun"].(int); ok {
		obj.Lun = int32(v)
	}
	if v, ok := in["iscsi_interface"].(string); ok {
		obj.ISCSIInterface = v
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandISCSIPersistentVolumeSource(l []interface{}) *v1.ISCSIPersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.ISCSIPersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.ISCSIPersistentVolumeSource{
		TargetPortal: in["target_portal"].(string),
		IQN:          in["iqn"].(string),
	}
	if v, ok := in["lun"].(int); ok {
		obj.Lun = int32(v)
	}
	if v, ok := in["iscsi_interface"].(string); ok {
		obj.ISCSIInterface = v
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandLocalObjectReference(l []interface{}) *v1.LocalObjectReference {
	if len(l) == 0 || l[0] == nil {
		return &v1.LocalObjectReference{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.LocalObjectReference{}
	if v, ok := in["name"].(string); ok {
		obj.Name = v
	}
	return obj
}

func expandSecretReference(l []interface{}) *v1.SecretReference {
	if len(l) == 0 || l[0] == nil {
		return &v1.SecretReference{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.SecretReference{}
	if v, ok := in["name"].(string); ok {
		obj.Name = v
	}
	if v, ok := in["namespace"].(string); ok {
		obj.Namespace = v
	}
	return obj
}

func expandNFSVolumeSource(l []interface{}) *v1.NFSVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.NFSVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.NFSVolumeSource{
		Server: in["server"].(string),
		Path:   in["path"].(string),
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandPersistentVolumeSource(l []interface{}) v1.PersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return v1.PersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := v1.PersistentVolumeSource{}
	if v, ok := in["gce_persistent_disk"].([]interface{}); ok && len(v) > 0 {
		obj.GCEPersistentDisk = expandGCEPersistentDiskVolumeSource(v)
	}
	if v, ok := in["aws_elastic_block_store"].([]interface{}); ok && len(v) > 0 {
		obj.AWSElasticBlockStore = expandAWSElasticBlockStoreVolumeSource(v)
	}
	if v, ok := in["host_path"].([]interface{}); ok && len(v) > 0 {
		obj.HostPath = expandHostPathVolumeSource(v)
	}
	if v, ok := in["local"].([]interface{}); ok && len(v) > 0 {
		obj.Local = expandLocalVolumeSource(v)
	}
	if v, ok := in["glusterfs"].([]interface{}); ok && len(v) > 0 {
		obj.Glusterfs = expandGlusterfsPersistentVolumeSource(v)
	}
	if v, ok := in["nfs"].([]interface{}); ok && len(v) > 0 {
		obj.NFS = expandNFSVolumeSource(v)
	}
	if v, ok := in["rbd"].([]interface{}); ok && len(v) > 0 {
		obj.RBD = expandRBDPersistentVolumeSource(v)
	}
	if v, ok := in["iscsi"].([]interface{}); ok && len(v) > 0 {
		obj.ISCSI = expandISCSIPersistentVolumeSource(v)
	}
	if v, ok := in["cinder"].([]interface{}); ok && len(v) > 0 {
		obj.Cinder = expandCinderPersistentVolumeSource(v)
	}
	if v, ok := in["ceph_fs"].([]interface{}); ok && len(v) > 0 {
		obj.CephFS = expandCephFSPersistentVolumeSource(v)
	}
	if v, ok := in["fc"].([]interface{}); ok && len(v) > 0 {
		obj.FC = expandFCVolumeSource(v)
	}
	if v, ok := in["flocker"].([]interface{}); ok && len(v) > 0 {
		obj.Flocker = expandFlockerVolumeSource(v)
	}
	if v, ok := in["flex_volume"].([]interface{}); ok && len(v) > 0 {
		obj.FlexVolume = expandFlexPersistentVolumeSource(v)
	}
	if v, ok := in["azure_file"].([]interface{}); ok && len(v) > 0 {
		obj.AzureFile = expandAzureFilePersistentVolumeSource(v)
	}
	if v, ok := in["vsphere_volume"].([]interface{}); ok && len(v) > 0 {
		obj.VsphereVolume = expandVsphereVirtualDiskVolumeSource(v)
	}
	if v, ok := in["quobyte"].([]interface{}); ok && len(v) > 0 {
		obj.Quobyte = expandQuobyteVolumeSource(v)
	}
	if v, ok := in["azure_disk"].([]interface{}); ok && len(v) > 0 {
		obj.AzureDisk = expandAzureDiskVolumeSource(v)
	}
	if v, ok := in["photon_persistent_disk"].([]interface{}); ok && len(v) > 0 {
		obj.PhotonPersistentDisk = expandPhotonPersistentDiskVolumeSource(v)
	}
	if v, ok := in["csi"].([]interface{}); ok && len(v) > 0 {
		obj.CSI = expandCSIPersistentDiskVolumeSource(v)
	}
	return obj
}

func expandPersistentVolumeSpec(l []interface{}) (*v1.PersistentVolumeSpec, error) {
	obj := &v1.PersistentVolumeSpec{}
	if len(l) == 0 || l[0] == nil {
		return obj, nil
	}
	in := l[0].(map[string]interface{})
	if v, ok := in["capacity"].(map[string]interface{}); ok && len(v) > 0 {
		c, err := expandMapToResourceList(v)
		if err != nil {
			return obj, err
		}
		obj.Capacity = *c
	}
	if v, ok := in["persistent_volume_source"].([]interface{}); ok && len(v) > 0 {
		obj.PersistentVolumeSource = expandPersistentVolumeSource(v)
	}
	if v, ok := in["access_modes"].(*schema.Set); ok && v.Len() > 0 {
		obj.AccessModes = expandPersistentVolumeAccessModes(v.List())
	}
	if v, ok := in["persistent_volume_reclaim_policy"].(string); ok {
		obj.PersistentVolumeReclaimPolicy = v1.PersistentVolumeReclaimPolicy(v)
	}
	if v, ok := in["storage_class_name"].(string); ok {
		obj.StorageClassName = v
	}
	if v, ok := in["node_affinity"].([]interface{}); ok && len(v) > 0 {
		obj.NodeAffinity = expandVolumeNodeAffinity(v)
	}
	if v, ok := in["mount_options"].(*schema.Set); ok && v.Len() > 0 {
		obj.MountOptions = schemaSetToStringArray(v)
	}
	if v, ok := in["volume_mode"].(string); ok {
		volumeMode := v1.PersistentVolumeMode(v)
		obj.VolumeMode = &volumeMode
	}
	return obj, nil
}

func expandPhotonPersistentDiskVolumeSource(l []interface{}) *v1.PhotonPersistentDiskVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.PhotonPersistentDiskVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.PhotonPersistentDiskVolumeSource{
		PdID: in["pd_id"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	return obj
}

func expandCSIPersistentDiskVolumeSource(l []interface{}) *v1.CSIPersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.CSIPersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.CSIPersistentVolumeSource{
		Driver:       in["driver"].(string),
		VolumeHandle: in["volume_handle"].(string),
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["volume_attributes"].(map[string]interface{}); ok && len(v) > 0 {
		obj.VolumeAttributes = expandStringMap(v)
	}
	if v, ok := in["controller_publish_secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.ControllerPublishSecretRef = expandSecretReference(v)
	}
	if v, ok := in["node_stage_secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.NodeStageSecretRef = expandSecretReference(v)
	}
	if v, ok := in["node_publish_secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.NodePublishSecretRef = expandSecretReference(v)
	}
	if v, ok := in["controller_expand_secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.ControllerExpandSecretRef = expandSecretReference(v)
	}
	return obj
}

func expandQuobyteVolumeSource(l []interface{}) *v1.QuobyteVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.QuobyteVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.QuobyteVolumeSource{
		Registry: in["registry"].(string),
		Volume:   in["volume"].(string),
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	if v, ok := in["user"].(string); ok {
		obj.User = v
	}
	if v, ok := in["group"].(string); ok {
		obj.Group = v
	}
	return obj
}

func expandRBDVolumeSource(l []interface{}) *v1.RBDVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.RBDVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.RBDVolumeSource{
		CephMonitors: expandStringSlice(in["ceph_monitors"].(*schema.Set).List()),
		RBDImage:     in["rbd_image"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["rbd_pool"].(string); ok {
		obj.RBDPool = v
	}
	if v, ok := in["rados_user"].(string); ok {
		obj.RadosUser = v
	}
	if v, ok := in["keyring"].(string); ok {
		obj.Keyring = v
	}
	if v, ok := in["secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.SecretRef = expandLocalObjectReference(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandRBDPersistentVolumeSource(l []interface{}) *v1.RBDPersistentVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.RBDPersistentVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.RBDPersistentVolumeSource{
		CephMonitors: expandStringSlice(in["ceph_monitors"].(*schema.Set).List()),
		RBDImage:     in["rbd_image"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	if v, ok := in["rbd_pool"].(string); ok {
		obj.RBDPool = v
	}
	if v, ok := in["rados_user"].(string); ok {
		obj.RadosUser = v
	}
	if v, ok := in["keyring"].(string); ok {
		obj.Keyring = v
	}
	if v, ok := in["secret_ref"].([]interface{}); ok && len(v) > 0 {
		obj.SecretRef = expandSecretReference(v)
	}
	if v, ok := in["read_only"].(bool); ok {
		obj.ReadOnly = v
	}
	return obj
}

func expandVsphereVirtualDiskVolumeSource(l []interface{}) *v1.VsphereVirtualDiskVolumeSource {
	if len(l) == 0 || l[0] == nil {
		return &v1.VsphereVirtualDiskVolumeSource{}
	}
	in := l[0].(map[string]interface{})
	obj := &v1.VsphereVirtualDiskVolumeSource{
		VolumePath: in["volume_path"].(string),
	}
	if v, ok := in["fs_type"].(string); ok {
		obj.FSType = v
	}
	return obj
}

func patchPersistentVolumeSpec(pathPrefix, prefix string, d *schema.ResourceData) (PatchOperations, error) {
	ops := make([]PatchOperation, 0)
	prefix += ".0."

	if d.HasChange(prefix + "capacity") {
		v := d.Get(prefix + "capacity").(map[string]interface{})
		capacity, err := expandMapToResourceList(v)
		if err != nil {
			return ops, err
		}
		ops = append(ops, &ReplaceOperation{
			Path:  pathPrefix + "/capacity",
			Value: capacity,
		})
	}

	if d.HasChange(prefix + "persistent_volume_source") {
		ops = append(ops, patchPersistentVolumeSource(
			pathPrefix,
			prefix+"persistent_volume_source.0.",
			d,
		)...)
	}

	if d.HasChange(prefix + "access_modes") {
		v := d.Get(prefix + "access_modes").(*schema.Set)
		ops = append(ops, &ReplaceOperation{
			Path:  pathPrefix + "/accessModes",
			Value: expandPersistentVolumeAccessModes(v.List()),
		})
	}
	if d.HasChange(prefix + "persistent_volume_reclaim_policy") {
		v := d.Get(prefix + "persistent_volume_reclaim_policy").(string)
		ops = append(ops, &ReplaceOperation{
			Path:  pathPrefix + "/persistentVolumeReclaimPolicy",
			Value: v1.PersistentVolumeReclaimPolicy(v),
		})
	}
	if d.HasChange(prefix + "storage_class_name") {
		o, n := d.GetChange(prefix + "storage_class_name")
		if v, ok := o.(string); ok && len(v) > 0 {
			ops = append(ops, &ReplaceOperation{
				Path:  pathPrefix + "/storageClassName",
				Value: n.(string),
			})
		} else {
			ops = append(ops, &AddOperation{
				Path:  pathPrefix + "/storageClassName",
				Value: n.(string),
			})
		}
	}
	if d.HasChange(prefix + "node_affinity") {
		v := d.Get(prefix + "node_affinity").([]interface{})
		nodeAffinity := expandVolumeNodeAffinity(v)
		ops = append(ops, &ReplaceOperation{
			Path:  pathPrefix + "/nodeAffinity",
			Value: nodeAffinity,
		})
	}
	if d.HasChange(prefix + "mount_options") {
		v := d.Get(prefix + "mount_options").(*schema.Set)
		ops = append(ops, &ReplaceOperation{
			Path:  pathPrefix + "/mountOptions",
			Value: expandPersistentVolumeAccessModes(v.List()),
		})
	}

	return ops, nil
}

func patchPersistentVolumeSource(pathPrefix, prefix string, d *schema.ResourceData) []PatchOperation {
	ops := make([]PatchOperation, 0)

	if d.HasChange(prefix + "gce_persistent_disk") {
		oldIn, newIn := d.GetChange(prefix + "gce_persistent_disk")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/gcePersistentDisk",
					Value: expandGCEPersistentDiskVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/gcePersistentDisk",
					Value: expandGCEPersistentDiskVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/gcePersistentDisk"})
		}
	}

	if d.HasChange(prefix + "aws_elastic_block_store") {
		oldIn, newIn := d.GetChange(prefix + "aws_elastic_block_store")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/awsElasticBlockStore",
					Value: expandAWSElasticBlockStoreVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/awsElasticBlockStore",
					Value: expandAWSElasticBlockStoreVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/awsElasticBlockStore"})
		}
	}

	if d.HasChange(prefix + "host_path") {
		oldIn, newIn := d.GetChange(prefix + "host_path")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/hostPath",
					Value: expandHostPathVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/hostPath",
					Value: expandHostPathVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/hostPath"})
		}
	}

	if d.HasChange(prefix + "glusterfs") {
		oldIn, newIn := d.GetChange(prefix + "glusterfs")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/glusterfs",
					Value: expandGlusterfsVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/glusterfs",
					Value: expandGlusterfsVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/glusterfs"})
		}
	}

	if d.HasChange(prefix + "nfs") {
		oldIn, newIn := d.GetChange(prefix + "nfs")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/nfs",
					Value: expandNFSVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/nfs",
					Value: expandNFSVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/nfs"})
		}
	}

	if d.HasChange(prefix + "rbd") {
		oldIn, newIn := d.GetChange(prefix + "rbd")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/rbd",
					Value: expandRBDVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/rbd",
					Value: expandRBDVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/rbd"})
		}
	}

	if d.HasChange(prefix + "iscsi") {
		oldIn, newIn := d.GetChange(prefix + "iscsi")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/iscsi",
					Value: expandISCSIVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/iscsi",
					Value: expandISCSIVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/iscsi"})
		}
	}

	if d.HasChange(prefix + "cinder") {
		oldIn, newIn := d.GetChange(prefix + "cinder")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/cinder",
					Value: expandCinderVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/cinder",
					Value: expandCinderVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/cinder"})
		}
	}

	if d.HasChange(prefix + "ceph_fs") {
		oldIn, newIn := d.GetChange(prefix + "ceph_fs")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/cephfs",
					Value: expandCephFSVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/cephfs",
					Value: expandCephFSVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/cephfs"})
		}
	}

	if d.HasChange(prefix + "fc") {
		oldIn, newIn := d.GetChange(prefix + "fc")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/fc",
					Value: expandFCVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/fc",
					Value: expandFCVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/fc"})
		}
	}

	if d.HasChange(prefix + "flocker") {
		oldIn, newIn := d.GetChange(prefix + "flocker")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/flocker",
					Value: expandFlockerVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/flocker",
					Value: expandFlockerVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/flocker"})
		}
	}

	if d.HasChange(prefix + "flex_volume") {
		oldIn, newIn := d.GetChange(prefix + "flex_volume")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/flexVolume",
					Value: expandFlexVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/flexVolume",
					Value: expandFlexVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/flexVolume"})
		}
	}

	if d.HasChange(prefix + "azure_file") {
		oldIn, newIn := d.GetChange(prefix + "azure_file")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/azureFile",
					Value: expandAzureFileVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/azureFile",
					Value: expandAzureFileVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/azureFile"})
		}
	}

	if d.HasChange(prefix + "vsphere_volume") {
		oldIn, newIn := d.GetChange(prefix + "vsphere_volume")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/vsphereVolume",
					Value: expandVsphereVirtualDiskVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/vsphereVolume",
					Value: expandVsphereVirtualDiskVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/vsphereVolume"})
		}
	}

	if d.HasChange(prefix + "quobyte") {
		oldIn, newIn := d.GetChange(prefix + "quobyte")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/quobyte",
					Value: expandQuobyteVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/quobyte",
					Value: expandQuobyteVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/quobyte"})
		}
	}

	if d.HasChange(prefix + "azure_disk") {
		oldIn, newIn := d.GetChange(prefix + "azure_disk")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/azureDisk",
					Value: expandAzureDiskVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/azureDisk",
					Value: expandAzureDiskVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/azureDisk"})
		}
	}

	if d.HasChange(prefix + "photon_persistent_disk") {
		oldIn, newIn := d.GetChange(prefix + "photon_persistent_disk")
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  pathPrefix + "/photonPersistentDisk",
					Value: expandPhotonPersistentDiskVolumeSource(newV),
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  pathPrefix + "/photonPersistentDisk",
					Value: expandPhotonPersistentDiskVolumeSource(newV),
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: pathPrefix + "/photonPersistentDisk"})
		}
	}

	if d.HasChange(prefix + "csi") {
		path := pathPrefix + "/csi"
		oldIn, newIn := d.GetChange(path)
		oldV, oldOk := oldIn.([]interface{})
		newV, newOk := newIn.([]interface{})
		value := expandCSIPersistentDiskVolumeSource(newV)

		if newOk && len(newV) > 0 {
			if oldOk && len(oldV) > 0 {
				ops = append(ops, &ReplaceOperation{
					Path:  path,
					Value: value,
				})
			} else {
				ops = append(ops, &AddOperation{
					Path:  path,
					Value: value,
				})
			}
		} else if oldOk && len(oldV) > 0 {
			ops = append(ops, &RemoveOperation{Path: path})
		}
	}

	return ops
}

func flattenVolumeNodeAffinity(in *v1.VolumeNodeAffinity) []interface{} {
	att := make(map[string]interface{})
	nodeSelector := map[string]interface{}{
		"node_selector_term": flattenNodeSelectorTerms(in.Required.NodeSelectorTerms),
	}
	att["required"] = []interface{}{nodeSelector}
	return []interface{}{att}
}

func expandVolumeNodeAffinity(l []interface{}) *v1.VolumeNodeAffinity {
	if len(l) == 0 || l[0] == nil {
		return &v1.VolumeNodeAffinity{}
	}
	in := l[0].(map[string]interface{})
	nodeSelectorList := in["required"].([]interface{})

	if len(nodeSelectorList) == 0 || nodeSelectorList[0] == nil {
		return &v1.VolumeNodeAffinity{}
	}
	nodeSelector := nodeSelectorList[0].(map[string]interface{})

	if len(nodeSelector) == 0 || nodeSelectorList[0] == nil {
		return &v1.VolumeNodeAffinity{}
	}
	obj := &v1.VolumeNodeAffinity{
		Required: &v1.NodeSelector{
			NodeSelectorTerms: expandNodeSelectorTerms(nodeSelector["node_selector_term"].([]interface{})),
		},
	}
	return obj
}
