/*
 * powerd.c    Work with the UPS to automate shutdowns
 * 
 * Copyright (C) 1997-2001      James Brents <James@nistix.com>
 * 
 * This program is a tool to automate your system 
 * to shutdown when the power goes out. Has the ability to notify remote 
 * clients on the network of the power outage.
 * 
 * You can also set the program to not do anything until a setable
 * ammount of time has past, for the occasional short power glitches.
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "powerd.h" 


static Client *clients;
static Listen *listens;

FileDescriptor *fdlist = NULL;
fd_set readfds;
fd_set writefds;
int nfds;

char ver[] = VERSION;
char logbuffer[MAXBUF];
uid_t root, user;
int serverfd=-1;
char *device;
char buf[MAXBUF];
char sendbuf[MAXBUF];
char mode = 0;
char previous_fail = 0;
char failed_and_notified = 0;
int outfor;
time_t outtime = 0;
char quiting = 1;
int mypower = 0;
int listenport = PORT;
char pfbit;
int pfsignal;
char *touser = NULL;

/* Source explanation. Since this is still BETA, I may need to explain my 
 * code. If it MONITOR mode, we call monitorups() and notifyclients(), in 
 * PEER mode, we call bindport() which is basically a full TCP server. It is 
 * like this (able to handle alot of connections at a time) in case of a 
 * Denial Of Service attack, or really bad networks. Ive tried to make this 
 * program be able to still notify the clients and receive the notification 
 * even when all hell is breaking loose.  Im trying to put in more comments 
 * and make the code easier to read, Please feel free to send patches ot 
 * me..  The current powerd programs out there are really limited, and too 
 * complicated. Im trying to make this simple, I hope you like it.
 */

int main(int argc, char **argv)
{
#ifndef DEBUG
    int i;
#endif
    char *me=argv[0];
    int delay = 16;

    if (argc<1 || (argc >1 && (!strcmp(argv[1], "--help") || !strcmp(argv[1], "-h")))) {
	printf("powerd %s by: James Brents <james@nistix.com> (DaSyonic)\n\n", ver);
	printf("%s: usage: \"%s\"\n", me, me);
	exit(1);
    }

    openconfig(&delay); /* Open the configuration file */

    errno=0;

#ifndef DEBUG
    switch(i=fork()) {
    case 0: /* Child */
	chdir("/");
	setsid();

	if ((root = getuid())) {
		fprintf(stderr, "%s: need to have root privlidges.", me);
	} else if (touser != NULL) {
		if (seteuid(user = uname2id(touser)) == -1)
			fprintf(stderr, "%s is not a valid username.", touser);
	}

	break;
    default: /* Parent */
	exit(0);
    }
#endif

    signal(SIGINT, quit_sig);
#ifdef SYSV_SIGNAL
    signal(SIGPWR, debughelp); /* Send SIGPWR to simulate a power outtage */
#endif

    snprintf(logbuffer, sizeof(logbuffer)-1, "powerd %s by James Brents started. %d second delay", ver, delay);
    log(LOG_INFO, LOG_DAEMON, logbuffer);

    if (mode == 0)
	monitorups(delay);
    else if (mode == 1)
	bindport();
    exit(0);
}

void monitorups(int delay) {
    int fd;
    uint bits, mask=~0, outb=(TIOCM_RTS|TIOCM_DTR);
//    int timeout = 1;

/* Okay, this is part of the code that needs changing. I doubt it works on 
 * many UPS's, it monitors the TIOCM_CD bit and when it goes high (1) then 
 * we shutdown. This isnt portable, but its how my UPS works. Since it 
 * works for me, I havnt spent alot of time allowing other configurations. 
 * This will change, and if someone can make a patch, or at least send me 
 * their configuration, please do so.
 */

    if (touser != NULL)
	seteuid(root);

    if ((fd=open(device, O_RDONLY | O_NDELAY)) < 0) {
	fprintf(stderr, "powerd: %s\n", strerror(errno));exit(-1);
    }

    if (touser != NULL)
	seteuid(user);

    usleep(1); // Ive found this solves problems with a few UPS's - namely 
               // really cheap BlackOut Buster UPS's
    ioctl(fd, TIOCMBIC, TIOCM_RTS);
    ioctl(fd, TIOCMBIC, TIOCM_DTR);
    usleep(1);
    ioctl(fd,TIOCMGET,&bits); bits&=mask; bits|=outb; ioctl(fd,TIOCMSET,&bits);
    ioctl(fd,TIOCMGET, &bits);

    do {

	ioctl(fd,TIOCMGET, &bits);
	notifyclients(failed_and_notified);
	if ((((bits & pfsignal) > 0?1:0)== pfbit) || mypower == 1) {
	    /* Power is OFF */
	    if (previous_fail && !failed_and_notified && 
			(outtime+delay < time(NULL))) {
		
		notifyinit(FAIL);
		failed_and_notified = 1;
	    } else if (!previous_fail) {
		previous_fail = 1;
		outtime = time(NULL);
		log(LOG_CRIT, LOG_DAEMON, "The power went out!!\n");;
	    }
	} else {
	    /* Power is ON */
	    if (previous_fail) {
		outfor = time(NULL)-outtime;
		previous_fail = 0;
		if (!failed_and_notified) {
		    snprintf(logbuffer, MAXBUF-1, "The power is back, Very short glitch; out %d second%s\n",
			outfor, outfor==1 ? "" : "s");
		    log(LOG_CRIT, LOG_DAEMON, logbuffer);
		}
	    }
	    if (failed_and_notified && (outtime+outfor+delay < time(NULL))) {
		previous_fail = 0;
		failed_and_notified = 0;

		notifyinit(OK);
	    }

	}
	usleep(250);
    }
    while(quiting);
    exit(0);
}

/* log to the syslog */
void log(int type, int facility, const char *entry, ...)
{
    openlog("powerd", LOG_CONS | LOG_PID, facility);
    syslog(type, entry);
    closelog();
}

/* Borrowed from Apache
 * returns the uid_t of a string
 */
uid_t uname2id(char *name) {
    struct passwd *ent;

    if(name[0] == '#')
        return(atoi(&name[1]));

    if(!(ent = getpwnam(name))) {
	    fprintf(stderr,"powerd: bad user name %s\n", name);
	    exit(1);
    }
    else return(ent->pw_uid);
}

/* init does the shutdown command - we just tell them
 * this requires root privlidges
 */
void notifyinit(int status) {
	unsigned int file;

	errno = 0;

	if (touser != NULL)
	    seteuid(root);

	if ((file = open(PWRSTAT, O_CREAT | O_WRONLY, 0644)) == -1) {
		fprintf(stderr, "Cant open %s: %s\n", PWRSTAT, strerror(errno));
		exit(-1);
	}
	write(file, status ? "FAIL\n" : "OK\n", status ? 5 : 3);
	close(file);
#ifdef SYSV_SIGNAL
	kill(1, SIGPWR);
#else
	bsdshutdown(status);
#endif
	if (touser != NULL)
	    seteuid(user);

	failed_and_notified = status;
	if (status == OK) {
	    if (outfor >= 60) {
		snprintf(logbuffer, MAXBUF-1, "The power is back! The power was out for %d minute%s, %d second%s\n",
			    outfor/60, outfor/60==1 ? "" : "s",
			    outfor%60, outfor%60==1 ? "" : "s");
		log(LOG_CRIT, LOG_DAEMON, logbuffer);
	    } else {
		snprintf(logbuffer, MAXBUF-1, "The power is back! The power was out for %d second%s\n",
			    outfor, outfor==1 ? "" : "s");
		log(LOG_CRIT, LOG_DAEMON, logbuffer);
	    }
	}
}

/* Lets bind to a port and become a server that can respond to MULTIPLE 
 * connections
 */
int bindport() {
    int len, addr_len, newfd, sin_size, numbytes;
    struct sockaddr_in serveraddr, remoteaddr;
    struct timeval mytv;
    FileDescriptor *list;    
    
    serverfd=socket(AF_INET, SOCK_STREAM, 0);
//    fcntl(serverfd, F_SETFL, O_NONBLOCK);
    if (serverfd==-1) {
	fprintf(stderr, "socket()");
	 exit(-1);
    }

    serveraddr.sin_family = AF_INET;
    serveraddr.sin_addr.s_addr = INADDR_ANY;
    serveraddr.sin_port = htons(listenport);

    if ((touser != NULL) && (listenport < 1024))
	seteuid(root);

    if (bind(serverfd, (struct sockaddr *)&serveraddr, sizeof(serveraddr))) {
	fprintf(stderr, "bind()\n"); 
	close(serverfd); 
	exit(-1);
    }
    if (listen(serverfd, 20)) {
	fprintf(stderr, "listen()\n");
	close(serverfd);
	exit(-1);
    }

    len = sizeof(serveraddr);

    if (getsockname(serverfd, (struct sockaddr *)&serveraddr, &len)) {
	fprintf(stderr, "getsockname()\n"); 
	close(serverfd); 
	serverfd=-1; 
	exit(-1); 
    }

    if ((touser != NULL) && (listenport < 1024))
	seteuid(user);

    addr_len = sizeof(struct sockaddr);

//    printf("Im here\n");
    while (1) {
	errno = 0;
	mytv.tv_sec = 5;
	mytv.tv_usec = 850000;

	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_SET(serverfd, &readfds);
	reinstate();

	if ((nfds = select(MAXCONNECTIONS, &readfds, NULL, NULL, &mytv)) == -1) {
	    fprintf(stderr, "select(): Oh no! %s\n", strerror(errno));
	}
//	printf("LOOP! %d\n", nfds);
	
	if (FD_ISSET(serverfd, &readfds)) {
	    FileDescriptor *tmp;
	    sin_size = sizeof(struct sockaddr_in);
	    if ((newfd=accept(serverfd, (struct sockaddr *)&remoteaddr, 
				&sin_size)) == -1) {
		perror("accept");
		continue;
	    }
//	    fcntl(newfd, F_SETFL, O_NONBLOCK);
//	    printf("Got connection from %s\n", inet_ntoa(remoteaddr.sin_addr));
	    tmp = addfd(newfd, remoteaddr.sin_addr);
	    addstatus(tmp, CONNECTED);
/*	    if ((numbytes=sendto(serverfd, "Hi there man", 13, 0,
	          (struct sockaddr *)&remoteaddr, sizeof(struct sockaddr))) == -1) {
		perror("sendto");
		exit(-1);
	    }*/
	    snprintf(logbuffer, sizeof(logbuffer)-1, "Connection from %s\n", inet_ntoa(tmp->in));
	    log(LOG_INFO, LOG_AUTHPRIV, logbuffer);
	    snprintf(sendbuf, MAXBUF, "PWRD: %s\n", ver);
	    send(newfd, sendbuf, strlen(sendbuf), 0);
	    continue;
	}
	if (fdlist == NULL) continue;
	list=fdlist;
	while (list) {
	    FileDescriptor *tmp, *lst;
	    lst = list;
	    list = list->next;
	    errno = 0;
	    if ((time(NULL) - lst->time) > TIMEOUT) {
//		printf("TIMEOUT for %d\n", lst->fd);
		snprintf(sendbuf, MAXBUF, "ERROR 550\n");
//		if (send(lst->fd, sendbuf, strlen(sendbuf), 0) == -1)
//			printf("SEND ERROR\n");
		close(lst->fd);
		tmp = lst;
		delfd(tmp);
		continue;
	    }
	    if (FD_ISSET(lst->fd, &readfds)) {
		if (((numbytes = recv(lst->fd, &buf, MAXBUF-2, 0)) <= 0) || numbytes > 800) {
//		    fprintf(stderr, "Closing user %d: %p\n", lst->fd, lst);
		    close(lst->fd);
		    tmp = lst;
		    delfd(tmp);
		    continue;
		}
		buf[numbytes-1] = '\0';
//		fprintf(stdout, "got something from %d (%s) (%s)\n", lst->fd, inet_ntoa(lst->in), buf);
		parse(lst, buf);
	    } else {
//		printf("Else got here for %d (%s)\n", lst->fd, inet_ntoa(lst->in));
	    }
	}
    }
    
    close(serverfd);
    
    return 1;
}

/* Add a file descriptor for a network socket
 */
FileDescriptor *addfd(int fd, struct in_addr in)
{
    FileDescriptor *new, *list;
    if ((new = (FileDescriptor *)malloc(sizeof(FileDescriptor))) == NULL) {
	snprintf(logbuffer, sizeof(logbuffer)-1, "Memory Failure\n");
	log(LOG_ERR, LOG_DAEMON, logbuffer);
	fprintf(stderr, "Memory failure!\n");
	exit(-1);	
    }
    new->fd = fd;
    new->in = in;
    new->next = NULL;
    new->prev = NULL;
    new->time = time(NULL);
    new->status = 0;
    
    if (fdlist == NULL) fdlist = new;
    else {
	for (list=fdlist; list->next; list=list->next);
//	printf("addfd %p %p %p\n", list, list->next, new);
	list->next = new;
	new->prev = list;
    }
//    printf("Added an FD, fdlist = %p %p %p %p %s\n", fdlist, list->prev, list->next, list, ctime(&(list->time)));
    FD_SET(new->fd, &readfds);
    FD_SET(new->fd, &writefds);
    return new;
}

/* Remove an FD socket */
void delfd(FileDescriptor *fd)
{
//    printf("DELFD: %p %p %p\n", fd->prev, fd, fd->next);
    if (fd->prev == NULL) { /* Very beginning */
//	printf("delfd case 1\n");
	fdlist = fd->next;
	if (fdlist)
	    fdlist->prev = NULL;
    } else if (fd->next == NULL) { /* Very end with previous list */
//	printf("delfd case 2\n");
	fd->prev->next = NULL;
    } else {			/* We can assume it is in the middle */
//	printf("delfd case 3\n");
	fd->prev->next = fd->next;
	fd->next->prev = fd->prev;
    }

//    printf("Deleted an FD %p\n", fd);
    FD_CLR(fd->fd, &readfds);
    FD_CLR(fd->fd, &writefds);
    free(fd);
}

/* Redo the FD_SET for all network sockets - Im not great with select(), 
 * if someone can improve this, im open to suggestions.
 */
void reinstate(void) {
    FileDescriptor *list;
    if (fdlist == NULL) return;
    for (list=fdlist; list; list=list->next) {
	FD_SET(list->fd, &readfds);
	FD_SET(list->fd, &writefds);
    }
    
}

/* Parse the input from DAEMON<->DAEMON communication 
 */
int parse(FileDescriptor *fd, char *buf) {
    char copy[1024];
    char *tmp;
    char *command;
    char *param;

    strcpy(copy, buf);
    
    tmp = buf;
    command = strtok(buf, " ");
    param = strtok(NULL, " ");

    if (!command || !param) {
	snprintf(sendbuf, MAXBUF, "ERROR PROT%s\n", ver);
	send(fd->fd, sendbuf, strlen(sendbuf), 0);
	close(fd->fd);
	delfd(fd);
	return 0;
    }

//    printf("Parsing: \'%s\' - \'%s\'\n", command, param);
    if (mode == 1) {
	if (!strcmp(command, "AUTHENTICATE")) {
//	    printf("Trying to authenticate\n");
	    auth(fd, param);
	} else if (!strcmp(command, "SHUTDOWN")) {
	    request(fd, param, SHUTDOWN);
	} else if (!strcmp(command, "CANCEL")) {
	    request(fd, param, CANCEL);
	} else if (!strcmp(command, "QUIT")) {
	    request(fd, param, QUIT);
	} else {
	    snprintf(sendbuf, MAXBUF, "ERROR PROT%s\n", ver);
	    send(fd->fd, sendbuf, strlen(sendbuf), 0);
	    snprintf(logbuffer, sizeof(logbuffer)-1, "Unknown Client from %s\n", inet_ntoa(fd->in));
	    log(LOG_NOTICE, LOG_AUTHPRIV, logbuffer);
	    close(fd->fd);
	    delfd(fd);
	    return 0;
	}
	return 1;
    } else if (mode == 0) {
	if (!strcmp(command, "AUTHED")) {
	    if (checkstatus(fd, AUTHENTICATED)) {
		getrid(fd, 1);
	    }
	    if (!strcmp(param, "OKAY")) {
		addstatus(fd, AUTHENTICATED);
//		printf("Were authenticated\n");
	    } else if (!strcmp(param, "FAIL")) {
		snprintf(logbuffer, sizeof(logbuffer)-1, "Failed authentication. Password rejected for %s\n", inet_ntoa(fd->in));
		log(LOG_NOTICE, LOG_AUTHPRIV, logbuffer);
		getrid(fd, 0);
//		printf("Were NOT authenticated\n");
	    }
	} else if (!strcmp(command, "PWRD:")) {
		addstatus(fd, CONNECTED);
	} else if (!strcmp(command, "WHATIDO")) {
	    snprintf(logbuffer, sizeof(logbuffer)-1, "Successfully sent %s notice to %s\n", param, inet_ntoa(fd->in));
	    log(LOG_INFO, LOG_AUTHPRIV, logbuffer);
	   addstatus(fd, WEAGREE);
//	   printf("I have set WEAGREE\n");
	} else if (!strcmp(command, "ERROR")) {
	    snprintf(logbuffer, sizeof(logbuffer)-1, "Error from %s: %s\n", inet_ntoa(fd->in), param);
	    log(LOG_NOTICE, LOG_AUTHPRIV, logbuffer);
		fprintf(stderr, "ERROR!! %s (we need to log this)\n", param);
	} else {
		fprintf(stderr, "EVEN BIGGER ERROR!!! %s (gotta log)\n", param);
	}
    }
    return 1;
}
    
/* When hitting control+c with DEBUG defined */
void quit_sig(int sig)
{
    close(serverfd);
    printf("Shutting down\n");
    sleep(1);
    exit(1);
}

/* Send the daemon SIGPWR to simulate a power outtage - used for debugging
 */
void debughelp(int sig)
{
    mypower = (mypower == 1) ? 0 : 1;
}

/* Open and parse the configuration file
 */
void openconfig(int *delay) {
    FILE *file;
    char buf[1024];
    char *command;
    char *parameter;
    char *password;
    int line=0;
    char *host, *port;
    int iport;
    struct hostent *hname;
    struct in_addr addr;

//    printf("Opening config\n");

    if ((file = fopen(CONFIG, "r")) == NULL) {
	fprintf(stderr, "Config file %s does not exist.\nPlease run the 'detectups' application, or use one of the examples included\nwith the application.\n", CONFIG);
	exit(-1);
    }
    
    while(fgets(buf, 1023, file) != NULL) {
	buf[strlen(buf)-1] = '\0';
	line++;
	if ((*buf == '#') || (*buf == '\n') || (*buf == '/'))
	    continue;
	command = strtok(buf, " ");
	parameter = strtok(NULL, " ");
	password = strtok(NULL, " ");

//	printf("Line%d: (%s) (%s) (%s)\n", line, command, parameter, password);
	if (!command || strlen(command) < 2 || strlen(parameter) < 2) {
//		printf("not taking line %d\n", line);
		continue;
	}

	if (!strcasecmp(command, "mode")) {
	    if (!strcasecmp(parameter, "peer")) {
		mode = 1;
	    } else if (!strcasecmp(parameter, "monitor")) {
		 mode = 0;
	    } else {
		fprintf(stderr, "Line %d of config file invalid! Aboring.\n", line);
		exit(-1);
	    }
	}
	else if (!strcasecmp(command, "monitor")) {
	    device = strdup(parameter);
//	    printf("Monitoring %s\n", device);
	} else if (!strcasecmp(command, "notify")) {
	    if (strlen(password) < 5) {
		fprintf(stderr, "Password is too short/invalid. Line %d\n", line);
		exit(-1);
	    } else {
		host = strtok(parameter, ":");
		if ((port = strtok(NULL, "")))
		    iport = atoi(port);
		else
		    iport = PORT;

		if ((hname = gethostbyname(host))) {
		    while(*hname->h_addr_list != 0) {
			memset(&addr.s_addr, 0, sizeof(addr.s_addr));
			memcpy(&addr.s_addr, *hname->h_addr_list, hname->h_length);
			addclient(inet_ntoa(addr), password, iport);
			hname->h_addr_list++;
		    }
		}
	    }
	} else if (!strcasecmp(command, "listen")) {
	    if (strlen(password) < 5) {
		fprintf(stderr, "Password is too short/invalid. Line %d\n", line);
		exit(-1);
	    } else if ((hname = gethostbyname(parameter))) {
		while(*hname->h_addr_list != 0) {
		    memset(&addr.s_addr, 0, sizeof(addr.s_addr));
		    memcpy(&addr.s_addr, *hname->h_addr_list, hname->h_length);
		    addlisten(inet_ntoa(addr), password);
		    hname->h_addr_list++;
		}
	    }
	} else if (!strcasecmp(command, "listenport")) {
	    if ((listenport = atoi(parameter)) < 1) {
		fprintf(stderr, "Listen port is invalid. Line %d\n", line);
		exit(-1);
	    }
	} else if (!strcasecmp(command, "powerfail")) {
	    if (!strcasecmp(parameter, "CAR")) pfsignal = TIOCM_CAR;
	    else if (!strcasecmp(parameter, "CTS")) pfsignal = TIOCM_CTS;
	    else if (!strcasecmp(parameter, "DSR")) pfsignal = TIOCM_DSR;
	    else if (!strcasecmp(parameter, "RNG")) pfsignal = TIOCM_RNG;
	    else {
		fprintf(stderr, "Config file: %s is not a valid signal. Line %d\n", parameter, line);
		exit(-1);
	    }
	    if (!strcasecmp(password, "HIGH")) pfbit = 1;
	    else if (!strcasecmp(password, "LOW")) pfbit = 0;
	    else {
		fprintf(stderr, "Config file: signal must be either HIGH or LOW. Line %d\n", line);
		exit(-1);
	    }
	} else if (!strcasecmp(command, "delay")) {
	    if ((*delay = atoi(parameter)) < 1) {
		fprintf(stderr, "Delay time is invalid. Line %d\n", line);
		exit(-1);
	    }
	} else if (!strcasecmp(command, "user")) {
	    if (touser != NULL)
		free(touser);
	    touser = strdup(parameter);
	}
	
    }
    fclose(file);
}

/* Add a LISTEN arguement from the config file
 */
void addlisten(char *host, char *password) {
    Listen *new, *llist;

    new = (Listen *)malloc(sizeof(Listen));

    new->host = strdup(host);
    new->password = strdup(password);
    new->next = NULL;

    if (listens == NULL) listens = new;
    else {
	for (llist=listens; llist->next; llist=llist->next);
//	printf("addlisten %p %p %p\n", llist, llist->next, new);
	llist->next = new;
    }
//    printf("Added a listen, listens = %p %p %p: %s %s\n", listens, llist->next, llist, host, password);
    
}

/* Add a NOTIFY arguement from the config file
 */
void addclient(char *host, char *password, int port) {
    Client *new, *clist;
//    char *port;

    new = (Client *)malloc(sizeof(Client));

    new->host = strdup(strtok(host, ":"));
    new->password = strdup(password);
//    if ((port = strtok(NULL, "")))
//	new->port = atoi(port);
//    else
//	new->port = PORT;
    new->port = port;
    new->next = NULL;
    if (clients == NULL) clients = new;
    else {
	for (clist=clients; clist->next; clist=clist->next);
//	printf("addclient %p %p %p\n", clist, clist->next, new);
	clist->next = new;
    }
//    printf("Added a client, clients = %s %s %d\n", host, password, port);
    
}

/* Lets try and see if they are allowed to shut us down ... we will 
 * encrypt this later on.
 */
void auth(FileDescriptor *fd, char *password) {
    Listen *list;

    for (list = listens; list; list=list->next) {
//	printf("Checking host %s %s\n", listens->host, inet_ntoa(fd->in));
	if (!strcmp(listens->host, inet_ntoa(fd->in))) {
	    if (!strcmp(password, listens->password)) {
		addstatus(fd, AUTHENTICATED);
		snprintf(sendbuf, MAXBUF, "AUTHED OKAY\n");
		send(fd->fd, sendbuf, strlen(sendbuf), 0);
		return;
	    } else {
		snprintf(logbuffer, sizeof(logbuffer)-1, "Failed authentication. Bad password from %s\n", inet_ntoa(fd->in));
		log(LOG_NOTICE, LOG_AUTHPRIV, logbuffer);
		snprintf(sendbuf, MAXBUF, "AUTHED FAIL\n");
		send(fd->fd, sendbuf, strlen(sendbuf), 0);
		close(fd->fd);
		delfd(fd);
		return;
	    }
	}
    }
    snprintf(logbuffer, sizeof(logbuffer)-1, "Failed authentication. Unknown host: %s\n", inet_ntoa(fd->in));
    log(LOG_NOTICE, LOG_AUTHPRIV, logbuffer);

    snprintf(sendbuf, MAXBUF, "AUTHED FAIL\n");
    send(fd->fd, sendbuf, strlen(sendbuf), 0);
    close(fd->fd);
    delfd(fd);
}
    
/* Okay, this isnt used yet, but it will be - it works
 */
int vhost(FileDescriptor *fd, char *param) {
    Listen *list;

    for (list = listens; list; list=list->next) {
	if (!strcmp(listens->host, inet_ntoa(fd->in))) {
	    addstatus(fd, VALIDHOST);
	    return 1;
	}
    }
    snprintf(sendbuf, MAXBUF, "AUTHED FAIL\n");
    send(fd->fd, sendbuf, strlen(sendbuf), 0);
    close(fd->fd);
    delfd(fd);
    return 0;
}

/* We got a command from a MONITOR daemon, lets do it
 */
void request(FileDescriptor *fd, char *param, int action) {
    if (checkstatus(fd, AUTHENTICATED)) {
//	printf("We now have to do %d\n", action);
    } else {
	snprintf(sendbuf, MAXBUF, "ERROR 551\n");
	send(fd->fd, sendbuf, strlen(sendbuf), 0);
	snprintf(logbuffer, sizeof(logbuffer)-1, "Command from unauthenticated host: %s\n", inet_ntoa(fd->in));
	log(LOG_NOTICE, LOG_AUTHPRIV, logbuffer);
	close(fd->fd);
	delfd(fd);
	return;
    }

    if (action == SHUTDOWN) {
	snprintf(sendbuf, MAXBUF, "WHATIDO SHUTDOWN\n");
	send(fd->fd, sendbuf, strlen(sendbuf), 0);
	outtime = time(NULL);
	log(LOG_CRIT, LOG_DAEMON, "We have been notified power is out!!\n");;
	notifyinit(FAIL);
//	printf("sending WHATIDO FAIL\n");
    } else if (action == CANCEL) {
	snprintf(sendbuf, MAXBUF, "WHATIDO CANCEL\n");
	send(fd->fd, sendbuf, strlen(sendbuf), 0);
	outfor = time(NULL)-outtime;
	notifyinit(OK);
//	printf("sending WHATIDO OK\n");
    } else if (action == QUIT) {
	close(fd->fd);
	delfd(fd);
	return;
    }
}

/* Okay, were monitoring the UPS, lets tell the clients whats going on */
int notifyclients(int status) {
    static int stat = 0;
    struct timeval tv;
    FileDescriptor *fd, *tmp, *ptr;
    Client *cli;
    struct sockaddr_in remote;
    struct in_addr inp;
    int newfd;
    int numbytes;

//    printf("Notifying clients now %d\n", status);

    if (stat != status) {
//	printf("Closing old FDs\n");
	stat = status;
	if (fdlist != NULL) {
	    ptr = fdlist;
	    while (ptr) {
		fd = ptr;
		ptr = ptr->next;
		close(fd->fd);
		delfd(fd);
		continue;
	    }
	}
										
	for(cli = clients; cli; cli = cli->next) {
	    if (!inet_aton(cli->host, &inp)) {
//		printf("inet_aton failed\n");
		continue;
	    }
	    if ((newfd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
		perror("socket");
		exit(1);
	    }
	    remote.sin_family = AF_INET;
	    remote.sin_port = htons(cli->port);
	    remote.sin_addr = inp;
	    bzero(&(remote.sin_zero), 8);
	    fcntl(newfd, F_SETFL, O_NONBLOCK);
//printf("gonna connect to %s\n", cli->host);
	    errno = 0;
	    if (connect(newfd, (struct sockaddr *)&remote,
				sizeof(struct sockaddr)) == -1) {
		if (!(errno == EINPROGRESS || errno == EALREADY)) {
//		    printf("Connect error!\n");
		    exit(-1);
		}
	    }
	    tmp = addfd(newfd, inp);
	    tmp->client = cli;
        }
    }
    FD_ZERO(&readfds);
    FD_ZERO(&writefds);
    reinstate();

    tv.tv_sec = 1;
    tv.tv_usec = 0;
    if ((nfds = select(MAXCONNECTIONS, &readfds, &writefds, NULL, &tv)) == 0) {
//	fprintf(stderr, "select(): Aint nothing!\n");
	return 0;
    }

    ptr = fdlist;	
    while (ptr) {
	fd = ptr;
	ptr = fd->next;
	errno = 0;
//printf("Gonna work on %p\n", fd);
	     
	if ((time(NULL) - fd->time) > TIMEOUT+10) {
//	    printf("TIMEOUT for %d\n", fd->fd);
	    snprintf(sendbuf, MAXBUF, "ERROR 550\n");
//	    if (send(fd->fd, sendbuf, strlen(sendbuf), 0) == -1)
//		printf("SEND ERROR\n");
	    close(fd->fd);
	    delfd(fd);
	    continue;
	}

	if (FD_ISSET(fd->fd, &readfds)) {
//	    printf("We shall work on readfds\n");
	    if ((numbytes = recv(fd->fd, &buf, MAXBUF-1, 0)) <= 0) {
//		fprintf(stderr, "Closing user %d: %p\n", fd->fd, fd);
		close(fd->fd);
		delfd(fd);
		continue;
	    }
	    buf[numbytes-1] = '\0';
//	    printf("WE GOT DIS: %s\n", buf);
	    parse(fd, buf);
	}
	if (FD_ISSET(fd->fd, &writefds)) {
//	    printf("Lets work on writefds!\n");
	    if (checkstatus(fd, CONNECTED) && !checkstatus(fd, SENTNOTICE)) {
		snprintf(sendbuf, MAXBUF-1, "AUTHENTICATE %s\n", fd->client->password); 
		send(fd->fd, sendbuf, strlen(sendbuf), 0);
		addstatus(fd, SENTNOTICE);
//		printf("Sent notice!\n");
		continue;
	    } else if (checkstatus(fd, AUTHENTICATED) && !checkstatus(fd, SENTCOMMAND)) {
		if (status == 1)
		    snprintf(sendbuf, MAXBUF, "SHUTDOWN now\n");
		else if (status == 0)
		    snprintf(sendbuf, MAXBUF, "CANCEL now\n");
		send(fd->fd, sendbuf, strlen(sendbuf), 0);
		addstatus(fd, SENTCOMMAND);
//		printf("Sent command!\n");
	    } else if (checkstatus(fd, WEAGREE)) {
		snprintf(sendbuf, MAXBUF, "QUIT now\n");
		send(fd->fd, sendbuf, strlen(sendbuf), 0);
//		printf("We agree!\n");
	    }
	}
//	printf("Lets continue on!\n");
    }
    return 1;
}

/* get rid of a FD with closing it and junk
 */
void getrid(FileDescriptor *fd, int i) {
    if (i == 1) {
	snprintf(sendbuf, MAXBUF, "ERROR PROT%s\n", ver);
	send(fd->fd, sendbuf, strlen(sendbuf), 0);
    }
    close(fd->fd);
    delfd(fd);
}

void bsdshutdown(int status) {
    char *argv[4];
    int pid;

    argv[0] = "shutdown";

    pid = fork();

    if (pid != 0)
	return;
    if (status == FAIL) {
	argv[1] = "+2";
	argv[2] = "Power is failing! Shutting down.";
	argv[3] = 0;
    } else {
	argv[1] = "-c";
	argv[2] = "Power restored. Shutdown Cancelled.";
	argv[3] = 0;
    }
    execve("/sbin/shutdown", argv, NULL);
}

/* Thats it. is the code messy? :) */
