/*
 *	file related utilities
 *
 *	Copyright (C) 2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <unistd.h>
#include <dirent.h>
#include <limits.h>
#include <errno.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <futils.h>
#include <logging.h>

int /* bool */
nc_isdir(const char *path)
{
	struct stat stb;

	if (stat(path, &stb) < 0)
		return 0;

	return S_ISDIR(stb.st_mode);
}

int /* bool */
nc_isreg(const char *path)
{
	struct stat stb;

	if (stat(path, &stb) < 0)
		return 0;

	return S_ISREG(stb.st_mode);
}

int /* bool */
nc_file_exists(const char *filename)
{
	return access(filename, F_OK) == 0; /* bool */
}

int /* bool */
nc_file_exists_fmt(const char *fmt, ...)
{
	char filename[PATH_MAX + 1] = {'\0'};
	va_list ap;

	va_start(ap, fmt);
	vsnprintf(filename, sizeof(filename), fmt, ap);
	va_end(ap);

	return nc_file_exists(filename);
}

int /* bool */
nc_file_executable(const char *filename)
{
	return access(filename, X_OK) == 0;
}

const char *
nc_basename(const char *pathname)
{
	const char *basename;

	if (!pathname || !(basename = strrchr(pathname, '/')))
		return pathname;
	else
		return basename + 1;
}

int
nc_readlink(const char *link, char **path)
{
	char buf[PATH_MAX + 1] = {'\0'};
	ssize_t ret;

	if(!link || !*link || !path)
		return -1;

	ret = readlink(link, buf, sizeof(buf));
	if(ret > 0 && (size_t)ret < sizeof(buf)) {
		buf[ret] = '\0';
		return nc_string_dup(path, buf);
	}
	return -1;
}

int
nc_scandir(const char *dirname, const char *match_prefix, nc_string_array_t *res)
{
	struct dirent *ent;
	unsigned int count;
	size_t pfxlen;
	DIR *dir;

	dir = opendir(dirname);
	if (!dir)
		return 0;

	count = res->count;
	pfxlen = match_prefix ? strlen(match_prefix) : 0;
	while ((ent = readdir(dir))) {
		if (ent->d_name[0] == '.')
			continue;

		if (!pfxlen || !strncmp(ent->d_name, match_prefix, pfxlen))
			nc_string_array_append(res, ent->d_name);
	}

	closedir(dir);
	return res->count - count;
}

int
nc_make_dot_suffix(nc_stringbuf_t *buff, const char *path, const char *suffix)
{
	char *pos;

	if(!buff || !path)
		return -1;

	suffix = suffix ? suffix : "";

	nc_stringbuf_clear(buff);

	pos = strrchr(path, '/');
	if(pos) {
		/* construct <dirname>/.<basename><suffix> */
		pos++;
		nc_stringbuf_puts_n(buff, path, pos - path);
		nc_stringbuf_putc(buff, '.');
		nc_stringbuf_puts(buff, pos);
		nc_stringbuf_puts(buff, suffix);
	} else {
		nc_stringbuf_putc(buff, '.');
		nc_stringbuf_puts(buff, path);
		nc_stringbuf_puts(buff, suffix);
	}
	return 0;
}

int
nc_rename_to_dot_backup(const char *filename, const char *dot_suffix)
{
	nc_stringbuf_t old = NC_STRINGBUF_INIT;
	int            ret = -1;

	/* construct /path/.<name>[suffix] from /path/name */
	if(nc_make_dot_suffix(&old, filename, dot_suffix) < 0) {
		nc_error("Unable to apply a backup suffix to filename: %s",
				filename);
		return -1;
	}

	/* just rename filename to its backup name */
	ret = 0;
	if(rename(filename, old.string) < 0) {
		if(errno != ENOENT)
			ret = -1;
	}

	nc_stringbuf_destroy(&old);
	return ret;
}

