# -*- coding: utf-8 -*-
# brld - orca brlapi bindings
# Authors:
#    Halim Sahin <halim.sahin@t-online.de>
#    Marco Skambraks <marco@skammel.de>

from ctypes import byref
from ctypes import c_int
from ctypes import CDLL
from ctypes import create_string_buffer
import os


# key values for orca
KEY_CMD_FWINLT = 23
KEY_CMD_FWINRT = 24
KEY_CMD_LNUP = 1
KEY_CMD_LNDN = 2
KEY_CMD_FREEZE = 32
KEY_CMD_TOP_LEFT = 11
KEY_CMD_BOT_LEFT = 12
KEY_CMD_HOME = 29
KEY_CMD_SIXDOTS = 34
KEY_CMD_ROUTE = 65536
KEY_CMD_CUTBEGIN = 131072
KEY_CMD_CUTLINE = 327680
KEY_FLG_TOGGLE_ON = 0 # 256
KEY_TYPE_CMD = 0

rangeType_all = 0
rangeType_command = 2

class getconf:
    """ class to parse sbl.conf and keymap files
    """

    keylist = {}
    port = 0
    auth = "default"

    def __init__(self):
        try:
            conffile = open("/etc/sbl.conf",'r')
        except:
            print "cannot open /etc/sbl.conf"
            return None

        for i in conffile.readlines():
            if i.replace(' ','')[0] == '#':
                continue
            if i.find('brld_port') >= 0:
                try:
                    self.port = int(i.replace('\n','').split('=')[1])
                except:
                    self.port = 0

            if i.find('brld_auth_key') >= 0:
                self.auth = i.replace('\n','').split('=')[1]
                
        conffile.close()

    def _kmline(self, line):
        """ parse a keymap line - return name and value
        """
        comma = line.find(',')
        cmd_name = line[0:comma]
        cmd_val = 0

        cmd_name = cmd_name.replace(' ','').replace('\n','').lower().split('=')
        if cmd_name[0] and cmd_name[1]:
            cmd_val = int(cmd_name[1].replace('b','').replace('+',''))
            self.keylist[cmd_name[0]] = cmd_val
            return True
            
        return False


    def brldport(self):
        """ returns the brld port
        """
        return self.port
    def brldauth(self):
        """ returns the brld auth key
        """
        return self.auth



    def loadkeymap(self, brlname):
        """ keymap parser - first try to open $HOME/.sbl
        """
        
        self.keylist.clear()

        try:
            kmfile = open(os.getenv("HOME") + "/.sbl/keymap/" \
                     + brlname + ".map", 'r')
        except:
            try:
                kmfile = open("/etc/sbl/keymap/" + brlname + ".map", 'r')
            except:
                print "no keymap found for ", brlname
                return False

        for i in kmfile.readlines():
            if i.replace(' ','')[0] == '#':
                continue

            self._kmline(i)

        kmfile.close()
        if not self.keylist:
            return False
            
        return True
        
    def getkey(self, keystr):
        """ looks for a key in keylist
        """
        
        try:
            return self.keylist[keystr]
        except:
            return False


class WriteStruct:
    """ structure for braille text and attributes
    """
    
    regionBegin = 1
    regionSize = 0
    text = ""
    cursor = 0 # cursorcell
    attrOr = ""

    def __init__(self):
        return None    


class Connection:
    """ handles brld connection
    """

    brld = None
    conf = None
    brlname = ""
    lastkey = 0
    fileDescriptor = -1
    x = c_int()
    y = c_int()        
    displaySize = (32,1)
    rangeType_all = 0
    rangeType_command = 2

    def ignoreKeys(self, key_type, set):
        return True

    def __init__(self, host=None, auth=None):

        tmpstr = create_string_buffer('\000' * 128)
        if self.fileDescriptor >= 0:
            return None


        try:
            if self.brld == None:
                self.brld = CDLL("libbrld.so.1")
        except:
            print "cannot open libbrld.so.1"
            return None

        self.conf = getconf()
        self.fileDescriptor = self.brld.brld_open("localhost", self.conf.brldport())

        if self.fileDescriptor < 0:
            print "brld: connection failed"
            raise IoError

        tmpstr = create_string_buffer('\000' * 128)
        tmpstr.value = self.conf.brldauth()
        if self.brld.brld_reg(2, tmpstr.value) < 0:
            print "brld: reg failed"
        else:
            if self.brld.brld_seteventmode() <0:
                print "brld: eventmode setfailed"
                return None

            if self.brld.brld_getxy(byref(self.x),byref(self.y)) <0:
                print "brld: error getting Displaysize"
                return None
                
            self.displaySize = (self.x.value, self.y.value)
            if self.brld.brld_getalias(byref(tmpstr)) <0:
                print "brld: error getting brlname"
                return None
            self.brlname = tmpstr.value
            self.conf.loadkeymap(self.brlname)

    def __del__(self):
        if self.fileDescriptor >= 0:
            self.brld.brld_close(self.fileDescriptor)
            self.fileDescriptor = -1
    
    def writeText(self,text, cursor = -1):

        s = create_string_buffer('\000' * 128)
        s.value = text.decode("UTF-8").encode("iso8859-1")
        
        if self.brld.brld_write(cursor, s, "") <0:
            print "brld: writeText error"
            return False
        
        return True

    def write(self, writestruct, displayNumber = None, regionBegin = None, regionSize=None, text=None, andMask = None, orMask = None, cursor = None, charset = None):

        s = create_string_buffer('\000' * 128)
        a = create_string_buffer('\000' * 128)

        len_diff = 0
        add_spaces = ""
        s.value = writestruct.text.replace(u"\u2022", "*").encode("iso8859-1")
        if len(writestruct.attrOr) < len(writestruct.text):
            len_diff = len(writestruct.text) - len(writestruct.attrOr)
            add_spaces = ' ' * len_diff

        a.value = writestruct.attrOr.replace('\x00',' ') + add_spaces

        if self.brld.brld_write(writestruct.cursor-1, s.value, a.value) <0:
            print "brld: write err"
            print "brlline= ", writestruct.text
            return False

        return True
        
    def readKey(self,wait = True):

        pressed = c_int()
        key = self.brld.brld_getkey(byref(pressed))
        if key == -4:
            print "brld: getkey no connection"
            raise IoError

        if key <= 0:
            return 0

#        print "key=",key," pressed=",pressed.value
        if key >= 100:
            return key
            
        if self.lastkey == 0 and pressed.value == 0:
            return 0
        if self.lastkey == 0 and pressed.value > 0:
            self.lastkey = key
        else:
            if self.lastkey > 0 and pressed.value > 0:
                self.lastkey = (self.lastkey * 1000) + key
            if self.lastkey == key and pressed.value == 0:
                self.lastkey = (key * 1000) + key
        
        return self.lastkey
    
    def expandKeyCode(self, code):
        """ expand keys to more meaningful values
        """
        
        if code == 0:
            return {}

#        print "vlaue=",code
        if code == self.conf.getkey("lnup"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_LNUP,"argument":0,"flags":0 }

        if code == self.conf.getkey("lndn"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_LNDN,"argument":0,"flags":0 }

        if code == self.conf.getkey("lnlft"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_FWINLT,"argument":0,"flags":0 }
        if code == self.conf.getkey("lnrgt"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_FWINRT,"argument":0,"flags":0 }
        if code == self.conf.getkey("jmpmark1"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_FREEZE,"argument":0,"flags":0 }
        if code == self.conf.getkey("topleft"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_TOP_LEFT,"argument":0,"flags":0 }
        if code == self.conf.getkey("botleft"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_BOT_LEFT,"argument":0,"flags":0 }
        if code == self.conf.getkey("cutbegin"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_CUTBEGIN,"argument":0,"flags":0 }
        if code == self.conf.getkey("paste"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_CUTLINE,"argument":0,"flags":0 }
        if code == self.conf.getkey("sixdotsonoff"):
            self.lastkey = 0
            return {"type":1,"command":KEY_CMD_SIXDOTS,"argument":0,"flags":0 }

        # clear lastkey if two keys are pressed
        if code > 1000:
            self.lastkey = 0
            return {}

        # values > 100 are routing keys with an offset
        if code >= 100:
            return {"type":1,"command":KEY_CMD_ROUTE, \
                    "argument":(code-self.conf.getkey("csrroutbeg")),"flags":0 }

    def acceptKeys(self, key_type, set):
        return True

    def enterTtyModeWithPath(self, path=[], driver=None):
        return True

    def enterTtyMode(self, tty=7, driver=None):
        print "brld: enterTtyMode"
        print tty, driver


    def leaveTtyMode(self):
        if self.fileDescriptor >= 0:
#            self.brld.brld_unseteventmode()
            self.brld.brld_close(self.fileDescriptor)
            self.fileDescriptor= -1

