/**********************************************************************
 * $my_monitor example -- PLI application using VPI routines
 *
 * C source to place value change callbacks on all nets in a module
 * instance, and print the simulation time and new value whenever
 * any net changes value.
 *
 * Usage:   $my_monitor(<module_instance_name>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include <malloc.h>    /* ANSI C standard memory allocation library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of routines in this PLI application */
int PLIbook_MyMonitor_calltf(), PLIbook_MyMonitor_compiletf(), 
    PLIbook_MyMonitor_callback();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_MyMonitor_register()
{
  s_vpi_systf_data tf_data;

  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$my_monitor";
  tf_data.calltf    = PLIbook_MyMonitor_calltf;
  tf_data.compiletf = PLIbook_MyMonitor_compiletf; 
  tf_data.sizetf    = NULL;
  tf_data.user_data = NULL;
  vpi_register_systf(&tf_data);
}

/**********************************************************************
 * compiletf application
 *********************************************************************/
int PLIbook_MyMonitor_compiletf(char *user_data)
{
  vpiHandle systf_handle, arg_iterator, arg_handle;
  int       tfarg_type;

  /* obtain a handle to the system task instance */
  systf_handle = vpi_handle(vpiSysTfCall, NULL);

  /* obtain handles to system task arguments */
  arg_iterator = vpi_iterate(vpiArgument, systf_handle);
  if (arg_iterator == NULL) {
    vpi_printf("ERROR: $my_monitor requires 1 argument\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  
  /* check the type of object in system task arguments */
  arg_handle = vpi_scan(arg_iterator);
  tfarg_type = vpi_get(vpiType, arg_handle);
  if (tfarg_type != vpiModule) {
    vpi_printf("ERROR: $my_monitor arg1 must be module instance\n");
    vpi_free_object(arg_iterator); /* free iterator memory */
    tf_dofinish(); /* abort simulation */
    return(0);
  }

  /* check that there is only 1 system task argument */
  arg_handle = vpi_scan(arg_iterator);
  if (arg_handle != NULL) {
    vpi_printf("ERROR: $my_monitor can only have 1 argument\n");
    vpi_free_object(arg_iterator); /* free iterator memory */
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
PLIbook_MyMonitor_calltf(char *user_data)
{
  vpiHandle   systf_h, arg_itr, mod_h, net_itr, net_h;
  s_vpi_time   time_s;
  s_vpi_value  value_s;
  s_cb_data    cb_data_s;
  char        *net_name_temp, *net_name_keep;

  /* setup value change callback options */
  time_s.type       = vpiScaledRealTime;
  value_s.format    = vpiBinStrVal;

  cb_data_s.reason  = cbValueChange;
  cb_data_s.cb_rtn  = PLIbook_MyMonitor_callback;
  cb_data_s.time    = &time_s;
  cb_data_s.value   = &value_s;
  
  /* obtain a handle to the system task instance */
  systf_h = vpi_handle(vpiSysTfCall, NULL);

  /* obtain handle to system task argument */
  /* compiletf has already verified only 1 arg with correct type */
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  mod_h = vpi_scan(arg_itr);
  vpi_free_object(arg_itr);  /* free iterator--did not scan to null */

  /* add value change callback for each net in module named in tfarg */
  vpi_printf("\nAdding monitors to all nets in module %s:\n\n",
             vpi_get_str(vpiDefName, mod_h));

  net_itr = vpi_iterate(vpiNet, mod_h);
  while ((net_h = vpi_scan(net_itr)) != NULL) {
    net_name_temp = vpi_get_str(vpiFullName, net_h);
    net_name_keep = malloc(strlen(net_name_temp)+1);
    strcpy(net_name_keep, net_name_temp);
    cb_data_s.obj = net_h;
    cb_data_s.user_data = net_name_keep;
    vpi_register_cb(&cb_data_s);
  }
}

/**********************************************************************
 * Value change callback application
 *********************************************************************/
PLIbook_MyMonitor_callback(p_cb_data cb_data_p)
{
  vpi_printf("At time %0.2f:\t %s = %s\n",
             cb_data_p->time->real,
             cb_data_p->user_data,
             cb_data_p->value->value.str);
}

/*********************************************************************/
