/**********************************************************************
 * $read_strengthval example -- PLI application using VPI routines
 *
 * C source to read logic values and strength levels of scalar nets.
 *
 * Usage: $read_strengthval(<net_name>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of routines in this PLI application */
int  PLIbook_ReadStrengthVal_calltf(),
     PLIbook_ReadStrengthVal_compiletf();
char PLIbook_DecodeBitValue();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_ReadStrengthVal_register()
{
  s_vpi_systf_data tf_data;
  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$read_strengthval";
  tf_data.calltf    = PLIbook_ReadStrengthVal_calltf;
  tf_data.compiletf = PLIbook_ReadStrengthVal_compiletf;
  tf_data.sizetf    = NULL;

  vpi_register_systf(&tf_data);
}
/*********************************************************************/

/**********************************************************************
 * compiletf application
 *********************************************************************/
int PLIbook_ReadStrengthVal_compiletf(char *user_data)
{
  vpiHandle systf_h, arg_itr, arg_h;
  int       arg_type;
  
  /* obtain a handle to the system task instance */
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  if (systf_h == NULL) {
    vpi_printf("ERROR: $read_strengthval failed to obtain systf handle\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }

  /* obtain handles to system task arguments */
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $read_strengthval requires 1 argument\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  
  /* check the type of object in system task arguments */
  arg_h = vpi_scan(arg_itr);
  arg_type = vpi_get(vpiType, arg_h);
  if (arg_type != vpiNet) {
    vpi_printf("ERROR: $read_strengthval arg must be a net\n");
    vpi_free_object(arg_itr); /* free iterator memory */
    tf_dofinish(); /* abort simulation */
    return(0);
  }

  /* check the bit size of object in system task arguments */
  if (vpi_get(vpiSize, arg_h) != 1) {
    vpi_printf("ERROR: $read_strengthval arg must be scalar\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }

  /* check that there are no more system task arguments */
  arg_h = vpi_scan(arg_itr);
  if (arg_h != NULL) {
    vpi_printf("ERROR: $read_strengthval can only have 1 argument\n");
    vpi_free_object(arg_itr); /* free iterator memory */
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_ReadStrengthVal_calltf(char *user_data) 
{
  vpiHandle   systf_h, arg_itr, arg_h, net_h;
  s_vpi_value net_val;           /* structure to receive net value */

  /* obtain a handle to the system task instance */
  systf_h = vpi_handle(vpiSysTfCall, NULL);

  /* obtain handle to system task argument
     compiletf has already verified only 1 arg with correct type */
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  net_h = vpi_scan(arg_itr);
  vpi_free_object(arg_itr);  /* free iterator memory */

  net_val.format = vpiStrengthVal;      /* set value format field */

  vpi_get_value(net_h, &net_val);       /* read net's strength value */

  vpi_printf("\nNet %s:  ", vpi_get_str(vpiName, net_h));
  vpi_printf("value=%c  strength0=%2x(hex)  strength1=%2x(hex)\n\n",
             PLIbook_DecodeBitValue(net_val.value.strength->logic),
             net_val.value.strength->s0,
             net_val.value.strength->s1);
  return(0);
}

/**********************************************************************
 * Function to convert VPI logic constant to a character
 *********************************************************************/
char PLIbook_DecodeBitValue(int bit_constant)
{
  switch (bit_constant) {
    case vpi0:        return('0'); break;
    case vpi1:        return('1'); break;
    case vpiZ:        return('Z'); break;
    case vpiX:        return('X'); break;
    case vpiL:        return('L'); break;
    case vpiH:        return('H'); break;
    case vpiDontCare: return('?'); break;
    default:          return('U'); /* undefined value passed in */
  }
}
/*********************************************************************/
