/*
 * Copyright 2010-2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package com.amazonaws.services.dynamodbv2.datamodeling;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertNull;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.junit.Test;

/**
 * Unit tests for {@code DynamoDBAnnotationRegistry}.
 */
public class DynamoDBAnnotationRegistryTest {

    /**
     * The annotation registry.
     */
    private final DynamoDBAnnotationRegistry registry = new DynamoDBAnnotationRegistry();

    /**
     * Gets the mappings of method to annotations.
     * @param clazz The class.
     * @return The maps.
     */
    private final Map<String,DynamoDBAnnotationRegistry.AnnotationMap> mapsOf(final Class<?> clazz) {
        final Map<String,DynamoDBAnnotationRegistry.AnnotationMap> maps = new HashMap<String,DynamoDBAnnotationRegistry.AnnotationMap>();
        for (final Method method : clazz.getMethods()) {
            if (method.getName().startsWith("get") && !Object.class.equals(method.getDeclaringClass())) {
                maps.put(method.getName(), registry.annotationsOf(method, null));
            }
        }
        return maps;
    }

    /**
     * Not key as {@code DynamoDBAutoGeneratedKey}.
     */
    @Test
    public void testAutoGeneratedKeyNotKey() {
        final Map<String,DynamoDBAnnotationRegistry.AnnotationMap> maps = mapsOf(AutoGeneratedKeyNotKeyClass.class);
        assertEquals(1, maps.size());
        assertFalse(maps.get("getKey").isHashKey());
        assertFalse(maps.get("getKey").isAutoGeneratedKey());
        assertNull(maps.get("getKey").getAttributeName());
    }

    /**
     * Not key as {@code DynamoDBAutoGeneratedKey}.
     */
    @DynamoDBTable(tableName="aws-java-sdk-util")
    public static class AutoGeneratedKeyNotKeyClass {
        private String key;

        @DynamoDBAutoGeneratedKey
        public String getKey() {
            return this.key;
        }
    }

    /**
     * Key as {@code DynamoDBAutoGeneratedKey} and {@code DynamoDBVersionAttribute}.
     */
    @Test
    public void testAutoGeneratedKeyAndVersionAttribute() {
        final Map<String,DynamoDBAnnotationRegistry.AnnotationMap> maps = mapsOf(AutoGeneratedKeyAndVersionAttributeClass.class);
        assertTrue(maps.get("getKey").isHashKey());
        assertTrue(maps.get("getKey").isAutoGeneratedKey());
        assertFalse(maps.get("getKey").isVersion());
    }

    /**
     * Key as {@code DynamoDBAutoGeneratedKey} and {@code DynamoDBVersionAttribute}.
     */
    @DynamoDBTable(tableName="aws-java-sdk-util")
    public static class AutoGeneratedKeyAndVersionAttributeClass {
        private String key;

        @DynamoDBHashKey
        @DynamoDBAutoGeneratedKey
        @DynamoDBVersionAttribute
        public String getKey() {
            return this.key;
        }
    }

    /**
     * Key as {@code DynamoDBAutoGeneratedKey} and {@code DynamoDBAutoGeneratedTimestamp}.
     */
    @Test(expected=DynamoDBMappingException.class)
    public void testAutoGeneratedKeyAndAutoGeneratedTimestamp() {
        mapsOf(AutoGeneratedKeyAndAutoGeneratedTimestampClass.class);
    }

    /**
     * Key as {@code DynamoDBAutoGeneratedKey} and {@code DynamoDBAutoGeneratedTimestamp}.
     */
    @DynamoDBTable(tableName="aws-java-sdk-util")
    public static class AutoGeneratedKeyAndAutoGeneratedTimestampClass {
        private String key;

        @DynamoDBHashKey
        @DynamoDBAutoGeneratedKey
        @DynamoDBAutoGeneratedTimestamp
        public String getKey() {
            return this.key;
        }
    }

    /**
     * Version attribute as {@code DynamoDBAutoGeneratedTimestamp}.
     */
    @Test(expected=DynamoDBMappingException.class)
    public void testVersionAttributeAndAutoGeneratedTimestamp() {
        mapsOf(VersionAttributeAndAutoGeneratedTimestampClass.class);
    }

    /**
     * Version attribute as {@code DynamoDBAutoGeneratedTimestamp}.
     */
    @DynamoDBTable(tableName="aws-java-sdk-util")
    public static class VersionAttributeAndAutoGeneratedTimestampClass {
        private String key;
        private String versionAttribute;

        @DynamoDBHashKey
        public String getKey() {
            return this.key;
        }

        @DynamoDBVersionAttribute
        @DynamoDBAutoGeneratedTimestamp
        public String getVersionAttribute() {
            return this.versionAttribute;
        }
    }

    /**
     * Key as {@code DynamoDBAutoGeneratedTimestamp}.
     */
    @Test(expected=DynamoDBMappingException.class)
    public void testHashKeyAndAutoGeneratedTimestamp() {
        mapsOf(HashKeyAndAutoGeneratedTimestampClass.class);
    }

    /**
     * Key as {@code DynamoDBAutoGeneratedTimestamp}.
     */
    @DynamoDBTable(tableName="aws-java-sdk-util")
    public static class HashKeyAndAutoGeneratedTimestampClass {
        private String key;

        @DynamoDBHashKey
        @DynamoDBAutoGeneratedTimestamp
        public String getKey() {
            return this.key;
        }
    }

    /**
     * Range key as {@code DynamoDBAutoGeneratedTimestamp}.
     */
    @Test(expected=DynamoDBMappingException.class)
    public void testRangeKeyAndAutoGeneratedTimestamp() {
        mapsOf(RangeKeyAndAutoGeneratedTimestampClass.class);
    }

    /**
     * Range key as {@code DynamoDBAutoGeneratedTimestamp}.
     */
    @DynamoDBTable(tableName="aws-java-sdk-util")
    public static class RangeKeyAndAutoGeneratedTimestampClass {
        private String key;
        private String rangeKey;

        @DynamoDBHashKey
        public String getKey() {
            return this.key;
        }

        @DynamoDBRangeKey
        @DynamoDBAutoGeneratedTimestamp
        public String getRangeKey() {
            return this.rangeKey;
        }
    }

}
