# Contributing to Terraform - Google Provider

For a set of general guidelines, see the [CONTRIBUTING.md](https://github.com/hashicorp/terraform/blob/master/.github/CONTRIBUTING.md) page in the main Terraform repository.

The following are certain Google Provider-specific things to be aware of when contributing.

## Go

See the root [README](../README.md) for information on which version of Go you need to use the provider. Generally, this will
be the latest stable release of Go.

We aim to make the Google Provider a good steward of Go practices. See https://github.com/golang/go/wiki/CodeReviewComments for common Go mistakes that you should attempt to avoid.

## Autogenerated Resources

We maintain 2 different versions of the Google Terraform provider; the [`google` provider](https://github.com/hashicorp/terraform-provider-google) and the [`google-beta` provider](https://github.com/hashicorp/terraform-provider-google-beta). The `google` provider supports GA ([general availability](https://cloud.google.com/terms/launch-stages)) features, and `google-beta` supports Beta features.

We are using code generation tool called [Magic Modules](https://github.com/googleCloudPlatform/magic-modules/) that uses a shared code base to generate both providers. Some Terraform resources are fully generated, whereas some resources are hand written and located in [the third_party/terraform/ folder in magic modules](https://github.com/GoogleCloudPlatform/magic-modules/tree/master/third_party/terraform/resources). Generated resources will have a prominent header at the top of the file identifying them. Hand written resources have a .go or .go.erb extension but will eventually be migrated into the code generation tool with the goal of having all resources fully generated.

For more details on Magic Modules please visit [the readme](https://github.com/GoogleCloudPlatform/magic-modules). For feature requests or bugs regarding those resources, please continue to file issues in the [terraform-provider-google issue tracker](https://github.com/hashicorp/terraform-provider-google/issues). PRs changing those resources will not be accepted.

## Beta vs GA providers

Fields that are only available in beta versions of the Google Cloud Platform API will need to be added only to the `google-beta` provider and excluded from the `google` provider. When adding beta features or resources you will need to use templating to exclude them from generating into the ga version. Look for `*.erb` files in [resources](https://github.com/GoogleCloudPlatform/magic-modules/tree/master/third_party/terraform/resources) for examples.

## Tests

### Running Tests

Configuring tests is similar to configuring the provider; see the [Provider Configuration Reference](https://www.terraform.io/docs/providers/google/provider_reference.html#configuration-reference) for more details. The following environment variables must be set in order to run tests:

```
GOOGLE_PROJECT
GOOGLE_CREDENTIALS|GOOGLE_CLOUD_KEYFILE_JSON|GCLOUD_KEYFILE_JSON|GOOGLE_USE_DEFAULT_CREDENTIALS
GOOGLE_REGION
GOOGLE_ZONE
```

To ensure that your tests are performed in a region and zone with wide support for GCP feature, `GOOGLE_REGION` should be set to `us-central1` and `GOOGLE_ZONE` to `us-central1-a`.

For certain tests, primarily those involving project creation, the following variables may also need to be set. Most tests do
not require their being set:

```
GOOGLE_ORG
GOOGLE_BILLING_ACCOUNT
```

To run a specific test, use `TESTARGS`, such as in:

```
make testacc TEST=./google TESTARGS='-run=TestAccContainerNodePool_basic'
```

The `TESTARGS` variable is regexp-like, so multiple tests can be run in parallel by specifying a common substring of those tests (for example, `TestAccContainerNodePool` to run all node pool tests).

To run all tests, you can omit the `TESTARGS` argument - but please keep in mind that that is quite a few tests and will take quite a long time and create some fairly expensive resources.  It usually is not advisable to run all tests.

### Writing Tests

Tests should confirm that a resource can be created, and that the resulting Terraform state has the correct values, as well as the created GCP resource.

Tests should confirm that the resource works in a variety of scenarios, and not just that it can be created in a basic fashion.

Resources that support update should have tests for update.

Resources that are importable should have a test that confirms that every field is importable. This should be part of an existing test (in the regular resource_test.go file) as an extra TestStep with the following format:
```
resource.TestStep{
	ResourceName:      "google_compute_backend_service.foobar",
	ImportState:       true,
	ImportStateVerify: true,
},
```

## Instructing terraform to use a local copy of the provider

There are three different consumption strategies you can utilize

* Using a local version of terraform

  1. If you are building Terraform from source with a Google provider built from source, Terraform will automatically use the
  local `terraform-provider-google` and `terraform-provider-google-beta` plugins in `$GOPATH/bin`.

* Using a release version of terraform

  2. When you run `terraform init` supply the binary output location using the [-plugin-dir](https://www.terraform.io/docs/commands/init.html#plugin-dir-path) parameter. ie `terraform init -plugin-dir=$GOPATH/bin`
      * note: terraform will not download additional providers remotely. All requested providers should be in the -plugin-dir
      * note: each time you rebuild the binary you will have to rerun `terraform init -plugin-dir=<your-binary-location>` as the hash is invalidated.
  3. Use the [provider discovery directory](https://www.terraform.io/docs/extend/how-terraform-works.html#discovery) at `~/.terraform.d/plugins`. Terraform will attempt to use the binaries here as a provider before trying to pull the provider remotely.
      * note: you can either copy the provider binary to this location or do a symlink to the build output
      * note: you will need to delete this binary or syslink to pull the provider remotely
        ```bash
        # helpful bash to system link to binary path.
        # this allows the binary to stay in sync whenever you run make build
        ln -s $GOPATH/bin/terraform-provider-google ~/.terraform.d/plugins/terraform-provider-google
        ln -s $GOPATH/bin/terraform-provider-google-beta ~/.terraform.d/plugins/terraform-provider-google-beta
        ```
### FAQ

* Why isn't it using my local provider?? I did everything right!
  * If you've already used a release version of a provider in a given directory by running `terraform init`, Terraform will not use the locally built copy; remove the release version from the `./.terraform/` to start using your locally built copy.

# Maintainer-specific information

## Reviewing / Merging Code

When reviewing/merging code, roughly follow the guidelines set in the
[Maintainer's Etiquette](https://github.com/hashicorp/terraform/blob/master/docs/maintainer-etiquette.md)
guide. One caveat is that they're fairly old and apply primarily to HashiCorp employees, but the general guidance about merging / changelogs is still relevant.

## Upstreaming community PRs to Magic Modules

Community contributors can contribute directly to [Magic Modules](https://github.com/googleCloudPlatform/magic-modules/), or
they can contribute directly to this repo. When a community member makes a contribution, we review the change locally and
"upstream" it by copying it to MM.

When contributors update handwritten files, we've got a couple bash fns to make the process simpler. Define the following in
your `.bashrc` or `.bash_profile`.

```bash
function tpgpatch1 {
  pr_username=$(echo $1 | cut -d ':' -f1)
  feature_branch=$(echo $1 | cut -d ':' -f2)
  git remote add $pr_username git@github.com:$pr_username/${PWD##*/}
  git fetch $pr_username
  git checkout $pr_username/$feature_branch
  git format-patch $(git merge-base HEAD master)
}

function tpgpatch2 {
  for patch in $GOPATH/src/github.com/hashicorp/terraform-provider-google*/*.patch; do
    echo "checking ${patch}"
        if git apply --stat $patch | grep "google/"; then
                git am -3 -i $patch -p2 --directory=third_party/terraform/resources/ --include="*.go"
        fi
        if git apply --stat $patch | grep "google-beta/"; then
                git am -3 -i $patch -p2 --directory=third_party/terraform/resources/ --include="*.go"
        fi
        if git apply --stat $patch | grep "markdown"; then
                git am -3 -i $patch --directory=third_party/terraform/ --include="*website/*"
        fi
  done
}
```

With those functions defined:

1. Check out both the provider and MM repo to `master`, committing/stashing any local changes
1. In the MM repo, run `git checkout -b {{branch}}` to create a branch for your upstreaming PR
1. Click the clipboard button next to the `author:branch` indicator on the PR to copy it.
1. Run `tpgpatch1 author:branch` from the provider repo
1. Run `tpgpatch2` from the MM repo
1. Remove the patch files from the provider repo

At this point, you should be checked out to a branch with the changes to handwritten files included in the MM repo. For
generated files and most compiled files, you'll need to perform the upstreaming manually. After getting your local branch
ready:

1. Open a PR in MM
1. Assign a reviewer- generally they'll rubberstamp the change, since it's already been approved
    * You can ignore CLA notices for `third_party`-only changes, it's not subject to it.
1. Once approved, merge downstreams.
    * Instead of the repo's downstream, merge the original PR if it's identical. Close the downstream instead of merging.
    If you do so, make sure to add a changelog block or `changelog: no-release-note` to the original PR.
1. Merge the MM PR using `merge-prs`
