/**
 * @typedef {import('mdast').Image} Image
 * @typedef {import('../types.js').Handle} Handle
 */

import {checkQuote} from '../util/check-quote.js'
import {safe} from '../util/safe.js'
import {track} from '../util/track.js'

image.peek = imagePeek

/**
 * @type {Handle}
 * @param {Image} node
 */
export function image(node, _, context, safeOptions) {
  const quote = checkQuote(context)
  const suffix = quote === '"' ? 'Quote' : 'Apostrophe'
  const exit = context.enter('image')
  let subexit = context.enter('label')
  const tracker = track(safeOptions)
  let value = tracker.move('![')
  value += tracker.move(
    safe(context, node.alt, {before: value, after: ']', ...tracker.current()})
  )
  value += tracker.move('](')

  subexit()

  if (
    // If there’s no url but there is a title…
    (!node.url && node.title) ||
    // If there are control characters or whitespace.
    /[\0- \u007F]/.test(node.url)
  ) {
    subexit = context.enter('destinationLiteral')
    value += tracker.move('<')
    value += tracker.move(
      safe(context, node.url, {before: value, after: '>', ...tracker.current()})
    )
    value += tracker.move('>')
  } else {
    // No whitespace, raw is prettier.
    subexit = context.enter('destinationRaw')
    value += tracker.move(
      safe(context, node.url, {
        before: value,
        after: node.title ? ' ' : ')',
        ...tracker.current()
      })
    )
  }

  subexit()

  if (node.title) {
    subexit = context.enter('title' + suffix)
    value += tracker.move(' ' + quote)
    value += tracker.move(
      safe(context, node.title, {
        before: value,
        after: quote,
        ...tracker.current()
      })
    )
    value += tracker.move(quote)
    subexit()
  }

  value += tracker.move(')')
  exit()

  return value
}

/**
 * @type {Handle}
 */
function imagePeek() {
  return '!'
}
