/*
 *	Translation between internal representation and SUSE ifcfg files
 *
 *	Copyright (C) 2009, 2010  Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <unistd.h>
#include <assert.h>
#include <net/if_arp.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <linux/sockios.h>
#include <linux/ethtool.h>

#include <sutils.h>
#include <futils.h>
#include <sysconfig.h>
#include <logging.h>
#include <ntypes.h>
#include <nutils.h>
#include <handle.h>
#include <vlan.h>
#include <bridge.h>
#include <bonding.h>
#include <sysfs.h>
#include <sysctl.h>
#include <cmd_pipe.h>
#include <udev_utils.h>
#include <nlutils.h>
#include "tutils.h"
#include "xml.h"


/* config handle */
extern nc_handle_t *	nc__suse_config_init(const char *, nc_service_t *);
static int		__nc_suse_config_interfaces_refresh(nc_handle_t *);
static int		__nc_suse_config_interfaces_delete(nc_handle_t *, nc_interface_t *);
static int		__nc_suse_config_interfaces_configure(nc_handle_t *, nc_interface_t *);
static int		__nc_suse_config_interface_start(nc_handle_t *, nc_interface_t *);
static int		__nc_suse_config_interface_stop(nc_handle_t *, nc_interface_t *);
static void		__nc_suse_config_close(nc_handle_t *);
static int		nc_suse_config_wicked_refresh(nc_handle_t *);

typedef struct {
	nc_handle_t		base;
	char *			root;
	nc_service_t *		service;
} nc_config_handle_t;

static struct nc_ops	nc__legacy_config_ops = {
	.interfaces_refresh	= __nc_suse_config_interfaces_refresh,
	.interfaces_configure	= __nc_suse_config_interfaces_configure,
	.interfaces_delete   	= __nc_suse_config_interfaces_delete,
	.interface_start	= __nc_suse_config_interface_start,
	.interface_stop		= __nc_suse_config_interface_stop,
	.close			= __nc_suse_config_close,
};
static struct nc_ops	nc__wicked_config_ops = {
	.interfaces_refresh	= nc_suse_config_wicked_refresh,
	.interfaces_configure	= __nc_suse_config_interfaces_configure,
	.interfaces_delete   	= __nc_suse_config_interfaces_delete,
	.interface_start	= __nc_suse_config_interface_start,
	.interface_stop		= __nc_suse_config_interface_stop,
	.close			= __nc_suse_config_close,
};

/* system handle */
extern nc_handle_t *	nc__suse_system_init(const char *, nc_service_t *);
static int		__nc_suse_system_interfaces_refresh(nc_handle_t *);
static int		__nc_suse_system_interface_status(nc_handle_t *nh, nc_interface_t *ifp);
static void		__nc_suse_system_close(nc_handle_t *);
static int		nc_suse_system_wicked_refresh(nc_handle_t *);
static int		nc_suse_system_wicked_status(nc_handle_t *, nc_interface_t *ifp);

typedef struct {
	nc_handle_t		base;
	char *			root;
	nc_service_t *		service;
} nc_system_handle_t;

static struct nc_ops	nc__legacy_system_ops = {
	.interfaces_refresh	= __nc_suse_system_interfaces_refresh,
	.interface_status	= __nc_suse_system_interface_status,
	.close			= __nc_suse_system_close,
};
static struct nc_ops	nc__wicked_system_ops = {
	.interfaces_refresh	= nc_suse_system_wicked_refresh,
	.interface_status	= nc_suse_system_wicked_status,
	.close			= __nc_suse_system_close,
};


static int		__suse_ifcfg_valid_name(const char *name);
static int		__suse_ifcfg_scan_files(const char *dirname, const char *prefix,
			                        nc_string_array_t *res);

static int		__process_indexed_variables(nc_interface_t *, nc_sysconfig_t *,
			        const char *,
			        int (*)(nc_interface_t *, nc_sysconfig_t *, const char *));

static int		try_vlan(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_bridge(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_bonding(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_wireless(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_ppp(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_tunnel(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_infiniband(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_ovs(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_team(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_macvlan(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
static int		try_macvtap(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);

static int		nc_discover_iface_type(const char *name, unsigned int *ifp_type,
								 unsigned int *arp_type);
static int		__nc_discover_vlan(nc_interface_t *ifp);

static int		__suse_config_read_ifcfg(nc_handle_t *nh, nc_interface_t *ifp,
						nc_sysconfig_t *sc);
static int		__suse_config_read_routes(nc_route_array_t *routes,
						  const char       *file,
						  const char       *ifname);
static int		__suse_system_read_state(nc_interface_t *ifp);


#define _SYSCONFIG_NETWORK_DIR		"/etc/sysconfig/network"
#define _CONFIG_GLOBAL  		"config"
#define _CONFIG_DHCP    		"dhcp"
#define _ROUTES_GLOBAL  		"routes"
#define _IFCFG_PREFIX   		"ifcfg-"
#define _IFROUTE_PREFIX 		"ifroute-"
#define _DOT_BACKUP_SUFFIX		""

static nc_intmap_t	__ifname_types[] = {
	{ "lo",			NC_IFTYPE_LOOPBACK	},
	{ "br",			NC_IFTYPE_BRIDGE	},
	{ "bond",		NC_IFTYPE_BOND		},
	{ "vlan",		NC_IFTYPE_VLAN		},
	{ "dummy",		NC_IFTYPE_DUMMY		},
	{ "tun",		NC_IFTYPE_TUN		},
	{ "tap",		NC_IFTYPE_TAP		},
	{ "sit",		NC_IFTYPE_SIT		},
	{ "gre",		NC_IFTYPE_GRE		},
	{ "ipip",		NC_IFTYPE_TUNNEL	},
	{ "ipip6",		NC_IFTYPE_TUNNEL6	},
	{ "ip6tunl",		NC_IFTYPE_TUNNEL6	},
	{ "slip",		NC_IFTYPE_SLIP		},
	{ "ppp",		NC_IFTYPE_PPP		},
	{ "isdn",		NC_IFTYPE_ISDN		},
	{ "ib",			NC_IFTYPE_INFINIBAND	},
	{ "tr",			NC_IFTYPE_TOKENRING	},
	{ "eth",		NC_IFTYPE_ETHERNET	},
	{ "seth",		NC_IFTYPE_ETHERNET	},
	{ "qeth",		NC_IFTYPE_ETHERNET	},
	{ "hsi",		NC_IFTYPE_ETHERNET	},
	{ "lcs",		NC_IFTYPE_ETHERNET	},
	{ "ctc",		NC_IFTYPE_SLIP		},
	{ "iucv",		NC_IFTYPE_SLIP		},
	{ "ovs",		NC_IFTYPE_OVS		},
	{ "team",		NC_IFTYPE_TEAM		},
	{ "macvlan",		NC_IFTYPE_MACVLAN	},
	{ "macvtap",		NC_IFTYPE_MACVLAN	},

	{  NULL,		NC_IFTYPE_UNKNOWN	}
};
static nc_intmap_t	__startmode_name_map[] = {
	{ "manual",		NC_STARTMODE_MANUAL	},
	{ "auto",		NC_STARTMODE_AUTO	},
	{ "on",			NC_STARTMODE_AUTO	},
	{ "boot",		NC_STARTMODE_AUTO	},
	{ "onboot",		NC_STARTMODE_AUTO	},
	{ "hotplug",		NC_STARTMODE_HOTPLUG	},
	{ "ifplugd",		NC_STARTMODE_IFPLUGD	},
	{ "nfsroot",		NC_STARTMODE_NFSROOT	},
	{ "off",		NC_STARTMODE_OFF	},

	{ NULL,			NC_STARTMODE_MANUAL	},
};
static struct __nc_arptype_iftype_map {
	unsigned int		type;
	unsigned int		arp_type;
}			__nc_arptype_iftype_map[] = {
	{ NC_IFTYPE_LOOPBACK,	ARPHRD_LOOPBACK		},
	{ NC_IFTYPE_ETHERNET,	ARPHRD_ETHER		},
	{ NC_IFTYPE_BRIDGE,	ARPHRD_ETHER		},
	{ NC_IFTYPE_BOND,	ARPHRD_ETHER		},
	{ NC_IFTYPE_VLAN,	ARPHRD_ETHER		},
	{ NC_IFTYPE_TAP,	ARPHRD_ETHER		},
	{ NC_IFTYPE_DUMMY,	ARPHRD_ETHER		},
	{ NC_IFTYPE_WIRELESS,	ARPHRD_ETHER		},
	{ NC_IFTYPE_INFINIBAND,	ARPHRD_INFINIBAND	},
	{ NC_IFTYPE_PPP,	ARPHRD_PPP		},
	{ NC_IFTYPE_SLIP,	ARPHRD_SLIP		},
	{ NC_IFTYPE_SLIP,	ARPHRD_CSLIP		},
	{ NC_IFTYPE_SIT,	ARPHRD_SIT		},
	{ NC_IFTYPE_GRE,	ARPHRD_IPGRE		},
	{ NC_IFTYPE_TUNNEL,	ARPHRD_TUNNEL		},
	{ NC_IFTYPE_TUNNEL6,	ARPHRD_TUNNEL6		},

	/* keep unknown/void type (default) at the end */
	{ NC_IFTYPE_UNKNOWN,	ARPHRD_VOID		},
};

unsigned int
nc_arphrd_type_to_iftype(unsigned int arp_type)
{
	struct __nc_arptype_iftype_map *map;
	size_t n, end;

	end = sizeof(__nc_arptype_iftype_map)/sizeof(__nc_arptype_iftype_map[0]);
	for(map = __nc_arptype_iftype_map, n = 0; n < end; ) {
		if(map->arp_type == arp_type)
			break;
		if(++n < end) map++;
	}
	return map->type;
}

int
nc_iftype_to_arphrd_type(unsigned int iftype)
{
	struct __nc_arptype_iftype_map *map;
	size_t n, end;

	end = sizeof(__nc_arptype_iftype_map)/sizeof(__nc_arptype_iftype_map[0]);
	for(map = __nc_arptype_iftype_map, n = 0; n < end; ) {
		if(map->type == iftype)
			break;
		if(++n < end) map++;
	}
	return map->arp_type;
}

static const char *
__suse_ifcfg_typename_get(unsigned int iftype)
{
	return nc_format_int_mapped(iftype, __ifname_types);
}

static int
__suse_ifcfg_startmode_set(nc_interface_t *ifp, const char *startmode)
{
	if (startmode && *startmode && nc_parse_int_mapped(startmode,
			__startmode_name_map, &ifp->startmode) < 0) {
		ifp->startmode = __startmode_name_map[0].value;
	}
	return 0;
}

static const char *
__suse_ifcfg_startmode_get(unsigned int startmode)
{
	const char *ptr;

	ptr = nc_format_int_mapped(startmode, __startmode_name_map);
	return ptr ? ptr : __startmode_name_map[0].name;
}

static int
__suse_ifcfg_bootproto_set(nc_interface_t *ifp, const char *bootproto)
{
	char *bp, *s, *p;
	int ret = 0;

	bp = NULL;
	if(bootproto && nc_string_dup(&bp, bootproto) != 0)
		return -1;

	if(bp == NULL || *bp == '\0' || !strcasecmp(bp, "static")) {
		goto cleanup;
	}
	if(!strcasecmp(bp, "none")) {
		ifp->ipv4.addrconf = 0; /*NC_ADDRCONF_DISABLED;*/
		ifp->ipv6.addrconf = 0; /*NC_ADDRCONF_DISABLED;*/
		goto cleanup;
	}
	if(!strcasecmp(bp, "ibft")) {
		ifp->ipv4.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_IBFT);
		ifp->ipv6.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_IBFT);
		goto cleanup;
	}

	/* parse "+" separated flags */
	p = NULL;
	for (s = strtok_r(bp, "+", &p); s; s = strtok_r(NULL, "+", &p)) {
		if(!strcasecmp(s, "dhcp")) {
			ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
			ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
		} else
		if(!strcasecmp(s, "dhcp4")) {
			ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
		} else
		if(!strcasecmp(s, "dhcp6")) {
			ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
		} else
		if(!strcasecmp(s, "autoip")) {
			ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
		} else {
			nc_warn("%s: unhandled BOOTPROTO \"%s\"", ifp->name, s);
			/* ret = -1; */
			goto cleanup;
		}
	}

cleanup:
	free(bp);
	return ret;
}

static const char *
__suse_ifcfg_bootproto_get(const nc_interface_t *ifp)
{
	if(ifp->ipv4.addrconf == 0 && ifp->ipv4.addrconf == 0) {
		return "none";
	}
	if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_IBFT) ||
	   NC_ADDRCONF_TEST(ifp->ipv6.addrconf, NC_ADDRCONF_IBFT)) {
		return "ibft";
	}
	if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_DHCP) &&
	   NC_ADDRCONF_TEST(ifp->ipv6.addrconf, NC_ADDRCONF_DHCP)) {
		if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_AUTOCONF))
			return "dhcp+autoip";
		else
			return "dhcp";
	}
	if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_DHCP)) {
		if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_AUTOCONF))
			return "dhcp4+autoip";
		else
			return "dhcp4";
	}
	if(NC_ADDRCONF_TEST(ifp->ipv6.addrconf, NC_ADDRCONF_DHCP)) {
		if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_AUTOCONF))
			return "dhcp6+autoip";
		else
			return "dhcp6";
	}
	if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_AUTOCONF)) {
		return "autoip";
	}
	/*
	if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_STATIC) ||
	   NC_ADDRCONF_TEST(ifp->ipv6.addrconf, NC_ADDRCONF_STATIC) ||
	   NC_ADDRCONF_TEST(ifp->ipv6.addrconf, NC_ADDRCONF_AUTOCONF)) {
		return "static";
	}
	*/
	return "static";	/* is a default */
}

static int
maybe_infiniband(const char *ifname, nc_vlan_t **vlan)
{
	unsigned int tag;
	size_t len, n;

	len = strlen(ifname);
	if(len <= 2 || strncmp(ifname, "ib", 2))
		return 1; /* too short or no ib prefix */

	for(n=2; n < len && ifname[n]; ++n) {
		if(!isdigit((unsigned char)ifname[n]))
			break;
	}

	if(ifname[n] == '\0') {
		if(vlan)
			*vlan = NULL;
		return 0;
	}

	if(ifname[n] != '.')
		return 1; /* ibX prefix, but no .YYYY suffix */

	if(nc_parse_uint(ifname + n + 1, &tag, 16) < 0)
		return -1;

	if(vlan) {
		nc_vlan_t *v = nc_vlan_new(NULL, tag);
		if(!v || nc_string_set(&v->interface_name, ifname, n) < 0) {
			free(v);
			return -1;
		}
		*vlan = v;
	}
	return 0;
}

static inline struct nc_ops *
nc__suse_system_service_ops(nc_service_t *service)
{
	if (!service || nc_string_empty(service->name))
		return NULL;

	if (nc_string_eq(service->name, "wicked.service"))
		return &nc__wicked_system_ops;
	if (nc_string_eq(service->name, "network.service"))
		return &nc__legacy_system_ops;

	return NULL;
}

nc_handle_t *
nc__suse_system_init(const char *root_dir, nc_service_t *service)
{
	nc_system_handle_t * nh;
	struct nc_ops *ops;

	ops = nc__suse_system_service_ops(service);
	if (!ops || !service) {
		nc_error("Unsupported network service %s", service ? service->name : "");
		return NULL;
	}

	nh = calloc(1, sizeof(*nh));
	if(nh) {
		if(root_dir && *root_dir) {
			if(nc_string_dup(&nh->root, root_dir) < 0) {
				free(nh);
				return NULL;
			}
		}
		nh->service = nc_service_ref(service);
		nh->base.op = ops;
		return &nh->base;
	}
	return NULL;
}

static inline nc_system_handle_t *
__to_system_handle(const nc_handle_t *nh)
{
	assert(nh->op == &nc__legacy_system_ops || nh->op == &nc__wicked_system_ops);
	return (nc_system_handle_t *)nh;
}

void
__nc_suse_system_close(nc_handle_t *nh)
{
	nc_system_handle_t *sh = __to_system_handle(nh);
	nc_string_free(&sh->root);
	nc_service_free(sh->service);
}

static int
__nc_suse_system_interface_add_addrs(struct nl_msg *msg, void *ptr)
{
	nc_handle_t *nh = (nc_handle_t *)ptr;
	nc_address_t *ap;
	int ifindex = -1;

	if(nc_netlink_parse_addr_msg(msg, &ap, &ifindex) == 0 && ap) {
		nc_interface_t *ifp;

		ifp = nc_interface_by_ifindex(nh, ifindex);
		if(!ifp || nc_address_array_append(&ifp->addrs, ap) < 0)
			nc_address_free(ap);

		ap = NULL;
		ifindex = -1;
	}
	return 0;  /* NL_OK: Just continue with next one */
}

static int
__nc_suse_system_interface_add_routes(struct nl_msg *msg, void *ptr)
{
	nc_handle_t *nh = (nc_handle_t *)ptr;
	nc_route_t *rp = NULL;
	int ifindex = -1;

	(void)__to_system_handle(ptr); // guard

	if(nc_netlink_parse_route_msg(msg, &rp, &ifindex) == 0 && rp) {
		nc_interface_t *ifp;

		ifp = nc_interface_by_ifindex(nh, ifindex);
		if(!ifp || nc_route_array_append(&ifp->routes, rp) < 0)
			nc_route_free(rp);

		rp = NULL;
		ifindex = -1;
	}
	return 0; /* NL_OK: Just continue with next one */
}

int
__nc_suse_system_interfaces_refresh(nc_handle_t *nh)
{
	nc_system_handle_t * sh = __to_system_handle(nh);
	nc_string_array_t files = NC_STRING_ARRAY_INIT;
	nc_stringbuf_t    base  = NC_STRINGBUF_INIT;
	nc_interface_t *    ifp = NULL;
	char path[PATH_MAX + 1] = {'\0'};
	unsigned int i;
	int count = 0;
	nc_netlink_t *nl;

	nc_stringbuf_printf(&base, "%s%s", (sh->root ? sh->root : ""), _SYSCONFIG_NETWORK_DIR);

	/* Wipe out all interface information */
	nc_interfaces_clear(nh);

	snprintf(path, sizeof(path), "%s", NC_SYSFS_CLASS_NET_PATH);
	if( nc_scandir(path, NULL, &files) < 0) {
		goto failure;
	}

	for (i = 0; i < files.count; ++i) {
		const char * ifname = files.data[i];
		unsigned int ifindex = 0;
		nc_stringbuf_t  cfg = NC_STRINGBUF_INIT;
		nc_sysconfig_t *nsc = NULL;
		nc_var_t *var;

		if(!nc_sysfs_netif_exists(ifname, "ifindex") ||
		    nc_sysfs_netif_get_uint(ifname, "ifindex", &ifindex) < 0)
			continue;

		ifp = nc_interface_new(ifname, ifindex);
		if(!ifp)
			continue;

		if(__suse_system_read_state(ifp) != 0) {
			nc_interface_free(ifp);
			continue;
		}

		if(base.string) {
			nc_stringbuf_printf(&cfg, "%s/%s%s", base.string, _IFCFG_PREFIX, ifname);
			if(cfg.string)
				nsc = nc_sysconfig_read(cfg.string);
		}
		if(nsc) {
			if((var = nc_sysconfig_get(nsc, "STARTMODE")) != NULL)
				__suse_ifcfg_startmode_set(ifp, var->value);
			nc_sysconfig_free(nsc);
		}
		nc_stringbuf_destroy(&cfg);

#if defined(NC_OPT_TRACE)
		nc_trace("ifsys: name=%s, type=[%u]%s, startmode=[%u]%s, bootproto=[%u,%u]%s",
			ifp->name, ifp->type, nc_format_int_mapped(ifp->type, __ifname_types),
			ifp->startmode, __suse_ifcfg_startmode_get(ifp->startmode),
			ifp->ipv4.addrconf, ifp->ipv6.addrconf, __suse_ifcfg_bootproto_get(ifp));
#endif
		if(nc_interfaces_add(nh, ifp) != 0) {
			nc_interface_free(ifp);
			continue;
		}

		count++;
	}

	nl = NULL;
	if (nl || (nl = nc_netlink_open())) {
		if(nc_netlink_get_addrs(nl, __nc_suse_system_interface_add_addrs, nh) < 0)
			nc_netlink_close(&nl);
	}
	if (nl || (nl = nc_netlink_open())) {
		if (nc_netlink_get_routes(nl, __nc_suse_system_interface_add_routes, nh) < 0)
			nc_netlink_close(&nl);
	}
	nc_netlink_close(&nl);

	nc_stringbuf_destroy(&base);
	nc_string_array_destroy(&files);
	return count;

failure:
	nc_stringbuf_destroy(&base);
	nc_string_array_destroy(&files);
	return -1;
}

static inline struct nc_ops *
nc_suse_config_service_ops(nc_service_t *service)
{
	if (!service || nc_string_empty(service->name))
		return NULL;

	if (nc_string_eq(service->name, "wicked.service"))
		return &nc__wicked_config_ops;
	if (nc_string_eq(service->name, "network.service"))
		return &nc__legacy_config_ops;

	return NULL;
}

nc_handle_t *
nc__suse_config_init(const char *root_dir, nc_service_t *service)
{
	nc_config_handle_t *nh;
	struct nc_ops *ops;

	ops = nc_suse_config_service_ops(service);
	if (!ops || !service) {
		nc_error("Unsupported network service %s", service ? service->name : "");
		return NULL;
	}

	nh = calloc(1, sizeof(*nh));
	if(nh) {
		if(root_dir && *root_dir) {
			if(nc_string_dup(&nh->root, root_dir) < 0) {
				free(nh);
				return NULL;
			}
		}

		nh->service = nc_service_ref(service);
		nh->base.op = ops;
		return &nh->base;
	}
	return NULL;
}

static inline nc_config_handle_t *
__to_config_handle(const nc_handle_t *nh)
{
	assert(nh->op == &nc__legacy_config_ops || nh->op == &nc__wicked_config_ops);
	return (nc_config_handle_t *)nh;
}

static void
__nc_suse_config_close(nc_handle_t *nh)
{
	nc_config_handle_t *ch = __to_config_handle(nh);
	nc_string_free(&ch->root);
	nc_service_free(ch->service);
}

static int
__nc_suse_config_prepend_root(const nc_config_handle_t *ch, const char *path,
                              nc_stringbuf_t *buff)
{
	if(!path || !buff)
		return -1;

	nc_stringbuf_clear(buff);
	if(ch && ch->root) {
		if(nc_stringbuf_puts(buff, ch->root) < 0)
			return -1;
	}
	if(nc_stringbuf_puts(buff, path) < 0) {
		nc_stringbuf_clear(buff);
		return -1;
	}
	return 0;
}


int
__nc_suse_config_interfaces_refresh(nc_handle_t *nh)
{
	nc_config_handle_t   *ch = __to_config_handle(nh);
	nc_string_array_t  files = NC_STRING_ARRAY_INIT;
	nc_stringbuf_t base_path = NC_STRINGBUF_INIT;
	nc_stringbuf_t file_name = NC_STRINGBUF_INIT;
	nc_var_array_t bind_errs = NC_VAR_ARRAY_INIT;
	unsigned int i;
	int ret, count = 0;

	/* Wipe out all interface and route information */
	nc_interfaces_clear(nh);
	nc_route_array_destroy(&nh->routes);

	if(__nc_suse_config_prepend_root(ch, _SYSCONFIG_NETWORK_DIR, &base_path) < 0)
		goto failure;

	if( __suse_ifcfg_scan_files(base_path.string, _IFCFG_PREFIX, &files) < 0) {
		goto failure;
	}

	if((nc_stringbuf_puts(&file_name, base_path.string) < 0) ||
	   (nc_stringbuf_putc(&file_name, '/') < 0)              ||
	   (nc_stringbuf_puts(&file_name, _ROUTES_GLOBAL) < 0))
		goto failure;

	if(__suse_config_read_routes(&nh->routes, file_name.string, NULL) < 0) {
		goto failure;
	}

	for (i = 0; i < files.count; ++i) {
		const char *conf = files.data[i];
		const char *name = files.data[i] + (sizeof(_IFCFG_PREFIX)-1);
		nc_interface_t *ifp;
		nc_sysconfig_t *sc;
		unsigned int j, k;

		ifp = nc_interface_new(name, 0);
		if(!ifp)
			continue;

		nc_stringbuf_clear(&file_name);
		if((nc_stringbuf_puts(&file_name, base_path.string) < 0) ||
		   (nc_stringbuf_putc(&file_name, '/') < 0)              ||
		   (nc_stringbuf_puts(&file_name, conf) < 0))
			continue;

		sc = nc_sysconfig_read(file_name.string);
		if(!sc) {
			/* bug or file removed between readdir & open? -> failure? */
			nc_error("Unable to parse %s", file_name.string);
			nc_interface_free(ifp);
			continue;
		}
		ret = __suse_config_read_ifcfg(nh, ifp, sc);
		if(ret < 0) {
			nc_interface_free(ifp);
			nc_sysconfig_free(sc);
			continue;
		} else
		if(ret > 0) {
			if(nc_error_once_check(nh, "/ifcfg/read/%s", name)) {
				nc_error("Interface config %s contains errors",
					file_name.string);
			}
		} else {
			nc_error_once_clear(nh, "/ifcfg/read/%s", name);
		}
		nc_sysconfig_free(sc);

		nc_stringbuf_clear(&file_name);
		if((nc_stringbuf_puts(&file_name, base_path.string) < 0) ||
		   (nc_stringbuf_putc(&file_name, '/')  < 0)             ||
		   (nc_stringbuf_puts(&file_name, _IFROUTE_PREFIX) < 0)  ||
		   (nc_stringbuf_puts(&file_name, name) < 0))
			continue;

		if(__suse_config_read_routes(&ifp->routes, file_name.string, name) < 0) {
			nc_interface_free(ifp);
			continue;
		}

		/* Hmm... move to some function */
		for(j = 0; j < nh->routes.count; ++j) {
			nc_route_t *rp = nh->routes.data[j];
			int same = 0;

			for(k = 0; !same && k < ifp->routes.count; ++k) {
				nc_route_t *ir = ifp->routes.data[k];
				/* TODO: assume some special one (unreachable) */
				if(rp->opts)
					continue;
				same = nc_address_equal(&rp->destination,
				                        &ir->destination);
			}
			if(same) /* same destination, use ifroute */
				continue;

			if(nc_string_eq(name, rp->nh.device)) {
				nc_route_array_append(&ifp->routes, nc_route_ref(rp));
				nc_route_array_remove_index(&nh->routes, j--);
				continue;
			}

			for(k = 0; k < ifp->addrs.count; ++k) {
				nc_address_t *ap = ifp->addrs.data[k];

				// FIXME: this check works for "common" IPv4 only;
				//        gw may have own interface route, ...
				if(nc_address_can_reach(ap, &rp->nh.gateway)) {
					nc_route_array_append(&ifp->routes, nc_route_ref(rp));
					nc_route_array_remove_index(&nh->routes, j--);
					break;
				}
			}
		}
#if defined(NC_OPT_TRACE)
		nc_trace("ifcfg: name=%s, type=[%u]%s, startmode=[%u]%s, bootproto=[%u,%u]%s",
			ifp->name, ifp->type, nc_format_int_mapped(ifp->type, __ifname_types),
			ifp->startmode, __suse_ifcfg_startmode_get(ifp->startmode),
			ifp->ipv4.addrconf, ifp->ipv6.addrconf, __suse_ifcfg_bootproto_get(ifp));
#endif
		if(nc_interfaces_add(nh, ifp) != 0) {
			nc_interface_free(ifp);
			continue;
		}

		count++;
	}

	ret = nc_interfaces_create_topology(nh, &bind_errs);
	if(ret < 0) {
		if(nc_error_once_check(nh, "/topology")) {
			nc_error("Cannot create interface configuration topology");
		}
		goto failure;
	} else {
		nc_error_once_clear(nh, "/topology");
	}

	/* OK, print/reset all bind errors */
	for(i = 0; i < nh->ifaces.count; ++i) {
		nc_interface_t *ifp = nh->ifaces.data[i];
		nc_var_t *v;

		if(!ifp || !ifp->name)
			continue;

		v = nc_var_array_get(&bind_errs, ifp->name);
		if(v && v->value) {
			if(nc_error_once_check(nh, "/bind/%s", ifp->name)) {
				nc_error("bind %s: %s", ifp->name, v->value);
			}
		} else {
			nc_error_once_clear(nh, "/bind/%s", ifp->name);
		}
	}

	nc_stringbuf_destroy(&base_path);
	nc_stringbuf_destroy(&file_name);
	nc_string_array_destroy(&files);
	nc_var_array_destroy(&bind_errs);
	return count;

failure:
	nc_stringbuf_destroy(&base_path);
	nc_stringbuf_destroy(&file_name);
	nc_string_array_destroy(&files);
	nc_var_array_destroy(&bind_errs);
	return -1;
}

static int
__suse_ifcfg_valid_name(const char *name)
{
	const char *bl_suffix[] = {
	    "~", ".old", ".bak", ".orig", ".scpmbackup",
	    ".rpmnew", ".rpmsave", ".rpmorig"
	};
	size_t nlen, slen, n;

	nlen = nc_string_len(name);
	if(nlen == 0)
		return 1;

	for(n=0; n<sizeof(bl_suffix)/sizeof(bl_suffix[0]); n++) {
		slen = nc_string_len(bl_suffix[n]);
		if(nlen < slen)
			continue;

		if(nc_string_eq(bl_suffix[n], name + (nlen - slen)))
			return 1;
	}
	return 0;
}

static int
__is_valid_ifname(const char *name)
{
	size_t len = nc_string_len(name ? name : "");
	size_t i;

	for(i = 0; i < len; ++i) {
		if(isalnum((unsigned char)name[i])  ||
		   name[i] == '-' || name[i] == '_' ||
		   name[i] == '.' || name[i] == ':')
			continue;
		return 0;
	}
	return i > 0 ? 1 : 0;
}

static int
__suse_ifcfg_scan_files(const char *dirname, const char *prefix, nc_string_array_t *res)
{
	nc_string_array_t temp;
	unsigned int i, count = res->count;
	size_t pfxlen = nc_string_len(prefix);

	nc_string_array_init(&temp);
	if( !nc_scandir(dirname, prefix, &temp))
		return 0;

	for(i = 0; i < temp.count; ++i) {
		const char *conf = temp.data[i];
		const char *name = temp.data[i] + pfxlen;

		if(__suse_ifcfg_valid_name(name) == 0)
			nc_string_array_append(res, conf);
	}
	nc_string_array_destroy(&temp);

	return res->count - count;
}

static int
__process_indexed_variables(nc_interface_t *ifp, nc_sysconfig_t *sc,
                                const char *basename,
				int (*func)(nc_interface_t *, nc_sysconfig_t *, const char *))
{
	nc_string_array_t names = NC_STRING_ARRAY_INIT;
	unsigned int i, pfxlen;
	int ret = 0;

	if (nc_sysconfig_find_matching(sc, basename, &names) < 0)
		return -1;

	pfxlen = strlen(basename);
	for (i = 0; i < names.count; ++i) {
		ret = func(ifp, sc, names.data[i] + pfxlen);
		if(ret != 0)
			break;
	}
	nc_string_array_destroy(&names);
	return ret;
}

static nc_var_t *
__find_indexed_variable(nc_sysconfig_t *sc, const char *basename, const char *suffix)
{
	char namebuf[NAME_MAX + 1];
	nc_var_t *res;

	snprintf(namebuf, sizeof(namebuf), "%s%s", basename, suffix);
	res = nc_sysconfig_get(sc, namebuf);
	if (res && (res->value == NULL || res->value[0] == '\0'))
		res = NULL;
	return res;
}

static unsigned int
__get_prefix_len(nc_interface_t *ifp, nc_sysconfig_t *sc, const char *suffix)
{
	nc_sockaddr_t netmask;
	nc_var_t *var;

	(void)ifp;
	var = __find_indexed_variable(sc, "PREFIXLEN", suffix);
	if(var && var->value) {
		unsigned int prefix_len = 0;
		if(nc_parse_uint(var->value, &prefix_len, 10) == 0)
			return prefix_len;
	}
	var = __find_indexed_variable(sc, "NETMASK", suffix);
	if(var && var->value) {
		if(nc_address_parse(&netmask, var->value, AF_INET) >= 0) {
			return nc_netmask_bits(&netmask);
		}
	}
	return 0;
}

static nc_address_t *
__get_ipaddr(nc_interface_t *ifp, nc_sysconfig_t *sc, const char *suffix)
{
	nc_var_t *var;
	nc_address_t *ap;
	char *address_string;
	nc_sockaddr_t addr;
	unsigned int prefix_len = 0;
	char *sp;

	var = __find_indexed_variable(sc, "IPADDR", suffix);
	if (!var || !var->value)
		return NULL;
	
	address_string = var->value[0] ? strdup(var->value) : NULL;
	if(!address_string)
		return NULL;
	if ((sp = strchr(address_string, '/')) != NULL) {
		*sp++ = '\0';
		nc_parse_uint(sp, &prefix_len, 10);
	} else {
		prefix_len = __get_prefix_len(ifp, sc, suffix);
	}

	if (nc_address_parse(&addr, address_string, AF_UNSPEC) < 0) {
		nc_error("Unable to parse %s=\"%s\"", var->name, address_string);
		free(address_string);
		return NULL;
	}
	free(address_string);

	if(prefix_len == 0 || prefix_len > nc_address_bits(addr.ss_family))
		return NULL;

	ap = nc_address_new(addr.ss_family, prefix_len, &addr);
	if(ap) {
		nc_address_array_append(&ifp->addrs, ap);
	}
	return ap;
}

static int
try_add_address(nc_interface_t *ifp, nc_sysconfig_t *sc, const char *suffix)
{
	nc_var_t *var;
	nc_address_t *ap;

	if (!(ap = __get_ipaddr(ifp, sc, suffix)))
		return 0; /* do not fail */

	if (ap->family == AF_INET) {
		var = __find_indexed_variable(sc, "BROADCAST", suffix);
		if(var)
			nc_address_parse(&ap->bcast_addr, var->value, AF_INET);
	}
	var = __find_indexed_variable(sc, "REMOTE_IPADDR", suffix);
	if (var)
		nc_address_parse(&ap->peer_addr, var->value, AF_UNSPEC);
	return 0;
}

static int
try_vlan(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	nc_vlan_t *	vlan = NULL;
	unsigned int 	tag;
	const char  *	ptr;
	nc_var_t *	v;

	v = nc_sysconfig_get(sc, "ETHERDEVICE");
	if(!v || !nc_string_len(v->value)) {
		return 1;
	}

	if(ifp->type != NC_IFTYPE_VLAN &&
	   ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/try_vlan/%s/type", ifp->name)) {
			nc_error("Interface config %s: matches vlan but is of another interface type",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_vlan/%s/type", ifp->name);

	/* make sure to reject self reference crap */
	if(nc_string_eq(ifp->name, v->value)) {
		if(nc_error_once_check(nh, "/try_vlan/%s/self", ifp->name)) {
			nc_error("Interface config %s: vlan base refers to itself",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_vlan/%s/self", ifp->name);

	ifp->type = NC_IFTYPE_VLAN;
	vlan = nc_vlan_new(v->value, 0);
	if( !vlan) {
		/* Hmm... */
		nc_error("Unable to allocate vlan structure");
		goto failure;
	}

	if(nc_sysconfig_get_uint(sc, "VLAN_ID", &tag) == 0) {
		/*
		** OK, we have a VLAN_ID variable -- this has
		** priority / overrides any name scheme.
		*/
		vlan->tag = tag;
	} else {
		ptr = strrchr(ifp->name, '.');
		if(ptr) {
			/* OK, default ifnameX.<TAG> scheme */
			ptr++;
		} else {
			/*
			 ** Either vlan<TAG> scheme or custom name.
			 */
			size_t len = nc_string_len(ifp->name);
			while(len > 0 && isdigit((unsigned char)ifp->name[len - 1])) {
				ptr = &ifp->name[--len];
			}
		}
		if(ptr && nc_parse_uint(ptr, &tag, 10) == 0) {
			vlan->tag = tag;
			nc_error_once_clear(nh, "/try_vlan/%s/tag/parse", ifp->name);
		} else {
			if(nc_error_once_check(nh, "/try_vlan/%s/tag/parse", ifp->name)) {
				nc_error("Interface config %s: cannot get vlan tag from interface name",
					 ifp->name);
			}
			goto failure;
		}
	}

	if(ifp->vlan) {
		if(ifp->vlan->tag != vlan->tag) {
			if(nc_error_once_check(nh, "/try_vlan/%s/tag/diff", ifp->name)) {
				nc_warn("Interface config %s: vlan tag %u differs from previously detected %u",
					 ifp->name, vlan->tag, ifp->vlan->tag);
			}
		} else {
			nc_error_once_clear(nh, "/try_vlan/%s/tag/diff", ifp->name);
		}

		if(!nc_string_eq(ifp->vlan->interface_name, vlan->interface_name)) {
			if(nc_error_once_check(nh, "/try_vlan/%s/base/diff", ifp->name)) {
				nc_warn("Interface config %s: vlan base %s differs from previously detected %s",
					 ifp->name, vlan->interface_name, ifp->vlan->interface_name);
			}
		} else {
			nc_error_once_clear(nh, "/try_vlan/%s/base/diff", ifp->name);
		}

		// prefer previous ifp->vlan (runtime detected)
		nc_vlan_free(vlan);
		return 0;
	}

	ifp->vlan = vlan;
	return 0;

failure:
	nc_vlan_free(vlan);
	ifp->invalid = 1;
	return -1;
}

static int
try_bridge(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	nc_bridge_t *bridge = NULL;
	nc_var_t *   v;
	char *value = NULL, *token;

	v = nc_sysconfig_get(sc, "BRIDGE");
	if(!v || !nc_string_eq(v->value, "yes")) {
		return 1;
	}

	if(ifp->type != NC_IFTYPE_BRIDGE &&
	   ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/try_bridge/%s/type", ifp->name)) {
			nc_error("Interface config %s: matches bridge but is of another interface type",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_bridge/%s/type", ifp->name);

	ifp->type = NC_IFTYPE_BRIDGE;
	bridge = nc_bridge_new();
	if( !bridge) {
		/* Hmm... */
		nc_error("Unable to allocate bridge structure");
		goto failure;
	}

	if( (v = nc_sysconfig_get(sc, "BRIDGE_STP")) != NULL && v->value)
		nc_bridge_set_stp(bridge, v->value);
	if( (v = nc_sysconfig_get(sc, "BRIDGE_FORWARDDELAY")) != NULL && v->value)
		nc_bridge_set_forward_delay(bridge, v->value);
	if( (v = nc_sysconfig_get(sc, "BRIDGE_AGEINGTIME")) != NULL && v->value)
		nc_bridge_set_ageing_time(bridge, v->value);
	if( (v = nc_sysconfig_get(sc, "BRIDGE_HELLOTIME")) != NULL && v->value)
		nc_bridge_set_hello_time(bridge, v->value);
	if( (v = nc_sysconfig_get(sc, "BRIDGE_MAXAGE")) != NULL && v->value)
		nc_bridge_set_max_age(bridge, v->value);
	if( (v = nc_sysconfig_get(sc, "BRIDGE_PRIORITY")) != NULL && v->value)
		nc_bridge_set_priority(bridge, v->value);

	if(nc_sysconfig_get_string(sc, "BRIDGE_PORTS", &value) == 0) {
		char *p = NULL;
		for (token = strtok_r(value, " \t", &p);
		     token;
		     token = strtok_r(NULL, " \t", &p)) {

			if(nc_string_eq(ifp->name, token)) {
				if(nc_error_once_check(nh, "/try_bridge/%s/%s/self", ifp->name, token)) {
					nc_error("Interface config %s: bridge port %s refers to bridge",
							ifp->name, token);
				}
				goto failure;
			}
			nc_error_once_clear(nh, "/try_bridge/%s/%s/self", ifp->name, token);

			if(nc_bridge_add_port(bridge, token) < 0) {
				if(nc_error_once_check(nh, "/try_bridge/%s/%s/uniq", ifp->name, token)) {
					nc_error("Interface config %s: cannot add bridge port %s - used twice",
							ifp->name, token);
				}
				goto failure;
			}
			nc_error_once_clear(nh, "/try_bridge/%s/%s/uniq", ifp->name, token);

		}
		nc_string_free(&value);
	}

	if (nc_sysconfig_get_string(sc, "BRIDGE_PORTPRIORITIES", &value) == 0) {
		char *p = NULL;
		unsigned int i = 0;
		for (token = strtok_r(value, " \t", &p);
		     token;
		     token = strtok_r(NULL, " \t", &p), ++i) {
			const char *port;
			if(i >= bridge->ports.count)
				break;
			if(token[0] == '-')
				token = "";
			port = bridge->ports.data[i]->name;
			if( nc_bridge_port_set_priority(bridge, port, token) < 0)
				break;
		}
		nc_string_free(&value);
	}
	if (nc_sysconfig_get_string(sc, "BRIDGE_PATHCOSTS", &value) == 0) {
		char *p = NULL;
		unsigned int i = 0;
		for (token = strtok_r(value, " \t", &p);
		     token;
		     token = strtok_r(NULL, " \t", &p), ++i) {
			const char *port;
			if(i >= bridge->ports.count)
				break;
			if(token[0] == '-')
				token = "";
			port = bridge->ports.data[i]->name;
			if( nc_bridge_port_set_path_cost(bridge, port, token) < 0)
				break;
		}
		nc_string_free(&value);
	}

	nc_bridge_free(ifp->bridge);
	ifp->bridge = bridge;
	return 0;

failure:
	nc_bridge_free(bridge);
	nc_string_free(&value);
	ifp->invalid = 1;
	return -1;
}

static int
try_add_bonding_slave(nc_interface_t *ifp, nc_sysconfig_t *sc, const char *suffix)
{
	nc_var_t *v;

	v = __find_indexed_variable(sc, "BONDING_SLAVE", suffix);
	if (!v || !nc_string_len(v->value))
		return -1;

	if(!ifp->bonding)
		return -1;

	if(nc_string_eq(ifp->name, v->value)) {
		nc_error("bond %s config references itself as slave in %s",
			ifp->name, v->name);
		return -1;
	}
	if(maybe_infiniband(v->value, NULL) == 0) {
		ifp->bonding->slave_type = NC_IFTYPE_INFINIBAND;
	} else {
		ifp->bonding->slave_type = NC_IFTYPE_ETHERNET;
	}
	return nc_bonding_add_slave(ifp->bonding, v->value);
}

static int
try_bonding(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	nc_bonding_t *bonding;
	nc_var_t *    v;

	v = nc_sysconfig_get(sc, "BONDING_MASTER");
	if(!v || !nc_string_eq(v->value, "yes")) {
		return 1;
	}

	if(ifp->type != NC_IFTYPE_BOND &&
	   ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/try_bonding/%s/type", ifp->name)) {
			nc_error("Interface config %s: matches bonding but is of another interface type",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_bonding/%s/type", ifp->name);

	ifp->type = NC_IFTYPE_BOND;
	bonding = nc_bonding_new();
	if( !bonding) {
		/* Hmm... */
		nc_error("Unable to allocate bonding structure");
		goto failure;
	} else {
		if(ifp->bonding)
			nc_bonding_free(ifp->bonding);
		ifp->bonding = bonding;
	}

	if(__process_indexed_variables(ifp, sc, "BONDING_SLAVE", try_add_bonding_slave) != 0) {
		if(nc_error_once_check(nh, "/try_bonding/%s/slaves", ifp->name)) {
			nc_error("Interface config %s: cannot process bonding slaves",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_bonding/%s/slaves", ifp->name);

	nc_sysconfig_get_string(sc, "BONDING_MODULE_OPTS", &ifp->bonding->module_opts);
	nc_bonding_parse_module_options(ifp->bonding, nh, ifp->name);

	return 0;

failure:
	ifp->invalid = 1;
	return -1;
}

static int
try_wireless(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	unsigned int type = NC_IFTYPE_UNKNOWN;
	nc_var_t *   v;

	v = nc_sysconfig_get(sc, "WIRELESS");
	if(v && nc_string_eq(v->value, "yes")) {
		type = NC_IFTYPE_WIRELESS;
	}
	v = nc_sysconfig_get(sc, "WIRELESS_MODE");
	if(v && nc_string_len(v->value)) {
		type = NC_IFTYPE_WIRELESS;
	}

	if(type == NC_IFTYPE_UNKNOWN)
		return 1;

	if(ifp->type != type && ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/try_wireless/%s/type", ifp->name)) {
			nc_error("Interface config %s: matches wireless but is of another interface type",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_wireless/%s/type", ifp->name);

	return 0;

failure:
	ifp->invalid = 1;
	return -1;
}

static int
try_ppp(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	unsigned int type = NC_IFTYPE_UNKNOWN;
	const char * tstr = NULL;
	nc_var_t *   v;

	v = nc_sysconfig_get(sc, "MODEM_DEVICE");
	if(v && nc_string_len(v->value)) {
		type = NC_IFTYPE_PPP;
		tstr = "ppp";
	}
	v = nc_sysconfig_get(sc, "PPPMODE");
	if(v && nc_string_len(v->value)) {
		type = NC_IFTYPE_PPP;
		tstr = "ppp";
	}
	v = nc_sysconfig_get(sc, "ENCAP");
	if(v && nc_string_len(v->value)) {
		type = NC_IFTYPE_ISDN;
		tstr = "isdn";
	}

	if(type == NC_IFTYPE_UNKNOWN)
		return 1;

	if(ifp->type != type && ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/try_ppp/%s/type", ifp->name)) {
			nc_error("Interface config %s: matches %s but is of another interface type",
				ifp->name, tstr);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_ppp/%s/type", ifp->name);

	return 0;

failure:
	ifp->invalid = 1;
	return -1;
}

static int
try_tunnel(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	unsigned int type = NC_IFTYPE_UNKNOWN;
	const char * tstr = NULL;
	nc_var_t *   v;

	v = nc_sysconfig_get(sc, "TUNNEL");
	if(v && v->value) {
		type = NC_IFTYPE_TUNNEL;
		tstr = "tunnel";

		if(nc_string_eq(v->value, "tap")) {
			type = NC_IFTYPE_TAP;
			tstr = "tap";
		} else
		if(nc_string_eq(v->value, "tun")) {
			type = NC_IFTYPE_TUN;
			tstr = "tun";
		}
		if(nc_string_eq(v->value, "sit")) {
			type = NC_IFTYPE_SIT;
			tstr = "sit";
		}
		if(nc_string_eq(v->value, "gre")) {
			type = NC_IFTYPE_GRE;
			tstr = "gre";
		}
	}

	if(type == NC_IFTYPE_UNKNOWN)
		return 1;

	if(ifp->type != type && ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/try_tunnel/%s/type", ifp->name)) {
			nc_error("Interface config %s: matches %s but is of another interface type",
				ifp->name, tstr);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_tunnel/%s/type", ifp->name);

	return 0;

failure:
	ifp->invalid = 1;
	return -1;
}

static int
try_infiniband(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	nc_vlan_t *  vlan = NULL;
	int          ret;

	ret = maybe_infiniband(ifp->name, &vlan);
	if(ret > 0)
		return 1;

	if(ifp->type != NC_IFTYPE_INFINIBAND && ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/try_infiniband/%s/type", ifp->name)) {
			nc_error("Interface config %s: matches infiniband but is of another interface type",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_infiniband/%s/type", ifp->name);

	ifp->type = NC_IFTYPE_INFINIBAND;
	if(ret < 0) {
		if(nc_error_once_check(nh, "/try_infiniband/%s/pkey", ifp->name)) {
			nc_error("Interface config %s: Unable to parse parent key",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/try_infiniband/%s/pkey", ifp->name);

	/*
	** TODO: for !netcf->strict
	**   IPOIB_MODE=<connected|datagram>,
	**   IPOIB_UMCAST=<0|1>
	v = nc_sysconfig_get(sc, "IPOIB_MODE");
	v = nc_sysconfig_get(sc, "IPOIB_UMCAST");
	*/
	(void)sc;

	/* TODO: currently we do not detect/verify pkey, so just use as is;
	 *       ibchildren are not like vlan, but like ethernet anyway...
	 */
	if(ifp->vlan)
		nc_vlan_free(ifp->vlan);
	ifp->vlan = vlan;

	return 0;

failure:
	ifp->invalid = 1;
	return -1;
}

static int
try_add_ovs_port(nc_interface_t *ifp, nc_sysconfig_t *sc, const char *suffix)
{
	nc_var_t *v;

	v = __find_indexed_variable(sc, "OVS_BRIDGE_PORT_DEVICE", suffix);
	if (!v || !nc_string_len(v->value))
		return 1;

	if (nc_string_eq(ifp->name, v->value))
		return -1;

	if (!ifp->bridge && !(ifp->bridge = nc_bridge_new()))
		return -1;

	return nc_bridge_add_port(ifp->bridge, v->value);
}

static int
try_ovs(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	nc_var_t *v;

	v = nc_sysconfig_get(sc, "OVS_BRIDGE");
	if (!nc_string_eq(ifp->name, "ovs-system") &&
	    (!v || !nc_string_eq(v->value, "yes")))
		return 1;

	if(ifp->type != NC_IFTYPE_OVS && ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/%s/%s/type", __func__, ifp->name)) {
			nc_error("Interface config %s: matches an ovs and another interface type",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/%s/%s/type", __func__, ifp->name);

	ifp->type = NC_IFTYPE_OVS;
	if (nc_string_eq(ifp->name, "ovs-system"))
		return 0;

	v = nc_sysconfig_get(sc, "OVS_BRIDGE_VLAN_PARENT");
	if(v && nc_string_len(v->value)) {
		unsigned int tag = 0;

		if(nc_sysconfig_get_uint(sc, "OVS_BRIDGE_VLAN_TAG", &tag) != 0 ||
		   !(ifp->vlan = nc_vlan_new(v->value, tag))) {
			if(nc_error_once_check(nh, "/%s/%s/vlan", __func__, ifp->name)) {
				nc_error("Interface config %s: cannot process ovs vlan bridge",
					ifp->name);
			}
			goto failure;
		}
		nc_error_once_clear(nh, "/%s/%s/vlan", __func__, ifp->name);
	}
	if(__process_indexed_variables(ifp, sc, "OVS_BRIDGE_PORT_DEVICE", try_add_ovs_port) < 0) {
		if(nc_error_once_check(nh, "/%s/%s/port", __func__, ifp->name)) {
			nc_error("Interface config %s: cannot process ovs bridge ports",
				ifp->name);
		}
		goto failure;
	} else {
		nc_error_once_clear(nh, "/%s/%s/port", __func__, ifp->name);
	}

	return 0;
failure:
	ifp->invalid = 1;
	return -1;
}

static int
try_add_team_port(nc_interface_t *ifp, nc_sysconfig_t *sc, const char *suffix)
{
	nc_var_t *v;

	v = __find_indexed_variable(sc, "TEAM_PORT_DEVICE", suffix);
	if (!v || !nc_string_len(v->value))
		return 1;

	if (!ifp->bonding)
		return -1;

	if (nc_string_eq(ifp->name, v->value))
		return -1;

	ifp->bonding->slave_type = NC_IFTYPE_ETHERNET;
	return nc_bonding_add_slave(ifp->bonding, v->value);
}

static int
try_team(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	nc_var_t *v;

	v = nc_sysconfig_get(sc, "TEAM_RUNNER");
	if(!v || !nc_string_len(v->value))
		return 1;

	if(ifp->type != NC_IFTYPE_TEAM && ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/%s/%s/type", __func__, ifp->name)) {
			nc_error("Interface config %s: matches team and another interface type",
				ifp->name);
		}
		goto failure;
	}
	nc_error_once_clear(nh, "/%s/%s/type", __func__, ifp->name);

	ifp->type = NC_IFTYPE_TEAM;
	ifp->bonding = nc_bonding_new();
	if(__process_indexed_variables(ifp, sc, "TEAM_PORT_DEVICE", try_add_team_port) < 0) {
		if(nc_error_once_check(nh, "/%s/%s/port", __func__, ifp->name)) {
			nc_error("Interface config %s: cannot process team ports",
				ifp->name);
		}
		goto failure;
	} else {
		nc_error_once_clear(nh, "/%s/%s/port", __func__, ifp->name);
	}

	return 0;
failure:
	ifp->invalid = 1;
	return -1;
}

static int
try_macvlan(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	nc_var_t *v;

	v = nc_sysconfig_get(sc, "MACVLAN_DEVICE");
	if(!v || !nc_string_eq(v->value, "yes"))
		return 1;

	if(ifp->type != NC_IFTYPE_MACVLAN && ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/%s/%s/type", __func__, ifp->name)) {
			nc_error("Interface config %s: matches macvlan and another interface type",
				ifp->name);
		}
		ifp->invalid = 1;
		return -1;
	}
	nc_error_once_clear(nh, "/%s/%s/type", __func__, ifp->name);

	ifp->type = NC_IFTYPE_MACVLAN;
	return 0;
}

static int
try_macvtap(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	nc_var_t *v;

	v = nc_sysconfig_get(sc, "MACVTAP_DEVICE");
	if(!v || !nc_string_eq(v->value, "yes"))
		return 1;

	if(ifp->type != NC_IFTYPE_MACVTAP && ifp->type != NC_IFTYPE_UNKNOWN) {
		if(nc_error_once_check(nh, "/%s/%s/type", __func__, ifp->name)) {
			nc_error("Interface config %s: matches macvtap and another interface type",
				ifp->name);
		}
		ifp->invalid = 1;
		return -1;
	}
	nc_error_once_clear(nh, "/%s/%s/type", __func__, ifp->name);

	/* cannot differentiate them, but a macvtap is derived from macvlan */
	ifp->type = NC_IFTYPE_MACVLAN;
	return 0;
}

int
nc_interface_guess_type(nc_interface_t *ifp)
{
	if(ifp->type != NC_IFTYPE_UNKNOWN)
		return ifp->type;

	if (ifp->name == NULL)
		return ifp->type;

	ifp->type = NC_IFTYPE_ETHERNET;
	if (!strcmp(ifp->name, "lo")) {
		ifp->type = NC_IFTYPE_LOOPBACK;
	} else {
		nc_intmap_t *map;

		for (map = __ifname_types; map->name; ++map) {
			unsigned int len = strlen(map->name);

			if (!strncmp(ifp->name, map->name, len) &&
			    isdigit(ifp->name[len])) {
				ifp->type = map->value;
				break;
			}
		}
	}
	return ifp->type;
}

static int
__nc_suse_config_parse_route_line(char *buffer, nc_route_t **rpp, const char *iface,
				  const char *file, unsigned int line)
{
	char *dest, *gw, *mask = NULL, *ifname = NULL, *opts = NULL, *p;
	nc_sockaddr_t dest_addr, gw_addr, mask_addr;
	unsigned int prefixlen = 255;

	p = NULL;
	dest = strtok_r(buffer, " \t", &p);
	if(!dest) {
		return 1; // empty line
	}

	gw = strtok_r(NULL, " \t", &p);
	if(gw)
		mask = strtok_r(NULL, " \t", &p);
	if(mask)
		ifname = strtok_r(NULL, " \t", &p);
	if(ifname) {
		opts = nc_string_strip_spaces(p);
		/* TODO: parse type and opts */
		if(opts && !*opts)
			opts = NULL;
	}

	if(!gw || nc_string_eq(gw, "-")) {
		/*
		 * This is a local interface route; e.g.
		 * ifroute-lo contains just 127/8 in it.
		 */
		memset(&gw_addr, 0, sizeof(gw_addr));
	} else
	if(nc_address_parse(&gw_addr, gw, AF_UNSPEC) < 0) {
		nc_error("Unable to parse gateway address \"%s\" in %s:%u",
				gw, file, line);
		return -1;
	}
		
	if(nc_string_eq(dest, "default")) {
		memset(&dest_addr, 0, sizeof(dest_addr));
		dest_addr.ss_family = gw_addr.ss_family;
		prefixlen = 0;
	} else {
		char *sp;

		if ((sp = strchr(dest, '/')) != NULL) {
			*sp++ = '\0';
			if(nc_parse_uint(sp, &prefixlen, 10) < 0) {
				nc_error("Unable to parse prefix length \"%s\""
					 " in %s:%u", sp, file, line);
				return -1;
			}
		}
		if(nc_address_parse(&dest_addr, dest, AF_UNSPEC) < 0) {
			nc_error("Unable to parse destination address \"%s\""
				 " in %s:%u", dest, file, line);
			return -1;
		}
		if (prefixlen == 255) {
			/* FIXME: this does not work: when dest is "128"
			 *        and the mask is "8" or "255.0.0.0"...
			 */
			if (!mask || !strcmp(mask, "-")) {
				/* No prefix and no mask given - assume the destination
				 * is a single address. Use the full address length
				 * as prefix.
				 */
				prefixlen = nc_address_bits(dest_addr.ss_family);
			} else {
				if(!strchr(mask, '.')) {
					if(nc_parse_uint(mask, &prefixlen, 10) < 0) {
						nc_error("Unable to parse prefix length \"%s\""
							 " in %s:%u", mask, file, line);
						return -1;
					}
				} else {
					if(nc_address_parse(&mask_addr, mask, AF_UNSPEC) < 0) {
						nc_error("Unable to parse netmask address \"%s\""
							 "in %s:%u", mask, file, line);
						return -1;
					}
					prefixlen = nc_netmask_bits(&mask_addr);
				}
			}
		}
		if(prefixlen > nc_address_bits(dest_addr.ss_family)) {
			nc_error("Unable to find prefix length in %s:%u", file, line);
			return -1;
		}
	}
	if(!nc_string_len(ifname) || nc_string_eq(ifname, "-")) {
		ifname = NULL;
	}
	if(nc_string_len(iface)) {
		/*
		 * we read from iface specific file (ifroutes-<iface>)
		 * and reject routes for another interfaces.
		 */
		if(ifname && !nc_string_eq(ifname, iface))
			return -1;
	}

	*rpp = nc_route_create(prefixlen, &dest_addr, &gw_addr,
				(ifname ? ifname : iface), opts);
	if(!*rpp) {
		nc_error("Unable to create route from %s:%u", file, line);
		return -1;
	}
	return 0;
}


static int
__suse_config_read_routes(nc_route_array_t *routes,
                          const char       *file,
                          const char       *iface)
{
	char buffer[BUFSIZ] = {'\0'};
	unsigned int line = 0;
	FILE *fpi;

	fpi = fopen(file, "re");
	if(fpi == NULL) {
		if (errno != ENOENT) {
			nc_error("Unable to open %s: %m", file);
			return -1;
		}
		return 1;
	}

	while (fgets(buffer, sizeof(buffer), fpi) != NULL) {
		nc_route_t *rp = NULL;
		int ret;

		line++;
		buffer[strcspn(buffer, "#\r\n")] = '\0';

		ret = __nc_suse_config_parse_route_line(buffer, &rp, iface,
							file, line);
		if(ret != 0) {
			/* empty line or invalid */
			continue;
		}

		if( nc_route_array_append(routes, rp) < 0) {
			/* invalid parameter or oom (-> assert) only ... */
			nc_route_free(rp);
			nc_error("Unable to add route to array");
			goto failure;
		}
	}

	fclose(fpi);
	return 0;
failure:
	nc_route_array_destroy(routes);
	fclose(fpi);
	return -1;
}

#if 0
static void
__suse_config_load_sysctls(nc_handle_t *nh, nc_interface_t *ifp, nc_var_array_t *sysctl)
{
	nc_stringbuf_t file_name = NC_STRINGBUF_INIT;
	nc_config_handle_t   *ch = __to_config_handle(nh);
	nc_sysconfig_t       *sc;

	if(__nc_suse_config_prepend_root(ch, "/etc/sysconfig/sysctl", &file_name) < 0)
		goto cleanup;

	sc = nc_sysconfig_read(file_name.string);
	if(sc) {
		nc_var_t *v;
		const char *val = "0";
		if( (v = nc_sysconfig_get(sc, "IPV6_DISABLE")) != NULL) {
			if(nc_string_eq(v->value, "yes")) {
				val = "1";
			}
			nc_var_array_set(sysctl, "net/ipv6/conf/all/disable_ipv6", val);
		}
		if( (v = nc_sysconfig_get(sc, "IPV6_FORWARD")) != NULL) {
			if(nc_string_eq(v->value, "yes")) {
				val = "1";
			}
			nc_var_array_set(sysctl, "net/ipv6/conf/all/forwarding", val);
		}
		nc_sysconfig_free(sc);
		sc = NULL;
	}

	if(__nc_suse_config_prepend_root(ch, "/etc/sysctl.conf", &file_name) < 0)
		goto cleanup;
	nc_sysctl_load(sysctl, file_name.string);

	if(__nc_suse_config_prepend_root(ch, "/etc/sysconfig/network/ifsysctl", &file_name) < 0)
		goto cleanup;
	nc_sysctl_load(sysctl, file_name.string);

	if((nc_stringbuf_putc(&file_name, '-') < 0) ||
	   (nc_stringbuf_puts(&file_name, ifp->name) < 0))
		goto cleanup;
	nc_sysctl_load(sysctl, file_name.string);

cleanup:
	nc_stringbuf_destroy(&file_name);
}

static void
__suse_adjust_ipv6_autoconf(nc_interface_t *ifp, nc_var_array_t *sysctl)
{
	const char *afmt;
	const char *names[] = {
		"all",
		"default",
		"$INTERFACE",
		"$SYSCTL_IF",
		NULL
	};
	const char **n;
	const nc_var_t *v;
	int disabled = 0;
	int accept_ra = -1;
	int forwarding = -1;

	/*
	 * 1000 ways to disable ipv6 autoconf
	 * in hope this works sometimes ... ;)
	 */
	afmt = "net/ipv6/conf/%s/disable_ipv6";
	for(n = names; *n; n++) {
		if( (v = nc_sysctl_get_fmt(sysctl, afmt, *n))) {
			if(nc_string_eq(v->value, "1"))
				disabled = 1;
		}
	}
	if( (v = nc_sysctl_get_fmt(sysctl, afmt, ifp->name))) {
		if(nc_string_eq(v->value, "1"))
			disabled = 1;
		else
			disabled = 0;
	}

	afmt = "net/ipv6/conf/%s/forwarding";
	for(n = names; *n; n++) {
		if( (v = nc_sysctl_get_fmt(sysctl, afmt, *n))) {
			if(nc_string_eq(v->value, "1"))
				forwarding = 1;
			else
				forwarding = 0;
		}
	}
	if( (v = nc_sysctl_get_fmt(sysctl, afmt, ifp->name))) {
		if(nc_string_eq(v->value, "1"))
			forwarding = 1;
		else
			forwarding = 0;
	}

	afmt = "net/ipv6/conf/%s/accept_ra";
	for(n = names; *n; n++) {
		if( (v = nc_sysctl_get_fmt(sysctl, afmt, *n))) {
			if(nc_string_eq(v->value, "0")) {
				accept_ra = 0;
			} else
			if(nc_string_eq(v->value, "1")) {
				accept_ra = 1;
			} else
			if(nc_string_eq(v->value, "2")) {
				accept_ra = 2;
			}
		}
	}
	if( (v = nc_sysctl_get_fmt(sysctl, afmt, ifp->name))) {
		if(nc_string_eq(v->value, "0")) {
				accept_ra = 0;
			} else
			if(nc_string_eq(v->value, "1")) {
				accept_ra = 1;
			} else
			if(nc_string_eq(v->value, "2")) {
				accept_ra = 2;
			}
	}

	if(forwarding == 1 && accept_ra != 2)
		accept_ra = 0;
	if(accept_ra != -1)
		disabled = !accept_ra;

	afmt = "net/ipv6/conf/%s/autoconf";
	for(n = names; *n; n++) {
		if( (v = nc_sysctl_get_fmt(sysctl, afmt, *n))) {
			if(nc_string_eq(v->value, "0"))
				disabled = 1;
			else
				disabled = 0;
		}
	}
	if( (v = nc_sysctl_get_fmt(sysctl, afmt, ifp->name))) {
		if(nc_string_eq(v->value, "0"))
			disabled = 1;
		else
			disabled = 0;
	}

	if(disabled) {
		ifp->ipv6.addrconf &= ~NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
	}
}
#endif

static int
__suse_config_read_ifcfg(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc)
{
	/*
	nc_var_array_t sysctl = NC_VAR_ARRAY_INIT;
	*/
	typedef int (*try_func_t)(nc_handle_t *nh, nc_interface_t *ifp, nc_sysconfig_t *sc);
	try_func_t *try_func, try_funcs[] = {
		try_vlan,
		try_bridge,
		try_bonding,
		try_wireless,
		try_ppp,
		try_tunnel,
		try_infiniband,
		try_ovs,
		try_team,
		try_macvlan,
		try_macvtap,
		NULL
	};
	nc_var_t *v;

	(void)nh;

	ifp->type = NC_IFTYPE_UNKNOWN;
	if(nc_discover_iface_type(ifp->name, &ifp->type, &ifp->arp_type) == 0) {
		switch(ifp->type) {
			case NC_IFTYPE_VLAN:
				if(__nc_discover_vlan(ifp) != 0) {
					ifp->invalid = 1;
					return -1;
				}

				if(try_vlan(nh, ifp, sc) < 0) {
					ifp->invalid = 1;
					return -1;
				}
			break;
			case NC_IFTYPE_BRIDGE:
				if(try_bridge(nh, ifp, sc) < 0) {
					ifp->invalid = 1;
					return -1;
				}
			break;
			case NC_IFTYPE_BOND:
				if(try_bonding(nh, ifp, sc) < 0) {
					ifp->invalid = 1;
					return -1;
				}
			break;
			case NC_IFTYPE_TEAM:
				if(try_team(nh, ifp, sc) < 0) {
					ifp->invalid = 1;
					return -1;
				}
			break;
			case NC_IFTYPE_OVS:
				if(try_ovs(nh, ifp, sc) < 0) {
					ifp->invalid = 1;
					return -1;
				}
			break;
			case NC_IFTYPE_MACVLAN:
			case NC_IFTYPE_MACVTAP:
				break;

			default:
				/* Hmm... just check sc vars ? */
				(void)try_vlan(nh, ifp, sc);
				(void)try_bridge(nh, ifp, sc);
				(void)try_bonding(nh, ifp, sc);

				/* we have a valid type, reset invalid flag? */
				ifp->invalid = 0;
			break;
		}
	}

	if(ifp->type == NC_IFTYPE_UNKNOWN) {
		/*
		 * Try to get type from config and fail when somethig
		 * goes wrong or multiple types are in it (conflicts).
		 */
		for(try_func = try_funcs; *try_func; ++try_func) {
			if((*try_func)(nh, ifp, sc) < 0) {
				ifp->invalid = 1;
				return 1;
			}
		}

		if(ifp->type == NC_IFTYPE_UNKNOWN) {
			nc_interface_guess_type(ifp);

			/*
			 * do not allow to guess them from name;
			 * we need the details for topology
			 */
			switch(ifp->type) {
				case NC_IFTYPE_VLAN:
				case NC_IFTYPE_BOND:
				case NC_IFTYPE_BRIDGE:
					ifp->invalid = 1;
					return 1;
				break;

				default:
				break;
			}
		}
	}

	v = nc_sysconfig_get(sc, "STARTMODE");
	if(v && nc_string_len(v->value)) {
		__suse_ifcfg_startmode_set(ifp, v->value);
	} else {
		__suse_ifcfg_startmode_set(ifp, "manual");
	}

	v = nc_sysconfig_get(sc, "BOOTPROTO");
	if(v && nc_string_len(v->value)) {
		__suse_ifcfg_bootproto_set(ifp, v->value);
	} else {
		__suse_ifcfg_bootproto_set(ifp, "static");
	}

	/*
	__suse_config_load_sysctls(nh, ifp, &sysctl);
	__suse_adjust_ipv6_autoconf(ifp, &sysctl);
	nc_var_array_destroy(&sysctl);
	*/

	v = nc_sysconfig_get(sc, "LLADDR");
	if( v && nc_string_len(v->value)) {
		nc_hw_address_parse(&ifp->hwaddr, ifp->type, v->value);
	} else {
		char *value = NULL;
		if(nc_udev_net_rule_hwaddr_for_name(ifp->name, &value) == 0 && value) {
			nc_hw_address_parse(&ifp->hwaddr, ifp->type, value);
			nc_string_free(&value);
		}
	}

	(void)nc_sysconfig_get_uint(sc, "MTU", &ifp->mtu);

	__process_indexed_variables(ifp, sc, "IPADDR", try_add_address);

	return 0;
}

static int
nc_ethtool(const char *ifname, struct ethtool_cmd *cmd)
{
	struct ifreq ifr;
	int sock, ret;

	if (!nc_string_len(ifname) || !cmd)
		return -1;

	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
	ifr.ifr_data = (void *)cmd;

	sock = socket(PF_INET, SOCK_DGRAM, 0);
	if (sock < 0)
		return -1;

	ret = ioctl(sock, SIOCETHTOOL, &ifr);
	close(sock);
	return ret;
}

static int
nc_ethtool_driver_info(const char *ifname, struct ethtool_drvinfo *drv_info)
{
	if (!drv_info)
		return -1;

	memset(drv_info, 0, sizeof(struct ethtool_drvinfo));
	drv_info->cmd = ETHTOOL_GDRVINFO;

	return nc_ethtool(ifname, (struct ethtool_cmd *)drv_info);
}

static int
ni_discover_iface_type_ethtool(const char *name, unsigned int *ifp_type)
{
	struct ethtool_drvinfo drv_info;

	if (!ifp_type || nc_ethtool_driver_info(name, &drv_info) < 0)
		return -1;

	/* just return unknown for types not supported by netcf;
	 * not needed to made them known, except when this are
	 * master container devices / parent types ...
	 */
	if (nc_string_eq(drv_info.driver, "bridge"))
		*ifp_type = NC_IFTYPE_BRIDGE;
	else
	if (nc_string_eq(drv_info.driver, "bonding"))
		*ifp_type = NC_IFTYPE_BOND;
	else
	if (nc_string_eq(drv_info.driver, "802.1Q VLAN Support"))
		*ifp_type = NC_IFTYPE_VLAN;
	else
	if (nc_string_eq(drv_info.driver, "dummy"))
		*ifp_type = NC_IFTYPE_DUMMY;
	else
	if (nc_string_eq(drv_info.driver, "team"))
		*ifp_type = NC_IFTYPE_DUMMY;
	else
	if (nc_string_eq(drv_info.driver, "openvswitch"))
		*ifp_type = NC_IFTYPE_OVS;
	else
	if (nc_string_eq(drv_info.driver, "macvlan"))
		/* macvtap is not visible here */
		*ifp_type = NC_IFTYPE_MACVLAN;
	else
	if (nc_string_eq(drv_info.driver, "vxlan"))
		*ifp_type = NC_IFTYPE_UNKNOWN;
	else
	if (nc_string_eq(drv_info.driver, "veth"))
		*ifp_type = NC_IFTYPE_UNKNOWN;
	else
	if (nc_string_eq(drv_info.driver, "tun")) {
		if (nc_string_eq(drv_info.bus_info, "tap"))
			*ifp_type = NC_IFTYPE_TAP;
		else
		if (nc_string_eq(drv_info.bus_info, "tun"))
			*ifp_type = NC_IFTYPE_TUN;
		else
			return 1;
	} else
		return 1;

	return 0;
}

static int
nc_discover_iface_type(const char *name, unsigned int *ifp_type, unsigned int *arp_type)
{

	assert(name && *name && ifp_type && arp_type);

	*ifp_type = NC_IFTYPE_UNKNOWN;
	*arp_type = ARPHRD_VOID;
	if(nc_sysfs_netif_get_uint(name, "type", arp_type) < 0) {
		nc_trace("Interface %s does not exists", name);
		return -1;
	}

	if (ni_discover_iface_type_ethtool(name, ifp_type) == 0)
		return 0;

	switch(*arp_type) {
	case ARPHRD_NONE:
		if(nc_sysfs_netif_exists(name, "tun_flags")) {
			*ifp_type = NC_IFTYPE_TUN;
		}
	break;
	case ARPHRD_ETHER:
		*ifp_type = NC_IFTYPE_ETHERNET;

		if(nc_sysfs_netif_exists(name, "wireless") ||
		   nc_sysfs_netif_exists(name, "phy80211")) {
			*ifp_type = NC_IFTYPE_WIRELESS;
		} else
		if(nc_sysfs_netif_exists(name, "bonding/mode")) {
			*ifp_type = NC_IFTYPE_BOND;
		} else
		if(nc_sysfs_netif_exists(name, "bridge/bridge_id")) {
			*ifp_type = NC_IFTYPE_BRIDGE;
		} else
		if(nc_file_exists_fmt("/proc/net/vlan/%s", name)) {
			*ifp_type = NC_IFTYPE_VLAN;
		} else
		if(nc_file_exists_fmt("/sys/devices/virtual/net/%s", name)) {
			if(nc_string_prefix_eq("dummy", name)) {
				*ifp_type = NC_IFTYPE_DUMMY;
			}
		} else
		if(nc_sysfs_netif_exists(name, "tun_flags")) {
			*ifp_type = NC_IFTYPE_TAP;
		}
	break;
	case ARPHRD_INFINIBAND:
		*ifp_type = NC_IFTYPE_INFINIBAND;

		/* bonding inherits the type from slaves */
		if(nc_sysfs_netif_exists(name, "bonding/mode")) {
			*ifp_type = NC_IFTYPE_BOND;
		}
	break;
	default:
		*ifp_type = nc_arphrd_type_to_iftype(*arp_type);
	break;
	}

	return *ifp_type == NC_IFTYPE_UNKNOWN ? 1 : 0;
}

static int
__nc_discover_bridge(nc_interface_t *ifp)
{
	nc_string_array_t ports;
	unsigned int i;

	if(!ifp || !ifp->name)
		return -1;

	if(ifp->type != NC_IFTYPE_BRIDGE)
		return 1;

	ifp->bridge = nc_bridge_new();
	if(!ifp->bridge)
		return -1;

	nc_sysfs_bridge_get_config(ifp->name, &ifp->bridge->config);
	nc_sysfs_bridge_get_status(ifp->name, &ifp->bridge->status);

	nc_string_array_init(&ports);
	nc_sysfs_bridge_get_port_names(ifp->name, &ports);
	for (i = 0; i < ports.count; ++i) {
		if(nc_bridge_add_port(ifp->bridge, ports.data[i]) < 0)
			return -1;
	}
	nc_string_array_destroy(&ports);

	for (i = 0; i < ifp->bridge->ports.count; ++i) {
		nc_bridge_port_t *port = ifp->bridge->ports.data[i];
		nc_sysfs_bridge_port_get_config(port->name, &port->config);
		nc_sysfs_bridge_port_get_status(port->name, &port->status);
	}
	return 0;
}

static int
__nc_discover_bond(nc_interface_t *ifp)
{
	nc_bonding_t *bonding;

	if(!ifp || !ifp->name)
		return -1;

	if(ifp->type != NC_IFTYPE_BOND) {
		return 1;
	}

	bonding = nc_bonding_new();
	if (!bonding || nc_bonding_parse_sysfs_attrs(ifp->name, bonding) < 0) {
		nc_error("can't retrieve bonding attribute from sysfs");
		return -1;
	}

	if(ifp->arp_type == ARPHRD_INFINIBAND) {
		bonding->slave_type = NC_IFTYPE_INFINIBAND;
	} else {
		bonding->slave_type = NC_IFTYPE_ETHERNET;
	}

	ifp->bonding = bonding;

	return 0;
}

static int
__nc_discover_vlan(nc_interface_t *ifp)
{
	char *real = NULL;
	unsigned int tag;

	if(!ifp || !ifp->name)
		return -1;

	if(ifp->type != NC_IFTYPE_VLAN)
		return 1;

	if(nc_vlan_get_settings(ifp->name, &real, &tag) < 0)
		return -1;

	if(real) {
		assert(!ifp->vlan);
		ifp->vlan = nc_vlan_new(real, tag);
		nc_string_free(&real);
		if(!ifp->vlan)
			return -1;
		return 0;
	} else {
		return 1;
	}
}

static int
__nc_discover_infiniband(nc_interface_t *ifp)
{
	char *real = NULL;
	unsigned int tag;

	if(!ifp || !ifp->name)
		return -1;

	if(ifp->type != NC_IFTYPE_INFINIBAND)
		return 1;

	if(nc_sysfs_netif_exists(ifp->name, "create_child"))
		return 0;

	if(nc_sysfs_netif_get_string(ifp->name, "parent", &real) != 0)
		return -1;

	if(nc_sysfs_netif_get_uint(ifp->name, "pkey", &tag) != 0) {
		nc_string_free(&real);
		return -1;
	}
	assert(!ifp->vlan);
	ifp->vlan = nc_vlan_new(real, tag);
	nc_string_free(&real);
	if(!ifp->vlan)
		return -1;
	return 0;
}

static int
__nc_discover_parent(nc_interface_t *ifp)
{
	char *parent = NULL;

	if(!ifp || !ifp->name)
		return -1;

	/* seems that all slave types link to their master under newer kernels */
	if(nc_sysfs_netif_get_linkbase(ifp->name, "master", &parent) == 0) {
		nc_string_free(&ifp->parent);
		ifp->parent = parent;
		return 0;
	}

	/* bridge ports do not link the bridge via master under older kernels  */
	if(nc_sysfs_get_bridge_parent(ifp->name, &parent) == 0) {
		nc_string_free(&ifp->parent);
		ifp->parent = parent;
		return 0;
	}
	return 0;
}

static void
__nc_discover_bootproto(nc_interface_t *ifp)
{
	char if_file[PATH_MAX + 1] = {'\0'};
	nc_sysconfig_t *sc;
	nc_var_t       *v;

	assert(ifp);

	if(ifp->parent) {
		ifp->ipv4.addrconf = 0; /*NC_ADDRCONF_DISABLED;*/
		ifp->ipv6.addrconf = 0; /*NC_ADDRCONF_DISABLED;*/
		return;
	}

	snprintf(if_file, sizeof(if_file), "/dev/.sysconfig/network/if-%s", ifp->name);
	sc = nc_sysconfig_read(if_file);
	if(sc) {
		v = nc_sysconfig_get(sc, "dhcp4_client");
		if(v && nc_string_len(v->value)) {
			ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
		}
		v = nc_sysconfig_get(sc, "dhcp6_client");
		if(v && nc_string_len(v->value)) {
			ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
		}
		nc_sysconfig_free(sc);
	}
	return;
}

static int
__suse_system_read_type(nc_interface_t *ifp)
{
	if(!ifp || !ifp->name)
		return -1;

	if(ifp->type != NC_IFTYPE_UNKNOWN)
		return 0;

	return nc_discover_iface_type(ifp->name, &ifp->type, &ifp->arp_type);
}

static int
__suse_system_read_state(nc_interface_t *ifp)
{
	char *value = NULL;
	int ret = 0;

	if(!ifp || !ifp->name) {
		return -1;
	}

	if(__suse_system_read_type(ifp) < 0) {
		return -1;
	}

	switch(ifp->type) {
	case NC_IFTYPE_BRIDGE:
		ret = __nc_discover_bridge(ifp);
	break;
	case NC_IFTYPE_BOND:
		ret = __nc_discover_bond(ifp);
	break;
	case NC_IFTYPE_VLAN:
		ret = __nc_discover_vlan(ifp);
	break;
	case NC_IFTYPE_INFINIBAND:
		ret = __nc_discover_infiniband(ifp);
	break;
	default:
	break;
	}
	if(ret < 0) {
		nc_error("Unable to discover interface details %s", ifp->name);
		return ret;
	}

	switch(ifp->type) {
	case NC_IFTYPE_BRIDGE:
	case NC_IFTYPE_BOND:
	case NC_IFTYPE_VLAN:
	case NC_IFTYPE_TAP:
	case NC_IFTYPE_DUMMY:
	case NC_IFTYPE_ETHERNET:
	case NC_IFTYPE_INFINIBAND:
		ret = __nc_discover_parent(ifp);
	break;

	default:
	break;
	}
	if(ret < 0)
		return ret;

	/* TODO: double check use of 'none' */
	__nc_discover_bootproto(ifp);

	if(nc_sysfs_netif_get_string(ifp->name, "address", &value) < 0) {
		nc_warn("failed to read hwaddr from sysfs");
	} else if(value && nc_string_len(value)) {
		nc_hw_address_parse(&ifp->hwaddr, ifp->type, value);
	}
	nc_string_free(&value);

	return ret;
}

void
nc_ifcfg_trace_dump(const nc_sysconfig_t *sc)
{
	unsigned int i;
	const nc_var_t *v;

	if(!sc)
		return;

	nc_trace("==> %s", sc->pathname);
	for(i = 0, v = sc->vars.data; i < sc->vars.count; ++i, ++v) {
		if(!v->name || !v->name[0])
			continue;
		if(v->value) {
			nc_trace("\tset %s='%s'", v->name, v->value);
		} else {
			nc_trace("\tdel %s=", v->name);
		}
	}
}

int
__build_indexed_name(nc_stringbuf_t *nsb, const char *prefix, unsigned int index)
{
	if(index) {
		if(nc_stringbuf_printf(nsb, "%s_%u", prefix, index) <= 0)
			return -1;
	} else {
		if(nc_stringbuf_puts(nsb, prefix) < 0)
			return -1;
	}
	return 0;
}

int
__nc_suse_config_vlan2ifcfg(nc_handle_t *nh, nc_interface_t *nfp, nc_sysconfig_t *sc)
{
	nc_vlan_t *vlan;
	char *ptr;
	size_t len;
	unsigned int tag;
	nc_interface_t *ifp;

	if(!nh || !sc || !nfp || !nfp->name || !(vlan = nfp->vlan)) {
		nc_error("No VLAN specific configuration found for %s", nfp->name);
		return -1;
	}

	if(nc_string_eq(nfp->name, vlan->interface_name)) {
		nc_error("VLAN interface %s can't refer to %s (itself) base interface",
			vlan->interface_name, nfp->name);
		return -1;
	}

	ptr = strrchr(nfp->name, '.');
	if(ptr) {
		/* vlan / ib reserved <parent>.<tag> scheme */
		if(nc_parse_uint(ptr + 1, &tag, 10) != 0) {
			nc_error("Unable to parse '%s' as VLAN tag from interface name '%s'"
				 " (reserved name scheme)", ptr + 1, nfp->name);
			return -1;
		}

		if(tag != vlan->tag) {
			nc_error("VLAN tag %u does not match tag %u encoded in interface name"
			         " (reserved name scheme)", vlan->tag, tag);
			return -1;
		}

		len = (size_t)(ptr - nfp->name);
		if(strncmp(nfp->name, vlan->interface_name, len) != 0) {
			nc_error("VLAN interface name %s does not contain base interface %s"
			         " (reserved name scheme)", nfp->name, vlan->interface_name);
			return -1;
		}

		if(nfp->type == NC_IFTYPE_INFINIBAND)
			return 0;

		if(nc_sysconfig_set(sc, "ETHERDEVICE", vlan->interface_name) < 0) {
			nc_error("Unable to set ETHERDEVICE=%s variable for VLAN interface %s",
				 vlan->interface_name, nfp->name);
			return -1;
		}
	} else {
		/* ib interfaces do not support anything else */
		if(nfp->type == NC_IFTYPE_INFINIBAND) {
			nc_error("Infiniband interface supports only <name>.<number> scheme");
			return -1;
		}

		/* it is either vlan<tag> scheme or some custom name */
		len = nc_string_len(nfp->name);
		while(len > 0 && isdigit((unsigned char)nfp->name[len - 1])) {
			ptr = &nfp->name[--len];
		}
		if(!(ptr && nc_parse_uint(ptr, &tag, 10) == 0 && vlan->tag == tag)) {
			if(nc_sysconfig_set_uint(sc, "VLAN_ID", vlan->tag) < 0) {
				nc_error("Unable to set VLAN_ID=%u variable for VLAN interface %s",
					 vlan->tag, nfp->name);
				return -1;
			}
		}

		if(nc_sysconfig_set(sc, "ETHERDEVICE", vlan->interface_name) < 0) {
			nc_error("Unable to set ETHERDEVICE=%s variable for VLAN interface %s",
				vlan->interface_name, nfp->name);
			return -1;
		}
	}

	/* vlan does not recurse, create new config when not already there */
	ifp = nc_interface_find(nh, NULL, -1, vlan->interface_name, 1, 0);
	if(!ifp) {
		ifp = nc_interface_new(vlan->interface_name, 0);
		if(nc_interfaces_add(nh, ifp) < 0) {
			if(ifp)
				nc_interface_free(ifp);
			return -1;
		}
		ifp->modified  = 1;
		ifp->deleted   = 0;
	}
	/* vlan base needs same start mode (manual, auto) and at least static bootproto */
	ifp->startmode = nfp->startmode;
	if(!ifp->ipv4.addrconf)
		ifp->ipv4.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
	if(!ifp->ipv6.addrconf)
		ifp->ipv6.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);

	return 0;
}


int
__nc_suse_config_bonding2ifcfg(nc_handle_t *nh, nc_interface_t *nfp, nc_sysconfig_t *sc)
{
	nc_stringbuf_t name  = NC_STRINGBUF_INIT;
	nc_bonding_t *bonding = nfp->bonding;
	unsigned int i;

	if(nc_sysconfig_set(sc, "BONDING_MASTER", "yes") < 0)
		goto error;

	nc_sysconfig_del_matching(sc, "BONDING_SLAVE");
	for (i = 0; i < bonding->slave_names.count; ++i) {
		if(!bonding->slave_names.data[i])
			goto error;

		/* always use index here */
		if(nc_stringbuf_printf(&name, "%s_%u", "BONDING_SLAVE", i) <= 0)
			goto error;

		if(nc_sysconfig_set(sc, name.string, bonding->slave_names.data[i]) < 0)
			goto error;
	}

	/*
	** FIXME: verify options for each mode
	*/
	nc_bonding_build_module_options(bonding);
	if(nc_sysconfig_set(sc, "BONDING_MODULE_OPTS", bonding->module_opts) < 0)
		goto error;

	// now mark all slaves modified and set parent
	for (i = 0; i < bonding->slave_names.count; ++i) {
		const char *slave = bonding->slave_names.data[i];
		nc_interface_t *ifp;

		ifp = nc_interface_find(nh, NULL, -1, slave, 1, 0);
		if(!ifp) {
			ifp = nc_interface_new(slave, 0);
			if(nc_interfaces_add(nh, ifp) < 0) {
				if(ifp)
					nc_interface_free(ifp);
				goto error;
			}
			ifp->deleted = 0;
			ifp->modified = 1;
		}
		if(nfp->startmode == NC_STARTMODE_MANUAL ||
		   nfp->startmode == NC_STARTMODE_HOTPLUG) {
			ifp->startmode = nfp->startmode;
		} else {
			ifp->startmode = NC_STARTMODE_AUTO;
		}
		ifp->ipv4.addrconf = 0; /*NC_ADDRCONF_DISABLED;*/
		ifp->ipv6.addrconf = 0; /*NC_ADDRCONF_DISABLED;*/
		nc_string_dup(&ifp->parent, nfp->name);
	}

	nc_stringbuf_destroy(&name);
	return 0;
error:
	nc_stringbuf_destroy(&name);
	return -1;
}

int
__nc_suse_config_bridge2ifcfg(nc_handle_t *nh, nc_interface_t *nfp, nc_sysconfig_t *sc)
{
	nc_stringbuf_t pname_buf  = NC_STRINGBUF_INIT;
	nc_stringbuf_t pprio_buf  = NC_STRINGBUF_INIT;
	nc_stringbuf_t pcost_buf  = NC_STRINGBUF_INIT;
	nc_bridge_t *bridge;
	unsigned int i;
	char *value = NULL;
	int ret = -1;

	if(!nh || !sc || !nfp || !(bridge = nfp->bridge))
		return ret;

	if(nc_sysconfig_set(sc, "BRIDGE", "yes") < 0)
		return ret;

	if (nc_bridge_get_stp(bridge, &value) >= 0) {
		if(nc_sysconfig_set(sc, "BRIDGE_STP", value) < 0)
			goto failure;
		nc_string_free(&value);
	}
	if (nc_bridge_get_forward_delay(bridge, &value) >= 0) {
		if(nc_sysconfig_set(sc, "BRIDGE_FORWARDDELAY", value) < 0)
			goto failure;
		nc_string_free(&value);
	}
	if (nc_bridge_get_ageing_time(bridge, &value) >= 0) {
		if(nc_sysconfig_set(sc, "BRIDGE_AGEINGTIME", value) < 0)
			goto failure;
		nc_string_free(&value);
	}
	if (nc_bridge_get_hello_time(bridge, &value) >= 0) {
		if(nc_sysconfig_set(sc, "BRIDGE_HELLOTIME", value) < 0)
			goto failure;
		nc_string_free(&value);
	}
	if (nc_bridge_get_priority(bridge, &value) >= 0) {
		if(nc_sysconfig_set(sc, "BRIDGE_PRIORITY", value) < 0)
			goto failure;
		nc_string_free(&value);
	}
	if (nc_bridge_get_max_age(bridge, &value) >= 0) {
		if(nc_sysconfig_set(sc, "BRIDGE_MAXAGE", value) < 0)
			goto failure;
		nc_string_free(&value);
	}

#if 1
	for (i = 0; i < bridge->ports.count; ++i) {
		const char *port = bridge->ports.data[i]->name;
		if(!port)
			goto failure;

		if(i) {
			if(nc_stringbuf_putc(&pname_buf, ' ') < 0)
				goto failure;
			if(nc_stringbuf_putc(&pprio_buf, ' ') < 0)
				goto failure;
			if(nc_stringbuf_putc(&pcost_buf, ' ') < 0)
				goto failure;
		}
		if(nc_stringbuf_puts(&pname_buf, port) < 0)
			goto failure;

		if (nc_bridge_port_get_priority(bridge, port, &value) > 0) {
			if(nc_stringbuf_puts(&pprio_buf, value) < 0)
				goto failure;
			nc_string_free(&value);
		} else {
			/* needs 11.4 / sle11-sp2 */
			if(nc_stringbuf_putc(&pprio_buf, '-') < 0)
				goto failure;
		}
		if (nc_bridge_port_get_path_cost(bridge, port, &value) > 0) {
			if(nc_stringbuf_puts(&pcost_buf, value) < 0)
				goto failure;
			nc_string_free(&value);
		} else {
			/* needs 11.4 / sle11-sp2 */
			if(nc_stringbuf_putc(&pcost_buf, '-') < 0)
				goto failure;
		}
	}
#else
	for (i = 0; i < bridge->ports.count; ++i) {
		const char *port = bridge->ports.data[i]->name;
		if(!port)
			goto failure;

		if(i) {
			if(nc_stringbuf_putc(&pname_buf, ' ') < 0)
				goto failure;
		}
		if(nc_stringbuf_puts(&pname_buf, port) < 0)
			goto failure;
	}
	for (i = 0; i < bridge->ports.count; ++i) {
		const char *port = bridge->ports.data[i]->name;
		if (nc_bridge_port_get_priority(bridge, port, &value) <= 0)
			break;
		if(i) {
			if(nc_stringbuf_putc(&pprio_buf, ' ') < 0)
				goto failure;
		}
		if(nc_stringbuf_puts(&pprio_buf, value) < 0)
			goto failure;
		nc_string_free(&value);
	}
	for (i = 0; i < bridge->ports.count; ++i) {
		const char *port = bridge->ports.data[i]->name;
		if (nc_bridge_port_get_path_cost(bridge, port, &value) <= 0)
			break;
		if(i) {
			if(nc_stringbuf_putc(&pcost_buf, ' ') < 0)
				goto failure;
		}
		if(nc_stringbuf_puts(&pcost_buf, value) < 0)
			goto failure;
		nc_string_free(&value);
	}
#endif
	if(nc_sysconfig_set(sc, "BRIDGE_PORTS",          pname_buf.string) < 0)
		goto failure;
	if(nc_sysconfig_set(sc, "BRIDGE_PORTPRIORITIES", pprio_buf.string) < 0)
		goto failure;
	if(nc_sysconfig_set(sc, "BRIDGE_PATHCOSTS",      pcost_buf.string) < 0)
		goto failure;

	// now mark all ports modified and set parent
	for (i = 0; i < bridge->ports.count; ++i) {
		nc_bridge_port_t * port = bridge->ports.data[i];
		nc_interface_t *ifp;

		assert(port && port->name);

		ifp = nc_interface_find(nh, NULL, -1, port->name, 1, 0);
		if(!ifp) {
			ifp = nc_interface_new(port->name, 0);
			if(nc_interfaces_add(nh, ifp) < 0) {
				if(ifp)
					nc_interface_free(ifp);
				goto failure;
			}
			ifp->deleted = 0;
			ifp->modified = 1;
		}
		if(ifp->type == NC_IFTYPE_VLAN) {
			ifp->startmode = NC_STARTMODE_AUTO;
			ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
		} else {
			if(nfp->startmode == NC_STARTMODE_MANUAL ||
			   nfp->startmode == NC_STARTMODE_HOTPLUG) {
				ifp->startmode = nfp->startmode;
			} else {
				ifp->startmode = NC_STARTMODE_AUTO;
			}
			ifp->ipv4.addrconf = 0; /*NC_ADDRCONF_DISABLED;*/
			ifp->ipv6.addrconf = 0; /*NC_ADDRCONF_DISABLED;*/
		}
		nc_string_dup(&ifp->parent, nfp->name);
	}

	// OK, done
	ret = 0;

failure:
	nc_string_free(&value);
	nc_stringbuf_destroy(&pname_buf);
	nc_stringbuf_destroy(&pprio_buf);
	nc_stringbuf_destroy(&pcost_buf);
	return ret;
}

static int
__nc_suse_config_addrs2ifcfg(nc_handle_t *nh, nc_interface_t *nfp, nc_sysconfig_t *sc)
{
	unsigned int i, index;

	if(!nh || !nfp || !sc)
		return -1;

	nc_sysconfig_del_matching(sc, "IPADDR");
	nc_sysconfig_del_matching(sc, "SCOPE");
	nc_sysconfig_del_matching(sc, "LABEL");
	nc_sysconfig_del_matching(sc, "NETWORK");
	nc_sysconfig_del_matching(sc, "NETMASK");
	nc_sysconfig_del_matching(sc, "BROADCAST");
	nc_sysconfig_del_matching(sc, "PREFIXLEN");
	nc_sysconfig_del_matching(sc, "REMOTE_IPADDR");
	nc_sysconfig_del_matching(sc, "IP_OPTIONS");

	/* bonding slave or bridge port -> no IP config
	** [even there may be some for route cleanup]...
	*/
	if(nfp->parent)
		return 0;

	/* in case there aren't any, init empty one */
	nc_sysconfig_set(sc, "IPADDR", "");

	for(i = 0, index = 0; i < nfp->addrs.count; ++i) {
		nc_stringbuf_t name  = NC_STRINGBUF_INIT;
		nc_stringbuf_t addr  = NC_STRINGBUF_INIT;
		nc_stringbuf_t value = NC_STRINGBUF_INIT;
		nc_address_t * ap    = nfp->addrs.data[i];

		if(!ap || ap->config_method != NC_ADDRCONF_STATIC) 
			continue;

		if(!ap->prefixlen ||
		   (ap->prefixlen > nc_address_bits(ap->family)))
			continue;

		if(nc_address_format(&ap->local_addr, &addr) == NULL)
			goto next_addr;

		if(nc_stringbuf_printf(&value, "%s/%u", addr.string,
							ap->prefixlen) <= 0)
				goto next_addr;

		if(__build_indexed_name(&name, "IPADDR", index++) < 0)
			goto next_addr;

		nc_sysconfig_set(sc, name.string, value.string);

	next_addr:
		nc_stringbuf_destroy(&name);
		nc_stringbuf_destroy(&addr);
		nc_stringbuf_destroy(&value);
	}
	return 0;
}

#if 0
static int
__nc_interface_get_slave_names(nc_handle_t *nh, nc_string_array_t *slaves, const char *parent)
{
	unsigned int i, count;
	nc_interface_t *sfp = NULL;

	assert(nh && slaves && parent);

	count = slaves->count;
	for(i = 0; i < nh->ifaces.count; ++i) {
		sfp = nh->ifaces.data[i];
		if(!sfp || !sfp->name)
			continue;

		if(nc_string_eq(parent, sfp->parent)) {
			if(nc_string_array_append(slaves, sfp->name) < 0)
				return -1;
		}
	}
	return slaves->count - count;
}
#endif

static int
__nc_suse_config_do_write_routes(const nc_route_array_t *routes, FILE *fpo)
{
	unsigned int i;

	if(!routes || !fpo)
		return -1;

	for(i = 0; i < routes->count; ++i) {
		//nc_stringbuf_t addr  = NC_STRINGBUF_INIT;
		nc_stringbuf_t dest  = NC_STRINGBUF_INIT;
		nc_stringbuf_t gway  = NC_STRINGBUF_INIT;
		nc_route_t *   rp    = routes->data[i];

		if(!rp /* || rp->config_method != NC_ADDRCONF_STATIC */)
			continue;

		/*
		 * we should not need most of the checks ...
		 * xml syntax (input) should catch it.
		 */
		if(rp->prefixlen > nc_address_bits(rp->family))
			continue;

		if(rp->nh.gateway.ss_family == AF_UNSPEC && !rp->nh.device)
			continue;

		if(rp->prefixlen == 0) {
			nc_stringbuf_puts(&dest, "default");
		} else {
			if(nc_address_format(&rp->destination, &dest) == NULL)
				goto next_route;
		}
		if(rp->nh.gateway.ss_family != AF_UNSPEC) {
			if(nc_address_format(&rp->nh.gateway, &gway) == NULL)
				goto next_route;
		} else if(rp->prefixlen == 0) {
			/* no family visible in "default - - eth0" */
			if(rp->family == AF_INET)
				nc_stringbuf_puts(&gway, "0.0.0.0");
			else
				nc_stringbuf_puts(&gway, "::");
		} else {
			nc_stringbuf_puts(&gway, "-");
		}

		if(rp->prefixlen == 0) {
#if defined(NC_OPT_TRACE)
			nc_trace("route: %s %s %s %s",
				dest.string, gway.string, "-",
				 (rp->nh.device ? rp->nh.device : "-"));
#endif
			fprintf(fpo, "%s %s %s %s\n",
				dest.string, gway.string, "-",
				(rp->nh.device ? rp->nh.device : "-"));
		} else {
#if defined(NC_OPT_TRACE)
			nc_trace("route: %s/%u %s %s %s",
				dest.string, rp->prefixlen, gway.string, "-",
				 (rp->nh.device ? rp->nh.device : "-"));
#endif
			fprintf(fpo, "%s/%u %s %s %s\n",
				dest.string, rp->prefixlen, gway.string, "-",
				(rp->nh.device ? rp->nh.device : "-"));
		}

	next_route:
		nc_stringbuf_destroy(&dest);
		nc_stringbuf_destroy(&gway);
	}

	return 0;
}

static int
__nc_suse_config_do_write_ifroutes(const nc_interface_array_t *nfps, FILE *fpo)
{
	unsigned int i;

	if(!nfps || !fpo)
		return -1;

	for(i = 0; i < nfps->count; ++i) {
		nc_interface_t *nfp = nfps->data[i];
		if(nfp && __nc_suse_config_do_write_routes(&nfp->routes, fpo))
			return -1;
	}
	return 0;
}

static int
__nc_suse_config_do_rewrite_routes(nc_handle_t *nh, nc_interface_array_t *nfps,
                                   FILE *fpo, FILE *fpi, const char *file)
{
	nc_route_array_t written = NC_ROUTE_ARRAY_INIT;
	nc_route_array_t dropped = NC_ROUTE_ARRAY_INIT;
	char rbuff[BUFSIZ] = {'\0'};
	char pbuff[BUFSIZ] = {'\0'};
	unsigned int rline = 0, wline = 0;
	unsigned int i, match;

	if(!nh || !nfps || !fpo || !fpi) {
		return -1;
	}

	while (fgets(rbuff, sizeof(rbuff), fpi) != NULL) {
		nc_route_t *rp = NULL;
		int ret;

		memcpy(pbuff, rbuff, sizeof(pbuff));

		rline++;
		pbuff[strcspn(pbuff, "#\r\n")] = '\0';

		rp = NULL;
		ret = __nc_suse_config_parse_route_line(pbuff, &rp, NULL,
							file, rline);
		if(ret != 0) {
			if(ret == 1) {
				/* empty line or similar */
	just_copy:
				fputs(rbuff, fpo);
				wline++;
			}
			continue;
		}

		if(nc_string_len(rp->nh.device)) {
			/*
			** matches the interface name of a new interface
			** -> drop, they'll be added later
			*/
			for(match = 0, i = 0; !match && i < nfps->count; ++i) {
				nc_interface_t *nfp = nfps->data[i];
				if(!nfp || !nfp->name)
					continue;
				match = nc_string_eq(rp->nh.device, nfp->name);
			}
			if(match) {
				nc_route_array_append(&dropped, rp);
				continue;
			}

			/*
			** matches the name of a deleted or modified
			** (=old config) interface
			*/
			for(match = 0, i = 0; !match && i < nh->ifaces.count; i++) {
				nc_interface_t *ifp = nh->ifaces.data[i];
				if(!ifp || !ifp->name)
					continue;

				if(!ifp->deleted && !ifp->modified)
					continue;

				match = nc_string_eq(rp->nh.device, ifp->name);
			}
			if(match) {
				nc_route_array_append(&dropped, rp);
				continue;
			}

			// some anothter interface, just copy it
			nc_route_array_append(&written, rp);
			goto just_copy;
		}

		/* match against already written _interface_ routes */
		for(match = 0, i = 0; !match && i < written.count; ++i) {
			nc_route_t *wr = written.data[i];
			if(!wr || wr->family != rp->family)
				continue;
			if(wr->nh.gateway.ss_family != AF_UNSPEC || !wr->nh.device)
				continue;

			match = nc_address_prefix_match(wr->prefixlen,
					&wr->destination, &rp->nh.gateway);
		}
		if(match) {
			// OK, then write it too
			nc_route_array_append(&written, rp);
			goto just_copy;
		}

		/* match against already dropped _interface_ routes */
		for(match = 0, i = 0; !match && i < dropped.count; ++i) {
			nc_route_t *dr = dropped.data[i];
			if(!dr || dr->family != rp->family)
				continue;
			if(dr->nh.gateway.ss_family != AF_UNSPEC || !dr->nh.device)
				continue;

			match = nc_address_prefix_match(dr->prefixlen,
					&dr->destination, &rp->nh.gateway);
		}
		if(match) {
			nc_route_array_append(&dropped, rp);
			continue;
		}

		/* match against routes/IPs of deleted/modified interfaces */
		for(match = 0, i = 0; !match && i < nh->ifaces.count; i++) {
			nc_interface_t *ifp = nh->ifaces.data[i];
			unsigned int j = 0;

			if(!ifp || !ifp->name)
				continue;

			if(!ifp->deleted && !ifp->modified)
				continue;

			/* exact route destination match ?? */
			(void)j;
			/*
			** it would throw away routes for same destination
			** (without explicit interface)
			** at the moment we just use them ... but they may
			** (should) be in nh->routes: match them?
			**
			for(j = 0; !match && j < ifp->routes.count; ++j) {
				nc_route_t *ir = ifp->routes.data[j];
				if(!ir || ir->family != rp->family)
					continue;
				match = nc_address_equal(&ir->destination,
				                         &rp->destination);
			}
			*/

			/* reachable through the network of this IP */
			if(!match) {
				match = nc_address_can_reach_one(&ifp->addrs,
				                                 &rp->nh.gateway);
			}
		}
		if(match) {
			nc_route_array_append(&dropped, rp);
			continue;
		}

		// OK, does not match any of the interfaces we handle
		nc_route_array_append(&written, rp);
		goto just_copy;
	}

	// Now write down the routes of new/modified interfaces
	if(__nc_suse_config_do_write_ifroutes(nfps, fpo) < 0)
		goto failure;

	nc_route_array_destroy(&written);
	nc_route_array_destroy(&dropped);
	return 0;

failure:
	nc_route_array_destroy(&written);
	nc_route_array_destroy(&dropped);
	return -1;
}

static int
__nc_suse_config_rewrite_routes(nc_handle_t *nh, nc_interface_array_t *nfps,
				const char *dot_backup)
{
	nc_config_handle_t *  ch  = __to_config_handle(nh);
	nc_stringbuf_t base_path  = NC_STRINGBUF_INIT;
	nc_stringbuf_t file_name  = NC_STRINGBUF_INIT;
	nc_stringbuf_t old_path   = NC_STRINGBUF_INIT;
	nc_stringbuf_t tmp_path   = NC_STRINGBUF_INIT;
	const char *   tmp_suffix = ".XXXXXX";
	FILE *fpi = NULL, *fpo = NULL;
	int  fd;

	if((__nc_suse_config_prepend_root(ch, _SYSCONFIG_NETWORK_DIR, &base_path) < 0) ||
	   (nc_stringbuf_puts(&file_name, base_path.string) < 0) ||
	   (nc_stringbuf_putc(&file_name, '/') < 0)              ||
	   (nc_stringbuf_puts(&file_name, _ROUTES_GLOBAL) < 0)) {
		nc_error("Unable to construct routes file name");
		goto failure;
	}

	if(nc_make_dot_suffix(&tmp_path, file_name.string, tmp_suffix) < 0) {
		nc_error("Unable to apply a temp suffix to filename: %s",
				file_name.string);
		goto failure;
	}

	if(dot_backup &&
	   (nc_make_dot_suffix(&old_path, file_name.string, dot_backup) < 0)) {
		nc_error("Unable to apply a backup suffix to filename: %s",
				file_name.string);
		goto failure;
	}

	fd = mkstemp(tmp_path.string);
	if(fd == -1) {
		nc_error("Unable to create temp file %s for writing: %m",
				tmp_path.string);
		goto failure;
	}
	fpo = fdopen(fd, "we");
	if (fpo == NULL) {
		close(fd);
		nc_error("Unable to open %s file descriptor for writing: %m",
				tmp_path.string);
		goto failure;
	}

	fpi = fopen(file_name.string, "re");
	if (fpi == NULL) {
		if (errno != ENOENT) {
			nc_error("Unable to open file '%s' for reading: %m",
				file_name.string);
			goto failure;
		}
		if(__nc_suse_config_do_write_ifroutes(nfps, fpo) < 0)
			goto failure;
	} else {
		if(__nc_suse_config_do_rewrite_routes(nh, nfps, fpo, fpi,
				file_name.string) < 0)
			goto failure;
	}
	fflush(fpo);

	if(fpi && old_path.string) {
		unlink(old_path.string);
		if(link(file_name.string, old_path.string) != 0) {
			nc_warn("Unable to create a backup of %s: %m",
					file_name.string);
			nc_stringbuf_clear(&old_path);
		}
	}

	if (rename(tmp_path.string, file_name.string) < 0) {
		nc_error("Unable to rename %s to %s: %m",
				tmp_path.string, file_name.string);
		goto failure;
	}

	if(fpi)
		fclose(fpi);
	fclose(fpo);
	nc_stringbuf_destroy(&tmp_path);
	nc_stringbuf_destroy(&old_path);
	nc_stringbuf_destroy(&file_name);
	nc_stringbuf_destroy(&base_path);
	return 0;

failure:
	if(fpi) {
		if(old_path.string)
			unlink(old_path.string);
		fclose(fpi);
	}
	if(fpo) {
		if(tmp_path.string)
			unlink(tmp_path.string);
		fclose(fpo);
	}
	nc_stringbuf_destroy(&tmp_path);
	nc_stringbuf_destroy(&old_path);
	nc_stringbuf_destroy(&file_name);
	nc_stringbuf_destroy(&base_path);
	return -1;
}

static int /* bool */
__hwaddr_differs_from_default(nc_interface_t *nfp)
{
	nc_hwaddr_t hwaddr;
	char *value = NULL;

	if(nc_udev_net_rule_hwaddr_for_name(nfp->name, &value) == 0 && value) {
		nc_hw_address_parse(&hwaddr, nfp->type, value);
		nc_string_free(&value);
		if(nc_hw_address_equal(&hwaddr, &nfp->hwaddr))
			return 0;
	}

	if(nc_sysfs_netif_get_string(nfp->name, "address", &value) == 0 && value) {
		nc_hw_address_parse(&hwaddr, nfp->type, value);
		nc_string_free(&value);
		if(nc_hw_address_equal(&hwaddr, &nfp->hwaddr))
			return 0;
	}

	return 1;
}

static int
__nc_suse_config_iface2ifcfg(nc_handle_t *nh, nc_interface_t *nfp, nc_sysconfig_t *nsc)
{
	nc_sysconfig_del(nsc, "INTERFACETYPE");

	if(!nc_sysconfig_get(nsc, "USERCONTROL")) {
		nc_sysconfig_set(nsc, "USERCONTROL", "no");
	}

	nc_sysconfig_set(nsc, "STARTMODE", __suse_ifcfg_startmode_get(nfp->startmode));
	nc_sysconfig_set(nsc, "BOOTPROTO", __suse_ifcfg_bootproto_get(nfp));

	switch(nfp->hwaddr.type) {
	case NC_IFTYPE_BOND:
		if(nfp->bonding->slave_type != NC_IFTYPE_INFINIBAND &&
		   nfp->hwaddr.len && __hwaddr_differs_from_default(nfp)) {
			nc_stringbuf_t mac = NC_STRINGBUF_INIT;
			nc_sysconfig_set(nsc, "LLADDR", nc_hw_address_format(&nfp->hwaddr, &mac));
			nc_stringbuf_destroy(&mac);
		} // else keep when already in old config
	break;
	case NC_IFTYPE_VLAN:
	case NC_IFTYPE_ETHERNET:
		if(nfp->hwaddr.len && __hwaddr_differs_from_default(nfp)) {
			nc_stringbuf_t mac = NC_STRINGBUF_INIT;
			nc_sysconfig_set(nsc, "LLADDR", nc_hw_address_format(&nfp->hwaddr, &mac));
			nc_stringbuf_destroy(&mac);
		} // else keep when already in old config
	break;

	case NC_IFTYPE_UNKNOWN:
	case NC_IFTYPE_BRIDGE:
	case NC_IFTYPE_LOOPBACK:
	case NC_IFTYPE_INFINIBAND:	/* read-only */
	default:
		nc_sysconfig_del(nsc, "LLADDR");
	break;
	}

	if(nfp->mtu) {
		nc_sysconfig_set_uint(nsc, "MTU", nfp->mtu);
	}

	switch(nfp->type) {
		case NC_IFTYPE_VLAN:
			if(nfp->startmode != NC_STARTMODE_MANUAL &&
			   nfp->startmode != NC_STARTMODE_AUTO) {
				nc_sysconfig_set(nsc, "STARTMODE", "auto");
			}
			nfp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			nfp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			nc_sysconfig_del_matching(nsc, "BRIDGE");
			nc_sysconfig_del_matching(nsc, "BONDING");
			if(nfp->bonding || nfp->bridge || !nfp->vlan)
				goto failure;
			if(__nc_suse_config_vlan2ifcfg(nh, nfp, nsc))
				goto failure;
		break;
		case NC_IFTYPE_BOND:
			if(nfp->startmode != NC_STARTMODE_MANUAL &&
			   nfp->startmode != NC_STARTMODE_AUTO) {
				nc_sysconfig_set(nsc, "STARTMODE", "auto");
			}
			if(!nfp->ipv4.addrconf)
				nfp->ipv4.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			if(!nfp->ipv6.addrconf)
				nfp->ipv6.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			nc_sysconfig_del(nsc, "ETHERDEVICE");
			nc_sysconfig_del_matching(nsc, "BRIDGE");
			if(nfp->vlan || nfp->bridge || !nfp->bonding)
				goto failure;
			if(__nc_suse_config_bonding2ifcfg(nh, nfp, nsc))
				goto failure;
		break;
		case NC_IFTYPE_BRIDGE:
			if(nfp->startmode != NC_STARTMODE_MANUAL &&
			   nfp->startmode != NC_STARTMODE_AUTO) {
				nc_sysconfig_set(nsc, "STARTMODE", "auto");
			}
			if(!nfp->ipv4.addrconf)
				nfp->ipv4.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			if(!nfp->ipv6.addrconf)
				nfp->ipv6.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			nc_sysconfig_del(nsc, "ETHERDEVICE");
			nc_sysconfig_del_matching(nsc, "BONDING");
			if(nfp->bonding || nfp->vlan || !nfp->bridge)
				goto failure;
			if(__nc_suse_config_bridge2ifcfg(nh, nfp, nsc))
				goto failure;
		break;

		default:
			nc_sysconfig_del(nsc, "ETHERDEVICE");
			nc_sysconfig_del_matching(nsc, "BRIDGE");
			nc_sysconfig_del_matching(nsc, "BONDING");
			if(nfp->type != NC_IFTYPE_INFINIBAND && nfp->vlan)
				goto failure;
			if(nfp->bonding || nfp->bridge)
				goto failure;
		break;
	}

	if(__nc_suse_config_addrs2ifcfg(nh, nfp, nsc) < 0)
		goto failure;

	return 0;
failure:
	return -1;
}

static const char *
__nc_interface_in_use(nc_handle_t *nh, nc_interface_t *ifp, const char *except)
{
	unsigned int i, n;
	nc_interface_t *pfp = NULL;

	assert(nh && ifp && ifp->name);

	/* when parent is set, just return it */
	if(ifp->parent) {
		if(!except || !nc_string_eq(ifp->parent, except))
			return ifp->parent;
	}

	/* otherwise search where ifp->name is used */
	for(pfp=nc_interface_first(nh, &i); pfp; pfp=nc_interface_next(nh, &i)) {

		if(nc_string_eq(ifp->name, pfp->name))
			continue;

		if(pfp->modified || pfp->deleted)
			continue;

		if(pfp->bonding) {
			for(n = 0; n < pfp->bonding->slave_names.count; ++n) {
				const char *s = pfp->bonding->slave_names.data[n];
				if(s && nc_string_eq(ifp->name, s)) {
					if(!except || !nc_string_eq(pfp->name, except))
						return pfp->name;
				}
			}
		}
		if(pfp->bridge) {
			for(n = 0; n < pfp->bridge->ports.count; ++n) {
				nc_bridge_port_t *p = pfp->bridge->ports.data[n];
				if(p && nc_string_eq(ifp->name, p->name)) {
					if(!except || !nc_string_eq(pfp->name, except))
						return pfp->name;
				}
			}
		}
		if(pfp->vlan) {
			if(nc_string_eq(ifp->name, pfp->vlan->interface_name)) {
				if(!except || !nc_string_eq(pfp->name, except))
					return pfp->name;
			}
		}

	}
	return NULL;
}

static const char *
__nc_interface_is_enslaved(nc_handle_t *nh, nc_interface_t *ifp)
{
	unsigned int i, n;
	nc_interface_t *pfp = NULL;

	assert(nh && ifp && ifp->name);

	/* when parent is set, just return it */
	if(ifp->parent) {
		return ifp->parent;
	}

	/* otherwise search where ifp->name is used */
	for(pfp=nc_interface_first(nh, &i); pfp; pfp=nc_interface_next(nh, &i)) {

		if(nc_string_eq(ifp->name, pfp->name))
			continue;

		if(pfp->modified || pfp->deleted)
			continue;

		if(pfp->bonding) {
			for(n = 0; n < pfp->bonding->slave_names.count; ++n) {
				const char *s = pfp->bonding->slave_names.data[n];
				if(s && nc_string_eq(ifp->name, s)) {
					return pfp->name;
				}
			}
		}
		if(pfp->bridge) {
			for(n = 0; n < pfp->bridge->ports.count; ++n) {
				nc_bridge_port_t *p = pfp->bridge->ports.data[n];
				if(p && nc_string_eq(ifp->name, p->name))
					return pfp->name;
			}
		}
	}
	return NULL;
}

static int /* bool */
__nc_suse_config_modify_allowed(const nc_interface_t *ifp, int quiet)
{
	if(!nc_string_len(ifp->name)) {
		if(!quiet)
			nc_error("Interface with a valid name required");
		return 0;
	}
	if(!__is_valid_ifname(ifp->name)) {
		if(!quiet)
			nc_error("Interface name '%s' contains suspect characters",
				ifp->name);
		return 0;
	}

	/*
	** do not allow to remove loopback,nfsroot,ibft interface config
	*/
	if(nc_string_eq(ifp->name, "all")) {
		if(!quiet)
			nc_error("Not allowed to modify interface \"%s\" -- it has a special meaning",
				ifp->name);
		return 0;
	}
	if(nc_string_eq(ifp->name, "lo")) {
		if(!quiet)
			nc_error("Not allowed to modify loopback interface %s config",
				ifp->name);
		return 0;
	}
	if(ifp->startmode == NC_STARTMODE_NFSROOT) {
		if(!quiet)
			nc_error("Not allowed to remove nfsroot interface %s config - use yast2",
				ifp->name);
		return 0;
	}
	if(ifp->startmode == NC_STARTMODE_IFPLUGD) {
		if(!quiet)
			nc_error("Not allowed to remove ifplugd interface %s config - use yast2",
				ifp->name);
		return 0;
	}
	if(NC_ADDRCONF_TEST(ifp->ipv4.addrconf, NC_ADDRCONF_IBFT) ||
	   NC_ADDRCONF_TEST(ifp->ipv6.addrconf, NC_ADDRCONF_IBFT)) {
		if(!quiet)
			nc_error("Not allowed to modify ibft interface %s config -- use ibft firmware",
				ifp->name);
		return 0;
	}
	return 1; /* OK */
}

static int
__nc_suse_config_interfaces_delete(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_config_handle_t *  ch = __to_config_handle(nh);
	nc_stringbuf_t base_path = NC_STRINGBUF_INIT;
	nc_stringbuf_t ifcfg_cur = NC_STRINGBUF_INIT;
	nc_stringbuf_t ifrte_cur = NC_STRINGBUF_INIT;
	nc_interface_array_t nfps = NC_INTERFACE_ARRAY_INIT;
	const char *parent = NULL;
	unsigned int i;
	int ret = -1;

	assert(nh && ifp && ifp->name);
	if(!(nh && ifp && ifp->name))
		return ret;

	/*
	** => do not allow to remove config of a slave/port
	**    or also of the etherdevice interface of a vlan
	*/
	parent = __nc_interface_in_use(nh, ifp, NULL);
	if(parent) {
		nc_error("Unable to delete: interface %s is in use by %s",
			ifp->name, parent);
		return ret;
	}

	if(__nc_suse_config_prepend_root(ch, _SYSCONFIG_NETWORK_DIR, &base_path) < 0)
		goto failure;

	/* OK, now go through and delete */
	for(i = 0; i < nh->ifaces.count; ++i) {
		nc_interface_t *dfp = nh->ifaces.data[i];

		/* paranoia ... ignore crap */
		if(!dfp || !dfp->name)
			continue;

		/* don't remove what were never requested to get removed */
		if(!dfp->deleted)
			continue;

		/* construct ifcfg-<name> file */
		if(nc_stringbuf_printf(&ifcfg_cur, "%s/%s%s", base_path.string,
						_IFCFG_PREFIX, dfp->name) <= 0) {
			nc_error("Unable to construct %s%s file name",
					_IFCFG_PREFIX, dfp->name);
			goto failure;
		}
		/* construct ifroute-<name> file */
		if(nc_stringbuf_printf(&ifrte_cur, "%s/%s%s", base_path.string,
						_IFROUTE_PREFIX, dfp->name) <= 0) {
			nc_error("Unable to construct %s%s file name",
						_IFROUTE_PREFIX, dfp->name);
			goto failure;
		}
		/* move to .ifcfg-<name>[suffix] file */
		if(nc_rename_to_dot_backup(ifcfg_cur.string, _DOT_BACKUP_SUFFIX) < 0) {
			nc_error("Unable to create a dot backup of filename: %s",
					ifcfg_cur.string);
			goto failure;
		}
		/* move to .ifroute-<name>[suffix] file */
		if(nc_rename_to_dot_backup(ifrte_cur.string, _DOT_BACKUP_SUFFIX) < 0) {
			nc_error("Unable to create a dot backup of filename: %s",
					ifrte_cur.string);
			goto failure;
		}
	}

	/* finally, rewrite routes config file (with empty nfps)... */
	/* [yast2 network still doesn't support ifroute-<iface>...] */
	if(__nc_suse_config_rewrite_routes(nh, &nfps, _DOT_BACKUP_SUFFIX) < 0)
		goto failure;

	/*
	** OK, all done
	*/
	ret = 0;
failure:
	nc_interface_array_destroy(&nfps);
	nc_stringbuf_destroy(&ifrte_cur);
	nc_stringbuf_destroy(&ifcfg_cur);
	nc_stringbuf_destroy(&base_path);
	return ret;
}

static int
__update_bond_slave_udev_rules(nc_interface_t *nfp, nc_interface_array_t *nfps)
{
	nc_var_array_t    bus_id_map = NC_VAR_ARRAY_INIT;
	int               ret = -1;
	unsigned int      i;

	/*
	 * We replace the MAC [ATTR{address}] match with a Bus-ID [KERNELS]
	 * match to support hotplugging of bonded ethernet slaves.
	 *
	 * - new device (replacement), gets a new name when matching by MAC:
	 *   -> bonding config does't match it any more
	 *
	 * There is hardware (e.g. mlx4) with multiple port interfaces with
	 * same Bus-ID [KERNELS] for all ports but a different ATTR{dev_id}.
	 * The replacing function checks that there is a match for dev_id
	 * (AFAIS, dev_id match is always there -- at least in 11.4 rules).
	 */
	if(nfp->type != NC_IFTYPE_BOND)
		return 0;

	/* Create a ifname -> bus-id "todo" map */
	for(i = 0; i < nfps->count; ++i) {
		unsigned int     cur_type = NC_IFTYPE_UNKNOWN;
		unsigned int     arp_type = ARPHRD_NONE;
		nc_interface_t  *sfp      = nfps->data[i];
		char            *bus_id   = NULL;

		/* skip the bonding interface itself */
		if(!sfp || nc_string_eq(nfp->name, sfp->name))
			continue;

		if(!nc_string_eq(nfp->name, sfp->parent))
			continue;

		if(nc_discover_iface_type(sfp->name, &cur_type, &arp_type) < 0) {
			continue;	/* assume it does not exists */
		}

		/* skip any update, we seem to use infiniband slaves */
		if(cur_type != NC_IFTYPE_ETHERNET) {
			ret = 1;
			goto cleanup;
		}

		/* can't get the bus-id? -> continue */
		if(nc_sysfs_netif_get_bus_id(sfp->name, &bus_id) < 0) {
			continue;
		}

		if(nc_var_array_set(&bus_id_map, sfp->name, bus_id) < 0) {
			free(bus_id);
			goto cleanup;
		}
		free(bus_id);
	}

	/* OK, lets update */
	if(bus_id_map.count && nc_udev_net_rule_hwaddr_to_bus_id(&bus_id_map) < 0)
		goto cleanup;

	ret = 0;

cleanup:
	nc_var_array_destroy(&bus_id_map);
	return ret;
}

static int
__nc_suse_config_interfaces_verify_config(nc_handle_t *nh, nc_interface_array_t *nfps, nc_interface_t *pfp, nc_interface_t *nfp);

inline static int /* bool */
__check_interface_container_type(unsigned int iftype, unsigned int type)
{
	switch(iftype) {
		case NC_IFTYPE_VLAN:
		case NC_IFTYPE_BOND:
		case NC_IFTYPE_BRIDGE:
			if(type == iftype)
		       		return 1;
		break;
		default:
		break;
	}
	return 0;
}

static int /* bool */
nc_check_interface_usage(nc_interface_array_t *guard, const char *ifname, unsigned int iftype)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	unsigned int i;

	if(guard->count > 0) {
		for(i = 0; i < guard->count; ++i) {
			nc_interface_t *p = guard->data[i];
			if(!p || !p->name)
				continue;

			if(i > 0)
				nc_stringbuf_puts(&buf, " -> ");
			nc_stringbuf_puts(&buf, p->name);

			if(__check_interface_container_type(iftype, p->type)) {
				nc_error("interface %s type %s already seen before %s",
					ifname, __suse_ifcfg_typename_get(iftype), buf.string);
				return 1;
			}
		}
		nc_stringbuf_puts(&buf, " -> ");
		nc_stringbuf_puts(&buf, ifname);
	}
	if(nc_interface_array_index_by_name(guard, ifname) != -1) {
		nc_error("interface already seen before: %s", buf.string);
		return 1;
	} else {
#if defined(NC_OPT_TRACE)
		nc_trace("interface chain: %s", buf.string ? buf.string : ifname);
#endif
		return 0;
	}
}

static int
__suse_config_interfaces_verify_bridge_port(nc_handle_t *nh, nc_interface_array_t *nfps, nc_interface_t *pfp, nc_interface_t *nfp, const char *port)
{
	nc_interface_t *  mfp;
	nc_interface_t *  cfp;

	(void)pfp;

	mfp = nc_interface_find(nh, NULL, -1, port, 1, 0);
	if(!mfp) {
		nc_error_detail_fmt(nh,
			"Port interface %s configuration missed", port);
		return -1;
	}

	if(__nc_suse_config_interfaces_verify_config(nh, nfps, nfp, mfp) < 0) {
		return -1;
	}

	switch(mfp->arp_type) {
		case ARPHRD_VOID:
			/* do not fail when the mfp iface itself does not exists, but it is a vlan
			 * or bond, since their verify config returned OK (underlying interfaces OK).
			 */
			if(!(mfp->type == NC_IFTYPE_VLAN || mfp->type == NC_IFTYPE_BOND)) {
				/* port interface does not exists in the system */
				nc_error_detail_fmt(nh,
					"Bridge %s port interface %s does not exists",
					nfp->name, port);
				return -1;
			}
		break;

		case ARPHRD_ETHER:
			/* port interface exists in the system and is ether */
		break;

		default:
			/* port interface exists in the system but is not ether */
			nc_error_detail_fmt(nh,
				"Cannot use interface %s as port in bridge %s",
				mfp->name, nfp->name);
			return -1;
		break;
	}

	if(!(nc_string_eq(__suse_ifcfg_bootproto_get(mfp), "static") &&
	      mfp->addrs.count == 0 && mfp->routes.count == 0)) {
		nc_error_detail_fmt(nh,
			"IP configuration not allowed for port interface %s",
			mfp->name);
		return -1;
	}

	/* is there a config file for this port already? */
	cfp = nc_interface_find(nh, NULL, -1, port, 0, 0);
	if(cfp) {
		unsigned int type = cfp->type;

		switch(type) {
		case NC_IFTYPE_ETHERNET:
		case NC_IFTYPE_VLAN:
		case NC_IFTYPE_BOND:
		break;
		/* we do not configure them, but it is cool
		 * to be able to use them when they exists...
		 */
		case NC_IFTYPE_TAP:
		case NC_IFTYPE_DUMMY:
			type = NC_IFTYPE_ETHERNET;
		break;
		default:
			nc_error_detail_fmt(nh,
				"Can't use %s as bridge %s port -- config exists with type %s",
				port, nfp->name, __suse_ifcfg_typename_get(cfp->type));
			return -1;
		break;
		}

		if(type != mfp->type) {
			nc_error_detail_fmt(nh,
				"Not allowed to change bridge %s port %s interface type (%u -> %u)",
				nfp->name, port, cfp->type, mfp->type);
			return -1;
		}

		/* check where port's cfp->parent points to our bridge name */
		if(cfp->parent && !nc_string_eq(nfp->name, cfp->parent)) {
			nc_error_detail_fmt(nh,
				"Can't use %s as port in %s -- it is in use by %s",
				port, nfp->name, cfp->parent);
			return -1;
		}
	}

	return 0;
}


static int
__suse_config_interfaces_verify_bridge(nc_handle_t *nh, nc_interface_array_t *nfps, nc_interface_t *pfp, nc_interface_t *nfp)
{
	nc_string_array_t ports = NC_STRING_ARRAY_INIT;
	unsigned int run_type = NC_IFTYPE_UNKNOWN;
	unsigned int arp_type = ARPHRD_NONE;
	unsigned int i;

	if(nc_discover_iface_type(nfp->name, &run_type, &arp_type) == 0) {
		if(run_type != NC_IFTYPE_BRIDGE) {
			nc_error_detail_fmt(nh,
				"Interface %s exists and is not a bridge",
				nfp->name);
			goto failure;
		}
	}

	nc_bridge_get_port_names(nfp->bridge, &ports);
	if(nfp->bridge->ports.count != ports.count) {
		nc_error_detail_fmt(nh,
			"Unable to get list of port names for bridge %s", nfp->name);
		goto failure;
	}

	if(!nc_string_array_is_uniq(&ports)) {
		nc_error_detail_fmt(nh,
			"Invalid bonding configuration %s - port list is not unique",
			nfp->name);
		goto failure;
	}

	/* bridge is allowed to have an empty port list (host-only bridge) */
	for(i = 0; i < ports.count; ++i) {
		const char *port = ports.data[i];

		if(!nc_string_len(port)) {
			nc_error_detail_fmt(nh,
				"Bridge %s contains a port with empty name", nfp->name);
			goto failure;
		}
		if(nc_string_eq(nfp->name, port)) {
			nc_error_detail_fmt(nh,
				"Invalid bridge interface config %s using own name as port",
				nfp->name);
			goto failure;
		}
		if(__suse_config_interfaces_verify_bridge_port(nh, nfps, pfp, nfp, port) < 0)
			goto failure;
	}

#if defined(NC_OPT_DELETE_UNUSED)
{
	nc_interface_t *ifp;
	nc_interface_t *cfp;

	/*
	 * finally, mark deleted all unneeded ports in the current config
	 */
	cfp = nc_interface_find(nh, NULL, -1, nfp->name, 0, 0);
	if(cfp) {
		for(i = 0; i < cfp->bridge->ports.count; ++i) {
			nc_bridge_port_t *port = cfp->bridge->ports.data[i];
			const char *inuse;

			if(!port || !port->name)
				continue;

			if(nc_string_array_index(&ports, port->name) != -1)
				continue;

			ifp = nc_interface_find(nh, NULL, -1, port->name, 0, 0);
			if(!ifp)
				continue;

			inuse = __nc_interface_in_use(nh, ifp, cfp->name);
			if(inuse) {
				nc_info("not deleting unused bridge %s port %s -- in use by %s",
					cfp->name, port->name, inuse);
			} else if(!__nc_suse_config_modify_allowed(ifp, 1)) {
				nc_info("not deleting unused bridge %s port %s -- not allowed to modify",
					cfp->name, port->name);
			} else {
				ifp->deleted = 1;
				nc_info("deleting unused bridge %s port %s",
					cfp->name, port->name);
			}
		}
	}
}
#endif

	nc_string_array_destroy(&ports);
	return 0;
failure:
	nc_string_array_destroy(&ports);
	return -1;
}

static int
__suse_config_interfaces_verify_bond_slave(nc_handle_t *nh, nc_interface_array_t *nfps, nc_interface_t *pfp, nc_interface_t *nfp,
						const char *slave)
{
	nc_interface_t *mfp;
	nc_interface_t *cfp;

	(void)pfp;

	/* check that a modified slave ifp exists */
	mfp = nc_interface_find(nh, NULL, -1, slave, 1, 0);
	if(!mfp) {
		nc_error_detail_fmt(nh,
			"Slave interface %s configuration missed", slave);
		return -1;
	}
	if(!nc_string_eq(mfp->parent, nfp->name)) {
		nc_error_detail_fmt(nh,
			"Bond %s slave interface %s points to parent %s",
			nfp->name, mfp->name, mfp->parent);
		return -1;
	}

	if(__nc_suse_config_interfaces_verify_config(nh, nfps, nfp, mfp) < 0) {
		return -1;
	}

	switch(mfp->arp_type) {
		case ARPHRD_VOID:
			/* slave interface does not exists in the system */
			if(mfp->startmode != NC_STARTMODE_HOTPLUG) {
				nc_error_detail_fmt(nh,
					"Slave interface %s does not exists - use hotplug mode",
					slave);
				return -1;
			}
		break;

		case ARPHRD_ETHER:
		case ARPHRD_INFINIBAND:
			/* slave interface (or its base) exists in the system */
			if(nfp->bonding->slave_type == NC_IFTYPE_UNKNOWN) {
				if(mfp->arp_type == ARPHRD_ETHER)
					nfp->bonding->slave_type =  NC_IFTYPE_ETHERNET;
				else
					nfp->bonding->slave_type =  NC_IFTYPE_INFINIBAND;
			}
			if(mfp->startmode != NC_STARTMODE_AUTO &&
			   mfp->startmode != NC_STARTMODE_HOTPLUG) {
				nc_error_detail_fmt(nh,
					"Slave interface %s needs start mode hotplug or auto/onboot",
					slave);
				return -1;
			}
		break;

		default:
			/* port interface exists in the system but is not ether */
			nc_error_detail_fmt(nh,
				"Cannot use interface %s as slave in bond %s",
				mfp->name, nfp->name);
			return -1;
		break;
	}

	if(!((nc_string_eq(__suse_ifcfg_bootproto_get(mfp), "static") ||
	      nc_string_eq(__suse_ifcfg_bootproto_get(mfp), "none")) &&
	      mfp->addrs.count == 0 && mfp->routes.count == 0)) {
		nc_error_detail_fmt(nh,
			"IP configuration not allowed for slave interface %s",
			slave);
		return -1;
	}

	/* is there a config file for this slave already? */
	cfp = nc_interface_find(nh, NULL, -1, slave, 0, 0);
	if(cfp) {
		switch(cfp->type) {
		case NC_IFTYPE_ETHERNET:
		case NC_IFTYPE_INFINIBAND:
		break;
		default:
			nc_error_detail_fmt(nh,
				"Cannot use %s as bonding %s slave -- config exists with type %s",
				slave, nfp->name, __suse_ifcfg_typename_get(cfp->type));
			return -1;
		break;
		}

		if(cfp->type != mfp->type) {
			nc_error_detail_fmt(nh,
				"Not allowed to change bond %s slave interface %s type (%u -> %u)",
				nfp->name, slave, cfp->type, mfp->type);
			return -1;
		}

		/* check where slave's cfp->parent points to our bridge name */
		if(cfp->parent && !nc_string_eq(nfp->name, cfp->parent)) {
			nc_error_detail_fmt(nh,
				"Can't use %s as slave in %s -- it is in use by %s",
				slave, nfp->name, cfp->parent);
			return -1;
		}
	}

	return 0;
}

static int
__suse_config_interfaces_verify_bond(nc_handle_t *nh, nc_interface_array_t *nfps, nc_interface_t *pfp, nc_interface_t *nfp)
{
	nc_string_array_t uniq = NC_STRING_ARRAY_INIT;
	nc_interface_t *  ifp;
	unsigned int      arp_type = ARPHRD_VOID;
	unsigned int      run_type = NC_IFTYPE_UNKNOWN;
	unsigned int      i;

	/* Let's catch it here already to save time ... */
	if(nfp->bonding->slave_names.count < 1) {
		nc_error_detail_fmt(nh,
			"Invalid bonding configuration %s - empty slave list",
			nfp->name);
		return -1;
	}
	if(!nc_string_array_is_uniq(&nfp->bonding->slave_names)) {
		nc_error_detail_fmt(nh,
			"Invalid bonding configuration %s - slave list is not unique",
			nfp->name);
		return -1;
	}

	/*
	 * When bonding gets created, its arp type defaults to ethernet, but
	 * this changes to the arp type of the first slave (e.g. infiniband)
	 * and stays with this type after all slaves were removed - until a
	 * new slave gets added again, where it may change to the slave type.
	 *
	 * Thus we cannot use the arp type of the bond, even it exists.
	 *
	 * The first existing slave iface we should add to the bond, will
	 * set slave_type and we check the slaves against in a second run.
	 * At least one slave has to exist in the system already.
	 */
	nfp->arp_type = ARPHRD_VOID;
	nfp->bonding->slave_type = NC_IFTYPE_UNKNOWN;
	if(nc_discover_iface_type(nfp->name, &run_type, &arp_type) == 0) {
		if(run_type != NC_IFTYPE_BOND) {
			nc_error_detail_fmt(nh,
				"Interface %s exists, but is not a bond", nfp->name);
			return -1;
		}
		/* Hmm... just catch this */
		if(arp_type != ARPHRD_ETHER && arp_type != ARPHRD_INFINIBAND) {
			nc_error_detail_fmt(nh,
				"Interface %s exists, but is using unknown arp type %u",
				nfp->name, arp_type);
			return -1;
		}
	}

	for(i = 0; i < nfp->bonding->slave_names.count; ++i) {
		const char *slave = nfp->bonding->slave_names.data[i];

		if(!nc_string_len(slave)) {
			nc_error_detail_fmt(nh,
				"Bond %s contains a slave with empty name", nfp->name);
			return -1;
		}

		if(nc_string_eq(nfp->name, slave)) {
			nc_error_detail_fmt(nh,
				"Invalid bond interface %s using own name as slave",
				nfp->name);
			return -1;
		}

		if(__suse_config_interfaces_verify_bond_slave(nh, nfps, pfp, nfp, slave) < 0)
			return -1; /* error already reported */
	}

	/*
	 * We need at least one existing slave -- it provides the type.
	 *
	 * When the bond does not exists (arp_type is void), set it to
	 * the type of existing slave, to allow parent to decide how
	 * to deal with it.
	 */
	switch(nfp->bonding->slave_type) {
		case NC_IFTYPE_ETHERNET:
			if(nfp->arp_type == ARPHRD_VOID)
				nfp->arp_type = ARPHRD_ETHER;
		break;
		case NC_IFTYPE_INFINIBAND:
			if(nfp->arp_type == ARPHRD_VOID)
				nfp->arp_type = ARPHRD_INFINIBAND;
		break;
		default:
			nc_error_detail_fmt(nh,
				"Invalid bonding configuration %s without any existing slave",
				nfp->name);
			return -1;
		break;
	}

	/*
	 * Now check that the types of all slave are same and
	 * catch multiple use of same ib parent: { ib0, ib0.8888 }
	 */
	for(i = 0; i < nfp->bonding->slave_names.count; ++i) {
		const char *slave = nfp->bonding->slave_names.data[i];
		assert(slave);	/* already checked before */

		ifp = nc_interface_find(nh, NULL, -1, slave, 1, 0);
		assert(ifp);	/* already checked before */

		if(ifp->type == NC_IFTYPE_INFINIBAND && ifp->vlan) {
			nc_string_array_append(&uniq, ifp->vlan->interface_name);
		}
		nc_string_array_append(&uniq, ifp->name);

		if(ifp->type == nfp->bonding->slave_type)
			continue;

		nc_error_detail_fmt(nh,
			"Bonding slave type mix - %s-type %s as slave in %s-bonding %s",
			__suse_ifcfg_typename_get(ifp->type), slave,
			__suse_ifcfg_typename_get(nfp->bonding->slave_type), nfp->name);
		return -1;
	}
	if(!nc_string_array_is_uniq(&uniq)) {
		nc_error_detail_fmt(nh,
			"Invalid bonding configuration %s - slave list is not unique",
			nfp->name);
		return -1;
	}

#if defined(NC_OPT_DELETE_UNUSED)
{
	nc_interface_t *  cfp;

	/*
	 * finally, mark all unneeded ports in the current config as deleted
	 */
	cfp = nc_interface_find(nh, NULL, -1, nfp->name, 0, 0);
	if(cfp) {
		for(i = 0; i < cfp->bonding->slave_names.count; ++i) {
			const char *slave = cfp->bonding->slave_names.data[i];
			const char *inuse;
			if(!slave)
				continue;

			if(nc_string_array_index(&nfp->bonding->slave_names, slave) != -1)
				continue;

			ifp = nc_interface_find(nh, NULL, -1, slave, 0, 0);
			if(!ifp)
				continue;

			inuse = __nc_interface_in_use(nh, ifp, cfp->name);
			if(inuse) {
				nc_info("not deleting unused bond %s slave %s -- in use by %s",
					cfp->name, slave, inuse);
			} else if(!__nc_suse_config_modify_allowed(ifp, 1)) {
				nc_info("not deleting unused bond %s slave %s -- not allowed to modify",
					cfp->name, slave);
			} else {
				ifp->deleted = 1;
				nc_info("deleting unused bond %s slave %s",
					cfp->name, slave);
			}
		}
	}
}
#endif

	return 0;
}

static int
__verify_ethvlan_run_details(nc_handle_t *nh, nc_interface_t *nfp)
{
	unsigned int tag = 0;
	char *base = NULL;

	if(nc_vlan_get_settings(nfp->name, &base, &tag) < 0) {
		nc_error_detail_fmt(nh,
			"iface %s is not a ethernet vlan", nfp->name);
		return -1;
	}

	if(!nc_string_eq(nfp->vlan->interface_name, base)) {
		nc_error_detail_fmt(nh,
			"base of ethernet vlan %s is %s, not of %s (config)",
			nfp->name, base, nfp->vlan->interface_name);
		nc_string_free(&base);
		return 1;
	}
	nc_string_free(&base);

	if(tag != nfp->vlan->tag) {
		nc_error_detail_fmt(nh,
			"tag of ethernet vlan %s is %u, not %u (config)",
			nfp->name, tag, nfp->vlan->tag);
		return 1;
	}
	return 0;
}

static int
__verify_ibchild_run_details(nc_handle_t *nh, nc_interface_t *nfp)
{
	unsigned int tag = 0;
	char *base = NULL;

	if(nc_sysfs_netif_get_uint(nfp->name, "pkey", &tag) < 0) {
		nc_error_detail_fmt(nh,
			"iface %s is not a infiniband child", nfp->name);
		return -1;
	}
	if(nc_sysfs_netif_get_string(nfp->name, "parent", &base) < 0) {
		nc_error("iface %s is not a infiniband child", nfp->name);
		return -1;
	}

	if(!nc_string_eq(nfp->vlan->interface_name, base)) {
		nc_error_detail_fmt(nh,
			"parent of infiniband child %s is %s, not of %s (config)",
			nfp->name, base, nfp->vlan->interface_name);
		nc_string_free(&base);
		return 1;
	}
	nc_string_free(&base);

	if(tag != nfp->vlan->tag) {
		nc_error_detail_fmt(nh,
			"key of infiniband child %s is 0x%x, not 0x%x (config)",
			nfp->name, tag, nfp->vlan->tag);
		return 1;
	}

	return 0;
}

static int
__suse_config_interfaces_verify_vlan(nc_handle_t *nh, nc_interface_array_t *nfps, nc_interface_t *pfp, nc_interface_t *nfp)
{
	nc_interface_t *cfp;
	nc_interface_t *mfp;
	unsigned int    vlan_run_type = NC_IFTYPE_UNKNOWN;
	unsigned int    base_run_type = NC_IFTYPE_UNKNOWN;
	unsigned int    base_arp_type = ARPHRD_VOID;
	unsigned int    pos;
	const char *    base = nfp && nfp->vlan ? nfp->vlan->interface_name : NULL;
	const char *    self = nfp && nfp->type == NC_IFTYPE_INFINIBAND ? "ib-child" : "vlan";
	int             exists = 1;

	(void)nfps;

	/* make sure there is a vlan base / ib-parent name */
	if(!nc_string_len(base)) {
		nc_error_detail_fmt(nh,
			"Invalid or empty base interface in %s interface %s",
			self, nfp->name);
		return -1;
	}
	/* and it differs from vlan / ib-child name */
	if(nc_string_eq(nfp->name, base)) {
		nc_error_detail_fmt(nh,
			"Invalid %s interface %s with own name as base",
			self, nfp->name);
		return -1;
	}

	/*
	 * runtime check of the vlan / ib-child
	 */
	nfp->arp_type = ARPHRD_VOID;
	if(nc_discover_iface_type(nfp->name, &vlan_run_type, &nfp->arp_type) == 0) {

		exists = 0;

		if(vlan_run_type != NC_IFTYPE_VLAN && vlan_run_type != NC_IFTYPE_INFINIBAND) {
			nc_error_detail_fmt(nh,
				"Interface %s exists and but is not a %s", nfp->name, self);
			return -1;
		}

		if(nfp->arp_type == ARPHRD_INFINIBAND) {
			if(__verify_ibchild_run_details(nh, nfp) != 0)
				return -1; /* error already reported */

			if(nfp->type == NC_IFTYPE_VLAN) {
				nc_error_detail_fmt(nh,
					"Interface %s is a infiniband child, not a vlan",
					nfp->name);
				return -1;
			}

			/* allow infiniband child bonding only */
			if(pfp && (pfp->type == NC_IFTYPE_BRIDGE || pfp->type == NC_IFTYPE_VLAN)) {
				nc_error_detail_fmt(nh,
					"Cannot use infiniband child %s in %s %s",
					nfp->name, (pfp->type == NC_IFTYPE_VLAN ? "vlan" : "bridge"),
					pfp->name);
				return -1;
			}
		}
		else
		if(nfp->arp_type == ARPHRD_ETHER) {
			if(__verify_ethvlan_run_details(nh, nfp) != 0)
				return -1; /* error already reported */

			if(nfp->type == NC_IFTYPE_INFINIBAND) {
				nc_error_detail_fmt(nh,
					"Interface %s is a ethernet vlan, not infiniband child",
					nfp->name);
				return -1;
			}

			/* disallow ethernet vlan bonding */
			if(pfp && pfp->type == NC_IFTYPE_BOND) {
				nc_error_detail_fmt(nh,
					"Cannot use ethernet vlan %s as bond %s slave",
					nfp->name, pfp->name);
				return -1;
			}
		}
	}

	/*
	 * runtime check of the vlan base/ib parent
	 */
	if(nc_discover_iface_type(base, &base_run_type, &base_arp_type) == 0) {

		exists = 0;

		if(nfp->arp_type == ARPHRD_VOID)
			nfp->arp_type = base_arp_type;

		if(base_arp_type == ARPHRD_INFINIBAND) {
			if(base_run_type != NC_IFTYPE_INFINIBAND) {
				nc_error_detail_fmt(nh,
					"Interface %s is not supported infiniband parent for %s",
					base, nfp->name);
				return -1;
			}

			/* allow infiniband child bonding only */
			if(pfp && (pfp->type == NC_IFTYPE_BRIDGE || pfp->type == NC_IFTYPE_VLAN)) {
				nc_error_detail_fmt(nh,
					"Cannot use infiniband child %s in %s %s",
					nfp->name, (pfp->type == NC_IFTYPE_VLAN ? "vlan" : "bridge"),
					pfp->name);
				return -1;
			}
		}
		else if(base_arp_type == ARPHRD_ETHER) {
			switch(base_run_type) {
				case NC_IFTYPE_ETHERNET:
				case NC_IFTYPE_BRIDGE:
				case NC_IFTYPE_BOND:
				break;
				default:
					nc_error_detail_fmt(nh,
						"Interface %s is not supported %s base for %s",
						self, base, nfp->name);
				break;
			}
			/* disallow ethernet vlan bonding */
			if(pfp && pfp->type == NC_IFTYPE_BOND) {
				nc_error_detail_fmt(nh,
					"Cannot use ethernet vlan %s as bond %s slave",
					nfp->name, pfp->name);
				return -1;
			}
		}
	}

	/* when there is already a config for the base */
	cfp = nc_interface_find(nh, NULL, -1, base, 0, 0);
	if(cfp) {
		if(cfp->type == NC_IFTYPE_VLAN) {
			nc_error_detail_fmt(nh,
				"QinQ configurations not supported (%s [%u] in %s [%u])",
				nfp->name, (nfp->vlan ? nfp->vlan->tag : 0),
				cfp->name, (cfp->vlan ? cfp->vlan->tag : 0));
			return -1;
		}
		if(cfp->type == NC_IFTYPE_INFINIBAND && cfp->vlan) {
			nc_error_detail_fmt(nh,
				"Cannot create infiniband child %s with %s as parent",
				nfp->name, cfp->name);
			return -1;
		}

		/* Hmm... nfsroot, ibft ? */
		if(cfp->startmode != NC_STARTMODE_AUTO) {
			nc_error_detail_fmt(nh,
				"Creation of %s %s not supported -- base %s not started at boot",
				self, nfp->name, cfp->name);
			return -1;
		}

		/* Bond or bridge may be not yet started, but disallow vlan on missed NICs */
		if(exists != 0 && !(cfp->type == NC_IFTYPE_BRIDGE || cfp->type == NC_IFTYPE_BOND)) {
			/* base interface does not exists in the system */
			nc_error_detail_fmt(nh,
				"Cannot find %s %s base interface %s", self, nfp->name, cfp->name);
			return -1;
		}
	} else {
		if(exists != 0) {
			/* base interface does not exists in the system - disallow */
			nc_error_detail_fmt(nh,
				"Cannot find %s %s base interface %s", self, nfp->name, base);
			return -1;
		}
		cfp = NULL;
	}

	/*
	 * OK... we have the choice to reject it or just change it...
	 * In case of ib-child bonding, hotplug is the default, so
	 * it may be not even requested ...
	 */
	if(nfp->startmode == NC_STARTMODE_HOTPLUG)
		nfp->startmode = NC_STARTMODE_AUTO;

	/*
	 * Usually there is no new config for the base, but there are cases there is one.
	 */
	mfp = nc_interface_find(nh, &pos, -1, base, 1, 0);
	if(mfp) {
		if(!mfp->name || !__is_valid_ifname(mfp->name)) {
			nc_error_detail_fmt(nh,
				"Interface name '%s' contains suspect characters",
				mfp->name);
			return -1;
		}
		if(nc_string_eq(mfp->name, "lo")) {
			nc_error_detail_fmt(nh,
				"Cannot create %s on loopback interface %s",
				self, mfp->name);
			return -1;
		}

		if(mfp->type == NC_IFTYPE_VLAN) {
			nc_error_detail_fmt(nh,
				"QinQ configurations not supported (%s [%u] in %s [%u])",
				nfp->name, (nfp->vlan ? nfp->vlan->tag : 0),
				mfp->name, (mfp->vlan ? mfp->vlan->tag : 0));
			return -1;
		}
		if(mfp->type == NC_IFTYPE_INFINIBAND) {
			nc_error_detail_fmt(nh,
				"Cannot create ethernet vlan %s on top infiniband (child) %s",
				nfp->name, mfp->name);
			return -1;
		}

		if(cfp && mfp->type != cfp->type) {
			nc_error_detail_fmt(nh,
				"Not allowed to change type of vlan base interface %s", mfp->name);
			return -1;
		}

		/* Hmm... nfsroot, ibft ? */
		if(mfp->startmode != NC_STARTMODE_AUTO) {
			nc_error_detail_fmt(nh,
				"Creation of %s %s not supported -- base %s not started at boot",
				self, nfp->name, mfp->name);
			return -1;
		}
	}

	/*
	 * OK, now verify the new vlan config file against its old config...
	 */
	cfp = nc_interface_find(nh, NULL, -1, nfp->name, 0, 0);
	if(cfp) {
		if(cfp->type != nfp->type) {
			nc_error_detail_fmt(nh,
				"Not allowed to change the type of interface %s", nfp->name);
			return -1;
		}
	}

	return 0;
}

static int
__suse_config_interfaces_verify_other(nc_handle_t *nh, nc_interface_array_t *nfps, nc_interface_t *pfp, nc_interface_t *nfp)
{
	nc_interface_t *cfp;
	unsigned int run_type = NC_IFTYPE_UNKNOWN;
	unsigned int arp_type = ARPHRD_NONE;
	int          exists;

	(void)nh;
	(void)nfps;

	/*
	 * infiniband may have an optional vlan (children) data and can be bonded only
	 *
	 * ethernet can be bridged and bonded, ethernet vlan is a separate type and
	 * can be used on top of any ethernet-type interface (bridge, bonding)...
	 *
	 * the type of infiniband configs (parents and children) may be ethernet ...
	 *
	 * I do not use vlan for ib children as vlans can be on top of bond/bridge,
	 * ib children not.
	 */

	/* when ib has a vlan element (== ib child) ... check as vlan */
	if(nfp->type == NC_IFTYPE_INFINIBAND) {
		if(nfp->vlan) {
			return __suse_config_interfaces_verify_vlan(nh, nfps, pfp, nfp);
		}
	} else if(nfp->vlan) {
		nc_error_detail_fmt(nh,
			"interface %s: invalid type-specific data", nfp->name);
		return -1;
	}

	exists = nc_discover_iface_type(nfp->name, &run_type, &arp_type);
	if(exists == 0) {
		/* it exists, check type */
		nfp->arp_type = arp_type;

		switch(run_type) {
			case NC_IFTYPE_ETHERNET:
			case NC_IFTYPE_INFINIBAND:
			break;
			default:
				nc_error_detail_fmt(nh,
					"Interface %s configuration not supported", nfp->name);
				return -1;
			break;
		}

		/* it may come as ethernet but be in fact ib [child -> vlan code] */
		if(run_type != nfp->type) {
			nfp->type = run_type;
		}
		if(run_type == NC_IFTYPE_INFINIBAND) {
			if(__nc_discover_infiniband(nfp) == 0 && nfp->vlan) {
				return __suse_config_interfaces_verify_vlan(nh, nfps, pfp, nfp);
			}
		}
	} else {
		/* it does not exist */
		nfp->arp_type = ARPHRD_VOID;

		if(maybe_infiniband(nfp->name, &nfp->vlan) == 0) {
			nfp->type = NC_IFTYPE_INFINIBAND;
			if(nfp->vlan) {
				return __suse_config_interfaces_verify_vlan(nh, nfps, pfp, nfp);
			}
		}

		/* OK, handle as normal NIC */
		if(pfp) {
			/* only bond can have optional NICs */
			if(pfp->type != NC_IFTYPE_BOND) {
				nc_error_detail_fmt(nh,
					"Interface %s does not exists", nfp->name);
				return -1;
			}
		}
		if(nfp->startmode != NC_STARTMODE_HOTPLUG && nfp->startmode != NC_STARTMODE_MANUAL) {
			nc_error_detail_fmt(nh, "Interface %s does not exists", nfp->name);
			return -1;
		}
	}

	if(pfp) {
		if(nfp->type == NC_IFTYPE_INFINIBAND) {
			if(!(pfp->type == NC_IFTYPE_BOND)) {
				nc_error_detail_fmt(nh,
					"interface %s: invalid parent %s",
					nfp->name, pfp->name);
				return -1;
			}
		} else {
			if(!(pfp->type == NC_IFTYPE_BRIDGE || pfp->type == NC_IFTYPE_BOND)) {
				nc_error_detail_fmt(nh,
					"interface %s: invalid parent %s",
					nfp->name, pfp->name);
				return -1;
			}
		}
	}

	/*
	 * OK, now verify the new config file against its old config...
	 */
	cfp = nc_interface_find(nh, NULL, -1, nfp->name, 0, 0);
	if(cfp) {
		unsigned int type = cfp->type;
		if(type == NC_IFTYPE_TAP || type == NC_IFTYPE_DUMMY) {
			/*
			 * we do not configure them, but it is cool
			 * to be able to use them when they exists...
			 */
			type = NC_IFTYPE_ETHERNET;
		}
		if(type != nfp->type) {
			nc_error_detail_fmt(nh,
				"Not allowed to change the type of interface %s", nfp->name);
			return -1;
		}
	}

	return 0;
}

static int
__nc_suse_config_interfaces_verify_config(nc_handle_t *nh, nc_interface_array_t *nfps, nc_interface_t *pfp, nc_interface_t *nfp)
{
	nc_interface_t *cfp;

	assert(nh && nfp && nfps);

#if defined(NC_OPT_TRACE)
	nc_trace("> VFY[%u] nfp=%s [%s], pfp=%s [%s]",
		nfps->count, nfp->name,
	       	__suse_ifcfg_typename_get(nfp->type),
		(pfp ? pfp->name : NULL),
	       	(pfp ? __suse_ifcfg_typename_get(pfp->type) : NULL));
#endif

	/* check if we can modify this interface */
	if(!__nc_suse_config_modify_allowed(nfp, 0)) {
		/* __nc_suse_config_modify_allowed reports error */
		return -1;
	}

	if(!nfp->modified) {
		nc_error_detail_fmt(nh,
			"Interface %s is not intended for modifications",
			nfp->name);
		return -1;
	}

	if(nc_check_interface_usage(nfps, nfp->name, NC_IFTYPE_UNKNOWN))
		return -1;

	if(nc_interface_array_append(nfps, nc_interface_ref(nfp)) < 0)
		return -1;

	if(pfp) {
		if(!nc_string_len(pfp->name)) {
			nc_error_detail_fmt(nh,
				"Parent interface without a valid name");
			return -1;
		}
		if(!nc_string_eq(nfp->parent, pfp->name)) {
			nc_error_detail_fmt(nh,
				"Parent interface name %s does not match parent reference %s",
				pfp->name, nfp->parent);
			return -1;
		}
	} else {
		if(nfp->parent) {
			nc_error_detail_fmt(nh,
				"Top-Level interface %s contains a parent %s reference",
				nfp->name, nfp->parent);
			return -1;
		}
	}

	/*
	 * find the unmodified interface (from config)
	 */
	cfp = nc_interface_find(nh, NULL, -1, nfp->name, 0, 0);
	if(cfp) {
		/*
		** check if we are a slave -- not allowed to change slaves
		** as long as there is a master referencing it
		*/
		const char * parent = __nc_interface_is_enslaved(nh, cfp);
		if(parent) {
			if(!pfp || !nc_string_eq(parent, pfp->name)) {
				nc_error_detail_fmt(nh,
					"Unable to modify: interface %s is in use by %s",
					cfp->name, parent);
				return -1;
			}
		}

		/* check if we are are allowed to modify this interface */
		if(!__nc_suse_config_modify_allowed(cfp, 0)) {
			/* __nc_suse_config_modify_allowed reports error */
			return -1;
		}
	}

	/* Now, check the new nfp slaves */
	switch(nfp->type) {
		case NC_IFTYPE_BRIDGE:
			if( !nfp->bridge || nfp->bonding || nfp->vlan) {
				nc_error_detail_fmt(nh,
					"interface %s: invalid bridge-type data", nfp->name);
				return -1;
			}

			/* no parent possible -- vlan does not recurse */
			if(pfp) {
				nc_error_detail_fmt(nh,
					"interface %s: invalid bridge parent %s",
					nfp->name, pfp->name);
				return -1;
			}

			if(__suse_config_interfaces_verify_bridge(nh, nfps, pfp, nfp) < 0)
				return -1;
		break;

		case NC_IFTYPE_BOND:
			if( !nfp->bonding || nfp->bridge || nfp->vlan) {
				nc_error_detail_fmt(nh,
					"interface %s: invalid bonding-type data", nfp->name);
				return -1;
			}

			if(pfp && !(pfp->type == NC_IFTYPE_BRIDGE)) {
				nc_error_detail_fmt(nh,
					"interface %s: cannot use bond in parent %s",
					nfp->name, pfp->name);
				return -1;
			}

			if(__suse_config_interfaces_verify_bond(nh, nfps, pfp, nfp) < 0)
				return -1;
		break;

		case NC_IFTYPE_VLAN:
			if( !nfp->vlan || nfp->bonding || nfp->bridge) {
				nc_error_detail_fmt(nh,
					"interface %s: invalid vlan-type data", nfp->name);
				return -1;
			}

			if(pfp && !(pfp->type == NC_IFTYPE_BRIDGE)) {
				nc_error_detail_fmt(nh,
					"interface %s: cannot use vlan in parent %s",
					nfp->name, pfp->name);
				return -1;
			}

			if(__suse_config_interfaces_verify_vlan(nh, nfps, pfp, nfp) < 0)
				return -1;
		break;

		case NC_IFTYPE_INFINIBAND:
		case NC_IFTYPE_ETHERNET:
			if( nfp->bonding || nfp->bridge) {
				nc_error_detail_fmt(nh,
					"interface %s: invalid type-specific data", nfp->name);
				return -1;
			}

			if(__suse_config_interfaces_verify_other(nh, nfps, pfp, nfp) < 0)
					return -1;

		break;

		default:
			nc_error_detail_fmt(nh,
				"Configuration of interface %s is not suppoted", nfp->name);
			return -1;
		break;
	}

#if defined(NC_OPT_TRACE)
	nc_trace("< VFY[%u] nfp=%s [%s], pfp=%s [%s]",
		nfps->count, nfp->name,
	       	__suse_ifcfg_typename_get(nfp->type),
		(pfp ? pfp->name : NULL),
	       	(pfp ? __suse_ifcfg_typename_get(pfp->type) : NULL));
#endif

	return 0;
}

static int
__nc_suse_config_interfaces_prepare_config(nc_handle_t *nh, nc_interface_t *nfp, nc_interface_t *pfp,
			const char *base_path, nc_sysconfig_array_t *nscs, nc_interface_array_t *nfps);

static int
__nc_suse_config_interfaces_prepare_vlan_base(nc_handle_t *nh, nc_interface_t *nfp, nc_interface_t *pfp,
			const char *base_path, nc_sysconfig_array_t *nscs, nc_interface_array_t *nfps)
{
	nc_interface_t *ifp = NULL;

	(void)pfp;

	/* check if there is a modified interface (usually not) */
	ifp = nc_interface_find(nh, NULL, -1, nfp->vlan->interface_name, 1, 0);
	if(ifp) {
		if(__nc_suse_config_interfaces_prepare_config(nh, ifp, NULL,
						base_path, nscs, nfps) < 0)
			return -1;
	} else {
		/* check if there is an already configured interface */
		if(nc_interface_find(nh, NULL, -1, nfp->vlan->interface_name, 0, 0) == NULL) {

			/* Create new one */
			if((ifp = nc_interface_new(nfp->vlan->interface_name, 0)) == NULL) {
				nc_error_detail_fmt(nh,
					"Unable to allocate %s interface configuration",
					nfp->vlan->interface_name);
				return -1;
			}
			ifp->modified  = 1;
			ifp->startmode = NC_STARTMODE_AUTO;
			ifp->ipv4.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			ifp->ipv6.addrconf = NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);
			if(nfp->type == NC_IFTYPE_INFINIBAND)
				ifp->type = NC_IFTYPE_INFINIBAND;
			else
				ifp->type = NC_IFTYPE_ETHERNET;

			/* add the interface to the guard/cleanup array + repo */
			if(nc_interface_array_append(nfps, ifp) < 0 ||
			   (ifp = nc_interface_ref(ifp)) == NULL ||
			   nc_interfaces_add(nh, ifp) < 0) {
				nc_error_detail_fmt(nh,
					"Unable to add %s interface configuration to repository",
					nfp->vlan->interface_name);
				if(ifp)
					nc_interface_free(ifp);
				return -1;
			}

			if(__nc_suse_config_interfaces_prepare_config(nh, ifp, NULL,
							base_path, nscs, nfps) < 0)
				return -1;
		} /* else: old config exists */
	} /* else: new config provided */

	return 0;
}

static int
__nc_suse_config_interfaces_prepare_config(nc_handle_t *nh, nc_interface_t *nfp, nc_interface_t *pfp,
			const char *base_path, nc_sysconfig_array_t *nscs, nc_interface_array_t *nfps)
{
	nc_stringbuf_t  cfg  = NC_STRINGBUF_INIT;
	nc_interface_t *ifp = NULL;
	nc_sysconfig_t *nsc = NULL;
	unsigned int i;

	assert(nh && nfp && base_path && nscs && nfps);

#if defined(NC_OPT_TRACE)
	for(i=0; i< nh->ifaces.count; ++i) {
		ifp = nh->ifaces.data[i];
		if(ifp) {
			nc_trace("> cfg iface[%u]{users=%u, type=%s, mode=%s, proto=%s, parent=%s}: ifname=%s%s%s",
				i,
				ifp->users,
				__suse_ifcfg_typename_get(ifp->type),
				__suse_ifcfg_startmode_get(ifp->startmode),
				__suse_ifcfg_bootproto_get(ifp),
				ifp->parent,
				ifp->name,
				(ifp->modified ? " modified" : ""),
				(ifp->deleted ? " deleted" : ""));
		} else {
			nc_trace("> cfg iface[%u]: NULL", i);
		}
	}
	nc_trace("------------------------------------------------");
#endif

	/* prepare sysconfig file name */
	if(nc_stringbuf_printf(&cfg, "%s/%s%s", base_path, _IFCFG_PREFIX, nfp->name) <= 0) {
		nc_error_detail_fmt(nh,
			"Unable to construct %s%s file name", _IFCFG_PREFIX, nfp->name);
		goto failure;
	}

	/* parse sysconfig when exists or create new */
	nsc = nc_sysconfig_read(cfg.string);
	if(!nsc) {
		nsc = nc_sysconfig_new(cfg.string);
		if(!nsc) {
			nc_stringbuf_destroy(&cfg);
			goto failure;
		}
	}
	nc_stringbuf_destroy(&cfg);

	/* Let's verify & prepare child configs first */
	switch(nfp->type) {
		case NC_IFTYPE_BRIDGE:
			for(i = 0; i < nfp->bridge->ports.count; ++i) {
				nc_bridge_port_t *port = nfp->bridge->ports.data[i];

				if(!port || !port->name)
					continue;

				ifp = nc_interface_find(nh, NULL, -1, port->name, 1, 0);
				if(!ifp) {
					nc_error_detail_fmt(nh,
						"Bridge %s port interface %s configuration missed",
						nfp->name, port->name);
					goto failure;
				}
				if(__nc_suse_config_interfaces_prepare_config(nh, ifp, nfp, base_path, nscs, nfps) < 0)
					goto failure;
			}
		break;

		case NC_IFTYPE_BOND:
			assert(nfp->bonding != NULL);

			for(i = 0; i < nfp->bonding->slave_names.count; ++i) {
				const char *slave = nfp->bonding->slave_names.data[i];

				if(!slave)
					continue;

				ifp = nc_interface_find(nh, NULL, -1, slave, 1, 0);
				if(!ifp) {
					nc_error_detail_fmt(nh,
						"Bonding %s slave interface %s configuration missed",
						nfp->name, slave);
					goto failure;
				}
				if(__nc_suse_config_interfaces_prepare_config(nh, ifp, nfp, base_path, nscs, nfps) < 0)
					goto failure;
			}
		break;

		case NC_IFTYPE_VLAN:
			/* vlan data is mandatory */
			if(!nfp->vlan || __nc_suse_config_interfaces_prepare_vlan_base(nh,
						nfp, pfp, base_path, nscs, nfps) < 0)
				goto failure;
		break;

		case NC_IFTYPE_INFINIBAND:
			/* vlan data is optional */
			if(nfp->vlan && __nc_suse_config_interfaces_prepare_vlan_base(nh,
						nfp, pfp, base_path, nscs, nfps) < 0)
				goto failure;
		break;

		default:
			/* nothing special */
		break;
	}

	/* Generate sysconfig file */
	if(__nc_suse_config_iface2ifcfg(nh, nfp, nsc) < 0) {
		nc_sysconfig_free(nsc);
		nsc = NULL;
		goto failure;
	}

	/* OK, sysconfig is ready now */
	if(nc_sysconfig_array_append(nscs, nsc) < 0) {
		nc_sysconfig_free(nsc);
		nsc = NULL;
		goto failure;
	}
	nsc = NULL;

#if defined(NC_OPT_TRACE)
	nc_trace("PREPARED CONFIG FOR %s", nfp->name);
#endif

	return 0;
failure:
	if(nsc)
		nc_sysconfig_free(nsc);
	return -1;
}

static int
__nc_suse_config_interfaces_configure(nc_handle_t *nh, nc_interface_t *nfp)
{
	nc_config_handle_t *  ch  = __to_config_handle(nh);
	nc_stringbuf_t base_path  = NC_STRINGBUF_INIT;
	nc_stringbuf_t ifcfg_cur  = NC_STRINGBUF_INIT;
	nc_stringbuf_t ifrte_cur  = NC_STRINGBUF_INIT;
	nc_interface_array_t nfps = NC_INTERFACE_ARRAY_INIT;
	nc_sysconfig_array_t nscs = NC_SYSCONFIG_ARRAY_INIT;
	nc_sysconfig_t *nsc = NULL;
	nc_interface_t *ifp = NULL;
	unsigned int i;
	int ret = -1;

	assert(nh && nfp && nfp->name);
	if(!(nh && nfp && nfp->name))
		goto failure;

#if defined(NC_OPT_TRACE)
	for(i=0; i< nh->ifaces.count; ++i) {
		ifp = nh->ifaces.data[i];
		if(ifp) {
			nc_trace(">>> iface[%u]: users=%u, parent=%s ifname=%s%s%s",
				i, ifp->users, ifp->parent, ifp->name,
				(ifp->modified ? " modified" : ""),
				(ifp->deleted ? " deleted" : ""));
		} else {
			nc_trace(">>> iface[%u]: NULL", i);
		}
	}
	nc_trace("================================================");
#endif

	if(__nc_suse_config_prepend_root(ch, _SYSCONFIG_NETWORK_DIR, &base_path) < 0)
		goto failure;

	/* configuration verify */
	if(__nc_suse_config_interfaces_verify_config(nh, &nfps, NULL, nfp) < 0)
		goto failure;

	/* make sure, there are no unverified interfaces marked modified left */
	for(ifp=nc_interface_first(nh, &i); ifp; ifp=nc_interface_next(nh, &i)) {
		if(!ifp->modified)
			continue;

		if(nc_interface_array_find(&nfps, NULL, -1, ifp->name, 1, 0) == NULL) {
			nc_error_detail_fmt(nh,
				"Interface %s is marked modified, but not related to %s",
				ifp->name, nfp->name);
			goto failure;
		}
	}

	if(__nc_suse_config_interfaces_prepare_config(nh, nfp, NULL, base_path.string, &nscs, &nfps) < 0)
		goto failure;

	/* OK, no error so far -- write sysconfig files */
	for(i = 0; i < nscs.count; i++) {
		nsc = nscs.data[i];

		nc_ifcfg_trace_dump(nsc);

		if(nc_sysconfig_rewrite(nsc, _DOT_BACKUP_SUFFIX) < 0)
			goto failure;
	}

	/* OK, no error so far -- write routes */
	if(__nc_suse_config_rewrite_routes(nh, &nfps, _DOT_BACKUP_SUFFIX) < 0)
		goto failure;

	for(ifp=nc_interface_array_first(&nfps, &i); ifp; ifp=nc_interface_array_next(&nfps, &i)) {
		if(!ifp->modified)
			continue;

		/* OK, update bonding slave udev rules */
		if(ifp->type == NC_IFTYPE_BOND) {
			__update_bond_slave_udev_rules(ifp, &nfps);
			/* optional, so don't fail */
		}

	}

	/*
	 * OK, last but not least, cleanup deleted slaves
	 */
	for(i = 0; i < nh->ifaces.count; ++i) {
		nc_interface_t *dfp = nh->ifaces.data[i];
		if(!dfp || !dfp->name)
			continue;

		if(dfp->modified || !dfp->deleted)
			continue;

		/* construct ifcfg-<name> file */
		if(nc_stringbuf_printf(&ifcfg_cur, "%s/%s%s", base_path.string,
						_IFCFG_PREFIX, dfp->name) <= 0) {
			nc_error_detail_fmt(nh,
				"Unable to construct %s%s file name",
				_IFCFG_PREFIX, dfp->name);
			goto failure;
		}
		/* construct ifroute-<name> file */
		if(nc_stringbuf_printf(&ifrte_cur, "%s/%s%s", base_path.string,
						_IFROUTE_PREFIX, dfp->name) <= 0) {
			nc_error_detail_fmt(nh,
				"Unable to construct %s%s file name",
				_IFROUTE_PREFIX, dfp->name);
			goto failure;
		}
		/* move to .ifcfg-<name>[suffix] file */
		if(nc_rename_to_dot_backup(ifcfg_cur.string, _DOT_BACKUP_SUFFIX) < 0) {
			nc_error_detail_fmt(nh,
				"Unable to create a dot backup of filename: %s",
				ifcfg_cur.string);
			goto failure;
		}
		/* move to .ifroute-<name>[suffix] file */
		if(nc_rename_to_dot_backup(ifrte_cur.string, _DOT_BACKUP_SUFFIX) < 0) {
			nc_error_detail_fmt(nh,
				"Unable to create a dot backup of filename: %s",
				ifrte_cur.string);
			goto failure;
		}
	}

	ret = 0; /* OK, all done */

failure:
	nc_sysconfig_array_destroy(&nscs);
	nc_interface_array_destroy(&nfps);
	nc_stringbuf_destroy(&base_path);
	nc_stringbuf_destroy(&ifcfg_cur);
	nc_stringbuf_destroy(&ifrte_cur);
	return ret;
}

static int
__child_ifaction_exec(nc_string_array_t *args)
{
	if(!args || args->count < 2 || !args->data[0] || *args->data[0] != '/')
		return -1;

	if(freopen("/dev/null", "r", stdin) == NULL)
		return -1;

	return execv(args->data[0], args->data);
}

static int
__execute_pipe_cmd(nc_string_array_t *args, int (*exec)(nc_string_array_t *args), int *code)
{
	char           buf[BUFSIZ];
	nc_cmd_pipe_t  cmd;
	int            status;

	if(!code || !exec || !args || args->count < 2 || !args->data[0] || *args->data[0] != '/') {
		nc_error("Invalid command execution arguments");
		return -1;
	}

	memset(&cmd, 0, sizeof(cmd));
	cmd.exec = exec;
	cmd.args = args;
	if(nc_cmd_pipe_exec(&cmd) == -1) {
		nc_error("Unable to execute %s %s", args->data[0], args->data[1]);
		return -1;
	}

	if(cmd.file) {
		memset(buf, 0, sizeof(buf));
		while(fgets(buf, sizeof(buf)-1, cmd.file)) {
			buf[strcspn(buf, "\r\n")] = '\0';
			nc_trace("%s %s: %s", args->data[0], args->data[1], buf);
		}
		fclose(cmd.file);
	} else {
		nc_error("Unable to read output of %s %s", args->data[0], args->data[1]);
		/* do not fail here */
	}

	status = -1;
	if(nc_cmd_pipe_wait(&cmd, &status, 0) == -1) {
		nc_error("waitpid failure: %m");
		return -1;
	}

	if(WIFEXITED(status)) {
		*code = WEXITSTATUS(status);
		nc_info("Command %s %s returned with exit code: %d",
			args->data[0], args->data[1], *code);
		return 0;
	} else {
		*code = -1;
		nc_error("Command %s %s terminated abormally: %d [%m]",
			args->data[0], args->data[1], status);
		return -1;
	}
}

static int
__suse_exec_ifcmd(nc_handle_t *nh, nc_interface_t *ifp, const char *ifcmd)
{
	nc_string_array_t    args = NC_STRING_ARRAY_INIT;
	int                  code = -1;

	assert(nh && ifp && ifp->name && ifcmd);

	nc_string_array_append(&args, ifcmd);
	nc_string_array_append(&args, ifp->name);
	if(args.count != 2) {
		nc_error_detail_fmt(nh,
			"Unable to construct %s command for interface %s",
			ifcmd, ifp->name);
		return -1;
	}

	if(__execute_pipe_cmd(&args, __child_ifaction_exec, &code) < 0) {
		nc_string_array_destroy(&args);
		nc_error_detail_fmt(nh,
			"Execution of '%s %s' failed", ifcmd, ifp->name);
		return -1;
	}
	nc_string_array_destroy(&args);

	return code;
}

static int
__suse_interface_status(nc_handle_t *nh, nc_interface_t *ifp)
{
	int code = -1;

	if(!ifp || !ifp->name || !__is_valid_ifname(ifp->name)) {
		nc_error_detail_fmt(nh,
			"Interface name '%s' contains suspect characters",
			ifp->name);
		return -1;
	}

	code = __suse_exec_ifcmd(nh, ifp, "/sbin/ifstatus");
	switch(code) {
		case 0:		/* R_SUCCESS */
		case 12:	/* R_DHCP_BG */
			ifp->ifflags |= NC_IFF_LINK_UP;
			return 0;

		default:
			ifp->ifflags &= ~NC_IFF_LINK_UP;
			return code;
	}
}

static int
__nc_suse_system_interface_status(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_system_handle_t * sh   = __to_system_handle(nh);
	if(sh->root) {
		nc_error_detail_fmt(nh,
			"Status check not supported when root_directory is set");
		return -1;
	}
	return __suse_interface_status(nh, ifp);
}

static inline int
__start_child_interfaces(nc_handle_t *nh, nc_interface_t *ifp)
{
	const char *         cname;
	nc_interface_t *     child;
	unsigned int         i;

	if(ifp->bonding) {
		for (i = 0; i < ifp->bonding->slave_names.count; ++i) {
			cname = ifp->bonding->slave_names.data[i];
			child = nc_interface_by_name(nh, cname);
			if(child) {
				(void)__nc_suse_config_interface_start(nh, child);
				/*
				 * ignore missed slaves -- ifup bondX will
				 * fail when not ever one is available...
				 */
			}
		}
	}
	else if(ifp->bridge) {
		for (i = 0; i < ifp->bridge->ports.count; ++i) {
			cname = ifp->bridge->ports.data[i]->name;
			child = nc_interface_by_name(nh, cname);
			if(child) {
				int ret = __nc_suse_config_interface_start(nh, child);
				if(ret != 0)
					return ret;
			}
		}
	}
	else if(ifp->vlan) {
		child = nc_interface_by_name(nh, ifp->vlan->interface_name);
		if(child) {
			return __nc_suse_config_interface_start(nh, child);
		}
	}
	return 0;
}

static int
__nc_suse_config_interface_start(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_config_handle_t * ch   = __to_config_handle(nh);
	int                  code = -1;

	if(ch->root) {
		nc_error_detail_fmt(nh,
			"Start not supported when root_directory is set");
		return -1;
	}

	if(!ifp || !ifp->name || !__is_valid_ifname(ifp->name)) {
		nc_error_detail_fmt(nh,
			"Interface name '%s' contains suspect characters",
			ifp->name);
		return -1;
	}

	/* automatically start interfaces we depend on */
	__start_child_interfaces(nh, ifp);

	code = __suse_exec_ifcmd(nh, ifp, "/sbin/ifup");

	switch(code) {
		case 0:		/* R_SUCCESS */
		case 12:	/* R_DHCP_BG */
			ifp->ifflags |= NC_IFF_LINK_UP;
			return 0;

		default:
			ifp->ifflags &= ~NC_IFF_LINK_UP;
			return code;
	}
}

static int
__nc_suse_config_interface_stop(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_config_handle_t * ch   = __to_config_handle(nh);
	int                  code = -1;

	if(ch->root) {
		nc_error_detail_fmt(nh,
			"Stop not supported when root_directory is set");
		return -1;
	}

	if(!ifp || !ifp->name || !__is_valid_ifname(ifp->name)) {
		nc_error_detail_fmt(nh,
			"Interface name '%s' contains suspect characters",
			ifp->name);
		return -1;
	}

	code = __suse_exec_ifcmd(nh, ifp, "/sbin/ifdown");

	switch(code) {
		case 0:		/* R_SUCCESS */
		case 7:		/* R_NODEV   */
			ifp->ifflags |= NC_IFF_LINK_UP;
			return 0;

		default:
			ifp->ifflags &= ~NC_IFF_LINK_UP;
			return code;
	}
}

/*
 * ===  wicked backend =========================================================
 */
#define WICKED_CLI_CMD				"/usr/sbin/wicked"
#define WICKED_CLI_SHOW_CONFIG			"show-config"
#define WICKED_CLI_SHOW_STATUS			"show-xml"

#define WICKED_NIF_XML_OBJECT_LIST		"/org/opensuse/Network/Interface/"
#define WICKED_NIF_XML_OBJECT_NODE		"object"
#define WICKED_NIF_XML_OBJECT_PATH		"path"
#define WICKED_NIF_XML_IFCFG_ORIGIN		"compat:suse:"

static const nc_intmap_t		nc_wicked_iftype_map[] = {
	{ "loopback",			NC_IFTYPE_LOOPBACK		},
	{ "ethernet",			NC_IFTYPE_ETHERNET		},
	{ "bridge",			NC_IFTYPE_BRIDGE		},
	{ "bond",			NC_IFTYPE_BOND			},
	{ "bonding",			NC_IFTYPE_BOND			},
	{ "vlan",			NC_IFTYPE_VLAN			},
/*	{ "vxlan",			NC_IFTYPE_UNKNOWN		},	*/
	{ "macvlan",			NC_IFTYPE_MACVLAN		},
	{ "macvtap",			NC_IFTYPE_MACVTAP		},
	{ "wireless",			NC_IFTYPE_WIRELESS		},
	{ "infiniband",			NC_IFTYPE_INFINIBAND		},
	{ "infiniband-child",		NC_IFTYPE_INFINIBAND		},
	{ "infiniband:child",		NC_IFTYPE_INFINIBAND		},
	{ "ppp",			NC_IFTYPE_PPP			},
	{ "slip",			NC_IFTYPE_SLIP			},
	{ "sit",			NC_IFTYPE_SIT			},
	{ "gre",			NC_IFTYPE_GRE			},
	{ "isdn",			NC_IFTYPE_ISDN			},
	{ "ipip",			NC_IFTYPE_TUNNEL		},
	{ "tunnel6",			NC_IFTYPE_TUNNEL6		},
	{ "tun",			NC_IFTYPE_TUN			},
	{ "tap",			NC_IFTYPE_TAP			},
	{ "dummy",			NC_IFTYPE_DUMMY			},
/*	{ "ctcm",			NC_IFTYPE_UNKNOWN		},	*/
/*	{ "iucv",			NC_IFTYPE_UNKNOWN		},	*/
	{ "team",			NC_IFTYPE_TEAM			},
/*	{ "openvswitch",		NC_IFTYPE_UNKNOWN		},	*/
/*	{ "ovs-system",			NC_IFTYPE_UNKNOWN		},	*/
	{ "ovs-bridge",			NC_IFTYPE_OVS			},
/*	{ "ovs",			NC_IFTYPE_UNKNOWN		},	*/
/*	{ "vrf",			NC_IFTYPE_UNKNOWN		},	*/
	{ NULL,				NC_IFTYPE_UNKNOWN		}
};

static const nc_intmap_t		nc_wicked_ifflag_map[] = {
	{ "device-up",			NC_IFF_DEVICE_UP		},
	{ "link-up",			NC_IFF_LINK_UP			},
	{ "network-up",			NC_IFF_NETWORK_UP		},
	{ "powersave",			NC_IFF_POWERSAVE		},
	{ "point-to-point",		NC_IFF_POINT_TO_POINT		},
	{ "arp",			NC_IFF_ARP_ENABLED		},
	{ "broadcast",			NC_IFF_BROADCAST_ENABLED	},
	{ "multicast",			NC_IFF_MULTICAST_ENABLED	},
/*	{ "ready",			NC_IFF_READY			},	*/
	{ NULL,				0				},
};

static const nc_intmap_t		nc_wicked_bond_mode_map[] = {
	{ "balance-rr",			NC_BOND_MODE_BALANCE_RR		},
	{ "active-backup",		NC_BOND_MODE_ACTIVE_BACKUP	},
	{ "balance-xor",		NC_BOND_MODE_BALANCE_XOR	},
	{ "broadcast",			NC_BOND_MODE_BROADCAST		},
	{ "ieee802-3ad",		NC_BOND_MODE_802_3AD		},
	{ "balance-tlb",		NC_BOND_MODE_BALANCE_TLB	},
	{ "balance-alb",		NC_BOND_MODE_BALANCE_ALB	},
	{ NULL,				0				}
};
static const nc_intmap_t		nc_wicked_bond_carrier_map[] = {
	{ "ioctl",			NC_BOND_CARRIER_DETECT_IOCTL	},
	{ "netif",			NC_BOND_CARRIER_DETECT_NETIF	},
	{ NULL,				0				}
};
static const nc_intmap_t		nc_wicked_bond_arp_validate_map[] = {
	{ "none",			NC_BOND_VALIDATE_NONE		},
	{ "active",			NC_BOND_VALIDATE_ACTIVE		},
	{ "backup",			NC_BOND_VALIDATE_BACKUP		},
	{ "all",			NC_BOND_VALIDATE_ALL		},
	{ NULL,				0				}
};

static const nc_intmap_t		nc_wicked_address_owner_map[] = {
	{ "dhcp",			NC_ADDRCONF_DHCP		},
	{ "static",			NC_ADDRCONF_STATIC		},
	{ "auto",			NC_ADDRCONF_AUTOCONF		},
	{ NULL,				0				}
};
static const nc_intmap_t		nc_wicked_route_scope_map[] = {
	{ "universe",			RT_SCOPE_UNIVERSE		},
	{ "site",			RT_SCOPE_SITE			},
	{ "link",			RT_SCOPE_LINK			},
	{ "host",			RT_SCOPE_HOST			},
	{ NULL,				RT_SCOPE_NOWHERE		}
};

static int	nc_wicked_cli_show_xml_config(const char *ifname, xml_document_t **doc);
static int	nc_wicked_cli_show_xml_status(const char *ifname, xml_document_t **doc);

static int
nc_wicked_nif_xml_common_iftype_vlan(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;
	nc_vlan_t *vlan;

	if (!(vlan = nc_vlan_new(NULL, 0)))
		goto failure;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "device"))
			nc_string_dup(&vlan->interface_name, child->cdata);
		else
		if (nc_string_eq(child->name, "tag"))
			nc_parse_uint(child->cdata, &vlan->tag, 10);
		else
		if (nc_string_eq(child->name, "address"))
			nc_hw_address_parse(&ifp->hwaddr, ifp->type, child->cdata);
	}

	if (!vlan->tag || nc_string_empty(vlan->interface_name))
		goto failure;

	if (ifp->vlan)
		nc_vlan_free(ifp->vlan);
	ifp->vlan = vlan;
	return 0;

failure:
	nc_vlan_free(vlan);
	ifp->invalid = 1;
	return -1;
}

static int
nc_wicked_nif_xml_common_iftype_bond_slave(const xml_node_t *node, nc_bonding_t *bond)
{
	xml_node_t *child;
	const char *name = NULL;
	const char *primary = NULL;

	if ((child = xml_node_get_child(node, "device")))
		name = child->cdata;

	if (nc_string_empty(name) || nc_bonding_add_slave(bond, name))
		return -1;

	if ((child = xml_node_get_child(node, "primary")))
		primary = child->cdata;

	if (!nc_string_empty(primary) && nc_string_empty(bond->primary_slave))
		nc_string_dup(&bond->primary_slave, primary);

	return 0;
}

static int
nc_wicked_nif_xml_common_iftype_bond_slaves(const xml_node_t *node, nc_bonding_t *bond)
{
	xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "slave"))
			nc_wicked_nif_xml_common_iftype_bond_slave(child, bond);
	}
	return 0;
}

static int
nc_wicked_nif_xml_common_iftype_bond_arp_target(const xml_node_t *node, nc_bonding_t *bond)
{
	xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (!nc_string_eq(child->name, "address"))
			continue;
		if (!nc_string_empty(child->cdata))
			nc_string_array_append(&bond->arpmon.targets, child->cdata);
	}
	return 0;
}

static int
nc_wicked_nif_xml_common_iftype_bond_arpmon(const xml_node_t *node, nc_bonding_t *bond)
{
	xml_node_t *child;

	bond->monitoring = NC_BOND_MONITOR_ARP;

	if ((child = xml_node_get_child(node, "interval"))) {
		nc_parse_uint(child->cdata, &bond->arpmon.interval, 10);
	}
	if ((child = xml_node_get_child(node, "validate"))) {
		nc_parse_int_mapped(child->cdata, nc_wicked_bond_arp_validate_map,
						&bond->arpmon.validate);
	}
	if ((child = xml_node_get_child(node, "targets"))) {
		nc_wicked_nif_xml_common_iftype_bond_arp_target(child, bond);
	}
	return 0;
}

static int
nc_wicked_nif_xml_common_iftype_bond_miimon(const xml_node_t *node, nc_bonding_t *bond)
{
	xml_node_t *child;

	bond->monitoring = NC_BOND_MONITOR_MII;

	if ((child = xml_node_get_child(node, "frequency"))) {
		nc_parse_uint(child->cdata, &bond->miimon.frequency, 10);
	}
	if ((child = xml_node_get_child(node, "carrier-detect"))) {
		nc_parse_int_mapped(child->cdata, nc_wicked_bond_carrier_map,
						&bond->miimon.carrier_detect);
	}
	if ((child = xml_node_get_child(node, "updelay"))) {
		nc_parse_uint(child->cdata, &bond->miimon.updelay, 10);
	}
	if ((child = xml_node_get_child(node, "downdelay"))) {
		nc_parse_uint(child->cdata, &bond->miimon.downdelay, 10);
	}
	return 0;
}

static int
nc_wicked_nif_xml_common_iftype_bond(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;
	nc_bonding_t *bond;

	if (!(bond = nc_bonding_new()))
		goto failure;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "mode"))
			nc_parse_int_mapped(child->cdata, nc_wicked_bond_mode_map, &bond->mode);
		else
		if (nc_string_eq(child->name, "miimon"))
			nc_wicked_nif_xml_common_iftype_bond_miimon(child, bond);
		else
		if (nc_string_eq(child->name, "arpmon"))
			nc_wicked_nif_xml_common_iftype_bond_arpmon(child, bond);
		else
		if (nc_string_eq(child->name, "slaves")) /* deprecated */
			nc_wicked_nif_xml_common_iftype_bond_slaves(child, bond);
		else
		if (nc_string_eq(child->name, "address"))        /* LLADDR in ifcfg */
			nc_hw_address_parse(&ifp->hwaddr, ifp->type, child->cdata);
	}

	if (ifp->bonding)
		nc_bonding_free(ifp->bonding);
	ifp->bonding = bond;
	return 0;

failure:
	ifp->invalid = 1;
	return -1;
}

static int
nc_wicked_nif_xml_common_iftype_bridge_port(const xml_node_t *node, nc_bridge_t *bridge)
{
	xml_node_t *child;
	const char *name = NULL;

	if ((child = xml_node_get_child(node, "device")))
		name = child->cdata;

	if (nc_string_empty(name) || nc_bridge_add_port(bridge, name))
		return -1;

	if ((child = xml_node_get_child(node, "priority")))
		nc_bridge_port_set_priority(bridge, name, child->cdata);

	if ((child = xml_node_get_child(node, "path-cost")))
		nc_bridge_port_set_path_cost(bridge, name, child->cdata);

	return 0;
}

static int
nc_wicked_nif_xml_common_iftype_bridge_ports(const xml_node_t *node, nc_bridge_t *bridge)
{
	xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "port"))
			nc_wicked_nif_xml_common_iftype_bridge_port(child, bridge);
	}
	return 0;
}

static int
nc_wicked_nif_xml_common_iftype_bridge(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;
	nc_bridge_t *bridge;

	if (!(bridge = nc_bridge_new()))
		goto failure;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "stp"))
			nc_bridge_set_stp(bridge, child->cdata);
		else
		if (nc_string_eq(child->name, "forward-delay"))
			nc_bridge_set_forward_delay(bridge, child->cdata);
		else
		if (nc_string_eq(child->name, "hello-time"))
			nc_bridge_set_hello_time(bridge, child->cdata);
		else
		if (nc_string_eq(child->name, "ageing-time") ||
			nc_string_eq(child->name, "aging-time")) /* typo in wicked */
			nc_bridge_set_ageing_time(bridge, child->cdata);
		else
		if (nc_string_eq(child->name, "max-age"))
			nc_bridge_set_max_age(bridge, child->cdata);
		else
		if (nc_string_eq(child->name, "priority"))
			nc_bridge_set_priority(bridge, child->cdata);
		else
		if (nc_string_eq(child->name, "ports"))	         /* deprecated */
			nc_wicked_nif_xml_common_iftype_bridge_ports(child, bridge);
		else
		if (nc_string_eq(child->name, "address"))        /* LLADDR in ifcfg */
			nc_hw_address_parse(&ifp->hwaddr, ifp->type, child->cdata);
	}

	if (ifp->bridge)
		nc_bridge_free(ifp->bridge);
	ifp->bridge = bridge;
	return 0;

failure:
	nc_bridge_free(bridge);
	ifp->invalid = 1;
	return -1;
}

static int
nc_wicked_nif_xml_common_iftype_ethernet(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "address"))        /* LLADDR in ifcfg */
			nc_hw_address_parse(&ifp->hwaddr, ifp->type, child->cdata);
	}

	return 0;
}

static int
nc_wicked_nif_xml_common_address(const xml_node_t *node, nc_interface_t *ifp, int af)
{
	const xml_node_t *child;
	nc_address_t *ap = NULL;
	unsigned int  plen = 0;
	unsigned int  flag;
	nc_sockaddr_t addr;
	const char *ip;

	if (!(child = xml_node_get_child(node, "local")))
		return 1;

	ip = child->cdata;
	if (nc_address_prefix_parse(&addr, &plen, ip, af))
		return -1;

	if (!(ap = nc_address_new(addr.ss_family, plen, &addr)))
		goto failure;

	if ((child = xml_node_get_child(node, "broadcast")) &&
	    nc_address_parse(&ap->bcast_addr, child->cdata, ap->family))
		goto failure;

	if ((child = xml_node_get_child(node, "peer")) &&
	    nc_address_parse(&ap->peer_addr, child->cdata, ap->family))
		goto failure;

	if ((child = xml_node_get_child(node, "owner")) &&
	    !nc_parse_int_mapped(node->name, nc_wicked_address_owner_map, &flag))
		ap->config_method = flag;

	if ((child = xml_node_get_child(node, "scope")) &&
	    !nc_parse_int_mapped(node->name, nc_wicked_route_scope_map, &flag))
		ap->scope = flag;

	if ((child = xml_node_get_child(node, "flags")) &&
	    !nc_parse_uint(child->cdata, &flag, 10))
		{} /* not used yet */

	if ((child = xml_node_get_child(node, "label")) &&
	    !nc_string_empty(child->cdata))
		snprintf(ap->label, sizeof(ap->label), "%s", child->cdata);

	if (nc_address_array_append(&ifp->addrs, ap) == 0)
		return 0;

failure:
	if (ap)
		nc_address_free(ap);
	return -1;
}

static int
nc_wicked_nif_xml_common_route(const xml_node_t *node, nc_interface_t *ifp, int af)
{
	const xml_node_t *child, *value;
	nc_string_array_t options = NC_STRING_ARRAY_INIT;
	nc_stringbuf_t opts = NC_STRINGBUF_INIT;
	const char *dev = ifp->name;
	unsigned int  plen = 0;
	nc_sockaddr_t dest;
	nc_sockaddr_t gway;
	unsigned int weight = 0;
	nc_route_t *rp = NULL;
	int ret = -1;

	memset(&dest, 0, sizeof(dest)); dest.ss_family = af;
	memset(&gway, 0, sizeof(gway));
	if ((value = xml_node_get_child(node, "destination"))) {
		if (nc_address_prefix_parse(&dest, &plen, value->cdata, af))
			goto failure;
	}
	if ((value = xml_node_get_child(node, "pref-source")) &&
		!nc_string_empty(value->cdata)) {
		nc_string_array_append(&options, "src");
		nc_string_array_append(&options, value->cdata);
	}
	if ((value = xml_node_get_child(node, "priority")) &&
		!nc_string_empty(value->cdata)) {
		nc_string_array_append(&options, "metric");
		nc_string_array_append(&options, value->cdata);
	}
	if ((child = xml_node_get_child(node, "nexthop"))) {
		if ((value = xml_node_get_child(child, "gateway"))) {
			if (nc_address_parse(&gway, value->cdata, af))
				goto failure;
		}
		if ((value = xml_node_get_child(child, "device")) &&
			!nc_string_empty(value->cdata)) {
			dev = value->cdata;
		}
		if ((value = xml_node_get_child(child, "weight")))
			nc_parse_uint(value->cdata, &weight, 10);
	}
	if ((child = xml_node_get_child(node, "kern"))) {
		if ((value = xml_node_get_child(child, "type")) &&
			nc_string_empty(value->cdata)) {
			nc_string_array_append(&options, value->name);
			nc_string_array_append(&options, value->cdata);
		}
		if ((value = xml_node_get_child(child, "scope")) &&
			nc_string_empty(value->cdata)) {
			nc_string_array_append(&options, value->name);
			nc_string_array_append(&options, value->cdata);
		}
		if ((value = xml_node_get_child(child, "table")) &&
			nc_string_empty(value->cdata)) {
			nc_string_array_append(&options, value->name);
			nc_string_array_append(&options, value->cdata);
		}
		if ((value = xml_node_get_child(child, "protocol")) &&
			nc_string_empty(value->cdata)) {
			nc_string_array_append(&options, value->name);
			nc_string_array_append(&options, value->cdata);
		}
	}

	nc_stringbuf_join(&opts, &options, " ");
	if (!(rp = nc_route_create(plen, &dest, &gway, dev, opts.string)))
		goto failure;

	rp->nh.weight = weight;
	if ((ret = nc_route_array_append(&ifp->routes, rp)) == 0)
		rp = NULL;

failure:
	nc_route_free(rp);
	nc_stringbuf_destroy(&opts);
	nc_string_array_destroy(&options);
	return ret;
}

static const char *
nc_wicked_nif_xml_config_get_ifname(const xml_node_t *node)
{
	xml_node_t *child;

	if (!(child = xml_node_get_child(node, "name")))
		return NULL;

	return child->cdata;
}

static const char *
nc_wicked_nif_xml_config_get_origin(const xml_node_t *node)
{
	return xml_node_get_attr(node, "origin");
}

static const char *
nc_wicked_nif_xml_config_get_uuid(const xml_node_t *node)
{
	return xml_node_get_attr(node, "origin");
}

static int
nc_wicked_nif_xml_config_origin(const xml_node_t *node, nc_interface_t *ifp)
{
	nc_string_dup(&ifp->config.origin, nc_wicked_nif_xml_config_get_origin(node));
	nc_string_dup(&ifp->config.uuid,   nc_wicked_nif_xml_config_get_uuid(node));
	return nc_string_empty(ifp->config.origin);
}

static int
nc_wicked_nif_xml_config_iftype_node(const xml_node_t *node, nc_interface_t *ifp)
{
	unsigned int iftype;

	if (nc_parse_int_mapped(node->name, nc_wicked_iftype_map, &iftype))
		return 1;

	ifp->type = iftype;
	switch (ifp->type) {
		case NC_IFTYPE_ETHERNET:
			return nc_wicked_nif_xml_common_iftype_ethernet(node, ifp);
		case NC_IFTYPE_BRIDGE:
			return nc_wicked_nif_xml_common_iftype_bridge(node, ifp);
		case NC_IFTYPE_BOND:
			return nc_wicked_nif_xml_common_iftype_bond(node, ifp);
		case NC_IFTYPE_VLAN:
			return nc_wicked_nif_xml_common_iftype_vlan(node, ifp);
		default:
			break;
	}
	return 1;
}

static int
nc_wicked_nif_xml_config_link(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "master"))
			nc_string_dup(&ifp->parent, child->cdata);
		else
		if (nc_string_eq(child->name, "mtu"))
			nc_parse_uint(child->cdata, &ifp->mtu, 10);
	}
	return 0;
}

static int
nc_wicked_nif_xml_config_ipv4(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;

	if ((child = xml_node_get_child(node, "enabled"))) {
		if (nc_parse_bool(child->cdata, &ifp->ipv4.enabled)) {
			if (!nc_string_empty(ifp->parent))
				ifp->ipv4.enabled = 0;
		}
	}

	if (!ifp->ipv4.enabled) {
		ifp->ipv4.addrconf = 0;
		return 0;
	}

	if ((child = xml_node_get_child(node, "forwarding")))
		nc_parse_bool(child->cdata, &ifp->ipv4.forwarding);

	return 0;
}

static int
nc_wicked_nif_xml_config_ipv6(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;
	int accept_ra = 1;
	int autoconf = 1;

	if ((child = xml_node_get_child(node, "enabled"))) {
		if (nc_parse_bool(child->cdata, &ifp->ipv6.enabled)) {
			if (!nc_string_empty(ifp->parent))
				ifp->ipv6.enabled = 0;
		}
	}

	if (!ifp->ipv6.enabled) {
		ifp->ipv6.addrconf = 0;
		return 0;
	}

	/* Act as router when forwarding is enabled.*/
	if ((child = xml_node_get_child(node, "forwarding")))
		nc_parse_bool(child->cdata, &ifp->ipv6.forwarding);

	/* Whether to accept IPv6 RA from other routers when we're in
	 * host mode (1) or explicitly enabled also in router mode (2).
	 */
	if ((child = xml_node_get_child(node, "accept-ra"))) {
		if (nc_string_eq(child->cdata, "disable"))
			accept_ra = 0;
		else
		if (nc_string_eq(child->cdata, "host"))
			accept_ra = 1;
		else
		if (nc_string_eq(child->cdata, "router"))
			accept_ra = 2;
	}

	/* Enabled autoconf is a permission flag for the kernel to
	 * autoconfigure (stateless) IPv6 address on RA prefix(es)
	 * having the AdvAutonomous flag set.
	 */
	if ((child = xml_node_get_child(node, "autoconf")))
		nc_parse_bool(child->cdata, &autoconf);

	/* Enable autoconf considering it's pre-requisite flags. */
	if (autoconf && accept_ra && accept_ra > ifp->ipv6.forwarding)
		ifp->ipv6.addrconf |=  NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
	else
		ifp->ipv6.addrconf &= ~NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);

	return 0;
}

static int
nc_wicked_nif_xml_config_dhcp4(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;
	int enabled = 0;

	if (!ifp->ipv4.enabled ||
	    !(child = xml_node_get_child(node, "enabled"))  ||
	    nc_parse_bool(child->cdata, &enabled) || !enabled) {
		ifp->ipv4.addrconf &= ~NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
		return 0;
	}

	ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
	return 0;
}

static int
nc_wicked_nif_xml_config_dhcp6(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;
	int enabled = 0;

	if (!ifp->ipv6.enabled ||
	    !(child = xml_node_get_child(node, "enabled"))  ||
	    nc_parse_bool(child->cdata, &enabled) || !enabled) {
		ifp->ipv6.addrconf &= ~NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
		return 0;
	}

	/* DHCPv6 can run in an "auto" mode (default) resolved into a
	 * - "managed" (request stateful address and other config),
	 * - "info" (non-address other-config only, e.g. dns + ntp),
	 * - ! remain silent (e.g. when RA is missed) !
	 * or also in explicitly requested "managed" / "info" mode
	 * in combination with the optional explicit request of a
	 * "prefix" mode to request a prefix network (pool).
	 * While "auto" mode depends on accept_ra sysctl and flags
	 * in the RA, prefix and explicit managed/info modes don't
	 * ([missed] RA override).
	 *
	 * Regardless of the above details, this node enables the use
	 * of the DHCPv6 client for address and/or prefix configuration.
	 */
	ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);
	return 0;
}

static int
nc_wicked_nif_xml_config_auto4(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;
	int enabled = 0;

	if (!ifp->ipv4.enabled ||
	    !(child = xml_node_get_child(node, "enabled"))  ||
	    nc_parse_bool(child->cdata, &enabled) || !enabled) {
		ifp->ipv4.addrconf &= ~NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
		return 1;
	}

	/* Optionally enabled ipv4 autoconf aka zeroconf aka ...
	 * either as dhcp4 fallback or independently.
	 */
	ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
	return 0;
}

static int
nc_wicked_nif_xml_config_auto6(const xml_node_t *node, nc_interface_t *ifp)
{
	(void)node;
	(void)ifp;
	/*
	 * Enablement in the kernel is made via sysctl's handled by
	 * config_ipv6() already.
	 * netcontrol does not consider further flags or tweaks.
	 *
	 * This node can disable the auto6 IP address and rdnss/dnssl
	 * option "on demand" autoconf "lease" processing in wicked or
	 * also enable wicked to explicitly wait for the RA, addresses
	 * and perhaps also rdnss/dnssl options to arrive.
	 *
	const xml_node_t *child;
	int enabled = 0;

	if (!ifp->ipv6.enabled ||
	    !(child = xml_node_get_child(node, "enabled"))  ||
	    nc_parse_bool(child->cdata, &enabled) || !enabled) {
		ifp->ipv6.addrconf &= ~NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
		return 1;
	}

	ifp->ipv6.addrconf |=  NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
	*/
	return 0;
}

static int
nc_wicked_nif_xml_config_static4(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	if (!ifp->ipv4.enabled)
		return 0;

	ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "address"))
			nc_wicked_nif_xml_common_address(child, ifp, AF_INET);
		else
		if (nc_string_eq(child->name, "route"))
			nc_wicked_nif_xml_common_route(child, ifp, AF_INET);
	}
	return 0;
}

static int
nc_wicked_nif_xml_config_static6(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	if (!ifp->ipv6.enabled)
		return 0;

	ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "address"))
			nc_wicked_nif_xml_common_address(child, ifp, AF_INET6);
		else
		if (nc_string_eq(child->name, "route"))
			nc_wicked_nif_xml_common_route(child, ifp, AF_INET6);
	}
	return 0;
}

static int
nc_wicked_nif_xml_config_control(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *persistent;
	const xml_node_t *startmode;
	int enabled = 0;

	startmode = xml_node_get_child(node, "mode");
	if (!startmode || nc_parse_int_mapped(startmode->cdata,
			__startmode_name_map, &ifp->startmode)) {
		ifp->startmode = NC_STARTMODE_MANUAL;
	}

	persistent = xml_node_get_child(node, "persistent");
	if (persistent && nc_parse_bool(persistent->cdata,
			&enabled) == 0 && enabled) {
		ifp->startmode = NC_STARTMODE_NFSROOT;
	}

	return 0;
}

static int
nc_wicked_nif_xml_config_update(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	nc_wicked_nif_xml_config_origin(node, ifp);
	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "name"))
			continue; /* we already have it */
		else
		if (nc_string_eq(child->name, "link"))
			nc_wicked_nif_xml_config_link(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv4"))
			nc_wicked_nif_xml_config_ipv4(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv6"))
			nc_wicked_nif_xml_config_ipv6(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv4:dhcp"))
			nc_wicked_nif_xml_config_dhcp4(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv6:dhcp"))
			nc_wicked_nif_xml_config_dhcp6(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv4:auto"))
			nc_wicked_nif_xml_config_auto4(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv6:auto"))
			nc_wicked_nif_xml_config_auto6(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv4:static"))
			nc_wicked_nif_xml_config_static4(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv6:static"))
			nc_wicked_nif_xml_config_static6(child, ifp);
		else
		if (nc_string_eq(child->name, "firewall"))
			continue; /* does not matter */
		else
		if (nc_string_eq(child->name, "control"))
			nc_wicked_nif_xml_config_control(child, ifp);
		else
			nc_wicked_nif_xml_config_iftype_node(child, ifp);
	}

	return 0;
}

static void
nc_wicked_nif_reset(nc_handle_t *nh)
{
	nc_interface_t *ifp;
	unsigned int i;

	for (i = 0; i < nh->ifaces.count; ++i) {
		ifp = nh->ifaces.data[i];
		nc_interface_reset(ifp);
		ifp->deleted = 1;
	}
}

static void
nc_wicked_nif_cleanup(nc_handle_t *nh)
{
	nc_interface_t *ifp;
	unsigned int i;

	for (i = 0; i < nh->ifaces.count; ++i) {
		ifp = nh->ifaces.data[i];
		if (!ifp->deleted)
			continue;
		nc_interface_array_remove_index(&nh->ifaces, i--);
	}
}

static int
nc_wicked_nif_xml_common_adjust_parent(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_interface_t *ref;

	if (!ifp || ifp->invalid)
		return -1;

	if (nc_string_empty(ifp->parent))
		return 0;

	if (!(ref = nc_interface_by_name(nh, ifp->parent)))
		return -1;

	switch (ref->type) {
		case NC_IFTYPE_BRIDGE:
			if (ref->bridge)
				nc_bridge_add_port(ref->bridge, ifp->name);
			break;
		case NC_IFTYPE_BOND:
			if (ref->bonding)
				nc_bonding_add_slave(ref->bonding, ifp->name);
			break;
		default:
			/* other, unsupported master type (e.g. ovs datapath) */
			break;
	}
	return 0;
}

static int
nc_wicked_nif_xml_config_adjust_bridge(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_bridge_port_t *port;
	nc_interface_t *ref;
	unsigned int i;

	if (!ifp || !ifp->bridge || ifp->invalid)
		return -1;

	for (i = 0; i < ifp->bridge->ports.count; ++i) {
		if (!(port = ifp->bridge->ports.data[i]))
			continue;

		if (!(ref = nc_interface_by_name(nh, port->name)))
			continue;

		if (nc_string_empty(ref->parent))
			nc_string_dup(&ref->parent, ifp->name);

		if (!nc_string_eq(ifp->name, ref->parent))
			ref->invalid = ifp->invalid = 1;
		else
		if (!nc_interface_guess_type(ref))
			ref->type = NC_IFTYPE_ETHERNET;
	}

	return 0;
}

static int
nc_wicked_nif_xml_config_adjust_bond(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_interface_t *ref;
	const char *name;
	unsigned int i;

	if (!ifp || !ifp->bonding || ifp->invalid)
		return -1;

	for (i = 0; i < ifp->bonding->slave_names.count; ++i) {
		if (!(name = ifp->bonding->slave_names.data[i]))
			continue;

		if (!(ref = nc_interface_by_name(nh, name)))
			continue;

		if (nc_string_empty(ref->parent))
			nc_string_dup(&ref->parent, ifp->name);

		if (!nc_string_eq(ifp->name, ref->parent))
			ref->invalid = ifp->invalid = 1;
		else
		switch (ifp->bonding->slave_type) {
			case NC_IFTYPE_INFINIBAND:
				if (ref->type == NC_IFTYPE_ETHERNET)
					ifp->invalid = 1;
				else
				if (ref->type == NC_IFTYPE_UNKNOWN)
					ref->type = NC_IFTYPE_INFINIBAND;
				break;

			case NC_IFTYPE_ETHERNET:
				if (ref->type == NC_IFTYPE_INFINIBAND)
					ifp->invalid = 1;
				else
				if (ref->type == NC_IFTYPE_UNKNOWN)
					ref->type = NC_IFTYPE_ETHERNET;
				break;

			case NC_IFTYPE_UNKNOWN:
				if (ref->type != NC_IFTYPE_INFINIBAND &&
				    !nc_interface_guess_type(ref))
					ref->type = NC_IFTYPE_ETHERNET;
				ifp->bonding->slave_type = ref->type;
				break;
			default:
				break;
		}
	}

	return 0;
}

static int
nc_wicked_nif_xml_config_adjust_vlan(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_interface_t *ref;

	if (!ifp || !ifp->vlan || ifp->invalid)
		return -1;

	if (!(ref = nc_interface_by_name(nh, ifp->vlan->interface_name)))
		return -1;

	if (ref->invalid)
		return -1;

	if (ref->type != NC_IFTYPE_UNKNOWN)
		return 0;

	if (!nc_interface_guess_type(ref))
		ref->type = NC_IFTYPE_ETHERNET;

	return 0;
}

static void
nc_wicked_nif_xml_config_adjust_tree(nc_handle_t *nh)
{
	nc_interface_t *ifp;
	unsigned int i;

	/* we need to complete & verify relantions, adjust (guess) unknown
	 * config types, mark invalid sub-trees like bogus bondings with
	 * infiniband and ethernet slave mix, ...
	 */
	for (i = 0; i < nh->ifaces.count; ++i) {
		if (!(ifp = nh->ifaces.data[i]))
			continue;
		if (nc_wicked_nif_xml_common_adjust_parent(nh, ifp))
			ifp->invalid = 1;
	}
	for (i = 0; i < nh->ifaces.count; ++i) {
		if (!(ifp = nh->ifaces.data[i]))
			continue;
		switch (ifp->type) {
			case NC_IFTYPE_BRIDGE:
				if (nc_wicked_nif_xml_config_adjust_bridge(nh, ifp))
					ifp->invalid = 1;
				break;
			case NC_IFTYPE_BOND:
				if (nc_wicked_nif_xml_config_adjust_bond(nh, ifp))
					ifp->invalid = 1;
				break;
			case NC_IFTYPE_VLAN:
				if (nc_wicked_nif_xml_config_adjust_vlan(nh, ifp))
					ifp->invalid = 1;
				break;
			default:
				break;
		}
	}
	for (i = 0; i < nh->ifaces.count; ++i) {
		if (!(ifp = nh->ifaces.data[i]))
			continue;

		nc_interface_guess_type(ifp);
		if (ifp->arp_type == ARPHRD_VOID)
			ifp->arp_type = nc_iftype_to_arphrd_type(ifp->type);
	}
}

static int
nc_suse_config_wicked_refresh(nc_handle_t *nh)
{
	const xml_node_t *root, *node;
	xml_document_t *doc = NULL;
	const char *ifname = NULL;
	const char *origin = NULL;
	nc_interface_t *ifp;

	if (nc_wicked_cli_show_xml_config("all", &doc) != 0 || !doc)
		return -1;

	root = xml_document_root(doc);
	nc_wicked_nif_reset(nh);
	for (node = root->children; node; node = node->next) {
		if (!nc_string_eq(node->name, "interface"))
			continue;

		ifname = nc_wicked_nif_xml_config_get_ifname(node);
		if (nc_string_empty(ifname))
			continue;

		origin = nc_wicked_nif_xml_config_get_origin(node);
		if (!nc_string_prefix_eq(WICKED_NIF_XML_IFCFG_ORIGIN, origin))
			continue;

		if (!(ifp = nc_interface_by_ifname(nh, ifname))) {
			if (!(ifp = nc_interface_new(ifname, 0))) {
				xml_document_free(doc);
				nc_error("Unable to allocate new interface structure for name %s",
					ifname);
				return -1;
			}
			nc_interfaces_add(nh, ifp);
		}
		ifp->deleted = 0;
		nc_wicked_nif_xml_config_update(node, ifp);
	}
	nc_wicked_nif_cleanup(nh);
	xml_document_free(doc);
	nc_wicked_nif_xml_config_adjust_tree(nh);

	return 0;
}

static int
nc_wicked_nif_xml_object_index(const xml_node_t *node, unsigned int *ifindex)
{
	const char *path;
	xml_node_t *child;

	if (!nc_string_eq(node->name, WICKED_NIF_XML_OBJECT_NODE))
		return -1;

	path = xml_node_get_attr(node, WICKED_NIF_XML_OBJECT_PATH);
	if (!nc_string_prefix_eq(WICKED_NIF_XML_OBJECT_LIST, path))
		return -1;

	if (!(child = xml_node_get_child(node, "interface")))
		return -1;
	if (!(child = xml_node_get_child(child, "index")))
		return -1;
	if (nc_parse_uint(child->cdata, ifindex, 10) != 0)
		return -1;

	return 0;
}

static int
nc_wicked_nif_xml_system_ifflags(const xml_node_t *node, nc_interface_t *ifp)
{
	char *s, *p = NULL, *tmp = NULL;
	const char *sep = ", |";
	unsigned int flag;

	if (nc_string_dup(&tmp, node->cdata) || !tmp)
		return -1;

	ifp->ifflags = 0;
	for (s = strtok_r(tmp, sep, &p); s; s = strtok_r(NULL, sep, &p)) {
		if (nc_parse_int_mapped(s, nc_wicked_ifflag_map, &flag))
			continue;
		ifp->ifflags |= flag;
	}
	return 0;
}

static int
nc_wicked_nif_xml_system_client_config(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "origin"))
			nc_string_dup(&ifp->config.origin, child->cdata);
		else
		if (nc_string_eq(child->name, "uuid"))
			nc_string_dup(&ifp->config.uuid, child->cdata);
	}
	return 0;
}

static int
nc_wicked_nif_xml_system_client_control(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "persistent"))
			ifp->startmode = NC_STARTMODE_NFSROOT;
	}
	return 0;
}

static int
nc_wicked_nif_xml_system_client_state(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "control"))
			nc_wicked_nif_xml_system_client_control(child, ifp);
		else
		if (nc_string_eq(child->name, "config"))
			nc_wicked_nif_xml_system_client_config(child, ifp);
	}
	return 0;
}

static int
nc_wicked_nif_xml_system_addresses(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "assigned-address"))
			nc_wicked_nif_xml_common_address(child, ifp, AF_UNSPEC);
	}
	return 0;
}

static int
nc_wicked_nif_xml_system_routes(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "assigned-route"))
			nc_wicked_nif_xml_common_route(child, ifp, AF_UNSPEC);
	}
	return 0;
}

static int
nc_wicked_nif_xml_system_ifinfo_node(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "name"))
			nc_string_dup(&ifp->name, child->cdata);
		else
		if (nc_string_eq(child->name, "index"))
			continue; /* we already have it */
		else
		if (nc_string_eq(child->name, "status"))
			nc_wicked_nif_xml_system_ifflags(child, ifp);
		else
		if (nc_string_eq(child->name, "link-type"))
			nc_parse_int_mapped(child->cdata, nc_wicked_iftype_map, &ifp->type);
		else
		if (nc_string_eq(child->name, "mtu"))
			nc_parse_uint(child->cdata, &ifp->mtu, 10);
		else
		if (nc_string_eq(child->name, "txqlen"))
			continue;
		else
		if (nc_string_eq(child->name, "master"))
			nc_string_dup(&ifp->parent, child->cdata);
		else
		if (nc_string_eq(child->name, "client-state"))
			nc_wicked_nif_xml_system_client_state(child, ifp);
		else
		if (nc_string_eq(child->name, "addresses"))
			nc_wicked_nif_xml_system_addresses(child, ifp);
		else
		if (nc_string_eq(child->name, "routes"))
			nc_wicked_nif_xml_system_routes(child, ifp);
	}
	return 0;
}

static int
nc_wicked_nif_xml_system_iftype_node(const xml_node_t *node, nc_interface_t *ifp)
{
	unsigned int type = ifp->type;

	if (nc_parse_int_mapped(node->name, nc_wicked_iftype_map, &type))
		return 1;

	if (type != ifp->type) {
		nc_error("%s: link type %s and type data %s node do not match",
			ifp->name, nc_format_int_mapped(ifp->type, nc_wicked_iftype_map),
			nc_format_int_mapped(type, nc_wicked_iftype_map));
		return -1;
	}

	switch (type) {
		case NC_IFTYPE_ETHERNET:
			return nc_wicked_nif_xml_common_iftype_ethernet(node, ifp);
		case NC_IFTYPE_BRIDGE:
			return nc_wicked_nif_xml_common_iftype_bridge(node, ifp);
		case NC_IFTYPE_BOND:
			return nc_wicked_nif_xml_common_iftype_bond(node, ifp);
		case NC_IFTYPE_VLAN:
			return nc_wicked_nif_xml_common_iftype_vlan(node, ifp);
		default:
			break;
	}
	return 1;
}

static int
nc_wicked_nif_xml_system_ipv4(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;

	if ((child = xml_node_get_child(node, "enabled"))) {
		if (nc_parse_bool(child->cdata, &ifp->ipv4.enabled)) {
			if (!nc_string_empty(ifp->parent))
				ifp->ipv4.enabled = 0;
		}
	}

	if (!ifp->ipv4.enabled)
		return 0;

	if ((child = xml_node_get_child(node, "forwarding")))
		nc_parse_bool(child->cdata, &ifp->ipv4.forwarding);

	return 0;
}

static int
nc_wicked_nif_xml_system_ipv6(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *child;
	int accept_ra = 1;
	int autoconf = 1;

	if ((child = xml_node_get_child(node, "enabled"))) {
		if (nc_parse_bool(child->cdata, &ifp->ipv6.enabled)) {
			if (!nc_string_empty(ifp->parent))
				ifp->ipv6.enabled = 0;
		}
	}

	if (!ifp->ipv6.enabled)
		return 0;

	if ((child = xml_node_get_child(node, "forwarding")))
		nc_parse_bool(child->cdata, &ifp->ipv6.forwarding);
	if ((child = xml_node_get_child(node, "accept-ra"))) {
		if (nc_string_eq(child->cdata, "disable"))
			accept_ra = 0;
		else
		if (nc_string_eq(child->cdata, "host"))
			accept_ra = 1;
		else
		if (nc_string_eq(child->cdata, "router"))
			accept_ra = 2;
	}
	if ((child = xml_node_get_child(node, "autoconf")))
		nc_parse_bool(child->cdata, &autoconf);

	if (autoconf && ifp->ipv6.forwarding && accept_ra > 1)
		ifp->ipv6.addrconf |=  NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
	else
	if (autoconf && !ifp->ipv6.forwarding && accept_ra > 0)
		ifp->ipv6.addrconf |=  NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);
	else
		ifp->ipv6.addrconf &= ~NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);

	return 0;
}

static int
nc_wicked_nif_xml_system_dhcp4(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *lease;
	xml_node_t *value;

	if (!(lease = xml_node_get_child(node, "lease")))
		return 0;

	if ((value = xml_node_get_child(lease, "state")))
		ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);

	return 0;
}

static int
nc_wicked_nif_xml_system_dhcp6(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *lease;
	xml_node_t *value;

	if (!(lease = xml_node_get_child(node, "lease")))
		return 0;

	if ((value = xml_node_get_child(lease, "state")))
		ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_DHCP);

	return 0;
}

static int
nc_wicked_nif_xml_system_auto4(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *lease;
	xml_node_t *value;

	if (!(lease = xml_node_get_child(node, "lease")))
		return 0;

	if ((value = xml_node_get_child(lease, "state")))
		ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);

	return 0;
}

static int
nc_wicked_nif_xml_system_auto6(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *lease;
	xml_node_t *value;

	if (!(lease = xml_node_get_child(node, "lease")))
		return 0;

	if ((value = xml_node_get_child(lease, "state")))
		ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_AUTOCONF);

	return 0;
}


static int
nc_wicked_nif_xml_system_static4(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *lease;
	xml_node_t *value;

	if (!(lease = xml_node_get_child(node, "lease")))
		return 0;

	if ((value = xml_node_get_child(lease, "state")))
		ifp->ipv4.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);

	return 0;
}

static int
nc_wicked_nif_xml_system_static6(const xml_node_t *node, nc_interface_t *ifp)
{
	xml_node_t *lease;
	xml_node_t *value;

	if (!(lease = xml_node_get_child(node, "lease")))
		return 0;

	if ((value = xml_node_get_child(lease, "state")))
		ifp->ipv6.addrconf |= NC_ADDRCONF_MASK(NC_ADDRCONF_STATIC);

	return 0;
}

static int
nc_wicked_nif_xml_system_update(const xml_node_t *node, nc_interface_t *ifp)
{
	const xml_node_t *child;

	for (child = node->children; child; child = child->next) {
		if (nc_string_eq(child->name, "interface"))
			nc_wicked_nif_xml_system_ifinfo_node(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv4"))
			nc_wicked_nif_xml_system_ipv4(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv6"))
			nc_wicked_nif_xml_system_ipv6(child, ifp);
		else
		if (nc_string_eq(child->name, "ethtool"))
			continue; /* we currenty don't use it */
		else
		if (nc_string_eq(child->name, "ipv4:dhcp"))
			nc_wicked_nif_xml_system_dhcp4(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv6:dhcp"))
			nc_wicked_nif_xml_system_dhcp6(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv4:auto"))
			nc_wicked_nif_xml_system_auto4(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv6:auto"))
			nc_wicked_nif_xml_system_auto6(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv4:static"))
			nc_wicked_nif_xml_system_static4(child, ifp);
		else
		if (nc_string_eq(child->name, "ipv6:static"))
			nc_wicked_nif_xml_system_static6(child, ifp);
		else
			nc_wicked_nif_xml_system_iftype_node(child, ifp);
	}
	return 0;
}

static int
nc_wicked_nif_xml_system_adjust_bridge(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_bridge_port_t *port;
	nc_interface_t *ref;
	unsigned int i;

	if (!ifp || !ifp->bridge || ifp->invalid)
		return -1;

	for (i = 0; i < ifp->bridge->ports.count; ++i) {
		if (!(port = ifp->bridge->ports.data[i]))
			continue;

		if (!(ref = nc_interface_by_name(nh, port->name)))
			continue;

		if (nc_string_empty(ref->parent))
			nc_string_dup(&ref->parent, ifp->name);

		if (!nc_string_eq(ifp->name, ref->parent))
			ref->invalid = ifp->invalid = 1;
	}

	return 0;
}

static int
nc_wicked_nif_xml_system_adjust_bond(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_interface_t *ref;
	const char *name;
	unsigned int i;

	if (!ifp || !ifp->bonding || ifp->invalid)
		return -1;

	for (i = 0; i < ifp->bonding->slave_names.count; ++i) {
		if (!(name = ifp->bonding->slave_names.data[i]))
			continue;

		if (!(ref = nc_interface_by_name(nh, name)))
			continue;

		if (nc_string_empty(ref->parent))
			nc_string_dup(&ref->parent, ifp->name);

		if (!nc_string_eq(ifp->name, ref->parent))
			ref->invalid = ifp->invalid = 1;
		else
		if (ifp->bonding->slave_type == NC_IFTYPE_UNKNOWN) {
			/* we always know the types from system */
			if (ref->type == NC_IFTYPE_INFINIBAND)
				ifp->bonding->slave_type = NC_IFTYPE_INFINIBAND;
			else
			if (ref->type == NC_IFTYPE_ETHERNET)
				ifp->bonding->slave_type = NC_IFTYPE_ETHERNET;
		}
	}
	if (ifp->bonding->slave_type == NC_IFTYPE_INFINIBAND)
		ifp->arp_type = ARPHRD_INFINIBAND;
	else
		ifp->arp_type = ARPHRD_ETHER;
	return 0;
}

static int
nc_wicked_nif_xml_system_adjust_vlan(nc_handle_t *nh, nc_interface_t *ifp)
{
	nc_interface_t *ref;

	if (!ifp || !ifp->vlan || ifp->invalid)
		return -1;

	if (!(ref = nc_interface_by_name(nh, ifp->vlan->interface_name)))
		return -1;

	if (ref->invalid)
		return -1;

	if (ref->type == NC_IFTYPE_UNKNOWN)
		return -1;

	if (!nc_interface_guess_type(ref))
		ref->type = NC_IFTYPE_ETHERNET;

	return 0;
}


static int
nc_wicked_nif_xml_system_adjust_tree(nc_handle_t *nh)
{
	nc_interface_t *ifp;
	unsigned int i;

	for (i = 0; i < nh->ifaces.count; ++i) {
		if (!(ifp = nh->ifaces.data[i]))
			continue;
		if (nc_wicked_nif_xml_common_adjust_parent(nh, ifp))
			ifp->invalid = 1;
	}
	for (i = 0; i < nh->ifaces.count; ++i) {
		if (!(ifp = nh->ifaces.data[i]))
			continue;
		switch (ifp->type) {
			case NC_IFTYPE_BOND:
				if (nc_wicked_nif_xml_system_adjust_bond(nh, ifp))
					ifp->invalid = 1;
				break;
			case NC_IFTYPE_BRIDGE:
				if (nc_wicked_nif_xml_system_adjust_bridge(nh, ifp))
					ifp->invalid = 1;
				break;
			case NC_IFTYPE_VLAN:
				if (nc_wicked_nif_xml_system_adjust_vlan(nh, ifp))
					ifp->invalid = 1;
				break;
			default:
				break;
		}
		if (ifp->arp_type == ARPHRD_VOID)
			ifp->arp_type = nc_iftype_to_arphrd_type(ifp->type);
	}
	return 0;
}

static int
nc_suse_system_wicked_refresh(nc_handle_t *nh)
{
	xml_document_t *doc = NULL;
	const xml_node_t *root, *node;
	unsigned int ifindex;
	nc_interface_t *ifp;

	if (nc_wicked_cli_show_xml_status("all", &doc) != 0 || !doc)
		return -1;

	root = xml_document_root(doc);
	nc_wicked_nif_reset(nh);
	for (node = root->children; node; node = node->next) {
		if (nc_wicked_nif_xml_object_index(node, &ifindex) || !ifindex)
			continue;

		if (!(ifp = nc_interface_by_ifindex(nh, ifindex))) {
			if (!(ifp = nc_interface_new(NULL, ifindex))) {
				xml_document_free(doc);
				nc_error("Unable to allocate new interface structure for index %u",
					ifindex);
				return -1;
			}
			nc_interfaces_add(nh, ifp);
		}

		ifp->deleted = 0;
		nc_wicked_nif_xml_system_update(node, ifp);
	}
	xml_document_free(doc);
	nc_wicked_nif_cleanup(nh);
	nc_wicked_nif_xml_system_adjust_tree(nh);

	return 0;
}

static int
nc_suse_system_wicked_status(nc_handle_t *nh, nc_interface_t *ifp)
{
	/*
	 * we're reading the complete state while refresh (above),
	 * so there is no need to query `show-xml $ifname` again.
	 */
	(void)nh;
	return !nc_interface_link_is_up(ifp);
}

static int
nc_wicked_cli_xml_child_exec(nc_string_array_t *args)
{
	/* redirect stdin + stderr and exec it */
	if (!freopen("/dev/null", "r", stdin))
		return -1;
	if (!freopen("/dev/null", "r", stderr))
		return -1;
	return execv(args->data[0], args->data);
}

static int
nc_wicked_cli_xml_call(nc_string_array_t *args, int *code, xml_document_t **doc)
{
	nc_stringbuf_t buf = NC_STRINGBUF_INIT;
	nc_cmd_pipe_t  cmd;
	int            status;

	if (!doc || !code || !args || args->count < 2 || !args->data[0] || *args->data[0] != '/') {
		nc_error("Invalid command execution arguments");
		return -1;
	}

	*doc = NULL;
	*code = -1;
	memset(&cmd, 0, sizeof(cmd));
	cmd.exec = nc_wicked_cli_xml_child_exec;
	cmd.args = args;
	if (nc_cmd_pipe_exec(&cmd) == -1) {
		nc_error("Unable to execute `'%s'`", nc_stringbuf_join(&buf, args,  "' '"));
		goto failure;
	}

	if (cmd.file) {
		*doc = xml_document_scan(cmd.file);
		fclose(cmd.file);
	}

	status = -1;
	if (nc_cmd_pipe_wait(&cmd, &status, 0) == -1) {
		nc_error("waitpid failure from `'%s'`: %m", nc_stringbuf_join(&buf, args,  "' '"));
		goto failure;
	}

	if (!WIFEXITED(status)) {
		nc_error("Command `'%s'` terminated abormally: %d [%m]",
			nc_stringbuf_join(&buf, args,  "' '"), status);
		goto failure;
	}

	*code = WEXITSTATUS(status);
	nc_info("Command `'%s'` returned with exit code: %d",
		nc_stringbuf_join(&buf, args,  "' '"), *code);

	nc_stringbuf_destroy(&buf);
	return 0;

failure:
	if (*doc) {
		xml_document_free(*doc);
		*doc = NULL;
	}
	nc_stringbuf_destroy(&buf);
	return -1;
}

static int
nc_wicked_cli_show_xml_config(const char *ifname, xml_document_t **doc)
{
	nc_string_array_t args = NC_STRING_ARRAY_INIT;
	int code = -1, ret;
	struct timeval started, elapsed;

	if (nc_string_empty(ifname))
		ifname = "all";

	nc_get_time(&started);

	nc_string_array_append(&args, WICKED_CLI_CMD);
	nc_string_array_append(&args, WICKED_CLI_SHOW_CONFIG);
	nc_string_array_append(&args, ifname);

	ret = nc_wicked_cli_xml_call(&args, &code, doc);
	nc_string_array_destroy(&args);

	nc_get_time_elapsed(&started, &elapsed);
	nc_debug_wicked("%s %s: elapsed time: %ld.%06lds",
			WICKED_CLI_SHOW_CONFIG, ifname,
			elapsed.tv_sec, elapsed.tv_usec);

	return ret;
}

static int
nc_wicked_cli_show_xml_status(const char *ifname, xml_document_t **doc)
{
	nc_string_array_t args = NC_STRING_ARRAY_INIT;
	int code = -1, ret;
	struct timeval started, elapsed;

	if (nc_string_empty(ifname))
		ifname = "all";

	nc_get_time(&started);

	nc_string_array_append(&args, WICKED_CLI_CMD);
	nc_string_array_append(&args, WICKED_CLI_SHOW_STATUS);
	nc_string_array_append(&args, ifname);

	ret = nc_wicked_cli_xml_call(&args, &code, doc);
	nc_string_array_destroy(&args);

	nc_get_time_elapsed(&started, &elapsed);
	nc_debug_wicked("%s %s: elapsed time: %ld.%06lds",
			WICKED_CLI_SHOW_STATUS, ifname,
			elapsed.tv_sec, elapsed.tv_usec);

	return ret;
}

