/**********************************************************************
 * $port_info example -- PLI application using VPI routines
 *
 * C source to print information about module ports.
 *
 * Usage: $port_info(<module_instance_name>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of routines in this PLI application */
int   PLIbook_PortInfo_compiletf(), PLIbook_PortInfo_calltf();
char *PLIbook_fetch_type_str_vpi();

/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_PortInfo_register()
{
  s_vpi_systf_data tf_data;
  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$port_info";
  tf_data.calltf    = PLIbook_PortInfo_calltf;
  tf_data.compiletf = PLIbook_PortInfo_compiletf;
  tf_data.sizetf    = NULL;

  vpi_register_systf(&tf_data);
}
/*********************************************************************/


/**********************************************************************
 * Compiletf application
 *********************************************************************/
int PLIbook_PortInfo_compiletf(char *user_data)
{
  vpiHandle systf_h, tfarg_itr, tfarg_h;

  systf_h = vpi_handle(vpiSysTfCall, NULL);
  if (systf_h == NULL) {
    vpi_printf("ERROR: list_pathout_ports could not obtain handle to systf call\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  tfarg_itr = vpi_iterate(vpiArgument, systf_h);
  if (systf_h == NULL) {
    vpi_printf("ERROR: list_pathout_ports could not obtain iterator to systf args\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  tfarg_h = vpi_scan(tfarg_itr);
  if (vpi_get(vpiType, tfarg_h) != vpiModule) {
    vpi_printf("ERROR: $list_pathout_ports arg must be module instance\n");
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  if (vpi_scan(tfarg_itr) != NULL) {
    tf_error("ERROR: $list_pathout_ports requires 1 argument\n");
    vpi_free_object(tfarg_itr); /* because not scanning until null */
    tf_dofinish(); /* abort simulation */
    return(0);
  }
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_PortInfo_calltf(char *user_data) 
{
  vpiHandle systf_h, arg_itr, mod_h,
            port_itr, port_h, lowconn_h, highconn_h;
  int       lowconn_type, highconn_type;

  /* get module handle from first system task argument.  Assume the  */
  /* compiletf routine has already verified correct argument type.   */
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  if (systf_h == NULL) {
    vpi_printf("ERROR: list_pathout_ports could not obtain handle to systf call\n");
    return(0);
  }
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  if (systf_h == NULL) {
    vpi_printf("ERROR: list_pathout_ports could not obtain iterator to systf args\n");
    return(0);
  }
  mod_h = vpi_scan(arg_itr);
  vpi_free_object(arg_itr); /* free itr since did not scan until nul */

  vpi_printf("\nModule %s (instance %s)\n",
             vpi_get_str(vpiDefName, mod_h), 
             vpi_get_str(vpiFullName, mod_h));

  port_itr = vpi_iterate(vpiPort, mod_h);
  if (!port_itr) {
    vpi_printf("   No ports found\n");
    return(0);
  }
  while (port_h = vpi_scan(port_itr)) {
    vpi_printf("  Port name is %s\n", vpi_get_str(vpiName, port_h));
    vpi_printf("    Size is %d\n", vpi_get(vpiSize, port_h));  
    switch (vpi_get(vpiDirection, port_h)) {
      case vpiInput:  vpi_printf("    Direction is input\n");  break;
      case vpiOutput: vpi_printf("    Direction is output\n"); break;
      case vpiInout:  vpi_printf("    Direction is inout\n");  break;
    }

    lowconn_h = vpi_handle(vpiLowConn, port_h);
    lowconn_type = vpi_get(vpiType, lowconn_h);
    vpi_printf("    Low conn data type is %s\n",
               PLIbook_fetch_type_str_vpi(lowconn_type));

    highconn_h = vpi_handle(vpiHighConn, port_h);
    if (!highconn_h) {
      vpi_printf("    No high conn\n");
      return(0);
    }
    highconn_type = vpi_get(vpiType, highconn_h);
    vpi_printf("    High conn data type is %s\n",
               PLIbook_fetch_type_str_vpi(highconn_type));
  }
  return(0);
}

/**********************************************************************
 * PLIbook_fetch_type_str_vpi()
 * Return the string name of a VPI constant.
 *********************************************************************/
char *PLIbook_fetch_type_str_vpi(int type)
{
  switch (type) {
    case  vpiNet            : return("vpiNet");
    case  vpiNetBit         : return("vpiNetBit");
    case  vpiReg            : return("vpiReg");
    case  vpiRegBit         : return("vpiRegBit");
    case  vpiPartSelect     : return("vpiPartSelect");
    case  vpiIntegerVar     : return("vpiIntegerVar");
    case  vpiTimeVar        : return("vpiTimeVar");
    case  vpiMemoryWord     : return("vpiMemoryWord");
    case  vpiVarSelect      : return("vpiVarSelect");
    case  vpiConstant       : return("vpiConstant");
    case  vpiParameter      : return("vpiParameter");
    case  vpiFuncCall       : return("vpiFuncCall");
    case  vpiSysFuncCall    : return("vpiSysFuncCall");
    case  vpiOperation      : return("vpiOperation");
    default                 : return("UNDEFINED TYPE");
  }
}
/*********************************************************************/
