/*
 *  suse-blinux - Braille-display support for linux
 *  Author: Marco Skambraks <marco@suse.de>
 *  SuSE GmbH Nuernberg
 *
 *          Modifications for new generation braille devices
 *          by Klaus Knopper <knoppix@knopper.net>
 *
 * suse-blinux based on brltty
 * special thanks to the Brltty-Team
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include "cmdkeys.h"
#include "../trans.h"
#include "variolow.h"
#include "../brl.h"
#include "../brl_driver.h"
#include "../inskey.h"
#define CHKREAD 1
#define KEYTBL "/etc/sbl/brltbl/german"
#define CMDTBL "/etc/sbl/brltbl/cmdtbl"
int keytbl (int ch);

int insert (char ch, int cmd);

void loadcmdkeys ();

void loadkeytbl ();

int evo_keys (int c, int *pressed);

unsigned int brl_dbg = 0;

static unsigned char txttrans[256];

static unsigned char lastbuff[88];

/* extern int DEBUG; */
extern unsigned char st_disp[4];

struct brlinfo htdevs[] = {
  {"ht40", "Handy-Tech 40", 40, 4, 0, 0, B19200},
  {"ht80", "Handy-Tech 80", 80, 4, 0, 0, B19200},
  {"brlwave", "Handy-Tech Braille-Wave", 40, 0, 0, 0, B19200},
  {"vario40-ht", "Baum Vario40 via Handy-Tech emu", 40, 4, 0, 0, B19200},
  {"vario80-ht", "Baum Vario80 via Handy-Tech emu", 80, 4, 0, 0, B19200},
  {"dm80p-ht", "Baum DM80Plus via Handy-Tech emu", 80, 4, 0, 0, B19200},
  {"brlstar40", "Handy-Tech Braille-Star 40", 40, 0, 0, 0, B19200},
  {"brlstar80", "Handy-Tech Braille-Star 80", 80, 0, 0, 0, B19200},
  {"ht20", "Handy-Tech 20", 20, 4, 0, 0, B19200},
  {"brllino", "Handy-Tech Braille-Lino", 20, 0, 0, 0, B19200},
  {"evo88", "Handy-Tech Evolution 88", 88, 0, 0, 0, B19200},
  {"evo64", "Handy-Tech Evolution 64", 64, 0, 0, 0, B19200},
  {"activebraille", "Handy-Tech Active Braille 40", 40, 0, 0, 0, B19200},
  {"activestar", "Handy-Tech Active Star 40", 40, 0, 0, 0, B19200},
  {"basicbraille", "Handy-Tech Basic Braille 40", 40, 0, 0, 0, B19200},
};

/* Index in htdevs[] */
#define HT40 0
#define HT80 1
#define BRLWAVE 2
#define BRLSTAR40 6
#define BRLSTAR80 7
#define EVO88 10
#define EVO64 11
#define ACTIVEBRAILLE 12
#define ACTIVESTAR 13
#define BASICBRAILLE 14

int devnr = 0;

struct brlinfo identbrl (const char *name, const char *dev)
{

  int devcnt;

  struct brlinfo retinfo;

  devcnt = sizeof (htdevs) / sizeof (struct brlinfo);
  retinfo.cols = 0;
  for (devnr = 0; devnr < devcnt; devnr++)
    if (!strcmp (name, htdevs[devnr].name))
     {
       printf ("Using: %s on %s\n", htdevs[devnr].fullname, dev);
       retinfo = htdevs[devnr];
       break;
     }

  return retinfo;
}

void brl_debug (unsigned int dbg)
{
  brl_dbg = dbg;
}

void initbrl (brldim * brl, const char *dev)
{
  int brl_fd;

  if (devnr == BRLWAVE || devnr == ACTIVEBRAILLE)
   {
     loadcmdkeys ();
     loadkeytbl ();
   }

  /* fill brldim structure with correct values, regardless of device open. */
  brl->x = BRLCOLS;
  brl->y = 1;
  if (devnr == ACTIVEBRAILLE)
    brl->usb = 1;

  if ((brl_fd = varioinit ((char *) dev)) >= 0)
   {
     brl->x = BRLCOLS;
     brl->brl_fd = brl_fd;
     if (brl_dbg >= 1)
       fprintf (stderr, "devnr: %i cols: %i \n", devnr, htdevs[devnr].cols);
   }
  else
   {
     brl->disp = NULL;
     brl->brl_fd = -1;
     return;
   }

  if (!(brl->disp = malloc (brl->x * brl->y)))
   {
     perror (dev);
     brl->brl_fd = -1;
     return;
   }

}

void closebrl (brldim * brl)
{
  varioclose ();
  if (brl->disp)
   {
     free (brl->disp);
     brl->brl_fd = -1;
     brl->disp = NULL;
   }
}

void writebrl (brldim * brl)
{
  char outbuff[BRLCOLS];

  int i;

  /* Idiot check one */
  if (!brl || !brl->disp)
   {
     return;
   }
  /* Only display something if the data actually differs, this could most
     likely cause some problems in redraw situations etc but since the darn
     thing wants to redraw quite frequently otherwise this still makes a
     better lookin result */
  for (i = 0; i < BRLCOLS; i++)
   {
     if (lastbuff[i] != brl->disp[i])
      {
        memcpy (lastbuff, brl->disp, BRLCOLS * sizeof (char));

        /* Redefine the given dot-pattern to match ours */
        variotranslate (brl->disp, outbuff, BRLCOLS);
        if (variodisplay (outbuff) < 0)
          closebrl (brl);
        break;
      }
   }
}

void setbrlstat (const unsigned char *st)
{
  const unsigned char lowbits[10] =
    { 64 | 128 | 32, 4, 4 | 64, 4 | 32, 4 | 32 | 128,
    4 | 128, 4 | 64 | 32, 4 | 64 | 32 | 128, 4 | 64 | 128, 64 | 32
  };
  const unsigned char highbits[10] =
    { 2 | 8 | 16, 1, 1 | 2, 1 | 8, 1 | 8 | 16, 1 | 16, 1 | 2 | 8,
    1 | 2 | 8 | 16, 1 | 2 | 16, 2 | 8
  };

  st_disp[0] = highbits[st[0] / 10] | lowbits[st[1] / 10];
  st_disp[1] = highbits[st[0] % 10] | lowbits[st[1] % 10];
  st_disp[2] = highbits[st[2] / 10];
  st_disp[3] = highbits[st[2] % 10];

/* variotranslate(st_disp,st,ST_MODULES); */
}

int readbrl (int *type)
{
  static int kbdmode = 0;

  static int space = 0;

  int c;

  /* Since we are nonblocking this will happen quite a lot */
  if ((c = varioget ()) == -1)
    return EOF;
  if (c == 0xfe)
   {
     usleep (10000);
     varioget ();
     return EOF;
   }

  if (c == VARIO_DISPLAY_DATA_ACK)
    return EOF;

  if (brl_dbg >= 2)
    fprintf (stderr, "Got char 0x%x from device.\n", (int) c);

/* if an evolution display is connected 
 * use the evo_keys function to process the key values
 */
  if (devnr == EVO64 || devnr == EVO88)
   {
     int keyretval = EOF;

     keyretval = evo_keys (c, type);
     return keyretval;
   }

  /* if no evolution display is connected use the default processing method */
  /* Cursorrouting Keys (press codes) */
  if (c >= 0x20 && c <= 0x77)
   {
     *type = 1;
     return c - 0x20 + 104;
   }

  if (brl_dbg >= 4)
    fprintf (stderr, "kbdmode = %d\n", kbdmode);
  if (kbdmode)
   {
     switch (c)
      {
      case B05_PRESS:
        *type = 1;
        return 5;

      case B05_REL:
        *type = 0;
        return 5;

      case B06_PRESS:
        *type = 1;
        return 6;

      case B06_REL:
        *type = 0;
        return 6;
      default:
        kbdmode = keytbl (c);
        return EOF;
      }                         /* switch */
   }                            /* if */

  if (space)
   {
     if (c == 0x90)
       space = 0;
     if (c == 0xc)
      {
        space = 0;
        kbdmode = 1;
      }
     return EOF;
   }

  switch (c)
   {
   case 0x10:
     if (devnr == BRLWAVE || devnr == ACTIVEBRAILLE)    /* "Left" on Active
                                                           Braille */
      {
        space = 1;
        return EOF;
      }
     if ((devnr == BRLSTAR40) || (devnr == BRLSTAR80))
      {
        *type = 1;
        return 5;
      }
   case 0x90:
     if (devnr == BRLWAVE || devnr == ACTIVEBRAILLE)
      {
        space = 0;
        return EOF;
      }
     if ((devnr == BRLSTAR40) || (devnr == BRLSTAR80))
      {
        *type = 0;
        return 5;
      }
   case VARIO_DISPLAY_DATA_ACK:
     return EOF;
   case B01_PRESS:
     *type = 1;
     return 1;
   case B01_REL:
     *type = 0;
     return 1;
   case B02_PRESS:
     *type = 1;
     return 2;
   case B02_REL:
     *type = 0;
     return 2;
   case B03_PRESS:
     *type = 1;
     return 3;
   case B03_REL:
     *type = 0;
     return 3;
   case B04_PRESS:
     *type = 1;
     return 4;
   case B04_REL:
     *type = 0;
     return 4;
   case B05_PRESS:
     *type = 1;
     if ((devnr == BRLSTAR40) || (devnr == BRLSTAR80))
       return 13;
     return 5;
   case B05_REL:
     *type = 0;
     if ((devnr == BRLSTAR40) || (devnr == BRLSTAR80))
       return 13;
     return 5;
   case B06_PRESS:
     *type = 1;
     if ((devnr == BRLSTAR40) || (devnr == BRLSTAR80))
       return 14;
     return 6;
   case B06_REL:
     *type = 0;
     if ((devnr == BRLSTAR40) || (devnr == BRLSTAR80))
       return 14;
     return 6;
   case B07_PRESS:
     *type = 1;
     return 7;
   case B07_REL:
     *type = 0;
     return 7;
   case B08_PRESS:
     *type = 1;
     return 8;
   case B08_REL:
     *type = 0;
     return 8;
   case B09_PRESS:
     *type = 1;
     return 9;
   case B09_REL:
     *type = 0;
     return 9;
   case B10_PRESS:
     *type = 1;
     return 10;
   case B10_REL:
     *type = 0;
     return 10;
     /* numpad B9 - b14 */
   case 0x12:
     *type = 1;
     return 11;
   case 0x92:
     *type = 0;
     return 11;
   case 0x02:
     *type = 1;
     return 12;
   case 0x82:
     *type = 0;
     return 12;
   case 0x11:
     *type = 1;
     return 13;
   case 0x91:
     *type = 0;
     return 13;
   case 0x01:
     *type = 1;
     return 14;
   case 0x81:
     *type = 0;
     return 14;
   case 0x09:
     *type = 1;
     return 15;
   case 0x89:
     *type = 0;
     return 15;
   case 0x0d:
     *type = 1;
     return 16;
   case 0x8d:
     *type = 0;
     return 16;

     /* numpad 0 - 9 */
   case 0x05:
     *type = 1;
     return 20;
   case 0x85:
     *type = 0;
     return 20;
   case 0x15:
     *type = 1;
     return 21;
   case 0x95:
     *type = 0;
     return 21;
   case 0x19:
     *type = 1;
     return 22;
   case 0x99:
     *type = 0;
     return 22;
   case 0x1d:
     *type = 1;
     return 23;
   case 0x9d:
     *type = 0;
     return 23;
   case 0x06:
     *type = 1;
     return 24;
   case 0x86:
     *type = 0;
     return 24;
   case 0x0a:
     *type = 1;
     return 25;
   case 0x8a:
     *type = 0;
     return 25;
   case 0x0e:
     *type = 1;
     return 26;
   case 0x8e:
     *type = 0;
     return 26;
   case 0x16:
     *type = 1;
     return 27;
   case 0x96:
     *type = 0;
     return 27;
   case 0x1a:
     *type = 1;
     return 28;
   case 0x9a:
     *type = 0;
     return 28;
   case 0x1e:
     *type = 1;
     return 29;
   case 0x9e:
     *type = 0;
     return 29;
   case 12:
     if (devnr == BRLSTAR40 || devnr == BRLWAVE || devnr == BRLSTAR80
         || devnr == ACTIVEBRAILLE)
      {
        *type = 1;
        return 11;
      }
     break;
   case 140:
     if (devnr == BRLSTAR40 || devnr == BRLWAVE || devnr == BRLSTAR80
         || devnr == ACTIVEBRAILLE)
      {
        *type = 0;
        return 11;
      }
     break;
   case 20:
     if (devnr == BRLSTAR40 || devnr == BRLWAVE || devnr == BRLSTAR80
         || devnr == ACTIVEBRAILLE)
      {
        *type = 1;
        return 12;
      }
     break;
   case 148:
     if (devnr == BRLSTAR40 || devnr == BRLWAVE || devnr == BRLSTAR80
         || devnr == ACTIVEBRAILLE)
      {
        *type = 0;
        return 12;
      }
     break;
   case 24:
     *type = 1;
     if (devnr == BRLSTAR40 || devnr == BRLSTAR80)
       return 6;
     break;
   case 152:
     *type = 0;
     if (devnr == BRLSTAR40 || devnr == BRLSTAR80)
       return 6;
     break;
   default:
     /* statuscell routing keys */
     if (c >= 0x70 && c <= 0x73)
      {
        *type = 1;
        return c - 0x70 + 100;
      }
     if (c & 0x100)
      {                         /* ATC routing keys */
        *type = 1;
        return c;
      }
     *type = 0;
     return EOF;
   }                            /* switch end */
  *type = 0;
  return EOF;
}

int keytbl (int ch)
{
  static int pressed_keys = 0;

  static char insch = 0;

  static int space = 0;

  static int was_cmd = 0;

  switch (ch)
   {
   case 20:
     inskey ((unsigned char *) "\r");
     break;
   case 0x10:                  /* space bar */
     space = 1;
/*  pressed_keys++; */
     break;
   case 0xf:                   /* dot 1 */
     insch |= 1;
     pressed_keys++;
     break;
   case 0xb:                   /* dot 2 */
     insch |= 2;
     pressed_keys++;
     break;
   case 0x7:                   /* dot 3 */
     insch |= 4;
     pressed_keys++;
     break;
   case 0x13:                  /* dot 4 */
     insch |= 8;
     pressed_keys++;
     break;
   case 0x17:                  /* dot 5 */
     insch |= 16;
     pressed_keys++;
     break;
   case 0x1b:                  /* dot 6 */
     insch |= 32;
     pressed_keys++;
     break;
   case 0x3:                   /* dot 7 */
     insch |= 64;
     pressed_keys++;
     break;
   case 0x1f:                  /* dot 8 */
     insch |= 128;
     pressed_keys++;
     break;
/* key release */
   case 0x90:                  /* space release */
     if (was_cmd)
       was_cmd = 0;
     else if (space)
       inskey ((unsigned char *) " ");
     space = 0;
     break;
   case 0x8f:
   case 0x9f:
   case 0x8b:
   case 0x87:
   case 0x93:
   case 0x97:
   case 0x83:
   case 0x9b:
     if (pressed_keys)
      {
        was_cmd = 1;
        if (brl_dbg >= 2)
          fprintf (stderr, "pressed_keys = %d ", pressed_keys);
        pressed_keys--;
        if (!pressed_keys)
         {
           if (brl_dbg >= 2)
             fprintf (stderr, "start insert ");
           insert (insch, space);
           insch = 0;
/*   space=0; */
         }

      }

     break;
   case 0xc:                   /* exit kbdmode */
     if (brl_dbg >= 1)
       fprintf (stderr, "exit kbdmode ");
     return 0;
   }

  return 1;

}

int insert (char ch, int cmd)
{

  char str[10] = "";

  sprintf (str, "%c", (char) txttrans[(int) ch]);
  if (cmd)
   {
     int i = 0;

     if (ch == 0)
       inskey ((unsigned char *) " ");
     else
       do
        {
          if (inscmd[i].brlkey == (int) ch)
           {
             inskey ((unsigned char *) inscmd[i].instr);
             break;
           }
          i++;
        }
       while (inscmd[i].brlkey != -1);

   }
  else
    inskey ((unsigned char *) str);

  return 0;
}

void loadkeytbl ()
{
  int tbl_fd = 0;

  if ((tbl_fd = open (KEYTBL, O_RDONLY)) >= 0
      && (read (tbl_fd, txttrans, 256) == 256))
   {
     if (brl_dbg >= 1)
       fprintf (stderr, "keytable loaded\n");
   }
  else
    fprintf (stderr, "error: loading keytable %s.\n", KEYTBL);
}

void loadcmdkeys ()
{

#define GETVAL(string) bintrans(vecsearch(strcpy(str,string)))
  char str[20];

  iniread (CMDTBL);

  inscmd[0].brlkey = GETVAL ("csrdn");
  inscmd[1].brlkey = GETVAL ("csrup");
  inscmd[2].brlkey = GETVAL ("csrlft");
  inscmd[3].brlkey = GETVAL ("csrrgt");
  inscmd[4].brlkey = GETVAL ("esc");
  inscmd[5].brlkey = GETVAL ("ctrla");
  inscmd[6].brlkey = GETVAL ("ctrlc");
  inscmd[7].brlkey = GETVAL ("ctrld");
  inscmd[8].brlkey = GETVAL ("ctrle");
  inscmd[9].brlkey = GETVAL ("ctrlz");
  inscmd[10].brlkey = GETVAL ("tab");
  inscmd[11].brlkey = GETVAL ("del");
  inscmd[12].brlkey = GETVAL ("backspace");
  inscmd[13].brlkey = GETVAL ("insert");
  inscmd[14].brlkey = GETVAL ("home");
  inscmd[15].brlkey = GETVAL ("end");
  inscmd[16].brlkey = GETVAL ("pgup");
  inscmd[17].brlkey = GETVAL ("pgdn");
  inscmd[18].brlkey = GETVAL ("f1");
  inscmd[19].brlkey = GETVAL ("f10");

}

int evo_keys (int key, int *pressed)
{
/* check the cursor-routing */
  if (key >= 0x20 && key <= 0x77)
   {
     *pressed = 1;
     return key - 0x20 + 100;
   }

  switch (key)
   {

   case 0x03:
     *pressed = 1;
     return 1;

   case 0x83:
     *pressed = 0;
     return 1;

   case 0x07:
     *pressed = 1;
     return 2;

   case 0x87:
     *pressed = 0;
     return 2;

   case 0x0b:
     *pressed = 1;
     return 3;

   case 0x8b:
     *pressed = 0;
     return 3;

   case 0x0f:
     *pressed = 1;
     return 4;

   case 0x8f:
     *pressed = 0;
     return 4;

   case 0x10:
     *pressed = 1;
     return 5;

   case 0x90:
     *pressed = 0;
     return 5;

   case 0x18:
     *pressed = 1;
     return 6;

   case 0x98:
     *pressed = 0;
     return 6;

   case 0x13:
     *pressed = 1;
     return 7;

   case 0x93:
     *pressed = 0;
     return 7;

   case 0x17:
     *pressed = 1;
     return 8;

   case 0x97:
     *pressed = 0;
     return 8;

   case 0x1b:
     *pressed = 1;
     return 9;

   case 0x9b:
     *pressed = 0;
     return 9;

   case 0x1f:
     *pressed = 1;
     return 10;

   case 0x9f:
     *pressed = 0;
     return 10;

   case 0x0c:                  /* ACTIVEBRAILLE left multifunction key, UP,
                                   pressed */
     *pressed = 1;
     return 11;

   case 0x8c:                  /* ACTIVEBRAILLE left multifunction key, UP,
                                   released */
     *pressed = 0;
     return 11;

   case 0x14:                  /* ACTIVEBRAILLE left multifunction key,
                                   DOWN, pressed */
     *pressed = 1;
     return 12;

   case 0x94:                  /* ACTIVEBRAILLE left multifunction key,
                                   DOWN, released */
     *pressed = 0;
     return 12;

   case 0x04:                  /* Right Multifunction key UP pressed */
     *pressed = 1;
     return 13;

   case 0x84:                  /* Right Multifunction key UP release */
     *pressed = 0;
     return 13;

   case 0x08:                  /* Right Multifunction key DOWN pressed */
     *pressed = 1;
     return 14;

   case 0x88:                  /* Right Multifunction key DOWN released */
     *pressed = 0;
     return 14;

     /* numpad B9 - b14 maped to b030 B035 */
   case 0x12:
     *pressed = 1;
     return 30;
   case 0x92:
     *pressed = 0;
     return 30;
   case 0x02:
     *pressed = 1;
     return 31;
   case 0x82:
     *pressed = 0;
     return 31;
   case 0x11:
     *pressed = 1;
     return 32;
   case 0x91:
     *pressed = 0;
     return 32;
   case 0x01:
     *pressed = 1;
     return 33;
   case 0x81:
     *pressed = 0;
     return 33;
   case 0x09:
     *pressed = 1;
     return 34;
   case 0x89:
     *pressed = 0;
     return 34;
   case 0x0d:
     *pressed = 1;
     return 35;
   case 0x8d:
     *pressed = 0;
     return 35;

     /* numpad 0 - 9 mapped to B020 - B029 */
   case 0x05:
     *pressed = 1;
     return 20;
   case 0x85:
     *pressed = 0;
     return 20;
   case 0x15:
     *pressed = 1;
     return 21;
   case 0x95:
     *pressed = 0;
     return 21;
   case 0x19:
     *pressed = 1;
     return 22;
   case 0x99:
     *pressed = 0;
     return 22;
   case 0x1d:
     *pressed = 1;
     return 23;
   case 0x9d:
     *pressed = 0;
     return 23;
   case 0x06:
     *pressed = 1;
     return 24;
   case 0x86:
     *pressed = 0;
     return 24;
   case 0x0a:
     *pressed = 1;
     return 25;
   case 0x8a:
     *pressed = 0;
     return 25;
   case 0x0e:
     *pressed = 1;
     return 26;
   case 0x8e:
     *pressed = 0;
     return 26;
   case 0x16:
     *pressed = 1;
     return 27;
   case 0x96:
     *pressed = 0;
     return 27;
   case 0x1a:
     *pressed = 1;
     return 28;
   case 0x9a:
     *pressed = 0;
     return 28;
   case 0x1e:
     *pressed = 1;
     return 29;
   case 0x9e:
     *pressed = 0;
     return 29;

   default:
     return EOF;
   }

}
