/*
 *	sysctl (config file) utilities
 *
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <sutils.h>
#include <sysctl.h>
#include <logging.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <errno.h>

void
nc_sysctl_rewrite_to_slash(char *str)
{
	char *p, sep = 0;

	if(!str)
		return;

	for(p=str; *p; p++) {
		switch(*p) {
		case '/':
			switch(sep) {
			case 0:
				sep = '/';
				break;
			case '/':
				break;
			default:
				*p = '.';
				break;
			}
			break;
		case '.':
			switch(sep) {
			case 0:
				sep = '.';
				*p = '/';
				break;
			case '.':
				*p = '/';
				break;
			default:
				break;
			}
		default:
			break;
		}
	}
}


void
nc_sysctl_rewrite_to_dot(char *str)
{
	char *p, sep = 0;

	if(!str)
		return;

	for(p=str; *p; p++) {
		switch(*p) {
		case '/':
			switch(sep) {
			case 0:
				sep = '/';
				*p = '.';
				break;
			case '/':
				*p = '.';
				break;
			default:
				break;
			}
			break;
		case '.':
			switch(sep) {
			case 0:
				sep = '.';
				break;
			case '.':
				break;
			default:
				*p = '/';
				break;
			}
		default:
			break;
		}
	}
}

int
nc_sysctl_load(nc_var_array_t *vars, const char *filename)
{
	char buffer[NAME_MAX + 1] = {'\0'};
	char *key, *val;
	FILE *fp;

	fp = fopen(filename, "re");
	if(fp == NULL) {
		if (errno != ENOENT) {
			nc_error("Unable to open %s: %m", filename);
			return -1;
		}
		return 1;
	}

	while (fgets(buffer, sizeof(buffer), fp) != NULL) {
		key = nc_string_strip_spaces(buffer);
		if(!key || !*key || *key == ';' || *key == '#')
			continue;

		val = strchr(key, '=');
		if(!val)
			continue;
		*val++ = '\0';

		key = nc_string_strip_spaces(key);
		val = nc_string_strip_spaces(val);

		if(key && *key && *val) {
			nc_sysctl_rewrite_to_slash(key);
			if(nc_var_array_set(vars, key, val) < 0) {
				fclose(fp);
				return -1;
			}
		}
	}

	fclose(fp);
	return 0;
}

const nc_var_t *
nc_sysctl_get_fmt(const nc_var_array_t *vars, const char *fmt, ...)
{
	nc_stringbuf_t  attr = NC_STRINGBUF_INIT;
	va_list         ap;
	const nc_var_t *v = NULL;

	if(!vars || !fmt)
		return v;

	va_start(ap, fmt);
	if(nc_stringbuf_vprintf(&attr, fmt, ap) < 0) {
		va_end(ap);
		return v;
	}
	va_end(ap);

	nc_sysctl_rewrite_to_slash(attr.string);

	v = nc_var_array_get(vars, attr.string);
	nc_stringbuf_destroy(&attr);

	return v;
}

int
nc_sysctl_set_fmt(nc_var_array_t *vars, const char *value, const char *fmt, ...)
{
	nc_stringbuf_t  attr = NC_STRINGBUF_INIT;
	va_list         ap;
	int             ret = -1;

	if(!vars || !fmt || !value)
		return ret;

	va_start(ap, fmt);
	if(nc_stringbuf_vprintf(&attr, fmt, ap) < 0) {
		va_end(ap);
		return ret;
	}
	va_end(ap);

	nc_sysctl_rewrite_to_slash(attr.string);

	ret = nc_var_array_set(vars, attr.string, value);
	nc_stringbuf_destroy(&attr);

	return ret;
}

