/*
 *	network related types
 *
 *	Copyright (C) 2010 Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011 SUSE LINUX Products GmbH, Nuernberg, Germany.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *	MA  02110-1301  USA
 *
 *	Authors: Olaf Kirch <okir@suse.de>
 *	         Marius Tomaschewski <mt@suse.de>
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	You can find the wicked project at http://gitorious.org/wicked/.
 *
 */
#ifndef __NETCONTROL_NTYPES_H
#define __NETCONTROL_NTYPES_H
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include <sys/types.h>
#include <netinet/in.h>
#include <net/if.h>

typedef enum nc_addrconf_mode {
	NC_ADDRCONF_DHCP,
	NC_ADDRCONF_STATIC,
	NC_ADDRCONF_AUTOCONF,
	NC_ADDRCONF_IBFT,

	__NC_ADDRCONF_MAX
} nc_addrconf_mode_t;

#define NC_ADDRCONF_MASK(mode)		(1 << (mode))
#define NC_ADDRCONF_TEST(mask, mode)	!!((mask) & NC_ADDRCONF_MASK(mode))

typedef enum nc_start_mode {
	NC_STARTMODE_MANUAL = 0,
	NC_STARTMODE_AUTO,
	NC_STARTMODE_HOTPLUG,
	NC_STARTMODE_IFPLUGD,
	NC_STARTMODE_NFSROOT,
	NC_STARTMODE_OFF,
} nc_start_mode_t;

enum {
	NC_IFF_DEVICE_UP		= 0x0001,
	NC_IFF_LINK_UP			= 0x0002,
	NC_IFF_POWERSAVE		= 0x0004,
	NC_IFF_NETWORK_UP		= 0x0008,
	NC_IFF_POINT_TO_POINT		= 0x0010,
	NC_IFF_ARP_ENABLED		= 0x0020,
	NC_IFF_BROADCAST_ENABLED	= 0x0040,
	NC_IFF_MULTICAST_ENABLED	= 0x0080,
};

typedef enum nc_iftype {
	NC_IFTYPE_UNKNOWN = 0,
	NC_IFTYPE_ETHERNET = 1,
	NC_IFTYPE_LOOPBACK,
	NC_IFTYPE_BRIDGE,
	NC_IFTYPE_BOND,
	NC_IFTYPE_VLAN,
	NC_IFTYPE_WIRELESS,
	NC_IFTYPE_INFINIBAND,
	NC_IFTYPE_PPP,
	NC_IFTYPE_SLIP,
	NC_IFTYPE_SIT,
	NC_IFTYPE_GRE,
	NC_IFTYPE_ISDN,
	NC_IFTYPE_TUNNEL,
	NC_IFTYPE_TUNNEL6,
	NC_IFTYPE_TOKENRING,
	NC_IFTYPE_FIREWIRE,
	NC_IFTYPE_TUN,
	NC_IFTYPE_TAP,
	NC_IFTYPE_DUMMY,
	NC_IFTYPE_OVS,
	NC_IFTYPE_TEAM,
	NC_IFTYPE_MACVLAN,
	NC_IFTYPE_MACVTAP,
} nc_iftype_t;

/*
 * Link layer address
  */
#define NC_MAXHWADDRLEN         64
typedef struct nc_hwaddr {
	unsigned short		type;
	unsigned short		len;
	unsigned char		data[NC_MAXHWADDRLEN];
} nc_hwaddr_t;

typedef union nc_sockaddr {
	sa_family_t		ss_family;
	struct sockaddr_storage ss;
	struct sockaddr_in      sin;
	struct sockaddr_in6     six;
} nc_sockaddr_t;

typedef struct nc_address {
	nc_addrconf_mode_t	config_method;		/* usually static, but can be dhcp or autoconf */
	unsigned int		seq;
	unsigned int		family;
	unsigned int		flags;
	int			scope;
	unsigned int		prefixlen;
	nc_sockaddr_t		local_addr;
	nc_sockaddr_t		peer_addr;
	nc_sockaddr_t		anycast_addr;
	nc_sockaddr_t		bcast_addr;
	char			label[IFNAMSIZ];	/* ipv4 only, ignored for ipv6 */
	time_t			expires;		/* when address expires (ipv6) */
} nc_address_t;

typedef struct nc_address_array {
	unsigned int		count;
	nc_address_t **		data;
} nc_address_array_t;
#define NC_ADDRESS_ARRAY_INIT { .count = 0, .data = NULL }

typedef struct nc_route_nexthop {
	struct nc_route_nexthop *next;
	nc_sockaddr_t		gateway;
	char *			device;
	unsigned int		weight;
	unsigned int		flags;
} nc_route_nexthop_t;

typedef struct nc_route {
	unsigned int		users;

	nc_addrconf_mode_t	config_method;		/* usually static, but can be dhcp or autoconf */
	unsigned int		seq;
	sa_family_t		family;
	unsigned int		prefixlen;
	nc_sockaddr_t		destination;
	nc_route_nexthop_t	nh;

	char *			opts;
	unsigned int		mtu;
	unsigned int		tos;
	unsigned int		priority;
	time_t			expires;		/* when route expires (ipv6) */
} nc_route_t;

typedef struct nc_route_array {
	unsigned int		count;
	nc_route_t **		data;
} nc_route_array_t;
#define NC_ROUTE_ARRAY_INIT { .count = 0, .data = NULL }

typedef struct nc_afinfo {
	sa_family_t		family;
	int			enabled;
	int			forwarding;

	unsigned int		addrconf;		/* bitmask of enabled addrconf modes */
} nc_afinfo_t;

typedef struct nc_vlan		nc_vlan_t;
typedef struct nc_bridge	nc_bridge_t;
typedef struct nc_bonding	nc_bonding_t;
typedef struct nc_interface	nc_interface_t;

struct nc_interface
{
	unsigned int		users;

	unsigned int		seq;
	unsigned int		modified : 1;
	unsigned int		deleted : 1;
	unsigned int		invalid : 1;


	char *			name;
	nc_iftype_t		type;

	unsigned int		arp_type;
	unsigned int		ifindex;
	unsigned int		ifflags;

	/* nc_uuid_t		uuid; */
	nc_hwaddr_t		hwaddr;
	nc_address_array_t 	addrs;
	nc_route_array_t 	routes;

	unsigned int		mtu;
	unsigned int		metric;
	unsigned int		txqlen;
	unsigned int		master;		/* ifindex */
	char *			qdisc;
	char *			kind;

	/* nc_link_stats_t *	link_stats; */

	nc_afinfo_t		ipv4;
	nc_afinfo_t		ipv6;

	char *			parent;
	nc_vlan_t *		vlan;
	nc_bridge_t *		bridge;
	nc_bonding_t *		bonding;

	struct {
		char *		origin;
		char *		uuid;
	}			config;
	unsigned int		startmode;
};

typedef struct nc_interface_array {
	unsigned int            count;
	nc_interface_t **       data;
} nc_interface_array_t;
#define NC_INTERFACE_ARRAY_INIT { .count = 0, .data = NULL }

#endif /* __NETCONTROL_NTYPES_H */
