/*
 *	Routines for bridge interface handling
 *
 *	Copyright (C) 2009, 2010  Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <logging.h>
#include <bridge.h>
#include <handle.h>


#define NC_BRIDGE_VALUE_NOT_SET		~0U
#define NC_BRIDGE_PORT_ARRAY_CHUNK	16

static int			__nc_bridge_str_to_uint(const char *, unsigned int *);
static int			__nc_bridge_uint_to_str(unsigned int, char **);
static int			__nc_bridge_str_to_time(const char *, unsigned long *);
static int			__nc_bridge_time_to_str(unsigned long, char **);

static nc_bridge_port_t *	__nc_bridge_port_new(const char *);
static nc_bridge_port_t *	__nc_bridge_port_clone(const nc_bridge_port_t *);
static void			__nc_bridge_port_free(nc_bridge_port_t *);

static void			nc_bridge_port_array_init(nc_bridge_port_array_t *);
static int			nc_bridge_port_array_copy(nc_bridge_port_array_t *,
					const nc_bridge_port_array_t *);
static void			nc_bridge_port_array_destroy(nc_bridge_port_array_t *);
static int 			__nc_bridge_port_array_realloc(nc_bridge_port_array_t *,
					unsigned int);
static int			__nc_bridge_port_array_append(nc_bridge_port_array_t *,
					nc_bridge_port_t *);



/*
 * Bridge option value conversion utilities
 * Returns -1 on error, 0 if the value is not set, and 1 otherwise.
 */
static int
__nc_bridge_str_to_uint(const char *str, unsigned int *val)
{
	if (!str || !*str) {
		*val = NC_BRIDGE_VALUE_NOT_SET;
		return 0;
	} else {
		char *end = NULL;
		unsigned int i = strtoul(str, &end, 0);

		if (*end == '\0') {
			*val = i;
			return 1;
		}
	}
	return -1;
}

static int
__nc_bridge_uint_to_str(unsigned int val, char **str)
{
	if (val == NC_BRIDGE_VALUE_NOT_SET) {
		nc_string_free(str);
		return 0;
	} else {
		char   buf[32];

		snprintf(buf, sizeof(buf), "%u", val);
		nc_string_dup(str, buf);
		return *str ? 1 : -1;
	}
}

static int
__nc_bridge_str_to_time(const char *str, unsigned long *val)
{
	if (!str || !*str) {
		*val = NC_BRIDGE_VALUE_NOT_SET;
		return 0;
	} else {
		char *end = NULL;
		double d = strtod(str, &end);

		if (*end == '\0') {
			*val = (unsigned long)(d * 100);
			return 1;
		}
	}
	return -1;
}

static int
__nc_bridge_time_to_str(unsigned long val, char **str)
{
	if (val == NC_BRIDGE_VALUE_NOT_SET) {
		nc_string_free(str);
		return 0;
	} else {
		char   buf[32];
		double d = (double)val;

		snprintf(buf, sizeof(buf), "%.2lf", (d / 100));
		nc_string_dup(str, buf);
		return *str ? 1 : -1;
	}
}

static nc_bridge_port_t *
__nc_bridge_port_new(const char *name)
{
	nc_bridge_port_t *newport;

	newport = calloc(1, sizeof(nc_bridge_port_t));
	assert(newport != NULL);
	if (!newport)
		return NULL;

	nc_string_dup(&newport->name, name);
	newport->config.priority = NC_BRIDGE_VALUE_NOT_SET;
	newport->config.path_cost = NC_BRIDGE_VALUE_NOT_SET;
	return newport;
}

static nc_bridge_port_t *
__nc_bridge_port_clone(const nc_bridge_port_t *port)
{
	nc_bridge_port_t *newport;

	newport = __nc_bridge_port_new(port->name);
	memcpy(&newport->config, &port->config, sizeof(newport->config));
	return newport;
}

static void
__nc_bridge_port_free(nc_bridge_port_t *port)
{
	nc_string_free(&port->name);
	if (port->status)
		nc_bridge_port_status_free(port->status);
	free(port);
}

static void
nc_bridge_port_array_init(nc_bridge_port_array_t *array)
{
	memset(array, 0, sizeof(*array));
}

static int
nc_bridge_port_array_copy(nc_bridge_port_array_t *dst, const nc_bridge_port_array_t *src)
{
	unsigned int i;

	nc_bridge_port_array_destroy(dst);
	for (i = 0; i < src->count; ++i) {
		if (__nc_bridge_port_array_append(dst,
			__nc_bridge_port_clone(src->data[i])) < 0)
			return -1;
	}
	return 0;
}

static void
nc_bridge_port_array_destroy(nc_bridge_port_array_t *array)
{
	while (array->count > 0)
		__nc_bridge_port_free(array->data[--array->count]);
	free(array->data);
	nc_bridge_port_array_init(array);
}

static int
__nc_bridge_port_array_realloc(nc_bridge_port_array_t *array, unsigned int newsize)
{
	nc_bridge_port_t **newdata;
	unsigned int i;

	newsize = (newsize + NC_BRIDGE_PORT_ARRAY_CHUNK);
	newdata = realloc(array->data, newsize * sizeof(nc_bridge_port_t));
	assert(newdata != NULL);
	if (!newdata)
		return -1;

	array->data = newdata;
	for (i = array->count; i < newsize; ++i)
		array->data[i] = NULL;
	return 0;
}

static int
__nc_bridge_port_array_append(nc_bridge_port_array_t *array, nc_bridge_port_t *port)
{
	if ((array->count % NC_BRIDGE_PORT_ARRAY_CHUNK) == 0)
		if(__nc_bridge_port_array_realloc(array, array->count) < 0)
			return -1;

	array->data[array->count++] = port;
	return 0;
}

static int
__nc_bridge_port_array_index(nc_bridge_port_array_t *array, const char *port)
{
	unsigned int i;
	for (i = 0; i < array->count; ++i) {
		if (!strcmp(port, array->data[i]->name))
			return i;
	}
	return -1;
}

static int
nc_bridge_port_array_remove_index(nc_bridge_port_array_t *array, unsigned int pos)
{
	unsigned int i;

	if (pos >= array->count)
		return -1;

	__nc_bridge_port_free(array->data[pos]);
	/* make it less cumbersome... */
	array->data[pos] = NULL;
	for (i = pos + 1; i < array->count; ++i) {
		array->data[i - 1] = array->data[i];
		array->data[i] = NULL;
	}
	array->count--;
	return 0;
}

/*
 * Add a port to the bridge configuration
 */
int
nc_bridge_add_port(nc_bridge_t *bridge, const char *ifname)
{
	if (!ifname || !*ifname)
		return -1;

	if (__nc_bridge_port_array_index(&bridge->ports, ifname) < 0) {
		return __nc_bridge_port_array_append(&bridge->ports,
			__nc_bridge_port_new(ifname));
	}
	return -1;
}

int
nc_bridge_del_port(nc_bridge_t *bridge, const char *ifname)
{
	unsigned int i;

	for (i = 0; i < bridge->ports.count; ++i) {
		if (!strcmp(bridge->ports.data[i]->name, ifname)) {
			nc_bridge_port_array_remove_index(&bridge->ports, i);
			return 0;
		}
	}
	return -1;
}

void
nc_bridge_get_port_names(const nc_bridge_t *bridge, nc_string_array_t *ports)
{
	unsigned int i;

	for (i = 0; i < bridge->ports.count; ++i)
		nc_string_array_append(ports, bridge->ports.data[i]->name);
}

/*
 * Get bridge options
 */
int
nc_bridge_get_stp(nc_bridge_t *bridge, char **value)
{
	if (bridge->config.stp_enabled == NC_BRIDGE_NO_STP)
		nc_string_dup(value, "off");
	else
		nc_string_dup(value, "on");
	return *value ? 1 : 0;
}

int
nc_bridge_get_forward_delay(nc_bridge_t *bridge, char **value)
{
	return __nc_bridge_time_to_str(bridge->config.forward_delay, value);
}

int
nc_bridge_get_ageing_time(nc_bridge_t *bridge, char **value)
{
	return __nc_bridge_time_to_str(bridge->config.ageing_time, value);
}

int
nc_bridge_get_hello_time(nc_bridge_t *bridge, char **value)
{
	return __nc_bridge_time_to_str(bridge->config.hello_time, value);
}

int
nc_bridge_get_max_age(nc_bridge_t *bridge, char **value)
{
	return __nc_bridge_time_to_str(bridge->config.max_age, value);
}

int
nc_bridge_get_priority(nc_bridge_t *bridge, char **value)
{
	return __nc_bridge_uint_to_str(bridge->config.priority, value);
}

int
nc_bridge_get(nc_bridge_t *bridge, unsigned int opt, char **value)
{
	switch (opt) {
	case NC_BRIDGE_STP_ENABLED:
		return nc_bridge_get_stp(bridge, value);
	case NC_BRIDGE_FORWARD_DELAY:
		return nc_bridge_get_forward_delay(bridge, value);
	case NC_BRIDGE_AGEING_TIME:
		return nc_bridge_get_ageing_time(bridge, value);
	case NC_BRIDGE_HELLO_TIME:
		return nc_bridge_get_hello_time(bridge, value);
	case NC_BRIDGE_MAX_AGE:
		return nc_bridge_get_max_age(bridge, value);
	case NC_BRIDGE_PRIORITY:
		return nc_bridge_get_priority(bridge, value);
	}
	return -1;
}

/*
 * Set bridge options
 */
int
nc_bridge_set_stp(nc_bridge_t *bridge, const char *value)
{
	/* brctl accepts "on" / "off" as well as "yes" / "no"
	 * note: it is a bool {0,!0} while write, just sysfs
	 * shows details {0=off,1=stp,2=rstp} in stp_state.
	 */
	if (!value || !*value) {
		bridge->config.stp_enabled = NC_BRIDGE_NO_STP;
		return 0;
	}
	if (!nc_parse_bool(value, &bridge->config.stp_enabled))
		return 0;

	return -1;
}

int
nc_bridge_set_forward_delay(nc_bridge_t *bridge, const char *value)
{
	return __nc_bridge_str_to_time(value, &bridge->config.forward_delay);
}

int
nc_bridge_set_ageing_time(nc_bridge_t *bridge, const char *value)
{
	return __nc_bridge_str_to_time(value, &bridge->config.ageing_time);
}

int
nc_bridge_set_hello_time(nc_bridge_t *bridge, const char *value)
{
	return __nc_bridge_str_to_time(value, &bridge->config.hello_time);
}

int
nc_bridge_set_max_age(nc_bridge_t *bridge, const char *value)
{
	return __nc_bridge_str_to_time(value, &bridge->config.max_age);
}

int
nc_bridge_set_priority(nc_bridge_t *bridge, const char *value)
{
	return __nc_bridge_str_to_uint(value, &bridge->config.priority);
}

int
nc_bridge_set(nc_bridge_t *bridge, unsigned int opt, const char *value)
{
	switch (opt) {
	case NC_BRIDGE_STP_ENABLED:
		return nc_bridge_set_stp(bridge, value);
	case NC_BRIDGE_FORWARD_DELAY:
		return nc_bridge_set_forward_delay(bridge, value);
	case NC_BRIDGE_AGEING_TIME:
		return nc_bridge_set_ageing_time(bridge, value);
	case NC_BRIDGE_HELLO_TIME:
		return nc_bridge_set_hello_time(bridge, value);
	case NC_BRIDGE_MAX_AGE:
		return nc_bridge_set_max_age(bridge, value);
	case NC_BRIDGE_PRIORITY:
		return nc_bridge_set_priority(bridge, value);
	}
	return -1;
}

/*
 * Get bridge port options
 */
int
nc_bridge_port_get_priority(nc_bridge_t *bridge, const char *port, char **value)
{
	int i = __nc_bridge_port_array_index(&bridge->ports, port);

	if (i < 0)
		return -1;
	return __nc_bridge_uint_to_str(bridge->ports.data[i]->config.priority, value);
}

int
nc_bridge_port_get_path_cost(nc_bridge_t *bridge, const char *port, char **value)
{
	int i = __nc_bridge_port_array_index(&bridge->ports, port);

	if (i < 0)
		return -1;
	return __nc_bridge_uint_to_str(bridge->ports.data[i]->config.path_cost, value);
}

int
nc_bridge_port_get(nc_bridge_t *bridge, const char *port, unsigned int opt, char **value)
{
	switch (opt) {
	case NC_BRIDGE_PORT_PRIORITY:
		return nc_bridge_port_get_priority(bridge, port, value);
	case NC_BRIDGE_PORT_PATH_COST:
		return nc_bridge_port_get_path_cost(bridge, port, value);
	}
	return -1;
}

/*
 * Set bridge port options
 */
int
nc_bridge_port_set_priority(nc_bridge_t *bridge, const char *port, const char *value)
{
	int i = __nc_bridge_port_array_index(&bridge->ports, port);

	if (i < 0)
		return -1;
	return __nc_bridge_str_to_uint(value, &bridge->ports.data[i]->config.priority);
}

int
nc_bridge_port_set_path_cost(nc_bridge_t *bridge, const char *port, const char *value)
{
	int i = __nc_bridge_port_array_index(&bridge->ports, port);

	if (i < 0)
		return -1;
	return __nc_bridge_str_to_uint(value, &bridge->ports.data[i]->config.path_cost);
}

int
nc_bridge_port_set(nc_bridge_t *bridge, const char *port, unsigned int opt, const char *value)
{
	switch (opt) {
	case NC_BRIDGE_PORT_PRIORITY:
		return nc_bridge_port_set_priority(bridge, port, value);
	case NC_BRIDGE_PORT_PATH_COST:
		return nc_bridge_port_set_path_cost(bridge, port, value);
	}
	return -1;
}

/*
 * Binding callback for the bridge config.
 * This looks up interface config for all ports, and binds it
 */
int
nc_bridge_bind(nc_interface_t *ifp, nc_handle_t *nh, nc_var_array_t *errs)
{
	nc_stringbuf_t err = NC_STRINGBUF_INIT;
	unsigned int   i;

	if(!ifp || !ifp->name || !nh || !errs)
		return -1;
	if(!ifp->bridge)
		return -1;
	if(ifp->type != NC_IFTYPE_BRIDGE)
		return -1;

	for (i = 0; i < ifp->bridge->ports.count; ++i) {
		nc_bridge_port_t *port = ifp->bridge->ports.data[i];
		const char       *ifname = port->name;
		nc_interface_t   *iface;

		iface = nc_interface_by_name(nh, ifname);
		if (iface == NULL) {
			nc_trace("no interface for bridge %s port %s -- missed config?",
				ifp->name, ifname);
			continue;
		}
		if(iface->invalid) {
			ifp->invalid = 1;
			nc_stringbuf_printf(&err,
				"Bridge port %s is invalid - marking bridge invalid",
				ifname);
			nc_var_array_set(errs, ifp->name, err.string);
			nc_stringbuf_destroy(&err);
			return 1;
		}
		if(!nc_string_len(iface->parent)) {
			nc_string_dup(&iface->parent, ifp->name);
			continue;
		}

		if(!nc_string_eq(iface->parent, ifp->name)) {
			ifp->invalid = 1;
			nc_stringbuf_printf(&err,
				"Bridge port %s references %s as parent",
				ifname, iface->parent);
			nc_var_array_set(errs, ifp->name, err.string);
			nc_stringbuf_destroy(&err);
			return 1;
		} else {
			nc_trace("Bridge %s port %s parent already set",
				ifp->name, ifname);
		}
	}
	return 0;
}

/*
 * Create a copy of a bridge's configuration
 */
nc_bridge_t *
nc_bridge_clone(const nc_bridge_t *src)
{
	nc_bridge_t *dst;

	dst = calloc(1, sizeof(nc_bridge_t));
	if (!dst)
		return NULL;

	memcpy(&dst->config, &src->config, sizeof(dst->config));
	if (nc_bridge_port_array_copy(&dst->ports, &src->ports) < 0)
		goto failed;

	return dst;

failed:
	nc_error("Error cloning bridge configuration");
	nc_bridge_free(dst);
	return NULL;
}

/*
 * Bridge constructor and new operator
 */
static void
__nc_bridge_init(nc_bridge_t *bridge)
{
	/* apply "not set" defaults */
	bridge->config.forward_delay = NC_BRIDGE_VALUE_NOT_SET;
	bridge->config.ageing_time = NC_BRIDGE_VALUE_NOT_SET;
	bridge->config.hello_time = NC_BRIDGE_VALUE_NOT_SET;
	bridge->config.max_age = NC_BRIDGE_VALUE_NOT_SET;
	bridge->config.priority = NC_BRIDGE_VALUE_NOT_SET;
}

nc_bridge_t *
nc_bridge_new(void)
{
	nc_bridge_t *bridge;

	bridge = calloc(1, sizeof(*bridge));
	__nc_bridge_init(bridge);
	return bridge;
}

/*
 * Bridge destructor and delete operator
 */
static void
__nc_bridge_destroy(nc_bridge_t *bridge)
{
	nc_bridge_port_array_destroy(&bridge->ports);

	if (bridge->status) {
		nc_bridge_status_free(bridge->status);
		bridge->status = NULL;
	}
}

void
nc_bridge_free(nc_bridge_t *bridge)
{
	if(bridge) {
		__nc_bridge_destroy(bridge);
		free(bridge);
	}
}

void
nc_bridge_status_free(nc_bridge_status_t *bs)
{
	if (bs->root_id)
		free(bs->root_id);
	if (bs->bridge_id)
		free(bs->bridge_id);
	if (bs->group_addr)
		free(bs->group_addr);
	free(bs);
}

void
nc_bridge_port_status_free(nc_bridge_port_status_t *ps)
{
	if (ps->designated_root)
		free(ps->designated_root);
	if (ps->designated_bridge)
		free(ps->designated_bridge);
	free(ps);
}
