/*
 *  Braille Daemon - a simple braille-display server
 *  Author: Marco Skambraks <marco@skammel.de>
 *  Skammel Solutions - Marburg
 *
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <dirent.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <linux/un.h>
#include <netinet/in.h>
#include <netdb.h>
#include <pthread.h>
#include <linux/kd.h>
#include <fcntl.h>
#include <linux/vt.h>
/*#include <iconv.h>*/
#include <getopt.h>
#include "brld.h"
#include "ringbuffer.h"
#include "load_config.h"

void terminate (int signum);

int load_brltbl (char *tblname);

void send_to_brl (unsigned char csr_form, int csrpos, char *text, char *attr,
                  int len);

int getdriver (char *libpath, char *brlname, char *brlport);

#define SLEEP 5000
#define TEXT_CON 0
#define NON_TEXT_CON 1

/* console mutex to protect con_nr and con_type */
pthread_mutex_t conMutex = PTHREAD_MUTEX_INITIALIZER;

/* mutex to protect read/write from/to keybuffer */
pthread_mutex_t key_buf_Mutex = PTHREAD_MUTEX_INITIALIZER;

/* mutex to protect device operations */
pthread_mutex_t brl_dev_Mutex = PTHREAD_MUTEX_INITIALIZER;

/* mutex to protect connection-counter */
pthread_mutex_t connMutex = PTHREAD_MUTEX_INITIALIZER;

/* current console and current console-type */
unsigned int con_type = 0, con_nr = 0;

unsigned int text_app = 0, x11_app = 0, admin_app = 0;

unsigned int active_app = 0;

struct brlinfo brl_info;
brldim brl_disp = { -1, 0, NULL, 0, 0 };

braille_driver *braille = NULL;

unsigned char texttbl[256] = "";

unsigned char attrtbl[256] = {
#include "attrib.def"
};

unsigned char *brltbl;

/* connection counter */
int connections = 0;

volatile sig_atomic_t brl_loop = 1, main_loop = 1, conn_loop = 1;

void io_event (int state);

/* connection thread - started if an application connects to the brld socket */
void *thread_conn (void *arg);

/* thread to handle incoming braille events and check current console/type */
void *thread_brl ();

/* timeout variables */
int conn_timeout = 5000;

int brl_text_timeout = 500;

brld_config config;

/* full-path to braille libs */
char brl_libpath[MAX_PATH_LEN] = "";

/* full-pathanme of braille table */
char brl_trans_tbl[MAX_PATH_LEN] = "";

/* config file */
char config_file[MAX_PATH_LEN] = "";

int do_fork = 1;

int verbose = 0;

struct option long_options[] = {
  {"nodaemon", 0, 0, 'n'},
  {"verbose", 1, 0, 'v'},
  {"braille", 1, 0, 'b'},
  {"device", 1, 0, 'd'},
  {"table", 1, 0, 't'},
  {"auth", 1, 0, 'a'},
  {"port", 1, 0, 'p'},
  {"config", 1, 0, 'c'},
  {"help", 0, 0, 'h'},
  {0, 0, 0, 0}
};

const char usage[] = " \
brld V0.7 - a simple braille-display-server\n \
Author: Marco Skambraks <marco@skammel.de> \n\n \
Usage: brld [options]\n \
-a --auth <authkey> - authentication key for brld\n \
-b --braille <brlname> - short name of your braille device\n \
-c --config <file> - load config-file (default /etc/sbl.conf)\n \
-d --device - device where the braille display is connected\n \
-h --help - print this help\n \
-n --nodaemon - don't fork\n \
-p --port <port> - port to listen on (default 8888)\n \
-t --table <brltbl> - braillle translationtable\n \
-v --verbose <level> - verbosity level (default 0)\n";

int main (int argc, char **argv)
{

  struct sockaddr_in sAddr;

  struct sockaddr_un sFile;

  struct hostent *hp = NULL;

  int listensock;

  int newsock;

  int result;

  pthread_t thread_id;

  int val;

  int c = 0;

  char tmp_brl[MAX_PATH_LEN] = "";

  char tmp_dev[MAX_PATH_LEN] = "";

  int tmp_port = 0;

  char tmp_auth_key[MAX_AUTH_KEY] = "";

  char tmp_tbl[MAX_PATH_LEN] = "";

  struct timeval tout;

  fd_set sock_set;

  key_init_buf ();
/* parse commandline options */

  while (1)
   {
     int option_index = 0;

     c =
       getopt_long (argc, argv, "nhv:b:d:t:a:p:c:", long_options,
                    &option_index);
     if (c == -1)
       break;

     switch (c)
      {
      case 0:
        printf ("option %s", long_options[option_index].name);
        if (optarg)
          printf (" with arg %s", optarg);
        printf ("\n");
        break;

      case 'a':
        strcpy (tmp_auth_key, optarg);
        break;

      case 'b':
        strcpy (tmp_brl, optarg);
        break;

      case 'c':
        strcpy (config_file, optarg);
        break;

      case 'd':
        strcpy (tmp_dev, optarg);
        break;
      case 'h':
        printf ("%s", usage);
        return 0;

      case 'n':
        do_fork = 0;
        break;
      case 'p':
        tmp_port = atoi (optarg);
        break;
      case 't':
        strcpy (tmp_tbl, optarg);
        break;
      case 'v':
        verbose = atoi (optarg);
        printf ("verbose=%d\n", verbose);
        break;

      case '?':
        return 2;
        break;

      }
   }
  if (optind < argc)
   {
     fprintf (stderr, "non-option: ");
     return 2;
     while (optind < argc)
       fprintf (stderr, "%s ", argv[optind++]);
     fprintf (stderr, "\n");

     return 2;
   }

/* set termination handler */
  signal (SIGTERM, terminate);
  signal (SIGINT, terminate);
  signal (SIGCHLD, SIG_IGN);
  signal (SIGHUP, SIG_IGN);
  signal (SIGPIPE, SIG_IGN);
  signal (SIGABRT, SIG_IGN);
  if (do_fork)
   {
     int i = 0;

     for (i = 0; i < 255; i++)
       close (i);

     switch (fork ())
      {
      case -1:
        fprintf (stderr, "error: fork failed\n");
        return 1;
      case 0:
        setsid ();
        /* become a daemon */
        break;
      default:
        return 0;
      }
   }

  if (!config_file[0])
    sprintf (config_file, "%s/sbl.conf", CONFDIR);
  if (getconf (&config, config_file))
   {
     if (verbose)
       printf ("error: loading configuration %s\n", config_file);
   }

  sprintf (brl_libpath, "%s/lib", PROGPATH);

  if (tmp_tbl[0])
    strcpy (config.brltbl, tmp_tbl);

  if (!config.brltbl[0])
    strcpy (config.brltbl, "german");

  sprintf (brl_trans_tbl, "%s/sbl/brltbl/%s", CONFDIR, config.brltbl);
  if (tmp_brl[0])
    strcpy (config.brlalias, tmp_brl);
  if (tmp_dev[0])
    strcpy (config.brlport, tmp_dev);
  if (getdriver (brl_libpath, config.brlalias, config.brlport))
   {
     fprintf (stderr, "error: get braille driver\n");
     return 3;
   }

  if (load_brltbl (brl_trans_tbl))
   {
     fprintf (stderr, "error: loading brltbl %s\n", brl_trans_tbl);
     return 4;
   }

  if (tmp_port)
    config.brld_port = tmp_port;
  if (tmp_auth_key[0])
    strcpy (config.auth_key, tmp_auth_key);

  if (!config.auth_key[0])
    strcpy (config.auth_key, "default");

  /* Try to initialize now, or later. */
  brl_ok ();

  brltbl = texttbl;

/* start braille-thread */
  result = pthread_create (&thread_id, NULL, thread_brl, NULL);
  if (result != 0)
   {
     fprintf (stderr, "error: could not create braille thread\n");
     return 5;
   }

  pthread_detach (thread_id);
  sched_yield ();

  /* if port is 0 - use unix socket */
  if (!config.brld_port)
   {
     listensock = socket (AF_UNIX, SOCK_STREAM, 0);
     fcntl (listensock, F_SETFD, FD_CLOEXEC);
   }
  else
    listensock = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP);

/*  set option to re-use the addr */
  val = 1;
  result =
    setsockopt (listensock, SOL_SOCKET, SO_REUSEADDR, &val, sizeof (val));
  if (result < 0)
   {
     fprintf (stderr, "error: socket options failed\n");
     return 6;
   }

  /* set socket to non-blocking mode */
  fcntl (listensock, F_SETFL, O_NONBLOCK);

  /* if brld_port is 0 - init a unix socket */
  if (!config.brld_port)        /* unix socket */
   {
     memset (&sFile, 0, sizeof (struct sockaddr_un));
     sFile.sun_family = AF_UNIX;
     strcpy (sFile.sun_path, BRLD_SOCK_NAME);
     unlink (BRLD_SOCK_NAME);
   }
  else
   {
     sAddr.sin_family = AF_INET;
     sAddr.sin_port = htons (config.brld_port);
     if ((hp = gethostbyname ("localhost")) == NULL)
      {
        fprintf (stderr, "can't get host (localhost)\n");
        return 7;
      }
     else
       memcpy (&sAddr.sin_addr, hp->h_addr, hp->h_length);
   }                            /* else - inet socket */

  result =
    bind (listensock,
          config.brld_port ? (struct sockaddr *) &sAddr : (struct sockaddr *)
          &sFile, config.brld_port ? sizeof (sAddr) : sizeof (sFile));
  if (result < 0)
   {
     fprintf (stderr, "error: bind failed\n");
     return 8;
   }
  /* if unix socket - change permissions */
  if (!config.brld_port)
    chmod (BRLD_SOCK_NAME, 0666);

  result = listen (listensock, 5);
  if (result < 0)
   {
     fprintf (stderr, "error: listen failed\n");
     return 9;
   }

  while (main_loop)
   {
     FD_ZERO (&sock_set);
     FD_SET (listensock, &sock_set);
     tout.tv_sec = 2;
     tout.tv_usec = 0;
     if (select (listensock + 1, &sock_set, NULL, NULL, &tout) > 0)
      {
        newsock = accept (listensock, NULL, NULL);

/* start a new thread if a client connects to this server */
        if (newsock >= 0)
         {
           result =
             pthread_create (&thread_id, NULL, thread_conn,
                             (void *) &newsock);
           if (result != 0)
            {
              fprintf (stderr, "error: could not create connection thread\n");
              return 10;
            }

           pthread_detach (thread_id);
           sched_yield ();

         }

        usleep (SLEEP);
      }

   }

  brl_loop = 0;
  conn_loop = 0;
  close (listensock);
  if (!config.brld_port)
    unlink (BRLD_SOCK_NAME);
  if (brl_disp.disp)
   {
     free (brl_disp.disp);
     brl_disp.disp = NULL;
   }

  unload_braille_driver ();
  sleep (1);
  return 0;
}

void *thread_conn (void *arg)
{
  int sock, *tmp_sock = (int *) arg;

  int event_mode = 0;

  char buffer[MAX_BUF_LEN];

  int nread = 0, cnt = 0;

  unsigned char csr_form = BLOCK_CSR;

  char tmpstr[MAX_BUF_LEN], ch = 0;

  int timeout = 0;

  struct timeval select_time;

  int brl_key = 0, brl_pressed = 0;

  unsigned int self_app = 0, reg = 0;

  fd_set select_set;

  int sel_ret = 0;

  if (verbose)
    printf ("child thread pid %i created\n", getpid ());

  sock = *tmp_sock;
  pthread_mutex_lock (&connMutex);
  connections++;
  pthread_mutex_unlock (&connMutex);

  while (conn_loop && timeout < conn_timeout)
   {

     cnt = 0;

     memset (buffer, 0, sizeof (buffer));
     memset (tmpstr, 0, sizeof (tmpstr));
     timeout++;

     if (event_mode && timeout >= (conn_timeout - 100))
       if (send (sock, "000 00\r\n", 8, MSG_DONTWAIT) > 0)
        {
          timeout = 0;
          if (verbose > 1)
            printf ("check conn app=%d\n", self_app);

        }

     do
      {
        ch = 0;
        FD_ZERO (&select_set);
        FD_SET (sock, &select_set);
        select_time.tv_sec = 0;
        select_time.tv_usec = 20 * 1000;
        if ((sel_ret =
             select (sock + 1, &select_set, NULL, NULL, &select_time)) > 0)
         {
           errno = 0;
           nread = recv (sock, &ch, 1, MSG_DONTWAIT);
           if ((errno && errno != 11) || nread < 0)
            {
              if (verbose >= 1)
                printf ("errno=%d %s\n", errno, strerror (errno));
              if (self_app == X11)
               {
                 x11_app = 0;
                 if (active_app == X11)
                   active_app = TEXT;
               }
              else
                text_app = 0;

              timeout = conn_timeout;
              reg = 0;
              cnt = 0;
              nread = 0;
              break;

            }
           if (nread > 0)
            {
              buffer[cnt] = ch;
              cnt++;

              if ((unsigned int) cnt >= sizeof (buffer))
               {

                 if (verbose > 0)
                   printf
                     ("buffer overflow - re-init buffer and reset counter\n");

                 cnt = 0;
                 memset (buffer, 0, sizeof (buffer));
               }

            }
           else
             break;
         }

      }
     while (ch != 0);

     if (sel_ret < 0)
      {
        if (verbose == 1)
          printf ("error = select <0 = %d\n", sel_ret);
        break;
      }

     if (cnt)
      {
        timeout = 0;
/* do the following twice - because \r\n could occur in reverse order 
*/
        if (buffer[strlen (buffer) - 1] == '\n'
            || buffer[strlen (buffer) - 1] == '\r')
          buffer[strlen (buffer) - 1] = 0;
        if (buffer[strlen (buffer) - 1] == '\n'
            || buffer[strlen (buffer) - 1] == '\r')
          buffer[strlen (buffer) - 1] = 0;

        if (verbose > 4)
          printf ("%d buf=%s\n", self_app, buffer);

        if (!strncmp (buffer, "getcon", 6))
         {
           pthread_mutex_lock (&conMutex);
           sprintf (tmpstr, "%02d %02d\r\n", con_nr, con_type);
           pthread_mutex_unlock (&conMutex);
           send (sock, tmpstr, strlen (tmpstr), 0);
         }
        else if (!strncmp (buffer, "seteventmode", 12))
         {
           event_mode = 1;
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strncmp (buffer, "unseteventmode", 14))
         {
           event_mode = 0;
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strncmp (buffer, "cursor ", 7))
         {
           if (!atoi (buffer + 7))
             csr_form = UNDERLINE_CSR;
           else
             csr_form = BLOCK_CSR;
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strcmp (buffer, "attrtbl"))
         {
           brltbl = attrtbl;
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strcmp (buffer, "texttbl"))
         {
           brltbl = texttbl;
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strncmp (buffer, "reg ", 4))
         {
           char tmp_auth_key[MAX_AUTH_KEY];

           if (strlen (buffer) > 7 && (strlen (buffer) - 7) < MAX_AUTH_KEY)
            {
              strcpy (tmp_auth_key, buffer + 7);
              buffer[6] = 0;
              if (!strcmp (config.auth_key, tmp_auth_key))
               {
                 if ((self_app = atoi (buffer + 4)))
                  {
                    pthread_mutex_lock (&conMutex);
                    if (self_app == TEXT && !text_app)
                     {
                       reg = 1;
                       text_app = 1;
                       if (con_type == TEXT_CON)
                         active_app = TEXT;
                     }
                    else if (self_app == X11 && !x11_app)
                     {
                       reg = 1;
                       x11_app = 1;
                       if (con_type == NON_TEXT_CON)
                         active_app = X11;
                     }
                    else if (self_app == ADMIN && !admin_app)
                     {
                       reg = 1;
                       admin_app = 1;
                     }
                    else
                     {
                       pthread_mutex_unlock (&conMutex);
                       send (sock, "error\r\n", 7, 0);
                       break;
                     }
                    pthread_mutex_unlock (&conMutex);

                  }
               }
            }

           if (!reg)
            {
              self_app = 0;
              send (sock, "error: registration failed\r\n", 28, 0);
            }
           else
             send (sock, "OK\r\n", 4, 0);

           if (verbose)
             printf ("got reg=%d\n", reg);
         }
        else if (!strncmp (buffer, "getxy", 5))
         {
           sprintf (tmpstr, "%02d %02d\r\n", brl_disp.x, brl_disp.y);

           if (verbose > 2)
             printf ("sending %s\n", tmpstr);

           send (sock, tmpstr, strlen (tmpstr), 0);
         }
        else if (!strncmp (buffer, "getkey", 6))
         {

           if (self_app == X11 && reg)
             fprintf (stderr, "orca getkey\n");

           brl_key = 0;
           brl_pressed = 0;
           if (reg && active_app == self_app)
            {

              pthread_mutex_lock (&key_buf_Mutex);
              key_get_buf (&brl_key, &brl_pressed);
              pthread_mutex_unlock (&key_buf_Mutex);

              if (brl_key > 0)
               {
                 sprintf (tmpstr, "%03d %02d\r\n", brl_key, brl_pressed);
                 send (sock, tmpstr, strlen (tmpstr), 0);
               }
              else
               {
                 send (sock, "NULL\r\n", 6, 0);
               }

            }
           else
             send (sock, "error: not allowed\r\n", 20, 0);

         }
        else if (!strncmp (buffer, "closebrl", 8))
         {
           /* not implemented */
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strncmp (buffer, "csrblock", 8))
         {
           /* not implemented */
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strncmp (buffer, "reset", 5))
         {

           if (brl_disp.brl_fd >= 0)
            {
              pthread_mutex_lock (&brl_dev_Mutex);
              braille->close (&brl_disp);
              brl_disp.brl_fd = -1;
              pthread_mutex_unlock (&brl_dev_Mutex);
              /* brl_ok(), periodically called by thread_brl, * will
                 automatically try to reconnect the * device. */
              brl_ok ();
            }
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strncmp (buffer, "openbrl", 7))
         {
           /* not implemented */
           send (sock, "OK\r\n", 4, 0);
         }
        else if (!strncmp (buffer, "getname", 7))
         {
           sprintf (tmpstr, "%s\r\n", brl_info.fullname);
           send (sock, tmpstr, strlen (tmpstr), 0);
         }
        else if (!strncmp (buffer, "getalias", 8))
         {
           sprintf (tmpstr, "%s\r\n", brl_info.name);
           send (sock, tmpstr, strlen (tmpstr), 0);
         }
        else if (!strncmp (buffer, "getstcells", 10))
         {
           sprintf (tmpstr, "%02d\r\n", brl_info.st_cells);
           send (sock, tmpstr, strlen (tmpstr), 0);
         }
        else if (!strncmp (buffer, "write ", 6))
         {
           int csrpos = 0, attr = 0, len = 0;

           if (reg && active_app == self_app)
            {

              attr = atoi (buffer + 14);
              buffer[14] = 0;
              len = atoi (buffer + 10);
              buffer[10] = 0;
              csrpos = atoi (buffer + 6);

              if (verbose > 1)
                printf ("p=%d attr=%d len=%d\n", csrpos, attr, len);

              send_to_brl (csr_form, csrpos, buffer + 17,
                           attr ? (buffer + 17 + len) : NULL, len);

            }

         }
        else if (!strncmp (buffer, "quit", 4))
         {
           if (self_app == X11)
            {
              x11_app = 0;
              if (active_app == X11)
                active_app = TEXT;
            }
           else
             text_app = 0;
           reg = 0;
           break;

         }

      }                         /* if cnt */
     else if (reg && active_app == self_app && event_mode && key_getbufcnt ())
      {
        brl_key = 0;
        brl_pressed = 0;
        if (reg && active_app == self_app)
         {
           pthread_mutex_lock (&key_buf_Mutex);
           key_get_buf (&brl_key, &brl_pressed);
           pthread_mutex_unlock (&key_buf_Mutex);

           if (brl_key > 0)
            {
              sprintf (tmpstr, "%03d %02d\r\n", brl_key, brl_pressed);
              send (sock, tmpstr, strlen (tmpstr), 0);
              timeout = 0;
            }

         }

      }

     usleep (SLEEP);
   }

  close (sock);
  pthread_mutex_lock (&connMutex);
  pthread_mutex_lock (&conMutex);
  connections--;
  if (reg && self_app == TEXT)
    text_app = 0;
  else if (reg && self_app == X11)
    x11_app = 0;
  else if (reg && self_app == ADMIN)
    admin_app = 0;
  reg = 0;
  pthread_mutex_unlock (&conMutex);
  pthread_mutex_unlock (&connMutex);

  if (verbose)
    printf ("child thread pid %d finished timeout=%d",
            (unsigned int) getpid (), timeout);

  return 0;
}

void *thread_brl ()
{
  int con_fd = 0;

  struct vt_stat vterm;

  struct timeval timeout;

  int mode = 0;

  int mode_fd = 0;

  int key = 0, pressed = 0;

  fd_set brlfd_set;

  int sel = 0;

  con_fd = open ("/dev/console", O_RDWR);
  while (brl_loop)
   {

     if (ioctl (con_fd, VT_GETSTATE, &vterm) < 0)
      {
        close (con_fd);
        con_fd = open ("/dev/console", O_RDWR);
        ioctl (con_fd, VT_GETSTATE, &vterm);
      }

/* check if console has changed */
     if (vterm.v_active != con_nr)
      {
        char tmpstr[MAX_BUF_LEN];

        sprintf (tmpstr, "/dev/tty%d", vterm.v_active);
        if (verbose > 3)
          printf ("console changed: %s\n", tmpstr);

        mode_fd = open (tmpstr, O_RDWR);
        ioctl (mode_fd, KDGETMODE, &mode);
        close (mode_fd);

        pthread_mutex_lock (&conMutex);
        con_nr = vterm.v_active;
        con_type = mode;

/* check if admin controls brld */
        if (active_app != ADMIN)
         {

/* if a text application is registerd and the console mode is text
 * give the text application the control of brld
 */
           if (text_app && con_type == TEXT_CON && active_app != TEXT)
            {
              active_app = TEXT;
              key_init_buf ();
            }
           else
/* if a x11 application is registered and console mode is none-text
 * give the x11 application the control of brld 
 */
           if (x11_app && con_type == NON_TEXT_CON && active_app != X11)
            {
              active_app = X11;
              key_init_buf ();
            }
           else
/* if only a text application is registerd
 * we give the control to it
 */
           if (text_app && !x11_app && active_app != TEXT)
            {
              active_app = TEXT;
              key_init_buf ();
            }
         }

        pthread_mutex_unlock (&conMutex);
      }

     if (!brl_ok ())
      {
        sleep (2);
        continue;
      }

     sel = 0;
     FD_ZERO (&brlfd_set);
     if (brl_disp.brl_fd >= 0 && !brl_disp.usb)
      {
        FD_SET (brl_disp.brl_fd, &brlfd_set);
        timeout.tv_sec = 1;
        timeout.tv_usec = (1000 * 200); /* 200 ms */
        sel = select (brl_disp.brl_fd + 1, &brlfd_set, NULL, NULL, &timeout);
      }

     if ((sel > 0 && brl_disp.brl_fd >= 0) || brl_disp.usb)
      {
        pthread_mutex_lock (&connMutex);
        if (connections)
         {
           pthread_mutex_lock (&brl_dev_Mutex);
           key = braille->read (&pressed);
           pthread_mutex_unlock (&brl_dev_Mutex);
         }
        pthread_mutex_unlock (&connMutex);

        if (key > 0)
         {
           pthread_mutex_lock (&key_buf_Mutex);
           key_put_buf (key, pressed);
           pthread_mutex_unlock (&key_buf_Mutex);
           key = 0;
           pressed = 0;
         }
        else
          usleep (10 * 1000);

      }

   }

  if (con_fd >= 0)
    close (con_fd);

  if (verbose)
    printf ("braille thread finished %d\n", brl_loop);

  return NULL;
}

void terminate (int signum)
{
  main_loop = 0;

  signal (signum, terminate);
}

int getdriver (char *libpath, char *brlname, char *brlport)
{
  struct brlinfo info = { "", "", 0, 0, 0, 0, 0 };
  DIR *dirp;

  struct dirent *entry;

  char libname[150] = "";

  dirp = opendir (libpath);
  while ((entry = readdir (dirp)))
   {

     if (!strncmp (entry->d_name, "libsbl", 6))
      {
        sprintf (libname, "%s/%s", libpath, entry->d_name);
        load_braille_driver (libname);
        info = braille->identify (brlname, brlport);

        if (verbose > 1)
          printf ("%s cols=%d\n", libname, info.cols);
        if (info.cols > 0)
          break;
        else
          unload_braille_driver ();
      }                         /* if */

   }                            /* while */
  closedir (dirp);

  if (!info.cols)
    return 1;

  braille->identifier = brlname;
  brl_info = info;

  return 0;
}

int load_brltbl (char *tblname)
{
  int tbl_fd;

  if (!tblname)
    return 1;

  if ((tbl_fd = open (tblname, O_RDONLY)) < 0)
    return 2;

  if (read (tbl_fd, texttbl, 256) != 256)
   {
     close (tbl_fd);
     return 3;
   }

  close (tbl_fd);
  return 0;
}

void send_to_brl (unsigned char csr_form, int csrpos, char *text, char *attr,
                  int len)
{
  int i;

/*
  char *inptr = text;
  char outbuf[MAX_BUF_LEN] = "";
  char *outptr = outbuf;
  int outlen = MAX_BUF_LEN - 1;
  */

  struct timeval timeout;

  fd_set select_set;

/*  iconv_t cd;*/

  if ((brl_disp.brl_fd < 0 && !brl_disp.usb) || brl_disp.disp == NULL)
    return;                     /* Not connected. */

  if (!brl_disp.usb)
   {
     timeout.tv_sec = 0;
     timeout.tv_usec = 50 * 1000;
     FD_ZERO (&select_set);
     FD_SET (brl_disp.brl_fd, &select_set);
   }

  memset (brl_disp.disp, ' ', brl_disp.x * brl_disp.y);

/*
  if (active_app == X11)
   {
     int tmp_len = len;

     cd = iconv_open ("ISO-8859-1", "UTF-8");
     iconv (cd, NULL, NULL, &outptr, (size_t *) & outlen);
     while ((int)
            iconv (cd, &inptr, (size_t *) & tmp_len, &outptr,
                      (size_t *) & outlen) < 0)
      { */
  /* ignore/skip unknown characters inptr++; tmp_len--; } iconv_close (cd);
     if (verbose > 2) printf ("conv=%s\n", outbuf); memcpy (brl_disp.disp,
     outbuf, (MAX_BUF_LEN - 1) - outlen); errno = 0;

     } else */
  memcpy (brl_disp.disp, text, len);

  /* translate text to braille */
  for (i = 0; i < brl_disp.x * brl_disp.y; i++)
    brl_disp.disp[i] = brltbl[brl_disp.disp[i]];

/* set attribute underline if needed */
  for (i = 0; i < len && attr; i++)
    if (attr[i] != ' ')
      brl_disp.disp[i] |= attr[i];

  /* set cursor */
  if (csrpos >= 0)
    brl_disp.disp[csrpos] |= csr_form;

  if (brl_disp.usb
      || (select (brl_disp.brl_fd + 1, NULL, &select_set, NULL, &timeout) >
          0))
   {
     pthread_mutex_lock (&brl_dev_Mutex);
     braille->write (&brl_disp);
     if (errno)                 /* Input/output error */
      {
        if (verbose > 2)
          printf ("err_brlwrite%d %s\n", errno, strerror (errno));
        braille->close (&brl_disp);
        brl_disp.brl_fd = -1;
        brl_disp.usb = 0;
      }
     pthread_mutex_unlock (&brl_dev_Mutex);
   }

}

/* tries to reconnect if braille device is offline, return 1 if device
 * is ready, otherwise return 0 */
int brl_ok ()
{
  int result = 1;

  if (brl_disp.brl_fd >= 0 || brl_disp.usb)
    return 1;
  pthread_mutex_lock (&brl_dev_Mutex);
  if (brl_disp.brl_fd < 0 && !brl_disp.usb)
   {

     if (brl_disp.disp)
      {
        free (brl_disp.disp);
        brl_disp.disp = NULL;
      }

     braille->initialize (&brl_disp, config.brlport);
     result = (brl_disp.brl_fd >= 0);
     if (brl_disp.brl_fd >= 0 || brl_disp.usb)
      {
        pthread_mutex_lock (&key_buf_Mutex);
        key_init_buf ();
        pthread_mutex_unlock (&key_buf_Mutex);
      }

     if (verbose > 8)
       printf
         ("brl_ok: usb=%d brl_disp.brl_fd=%d, brl_disp.x=%d, brl_disp.y=%d\n",
          brl_disp.usb, brl_disp.brl_fd, brl_disp.x, brl_disp.y);
     if (result || brl_disp.usb)
      {
        memset (brl_disp.disp, ' ', brl_disp.x * brl_disp.y);
      }
   }

  pthread_mutex_unlock (&brl_dev_Mutex);
  return brl_disp.usb ? 1 : result;
}
