/*
 *	Routines for bridge interface handling
 *
 *	Copyright (C) 2009, 2010  Olaf Kirch <okir@suse.de>
 *	Copyright (C) 2011-2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Olaf Kirch
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#ifndef __NETCONTROL_BRIDGE_H
#define __NETCONTROL_BRIDGE_H
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif

#include <sutils.h>
#include <ntypes.h>
#include <handle.h>

enum {
	NC_BRIDGE_NO_STP = 0,			/* no spanning tree */
	NC_BRIDGE_STP,				/* old STP in kernel */
	NC_BRIDGE_RSTP,				/* new RSTP in userspace */
};

enum {
	NC_BRIDGE_STP_ENABLED	= 1,		/* bridge config options */
	NC_BRIDGE_FORWARD_DELAY	= 2,
	NC_BRIDGE_AGEING_TIME	= 3,
	NC_BRIDGE_HELLO_TIME	= 4,
	NC_BRIDGE_MAX_AGE	= 5,
	NC_BRIDGE_PRIORITY	= 6,
	NC_BRIDGE_PORT_PRIORITY	= 7,		/* bridge port config options */
	NC_BRIDGE_PORT_PATH_COST= 8,
};

typedef struct nc_bridge_port_config {
	unsigned int		priority;
	unsigned int		path_cost;
} nc_bridge_port_config_t;

typedef struct nc_bridge_port_status {
	unsigned int		priority;
	unsigned int		path_cost;

	int			state;
	unsigned int		port_id;
	unsigned int		port_no;

	char *			designated_root;
	char *			designated_bridge;

	unsigned int		designated_cost;
	unsigned int		designated_port;

	unsigned int		change_ack;
	unsigned int		hairpin_mode;
	unsigned int		config_pending;

	unsigned long		hold_timer;
	unsigned long		message_age_timer;
	unsigned long		forward_delay_timer;
} nc_bridge_port_status_t;

typedef struct nc_bridge_port {
	char *			name;
	nc_bridge_port_config_t config;
	nc_bridge_port_status_t	*status;
} nc_bridge_port_t;

typedef struct nc_bridge_port_array {
	unsigned int		count;
	nc_bridge_port_t **	data;
} nc_bridge_port_array_t;

typedef struct nc_bridge_config {
	unsigned int		priority;
	int			stp_enabled;
	unsigned long		forward_delay;	/* time in 1/100 sec */
	unsigned long		ageing_time;	/* time in 1/100 sec */
	unsigned long		hello_time;	/* time in 1/100 sec */
	unsigned long		max_age;	/* time in 1/100 sec */
} nc_bridge_config_t;

typedef struct nc_bridge_status {
	unsigned int		priority;
	int			stp_state;
	unsigned long		forward_delay;
	unsigned long		ageing_time;
	unsigned long		hello_time;
	unsigned long		max_age;

	char *			root_id;
	char *			bridge_id;
	char *			group_addr;

	unsigned int		root_port;
	unsigned int		root_path_cost;
	unsigned int		topology_change;
	unsigned int		topology_change_detected;

	unsigned long		gc_timer;
	unsigned long		tcn_timer;
	unsigned long		hello_timer;
	unsigned long		topology_change_timer;
} nc_bridge_status_t;

struct nc_bridge {
	nc_bridge_config_t	config;
	nc_bridge_status_t *	status;
	nc_bridge_port_array_t	ports;
};

extern int		nc_bridge_bind(nc_interface_t *, nc_handle_t *, nc_var_array_t *);

extern nc_bridge_t *	nc_bridge_new(void);
extern void		nc_bridge_free(nc_bridge_t *);
extern void		nc_bridge_status_free(nc_bridge_status_t *);
extern void		nc_bridge_port_status_free(nc_bridge_port_status_t *);
extern nc_bridge_t *	nc_bridge_clone(const nc_bridge_t *);
extern int		nc_bridge_add_port(nc_bridge_t *, const char *);
extern int		nc_bridge_del_port(nc_bridge_t *, const char *);
extern void		nc_bridge_get_port_names(const nc_bridge_t *, nc_string_array_t *);
extern int		nc_bridge_get(nc_bridge_t *, unsigned int, char **);
extern int		nc_bridge_get_stp(nc_bridge_t *, char **);
extern int		nc_bridge_get_forward_delay(nc_bridge_t *, char **);
extern int		nc_bridge_get_ageing_time(nc_bridge_t *, char **);
extern int		nc_bridge_get_hello_time(nc_bridge_t *, char **);
extern int		nc_bridge_get_max_age(nc_bridge_t *, char **);
extern int		nc_bridge_get_priority(nc_bridge_t *, char **);
extern int		nc_bridge_set_stp(nc_bridge_t *, const char *);
extern int		nc_bridge_set_forward_delay(nc_bridge_t *, const char *);
extern int		nc_bridge_set_ageing_time(nc_bridge_t *, const char *);
extern int		nc_bridge_set_hello_time(nc_bridge_t *, const char *);
extern int		nc_bridge_set_max_age(nc_bridge_t *, const char *);
extern int		nc_bridge_set_priority(nc_bridge_t *, const char *);
extern int		nc_bridge_port_get(nc_bridge_t *, const char *, unsigned int, char **);
extern int		nc_bridge_port_get_priority(nc_bridge_t *,const char *, char **);
extern int		nc_bridge_port_get_path_cost(nc_bridge_t *,const char *, char **);
extern int		nc_bridge_port_set_priority(nc_bridge_t *,const char *, const char *);
extern int		nc_bridge_port_set_path_cost(nc_bridge_t *,const char *, const char *);

extern int		nc_sysfs_bridge_get_config(const char *, nc_bridge_config_t *);
extern int		nc_sysfs_bridge_get_status(const char *, nc_bridge_status_t **);
extern int		nc_sysfs_bridge_get_port_names(const char *, nc_string_array_t *);
extern int		nc_sysfs_bridge_port_get_config(const char *, nc_bridge_port_config_t *);
extern int		nc_sysfs_bridge_port_get_status(const char *, nc_bridge_port_status_t **);
extern int		nc_sysfs_get_bridge_parent(const char *name, char **parent);

#endif /* __NETCONTROL_BRIDGE_H */
