/* Copyright (C) 2004, 2005, 2006  Thorsten Kukuk
   This file is part of the nscd interface library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License version 2.1 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include <errno.h>
#include <assert.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>

#include "libnscd.h"

/* Version number of the daemon interface */
#define NSCD_VERSION 2
/* Path for the Unix domain socket.  */
#define _PATH_NSCDSOCKET1 "/var/run/nscd/socket"
#define _PATH_NSCDSOCKET2 "/var/run/.nscd_socket"

/* Available services.  */
typedef enum
{
  GETPWBYNAME,
  GETPWBYUID,
  GETGRBYNAME,
  GETGRBYGID,
  GETHOSTBYNAME,
  GETHOSTBYNAMEv6,
  GETHOSTBYADDR,
  GETHOSTBYADDRv6,
  LASTDBREQ = GETHOSTBYADDRv6,
  SHUTDOWN,             /* Shut the server down.  */
  GETSTAT,              /* Get the server statistic.  */
  INVALIDATE,           /* Invalidate one special cache.  */
  LASTREQ
} request_type;

/* Header common to all requests */
typedef struct
{
  int version;          /* Version number of the daemon interface.  */
  request_type type;    /* Service requested.  */
#if defined(__alpha__)
  int64_t key_len;      /* Key length is 64bit on Alpha.  */
#else
  int32_t key_len;      /* Key length, 32bit on most plattforms.  */
#endif
} request_header;

/* Create a socket connected to a name.  */
static int
nscd_open_socket (void)
{
  struct sockaddr_un addr;
  int sock;

  sock = socket (PF_UNIX, SOCK_STREAM, 0);
  if (sock < 0)
    return -1;

  addr.sun_family = AF_UNIX;
  assert (sizeof (addr.sun_path) >= sizeof (_PATH_NSCDSOCKET1));
  strcpy (addr.sun_path, _PATH_NSCDSOCKET1);
  if (connect (sock, (struct sockaddr *) &addr, sizeof (addr)) < 0)
    {
      assert (sizeof (addr.sun_path) >= sizeof (_PATH_NSCDSOCKET2));
      strcpy (addr.sun_path, _PATH_NSCDSOCKET2);
      if (connect (sock, (struct sockaddr *) &addr, sizeof (addr)) < 0)
	{
	  int err = errno;
	  close (sock);
	  errno = err;
	  return -1;
	}
    }
  return sock;
}

int
nscd_flush_cache (const char *service)
{
  int sock = nscd_open_socket ();
  request_header req;
  ssize_t nbytes;
  int32_t resp;
  struct iovec iov[2];

  if (sock == -1)
    return -1;

  req.version = NSCD_VERSION;
  req.type = INVALIDATE;
  req.key_len = strlen (service) + 1;

  iov[0].iov_base = &req;
  iov[0].iov_len = sizeof (req);
  iov[1].iov_base = strdup (service);
  iov[1].iov_len = req.key_len;

  nbytes = TEMP_FAILURE_RETRY (writev (sock, iov, 2));
  if (nbytes != (ssize_t)(iov[0].iov_len + iov[1].iov_len))
    {
      int err = errno;
      close (sock);
      free (iov[1].iov_base);
      errno = err;
      return -1;
    }
  free (iov[1].iov_base);

  /* Wait for ACK. Older nscd just closed the socket, ignore that.  */
  resp = 0; /* set to 0 for old nscd implementations. */
  nbytes = TEMP_FAILURE_RETRY (read (sock, &resp, sizeof (resp)));
  if (nbytes != 0 && nbytes != sizeof (resp))
    {
      int err = errno;
      close (sock);
      errno = err;
      return -1;
    }

  close (sock);
  if (resp > 0)
    {
      errno = -resp;
      return -1;
    }
  else
    return 0;
}
