// Copyright 2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License"). You may not
// use this file except in compliance with the License. A copy of the
// License is located at
//
// http://aws.amazon.com/apache2.0/
//
// or in the "license" file accompanying this file. This file is distributed
// on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
// either express or implied. See the License for the specific language governing
// permissions and limitations under the License.

package main

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"strings"
	"text/template"

	"github.com/aws/amazon-ssm-agent/agent/appconfig"
)

const versiongoTemplate = `// This is an autogenerated file and should not be edited.
// Please DO NOT commit any changes to this file.

// Package version contains constants to indicate the current version of the
// agent. It is autogenerated.
package version

// Version is the version of the Agent
const Version = "{{.Version}}"
`
const releaseNotesFile = "RELEASENOTES.md"

type versionInfo struct {
	Version string
}

// version-gen is a simple program that generates the agent's version file,
// containing information about the agent's version
func main() {

	licenseStr, err := ioutil.ReadFile(filepath.Join("Tools", "src", "LICENSE"))
	if err != nil {
		log.Fatalf("Error reading LICENSE file. %v", err)
	}

	versionContent, err := ioutil.ReadFile(filepath.Join("VERSION"))
	if err != nil {
		log.Fatalf("Error reading VERSION file. %v", err)
	}

	versionStr := strings.TrimSpace(string(versionContent))

	fmt.Printf("Agent Version: %v", versionStr)
	if err := ioutil.WriteFile(filepath.Join("VERSION"), []byte(versionStr), appconfig.ReadWriteAccess); err != nil {
		log.Fatalf("Error writing to VERSION file. %v", err)
	}

	// default values
	info := versionInfo{
		Version: versionStr,
	}

	var newVersion bytes.Buffer
	versionFilePath := filepath.Join("agent", "version", "version.go")

	t := template.Must(template.New("version").Parse(string(licenseStr) + versiongoTemplate))
	err = t.Execute(&newVersion, info)
	if err != nil {
		log.Fatalf("Error applying template: %v", err)
	}

	oldContent, err := ioutil.ReadFile(versionFilePath)
	if err != nil {
		log.Fatalf("Error reading old version file: %v", err)
	}

	if newVersion.String() != string(oldContent) {
		outFile, err := os.Create(versionFilePath)
		if err != nil {
			log.Fatalf("Unable to create output version file: %v", err)
		}
		defer outFile.Close()

		err = t.Execute(outFile, info)
		if err != nil {
			log.Fatalf("Error applying template: %v", err)
		}
	}

	// Adding the version generated to the release notes
	releaseNoteInFile, err := ioutil.ReadFile(filepath.Join(releaseNotesFile))
	if err != nil {
		log.Fatalf("Unable to open RELEASENOTES.md file ")
	}

	releaseNoteLines := strings.Split(string(releaseNoteInFile), "\n")

	//Adding version to the top of the file
	releaseNoteLines[0] = versionStr

	releaseNoteOutFile := strings.Join(releaseNoteLines, "\n")

	if err = ioutil.WriteFile(filepath.Join(releaseNotesFile), []byte(releaseNoteOutFile), appconfig.ReadWriteAccess); err != nil {
		log.Fatalf("Error writing to RELEASENOTES.md file. %v", err)
	}

}
