package tests

import (
	"fmt"
	"net/http"
	"testing"

	"github.com/Azure/azure-sdk-for-go/services/preview/sql/mgmt/v3.0/sql"
	"github.com/hashicorp/terraform-plugin-sdk/helper/resource"
	"github.com/hashicorp/terraform-plugin-sdk/terraform"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/helpers/azure"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/acceptance"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/internal/clients"
	"github.com/terraform-providers/terraform-provider-azurerm/azurerm/utils"
)

func TestAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_basic(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_mssql_database_vulnerability_assessment_rule_baseline", "test")

	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaselineDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_basic(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaselineExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "rule_id", "VA2002"),
					resource.TestCheckResourceAttr(data.ResourceName, "baseline_name", "default"),
					resource.TestCheckResourceAttr(data.ResourceName, "baseline_result.#", "2"),
				),
			},
			data.ImportStep(),
		},
	})
}

func TestAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_primary(t *testing.T) {
	data := acceptance.BuildTestData(t, "azurerm_mssql_database_vulnerability_assessment_rule_baseline", "test")
	resource.ParallelTest(t, resource.TestCase{
		PreCheck:     func() { acceptance.PreCheck(t) },
		Providers:    acceptance.SupportedProviders,
		CheckDestroy: testCheckAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaselineDestroy,
		Steps: []resource.TestStep{
			{
				Config: testAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_primary(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaselineExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "rule_id", "VA2065"),
					resource.TestCheckResourceAttr(data.ResourceName, "baseline_name", "master"),
					resource.TestCheckResourceAttr(data.ResourceName, "baseline_result.#", "1"),
				),
			},
			data.ImportStep(),
			{
				Config: testAccAzureRMDatabaseVulnerabilityAssessmentRuleBaseline_update(data),
				Check: resource.ComposeTestCheckFunc(
					testCheckAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaselineExists(data.ResourceName),
					resource.TestCheckResourceAttr(data.ResourceName, "rule_id", "VA2065"),
					resource.TestCheckResourceAttr(data.ResourceName, "baseline_name", "master"),
					resource.TestCheckResourceAttr(data.ResourceName, "baseline_result.#", "2"),
				),
			},
			data.ImportStep(),
		},
	})
}

func testCheckAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaselineExists(resourceName string) resource.TestCheckFunc {
	return func(s *terraform.State) error {
		client := acceptance.AzureProvider.Meta().(*clients.Client).MSSQL.DatabaseVulnerabilityAssessmentRuleBaselinesClient
		ctx := acceptance.AzureProvider.Meta().(*clients.Client).StopContext

		rs, ok := s.RootModule().Resources[resourceName]
		if !ok {
			return fmt.Errorf("resource not found: %s", resourceName)
		}

		id, err := azure.ParseAzureResourceID(rs.Primary.ID)
		if err != nil {
			return err
		}

		resourceGroupName := id.ResourceGroup
		serverName := id.Path["servers"]
		databaseName := rs.Primary.Attributes["database_name"]
		ruleId := rs.Primary.Attributes["rule_id"]
		baselineName := sql.VulnerabilityAssessmentPolicyBaselineName(rs.Primary.Attributes["baseline_name"])

		resp, err := client.Get(ctx, resourceGroupName, serverName, databaseName, ruleId, baselineName)
		if err != nil {
			if utils.ResponseWasNotFound(resp.Response) {
				return fmt.Errorf("baseline was not found for "+
					"resource group %q, sql server %q, database %q, rule id %q, baseline name %q",
					resourceGroupName, serverName, databaseName, ruleId, baselineName)
			}

			return err
		}

		return nil
	}
}

func testCheckAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaselineDestroy(s *terraform.State) error {
	client := acceptance.AzureProvider.Meta().(*clients.Client).MSSQL.DatabaseVulnerabilityAssessmentRuleBaselinesClient
	ctx := acceptance.AzureProvider.Meta().(*clients.Client).StopContext

	for _, rs := range s.RootModule().Resources {
		if rs.Type != "azurerm_mssql_database_vulnerability_assessment_rule_baseline" {
			continue
		}

		id, err := azure.ParseAzureResourceID(rs.Primary.ID)
		if err != nil {
			return err
		}

		resourceGroupName := id.ResourceGroup
		serverName := id.Path["servers"]
		databaseName := rs.Primary.Attributes["database_name"]
		ruleId := rs.Primary.Attributes["rule_id"]
		baselineName := sql.VulnerabilityAssessmentPolicyBaselineName(rs.Primary.Attributes["baseline_name"])

		resp, err := client.Get(ctx, resourceGroupName, serverName, databaseName, ruleId, baselineName)
		if err != nil {
			return nil
		}

		if resp.StatusCode != http.StatusNotFound {
			return fmt.Errorf("Database Vulnerability Assessment Rule Baseline still exists:\n%#v", resp.DatabaseVulnerabilityAssessmentRuleBaselineProperties)
		}
	}

	return nil
}

func testAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_basic(data acceptance.TestData) string {
	database := testAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_database(data)

	return fmt.Sprintf(`
%s

resource "azurerm_mssql_database_vulnerability_assessment_rule_baseline" "test" {
  server_vulnerability_assessment_id = azurerm_mssql_server_vulnerability_assessment.test.id
  database_name                      = azurerm_sql_database.test.name
  rule_id                            = "VA2002"
  baseline_name                      = "default"
  baseline_result {
    result = [
      "SCHEMA",
      "dbo",
      "CONTROL",
      "SQL_USER",
      "adminuser1"
    ]
  }
  baseline_result {
    result = [
      "SCHEMA",
      "dbo",
      "CONTROL",
      "SQL_USER",
      "adminuser2"
    ]
  }
}
`, database)
}

func testAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_primary(data acceptance.TestData) string {
	database := testAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_database(data)

	return fmt.Sprintf(`
%s

resource "azurerm_mssql_database_vulnerability_assessment_rule_baseline" "test" {
  server_vulnerability_assessment_id = azurerm_mssql_server_vulnerability_assessment.test.id
  database_name                      = azurerm_sql_database.test.name
  rule_id                            = "VA2065"
  baseline_name                      = "master"
  baseline_result {
    result = [
      "clientip",
      "255.255.255.255",
      "255.255.255.255"
    ]
  }
}
`, database)
}

func testAccAzureRMDatabaseVulnerabilityAssessmentRuleBaseline_update(data acceptance.TestData) string {
	database := testAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_database(data)

	return fmt.Sprintf(`
%s

resource "azurerm_mssql_database_vulnerability_assessment_rule_baseline" "test" {
  server_vulnerability_assessment_id = azurerm_mssql_server_vulnerability_assessment.test.id
  database_name                      = azurerm_sql_database.test.name
  rule_id                            = "VA2065"
  baseline_name                      = "master"
  baseline_result {
    result = [
      "clientips",
      "255.255.255.255",
      "255.255.255.255"
    ]
  }
  baseline_result {
    result = [
      "myip",
      "255.255.255.0",
      "255.255.255.0"
    ]
  }
}
`, database)
}

func testAccAzureRMMssqlDatabaseVulnerabilityAssessmentRuleBaseline_database(data acceptance.TestData) string {
	return fmt.Sprintf(`
provider "azurerm" {
  features {}
}

resource "azurerm_resource_group" "test" {
  name     = "acctestRG-mssql-%d"
  location = "%s"
}

resource "azurerm_sql_server" "test" {
  name                         = "acctestsqlserver%d"
  resource_group_name          = azurerm_resource_group.test.name
  location                     = azurerm_resource_group.test.location
  version                      = "12.0"
  administrator_login          = "mradministrator"
  administrator_login_password = "thisIsDog11"
}

resource "azurerm_storage_account" "test" {
  name                     = "accsa%d"
  resource_group_name      = azurerm_resource_group.test.name
  location                 = "%s"
  account_tier             = "Standard"
  account_replication_type = "GRS"
}

resource "azurerm_storage_container" "test" {
  name                  = "acctestsc%d"
  storage_account_name  = azurerm_storage_account.test.name
  container_access_type = "private"
}

resource "azurerm_mssql_server_security_alert_policy" "test" {
  resource_group_name = azurerm_resource_group.test.name
  server_name         = azurerm_sql_server.test.name
  state               = "Enabled"
}

resource "azurerm_mssql_server_vulnerability_assessment" "test" {
  server_security_alert_policy_id = azurerm_mssql_server_security_alert_policy.test.id
  storage_container_path          = "${azurerm_storage_account.test.primary_blob_endpoint}${azurerm_storage_container.test.name}/"
  storage_account_access_key      = azurerm_storage_account.test.primary_access_key
}

resource "azurerm_sql_database" "test" {
  name                = "acctestdb%d"
  resource_group_name = azurerm_resource_group.test.name
  server_name         = azurerm_sql_server.test.name
  location            = azurerm_resource_group.test.location
  edition             = "Standard"
}
`, data.RandomInteger, data.Locations.Primary, data.RandomInteger, data.RandomInteger, data.Locations.Primary, data.RandomInteger, data.RandomInteger)
}
