/*
 * Copyright IBM Corp. 2021
 *
 * libzpc is free software; you can redistribute it and/or modify
 * it under the terms of the MIT license. See LICENSE for details.
 */

#ifndef TESTLIB_H
# define TESTLIB_H

# include "gtest/gtest.h"

# ifdef __cplusplus
/* *INDENT-OFF* */
extern "C" {
/* *INDENT-ON* */
# endif

# include <stddef.h>

#include "zpc/ecc_key.h"

# define UNUSED(x)	(void)(x)
# define NMEMB(x)	(sizeof(x) / sizeof(x[0]))

/*
 * GTEST_SKIP_ assumes the caller to be the test function that is to
 * be skipped. So this one has to be implemented as a macro.
 */
# define TESTLIB_ENV_AES_KEY_CHECK()													        \
do {		                                                                                    \
        const char *apqns[257];                                                                       \
        const char *mkvp;															            \
        int size, type, rc;																        \
																					            \
        size = testlib_env_aes_key_size();									                    \
        switch (size) {                                                                         \
        case 128:   /* fall-through */                                                          \
        case 192:   /* fall-through */                                                          \
        case 256:   /* fall-through */                                                          \
            break;                                                                              \
        case -1:														                        \
            GTEST_SKIP_("ZPC_TEST_AES_KEY_SIZE environment variable not set.");                 \
            break;                                                                              \
        default:                                                                                \
            GTEST_SKIP_("ZPC_TEST_AES_KEY_SIZE environment variable set to invalid key-size."); \
            break;                                                                              \
        }			                                                                            \
        											                                            \
        type = testlib_env_aes_key_type();											            \
        if (type == -1)																            \
                GTEST_SKIP_("ZPC_TEST_AES_KEY_TYPE environment variable not set.");	            \
                                                                                                \
        mkvp = testlib_env_aes_key_mkvp();                                                      \
        rc = testlib_env_aes_key_apqns(apqns);                                                  \
        if (rc == 0 && mkvp != NULL)                                                            \
            GTEST_SKIP_("Both ZPC_TEST_AES_KEY_MKVP and ZPC_TEST_AES_KEY_APQNS environment variables set.");    \
        if (rc != 0 && mkvp == NULL)                                                            \
            GTEST_SKIP_("ZPC_TEST_AES_KEY_MKVP and ZPC_TEST_AES_KEY_APQNS environment variables unset.");       \
} while (0)

# define TESTLIB_AES_ECB_HW_CAPS_CHECK()                                       \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_ecb *ctx;                                               \
                                                                               \
        rc = zpc_aes_ecb_alloc(&ctx);                                          \
        switch (rc) {                                                          \
        case ZPC_ERROR_DEVPKEY:                                                \
            GTEST_SKIP_("HW_CAPS check (AES-ECB): opening /dev/pkey failed."); \
            break;                                                             \
        case ZPC_ERROR_HWCAPS:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-ECB): no hw capabilities for AES-ECB."); \
            break;                                                             \
        case ZPC_ERROR_MALLOC:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-ECB): cannot allocate AES ctx object."); \
            break;                                                             \
        default:                                                               \
            zpc_aes_ecb_free(&ctx);                                            \
            break;                                                             \
        }                                                                      \
} while (0)

# define TESTLIB_AES_CBC_HW_CAPS_CHECK()                                       \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_cbc *ctx;                                               \
                                                                               \
        rc = zpc_aes_cbc_alloc(&ctx);                                          \
        switch (rc) {                                                          \
        case ZPC_ERROR_DEVPKEY:                                                \
            GTEST_SKIP_("HW_CAPS check (AES-CBC): opening /dev/pkey failed."); \
            break;                                                             \
        case ZPC_ERROR_HWCAPS:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-CBC): no hw capabilities for AES-CBC."); \
            break;                                                             \
        case ZPC_ERROR_MALLOC:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-CBC): cannot allocate AES ctx object."); \
            break;                                                             \
        default:                                                               \
            zpc_aes_cbc_free(&ctx);                                            \
            break;                                                             \
        }                                                                      \
} while (0)

# define TESTLIB_AES_CMAC_HW_CAPS_CHECK()                                      \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_cmac *ctx;                                              \
                                                                               \
        rc = zpc_aes_cmac_alloc(&ctx);                                         \
        switch (rc) {                                                          \
        case ZPC_ERROR_DEVPKEY:                                                \
            GTEST_SKIP_("HW_CAPS check (AES-CMAC): opening /dev/pkey failed.");\
            break;                                                             \
        case ZPC_ERROR_HWCAPS:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-CMAC): no hw capabilities for AES-CMAC."); \
            break;                                                             \
        case ZPC_ERROR_MALLOC:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-CMAC): cannot allocate AES ctx object."); \
            break;                                                             \
        default:                                                               \
            zpc_aes_cmac_free(&ctx);                                           \
            break;                                                             \
        }                                                                      \
} while (0)

# define TESTLIB_AES_CCM_HW_CAPS_CHECK()                                       \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_ccm *ctx;                                               \
                                                                               \
        rc = zpc_aes_ccm_alloc(&ctx);                                          \
        switch (rc) {                                                          \
        case ZPC_ERROR_DEVPKEY:                                                \
            GTEST_SKIP_("HW_CAPS check (AES-CCM): opening /dev/pkey failed."); \
            break;                                                             \
        case ZPC_ERROR_HWCAPS:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-CCM): no hw capabilities for AES-CCM."); \
            break;                                                             \
        case ZPC_ERROR_MALLOC:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-CCM): cannot allocate AES ctx object."); \
            break;                                                             \
        default:                                                               \
            zpc_aes_ccm_free(&ctx);                                            \
            break;                                                             \
        }                                                                      \
} while (0)

# define TESTLIB_AES_GCM_HW_CAPS_CHECK()                                       \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_gcm *ctx;                                               \
                                                                               \
        rc = zpc_aes_gcm_alloc(&ctx);                                          \
        switch (rc) {                                                          \
        case ZPC_ERROR_DEVPKEY:                                                \
            GTEST_SKIP_("HW_CAPS check (AES-GCM): opening /dev/pkey failed."); \
            break;                                                             \
        case ZPC_ERROR_HWCAPS:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-GCM): no hw capabilities for AES-GCM."); \
            break;                                                             \
        case ZPC_ERROR_MALLOC:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-GCM): cannot allocate AES ctx object."); \
            break;                                                             \
        default:                                                               \
            zpc_aes_gcm_free(&ctx);                                            \
            break;                                                             \
        }                                                                      \
} while (0)

# define TESTLIB_AES_XTS_HW_CAPS_CHECK()                                       \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_xts *ctx;                                               \
                                                                               \
        rc = zpc_aes_xts_alloc(&ctx);                                          \
        switch (rc) {                                                          \
        case ZPC_ERROR_DEVPKEY:                                                \
            GTEST_SKIP_("HW_CAPS check (AES-XTS): opening /dev/pkey failed."); \
            break;                                                             \
        case ZPC_ERROR_HWCAPS:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-XTS): no hw capabilities for AES-XTS."); \
            break;                                                             \
        case ZPC_ERROR_MALLOC:                                                 \
            GTEST_SKIP_("HW_CAPS check (AES-XTS): cannot allocate AES ctx object."); \
            break;                                                             \
        default:                                                               \
            zpc_aes_xts_free(&ctx);                                            \
            break;                                                             \
        }                                                                      \
} while (0)

# define TESTLIB_AES_XTS_KEY_SIZE_CHECK(size)                                  \
do {                                                                           \
        if (size != 128 && size != 256) {                                      \
            GTEST_SKIP_("Key size check (AES-XTS): only 128 and 256 bits supported by CPACF."); \
        }                                                                      \
} while (0)

# define TESTLIB_AES_SW_CAPS_CHECK(type)                                       \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_key *aes_key;                                           \
                                                                               \
        rc = zpc_aes_key_alloc(&aes_key);                                      \
        if (rc != 0)                                                           \
            GTEST_SKIP_("SW_CAPS check (AES): Cannot allocate key object.");   \
                                                                               \
        rc = zpc_aes_key_set_type(aes_key, type);                              \
        if ((type == ZPC_AES_KEY_TYPE_CCA_DATA ||                              \
             type == ZPC_AES_KEY_TYPE_CCA_CIPHER) &&                           \
            rc == ZPC_ERROR_CCA_HOST_LIB_NOT_AVAILABLE) {                      \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("SW_CAPS check (AES): CCA host lib not available or too old."); \
        }                                                                      \
        if (type == ZPC_AES_KEY_TYPE_EP11 &&                                   \
            rc == ZPC_ERROR_EP11_HOST_LIB_NOT_AVAILABLE) {                     \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("SW_CAPS check (AES): EP11 host lib not available or too old."); \
        }                                                                      \
        if (rc != 0) {                                                         \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("SW_CAPS check (AES): Unexpected error when setting key type."); \
        }                                                                      \
        zpc_aes_key_free(&aes_key);                                            \
} while (0)

# define TESTLIB_AES_KERNEL_CAPS_CHECK()                                       \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_key *aes_key;                                           \
                                                                               \
        rc = zpc_aes_key_alloc(&aes_key);                                      \
        if (rc != 0)                                                           \
            GTEST_SKIP_("KERNEL_CAPS check (AES): Cannot allocate key object."); \
                                                                               \
        rc = zpc_aes_key_set_mkvp(aes_key, NULL); /* cannot fail */            \
        rc = zpc_aes_key_set_size(aes_key, 128); /* cannot fail */             \
        rc = zpc_aes_key_generate(aes_key);                                    \
        if (rc == ZPC_ERROR_IOCTLGENPROTK) {                                   \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("KERNEL_CAPS check (AES): ioctl PKEY_GENPROTK not supported by kernel."); \
        }                                                                      \
        if (rc == ZPC_ERROR_IOCTLGENSECK2) {                                   \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("KERNEL_CAPS check (AES): ioctl PKEY_GENSECK2 not supported by kernel."); \
        }                                                                      \
        if (rc != 0) {                                                         \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("KERNEL_CAPS check (AES): Unexpected error when generating test key."); \
        }                                                                      \
                                                                               \
        zpc_aes_key_free(&aes_key);                                            \
} while (0)

# define TESTLIB_AES_NEW_MK_CHECK(type,mkvp,apqns)                             \
do {                                                                           \
        int rc;                                                                \
        struct zpc_aes_key *aes_key;                                           \
                                                                               \
        rc = zpc_aes_key_alloc(&aes_key);                                      \
        if (rc != 0)                                                           \
            GTEST_SKIP_("NEW_MK check (AES): Cannot allocate key object.");    \
                                                                               \
        rc = zpc_aes_key_set_type(aes_key, type);                              \
        if ((type == ZPC_AES_KEY_TYPE_CCA_DATA ||                              \
             type == ZPC_AES_KEY_TYPE_CCA_CIPHER) &&                           \
            rc == ZPC_ERROR_CCA_HOST_LIB_NOT_AVAILABLE) {                      \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("NEW_MK check (AES): CCA host lib not available or too old."); \
        }                                                                      \
        if (type == ZPC_AES_KEY_TYPE_EP11 &&                                   \
            rc == ZPC_ERROR_EP11_HOST_LIB_NOT_AVAILABLE) {                     \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("NEW_MK check (AES): EP11 host lib not available or too old."); \
        }                                                                      \
        if (mkvp != NULL) {                                                    \
            rc = zpc_aes_key_set_mkvp(aes_key, mkvp);                          \
            if (rc != 0) {                                                     \
                zpc_aes_key_free(&aes_key);                                    \
                GTEST_SKIP_("NEW_MK check (AES): error setting mkvp.");        \
            }                                                                  \
        } else {                                                               \
            rc = zpc_aes_key_set_apqns(aes_key, apqns);                        \
            if (rc != 0) {                                                     \
                zpc_aes_key_free(&aes_key);                                    \
                GTEST_SKIP_("NEW_MK check (AES): error setting apqns.");       \
            }                                                                  \
        }                                                                      \
        rc = zpc_aes_key_set_size(aes_key, 128); /* cannot fail */             \
        rc = zpc_aes_key_generate(aes_key);                                    \
        if (rc != 0) {                                                         \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("NEW_MK check (AES): unexpected error when generating test key."); \
        }                                                                      \
        rc = zpc_aes_key_reencipher(aes_key, ZPC_AES_KEY_REENCIPHER_CURRENT_TO_NEW); \
        if (rc != 0) {                                                         \
            zpc_aes_key_free(&aes_key);                                        \
            GTEST_SKIP_("NEW_MK check (AES): new MK not set for this APQN/MKVP."); \
        }                                                                      \
        zpc_aes_key_free(&aes_key);                                            \
} while (0)

# define TESTLIB_ENV_EC_KEY_CHECK()                                                           \
do {                                                                                          \
        const char *apqns[257];                                                               \
        const char *mkvp;                                                                     \
        int type, rc;                                                                         \
        int curve = testlib_env_ec_key_curve();                                               \
                                                                                              \
        switch (curve) {                                                                      \
        case ZPC_EC_CURVE_P256:      /* fall-through */                                       \
        case ZPC_EC_CURVE_P384:      /* fall-through */                                       \
        case ZPC_EC_CURVE_P521:      /* fall-through */                                       \
        case ZPC_EC_CURVE_ED25519:   /* fall-through */                                       \
        case ZPC_EC_CURVE_ED448:     /* fall-through */                                       \
            break;                                                                            \
        case ZPC_EC_CURVE_INVALID:                                                            \
            GTEST_SKIP_("ZPC_TEST_EC_KEY_CURVE environment variable set to invalid value.");  \
            break;                                                                            \
        default:                                                                              \
            GTEST_SKIP_("ZPC_TEST_EC_KEY_CURVE environment variable not set.");               \
            break;                                                                            \
        }                                                                                     \
                                                                                              \
        type = testlib_env_ec_key_type();                                                     \
        if (type == -1)	                                                                      \
                GTEST_SKIP_("ZPC_TEST_EC_KEY_TYPE environment variable not set.");            \
                                                                                              \
        mkvp = testlib_env_ec_key_mkvp();                                                     \
        rc = testlib_env_ec_key_apqns(apqns);                                                 \
        if (rc == 0 && mkvp != NULL)                                                          \
            GTEST_SKIP_("Both ZPC_TEST_EC_KEY_MKVP and ZPC_TEST_EC_KEY_APQNS environment variables set.");  \
        if (rc != 0 && mkvp == NULL)                                                            \
            GTEST_SKIP_("ZPC_TEST_EC_KEY_MKVP and ZPC_TEST_EC_KEY_APQNS environment variables unset.");     \
} while (0)

# define TESTLIB_EC_HW_CAPS_CHECK()                                            \
do {                                                                           \
        int rc;                                                                \
        struct zpc_ecdsa_ctx *ctx;                                             \
                                                                               \
        rc = zpc_ecdsa_ctx_alloc(&ctx);                                        \
        switch (rc) {                                                          \
        case ZPC_ERROR_DEVPKEY:                                                \
            GTEST_SKIP_("HW_CAPS check: opening /dev/pkey failed.");           \
            break;                                                             \
        case ZPC_ERROR_HWCAPS:                                                 \
            GTEST_SKIP_("HW_CAPS check: no hw capabilities for ECDSA.");       \
            break;                                                             \
        case ZPC_ERROR_MALLOC:                                                 \
            GTEST_SKIP_("HW_CAPS check: cannot allocate ECDSA ctx object.");   \
            break;                                                             \
        default:                                                               \
            zpc_ecdsa_ctx_free(&ctx);                                          \
            break;                                                             \
        }                                                                      \
} while (0)

# define TESTLIB_EC_SW_CAPS_CHECK(type)                                        \
do {                                                                           \
        int rc;                                                                \
        struct zpc_ec_key *ec_key;                                             \
                                                                               \
        rc = zpc_ec_key_alloc(&ec_key);                                        \
        if (rc != 0)                                                           \
            GTEST_SKIP_("SW_CAPS check (EC): Cannot allocate key object.");    \
                                                                               \
        rc = zpc_ec_key_set_type(ec_key, type);                                \
        if (type == ZPC_EC_KEY_TYPE_CCA &&                                     \
            rc == ZPC_ERROR_CCA_HOST_LIB_NOT_AVAILABLE) {                      \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("SW_CAPS check (EC): CCA host lib not available or too old (min CCA 7.0)."); \
        }                                                                      \
        if (type == ZPC_EC_KEY_TYPE_EP11 &&                                    \
            rc == ZPC_ERROR_EP11_HOST_LIB_NOT_AVAILABLE) {                     \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("SW_CAPS check (EC): EP11 host lib not available or too old (min EP11 3.0)."); \
        }                                                                      \
        if (rc != 0) {                                                         \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("HW_CAPS check (EC): unexpected error when setting key type."); \
        }                                                                      \
        zpc_ec_key_free(&ec_key);                                              \
} while (0)

# define TESTLIB_EC_KERNEL_CAPS_CHECK(type,mkvp,apqns)                         \
do {                                                                           \
        int rc;                                                                \
        struct zpc_ec_key *ec_key;                                             \
                                                                               \
        rc = zpc_ec_key_alloc(&ec_key);                                        \
        if (rc != 0)                                                           \
            GTEST_SKIP_("KERNEL_CAPS check (EC): Cannot allocate key object."); \
                                                                               \
        rc = zpc_ec_key_set_mkvp(ec_key, NULL); /* cannot fail */              \
        rc = zpc_ec_key_set_curve(ec_key, ZPC_EC_CURVE_P256); /* cannot fail */ \
        rc = zpc_ec_key_set_type(ec_key, type);                                \
        if (rc != 0) {                                                         \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("KERNEL_CAPS check (EC): error setting key type.");    \
        }                                                                      \
        if (mkvp != NULL) {                                                    \
            rc = zpc_ec_key_set_mkvp(ec_key, mkvp);                            \
            if (rc != 0) {                                                     \
                zpc_ec_key_free(&ec_key);                                      \
                GTEST_SKIP_("KERNEL_CAPS check (EC): error setting mkvp.");    \
            }                                                                  \
        } else {                                                               \
            rc = zpc_ec_key_set_apqns(ec_key, apqns);                          \
            if (rc != 0) {                                                     \
                zpc_ec_key_free(&ec_key);                                      \
                GTEST_SKIP_("KERNEL_CAPS check (EC): error setting apqns.");   \
            }                                                                  \
        }                                                                      \
        rc = zpc_ec_key_generate(ec_key);                                      \
        if (rc == ZPC_ERROR_IOCTLBLOB2PROTK3) {                                \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("KERNEL_CAPS check (EC): ioctl PKEY_KBLOB2PROTK3 not supported by kernel."); \
        }                                                                      \
        if (rc != 0) {                                                         \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("KERNEL_CAPS check (EC): Unexpected error when generating test key."); \
        }                                                                      \
                                                                               \
        zpc_ec_key_free(&ec_key);                                              \
} while (0)

/**
 * The reencipher tests require current and new master keys set in the
 * related CCA or EP11 APQNs. For CCA, also "reencipher old to current" is
 * supported, but currently not tested to keep tests independent of key types.
 * Either mkvp or apqns must be specified (not NULL)
 */
# define TESTLIB_EC_NEW_MK_CHECK(type,mkvp,apqns)                              \
do {                                                                           \
        int rc;                                                                \
        struct zpc_ec_key *ec_key;                                             \
                                                                               \
        rc = zpc_ec_key_alloc(&ec_key);                                        \
        if (rc != 0)                                                           \
            GTEST_SKIP_("NEW_MK check (EC): Cannot allocate key object.");     \
                                                                               \
        rc = zpc_ec_key_set_type(ec_key, type);                                \
        if (type == ZPC_EC_KEY_TYPE_CCA &&                                     \
            rc == ZPC_ERROR_CCA_HOST_LIB_NOT_AVAILABLE) {                      \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("NEW_MK check (EC): CCA host lib not available or too old (min CCA 7.0).");  \
        }                                                                      \
        if (type == ZPC_EC_KEY_TYPE_EP11 &&                                    \
            rc == ZPC_ERROR_EP11_HOST_LIB_NOT_AVAILABLE) {                     \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("NEW_MK check (EC): EP11 host lib not available or too old (min EP11 3.0)."); \
        }                                                                      \
        if (mkvp != NULL) {                                                    \
            rc = zpc_ec_key_set_mkvp(ec_key, mkvp);                            \
            if (rc != 0) {                                                     \
                zpc_ec_key_free(&ec_key);                                      \
                GTEST_SKIP_("NEW_MK check (EC): error setting mkvp.");         \
            }                                                                  \
        } else {                                                               \
            rc = zpc_ec_key_set_apqns(ec_key, apqns);                          \
            if (rc != 0) {                                                     \
                zpc_ec_key_free(&ec_key);                                      \
                GTEST_SKIP_("NEW_MK check (EC): error setting apqns.");        \
            }                                                                  \
        }                                                                      \
        rc = zpc_ec_key_set_curve(ec_key, ZPC_EC_CURVE_P256);/* cannot fail */ \
        rc = zpc_ec_key_generate(ec_key);                                      \
        if (rc != 0) {                                                         \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("NEW_MK check (EC): error generating test key.");      \
        }                                                                      \
        rc = zpc_ec_key_reencipher(ec_key, ZPC_EC_KEY_REENCIPHER_CURRENT_TO_NEW); \
        if (rc != 0) {                                                         \
            zpc_ec_key_free(&ec_key);                                          \
            GTEST_SKIP_("NEW_MK check (EC): new MK not set for this APQN/MKVP."); \
        }                                                                      \
        zpc_ec_key_free(&ec_key);                                              \
} while (0)

typedef unsigned char u8;
typedef unsigned short u16;
typedef unsigned int u32;
typedef unsigned long long u64;

struct EC_TEST_VECTOR {
	int curve;
	unsigned char privkey[66];
	int privkey_len;
	unsigned char pubkey[132];
	int pubkey_len;
	unsigned char msg[64];
	int msg_len;
	unsigned char sig[144];
	int sig_len;
};
extern const struct EC_TEST_VECTOR ec_tv[5];

const char *testlib_env_aes_key_mkvp(void);
int testlib_env_aes_key_apqns(const char *[257]);
void testlib_env_aes_key_check(void);
int testlib_env_aes_key_size(void);
int testlib_env_aes_key_type(void);
unsigned int testlib_env_aes_key_flags(void);

const char *testlib_env_ec_key_mkvp(void);
int testlib_env_ec_key_apqns(const char *[257]);
void testlib_env_ec_key_check(void);
zpc_ec_curve_t testlib_env_ec_key_curve(void);
int testlib_env_ec_key_type(void);
unsigned int testlib_env_ec_key_flags(void);

unsigned char *testlib_hexstr2buf(const char *, size_t *);
unsigned char *testlib_hexstr2fixedbuf(const char *hexstr, size_t tolen);
char *testlib_buf2hexstr(const unsigned char *, size_t);

# ifdef __cplusplus
/* *INDENT-OFF* */
}
/* *INDENT-ON* */
# endif
#endif
