/*
 *	service detection for backend selection
 *
 *	Copyright (C) 2021 SUSE LCC
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *	Authors:
 *		Marius Tomaschewski
 *
 *	libnetcontrol contains source code which is based on wicked.
 *	Wicked is licensed under the GPL-2.0+, but permission has been
 *	granted by the authors of wicked to use the code derived from
 *	wicked under the LGPL-2.1+ in libnetcontrol.
 *	See the wicked project at <https://github.com/openSUSE/wicked>.
 *
 */
#if defined(HAVE_CONFIG_H)
#include <config.h>
#endif
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>
#include <limits.h>

#include "sutils.h"
#include "futils.h"
#include "logging.h"
#include "cmd_pipe.h"
#include "sysconfig.h"
#include "service.h"

int /* bool */
nc_service_init(nc_service_t *service, const char *name)
{
	if (!service)
		return -1;

	service->refcount = 1;
	return nc_string_dup(&service->name, name);
}

void
nc_service_destroy(nc_service_t *service)
{
	if (service) {
		nc_string_free(&service->name);
	}
}

nc_service_t *
nc_service_ref(nc_service_t *service)
{
	if (service) {
		assert(service->refcount != 0);
		if(service->refcount != 0)
			service->refcount++;
		else
			return NULL;
	}
	return service;
}

nc_service_t *
nc_service_new(const char *name)
{
	nc_service_t *

	service = calloc(1, sizeof(*service));
	if (service)  {
		if (!nc_service_init(service, name))
			return service;
		free(service);
	}
	return service;
}

void
nc_service_free(nc_service_t *service)
{
	if (service) {
		assert(service->refcount != 0);
		service->refcount--;
		if (service->refcount == 0) {
			nc_string_free(&service->name);
			free(service);
		}
	}
}

#if defined(__NETCONTROL_USE_SERVICE_ALIAS)
static inline const char *
nc_service_detect_systemctl_path(void)
{
	static const char *paths[] = {
		"/usr/bin/systemctl",
		"/bin/systemctl",
		NULL
	};
	const char **path;

	for (path = paths; *path; ++path) {
		if (nc_file_executable(*path))
			return *path;
	}
	return NULL;
}

static int
nc_service_detect_systemd_network__child_exec(nc_string_array_t *args)
{
	/* in child process: prepare and execute the command */
	if(!args || args->count < 2 || !args->data[0] || *args->data[0] != '/')
		return -1;

	/* we need the stdout + return code */
	if(!freopen("/dev/null", "r", stdin) ||
	   !freopen("/dev/null", "r", stderr))
		return -1;

	return execv(args->data[0], args->data);
}

static int
nc_service_detect_systemd_network__parse_id(FILE *input, char **service)
{
	char buf[BUFSIZ], *ptr;
	int  ret = -1;

	memset(buf, 0, sizeof(buf));
	while(fgets(buf, sizeof(buf)-1, input)) {
		buf[strcspn(buf, "\r\n")] = '\0';
		if (*service != NULL)
			continue;

		ptr = nc_string_strip_spaces(buf);
		if (nc_string_prefix_eq("Id=", ptr)) {
			ptr += nc_string_len("Id=");
			ret = nc_string_dup(service, ptr);
		}
	}
	return ret;
}

static int
nc_service_detect_systemd_network(const char *root, char **service)
{
	nc_string_array_t	args = NC_STRING_ARRAY_INIT;
	int			status = -1;
	int			code = -1;
	const char *		systemctl;
	const char *		info;
	nc_cmd_pipe_t		cmd;

	nc_string_free(service);
	if (root) {
		nc_error("Not (yet) supported when root directory is set");
		return -1;
	}
	if (!(systemctl = nc_service_detect_systemctl_path())) {
		nc_error("Unable to find systemctl utility path");
		return -1;
	}

	info = "systemctl -p Id show network.service";
	nc_string_array_append(&args, systemctl);
	nc_string_array_append(&args, "--no-pager");
	nc_string_array_append(&args, "-p");
	nc_string_array_append(&args, "Id");
	nc_string_array_append(&args, "show");
	nc_string_array_append(&args, "network.service");
	if (args.count != 6) {
		nc_string_array_destroy(&args);
		nc_error("Unable to construct '%s' command argument list", info);
		return -1;
	}

	memset(&cmd, 0, sizeof(cmd));
	cmd.exec = nc_service_detect_systemd_network__child_exec;
	cmd.args = &args;
	if (nc_cmd_pipe_exec(&cmd) == -1) {
		nc_error("Unable to execute '%s'", info);
		return -1;
	}

	if (cmd.file) {
		nc_service_detect_systemd_network__parse_id(cmd.file, service);
		fclose(cmd.file);
	}
	nc_string_array_destroy(&args);

	if (nc_cmd_pipe_wait(&cmd, &status, 0) == -1) {
		nc_error("waitpid failure: %m");
		return -1;
	}

	if( !nc_string_len(*service)) {
		nc_error("Unable to read output of '%s'", info);
		return -1;
	}

	if(WIFEXITED(status)) {
		code = WEXITSTATUS(status);
		nc_info("Command '%s' returned with exit code: %d",
			info, code);
	} else {
		code = -1;
		nc_error("Command '%s' terminated abormally: %d [%m]",
			info, status);
	}

	if (code == 0 && !nc_string_empty(*service))
		return 0;

	nc_string_free(service);
	return code > 0 ? code : -1;
}

#else

static int
nc_service_detect_systemv_network(const char *root, char **service)
{
	char path[PATH_MAX + 1] = {'\0'};
	nc_sysconfig_t *sc;
	const nc_var_t *v;

	snprintf(path, sizeof(path), "%s%s/%s", (root ? root : ""),
		"/etc/sysconfig/network", "config");

	if (!(sc = nc_sysconfig_read(path)))
		return -1;

	v = nc_sysconfig_get(sc, "NETWORKMANAGER");
	/*
	 * we map this <= sle11 variable to >= sle12
	 * .service: sysconfig LSB /etc/init.d/network
	 * script were network.service on the openSUSE
	 * versions between sle11 and sle12, splitted
	 * to network.service + NetworkManager.service.
	 */
	if (v && nc_string_eq(v->value, "yes")) {
		nc_sysconfig_free(sc);
		return nc_string_dup(service, "NetworkManager.service");
	} else {
		nc_sysconfig_free(sc);
		return nc_string_dup(service, "network.service");
	}
}

#endif

nc_service_t *
nc_service_detect(const char *root_dir)
{
	nc_service_t *service;

	if (!(service = nc_service_new(NULL)))
		return NULL;

#if defined(__NETCONTROL_USE_SERVICE_ALIAS)
	if (nc_service_detect_systemd_network(root_dir, &service->name) == 0)
		return service;
#else
	if (nc_service_detect_systemv_network(root_dir, &service->name) == 0)
		return service;
#endif

	nc_service_free(service);
	return NULL;
}
