/*
 * livepatch_bsc1203606
 *
 * Fix for CVE-2022-41218, bsc#1203606
 *
 *  Upstream commit:
 *  None yet
 *
 *  SLE12-SP4, SLE12-SP5, SLE15 and SLE15-SP1 commit:
 *  231362ab4c29ef0e20575dbd887a1f782bb9a972
 *
 *  SLE15-SP2 and -SP3 commit:
 *  260d9857f92ce083e1b970d0cb1dbf82af6d7d26
 *
 *  SLE15-SP4 commit:
 *  bdcd7abd6fce9d77cd1f5d7d02665858820fcbde
 *
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_DVB_CORE)

#if !IS_MODULE(CONFIG_DVB_CORE)
#error "Live patch supports only CONFIG_DVB_CORE=m"
#endif

/* klp-ccp: from drivers/media/dvb-core/dmxdev.c */
#define pr_fmt(fmt) "dmxdev: " fmt

#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/ioctl.h>
#include <linux/wait.h>
#include <linux/uaccess.h>
#include <media/dmxdev.h>

/* klp-ccp: from include/media/dvbdev.h */
static void (*klpe_dvb_unregister_device)(struct dvb_device *dvbdev);

/* klp-ccp: from include/media/dvb_ringbuffer.h */
static void (*klpe_dvb_ringbuffer_init)(struct dvb_ringbuffer *rbuf, void *data,
				size_t len);

/* klp-ccp: from include/media/dmxdev.h */
void klpp_dvb_dmxdev_release(struct dmxdev *dmxdev);

/* klp-ccp: from drivers/media/dvb-core/dmxdev.c */
#include <media/dvb_vb2.h>

static inline void dvb_dmxdev_filter_state_set(struct dmxdev_filter
					       *dmxdevfilter, int state)
{
	spin_lock_irq(&dmxdevfilter->dev->lock);
	dmxdevfilter->state = state;
	spin_unlock_irq(&dmxdevfilter->dev->lock);
}

static void (*klpe_dvb_dmxdev_filter_timeout)(struct timer_list *t);

int klpp_dvb_demux_open(struct inode *inode, struct file *file)
{
	struct dvb_device *dvbdev = file->private_data;
	struct dmxdev *dmxdev = dvbdev->priv;
	int i;
	struct dmxdev_filter *dmxdevfilter;

	if (!dmxdev->filter)
		return -EINVAL;

	if (mutex_lock_interruptible(&dmxdev->mutex))
		return -ERESTARTSYS;

	/*
	 * Fix CVE-2022-41218
	 *  +4 lines
	 */
	if (dmxdev->exit) {
		mutex_unlock(&dmxdev->mutex);
		return -ENODEV;
	}

	for (i = 0; i < dmxdev->filternum; i++)
		if (dmxdev->filter[i].state == DMXDEV_STATE_FREE)
			break;

	if (i == dmxdev->filternum) {
		mutex_unlock(&dmxdev->mutex);
		return -EMFILE;
	}

	dmxdevfilter = &dmxdev->filter[i];
	mutex_init(&dmxdevfilter->mutex);
	file->private_data = dmxdevfilter;

#ifdef CONFIG_DVB_MMAP
#error "klp-ccp: non-taken branch"
#else
	dmxdev->may_do_mmap = 0;
#endif
	(*klpe_dvb_ringbuffer_init)(&dmxdevfilter->buffer, NULL, 8192);
	dvb_vb2_init(&dmxdevfilter->vb2_ctx, "demux_filter",
		     file->f_flags & O_NONBLOCK);
	dmxdevfilter->type = DMXDEV_TYPE_NONE;
	dvb_dmxdev_filter_state_set(dmxdevfilter, DMXDEV_STATE_ALLOCATED);
	timer_setup(&dmxdevfilter->timer, (*klpe_dvb_dmxdev_filter_timeout), 0);

	dvbdev->users++;

	mutex_unlock(&dmxdev->mutex);
	return 0;
}

void klpp_dvb_dmxdev_release(struct dmxdev *dmxdev)
{
	/*
	 * Fix CVE-2022-41218
	 *  -1 line, +3 lines
	 */
	mutex_lock(&dmxdev->mutex);
	dmxdev->exit = 1;
	mutex_unlock(&dmxdev->mutex);

	if (dmxdev->dvbdev->users > 1) {
		wait_event(dmxdev->dvbdev->wait_queue,
				dmxdev->dvbdev->users == 1);
	}
	if (dmxdev->dvr_dvbdev->users > 1) {
		wait_event(dmxdev->dvr_dvbdev->wait_queue,
				dmxdev->dvr_dvbdev->users == 1);
	}

	(*klpe_dvb_unregister_device)(dmxdev->dvbdev);
	(*klpe_dvb_unregister_device)(dmxdev->dvr_dvbdev);

	vfree(dmxdev->filter);
	dmxdev->filter = NULL;
	dmxdev->demux->close(dmxdev->demux);
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1203606.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "dvb_core"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "dvb_dmxdev_filter_timeout", (void *)&klpe_dvb_dmxdev_filter_timeout,
	  "dvb_core" },
	{ "dvb_ringbuffer_init", (void *)&klpe_dvb_ringbuffer_init,
	  "dvb_core" },
	{ "dvb_unregister_device", (void *)&klpe_dvb_unregister_device,
	  "dvb_core" },
};

static int livepatch_bsc1203606_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	ret = klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1203606_module_nb = {
	.notifier_call = livepatch_bsc1203606_module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1203606_init(void)
{
	int ret;
	struct module *mod;

	ret = klp_kallsyms_relocs_init();
	if (ret)
		return ret;

	ret = register_module_notifier(&livepatch_bsc1203606_module_nb);
	if (ret)
		return ret;

	rcu_read_lock_sched();
	mod = (*klpe_find_module)(LIVEPATCHED_MODULE);
	if (!try_module_get(mod))
		mod = NULL;
	rcu_read_unlock_sched();

	if (mod) {
		ret = klp_resolve_kallsyms_relocs(klp_funcs,
						  ARRAY_SIZE(klp_funcs));
	}

	if (ret)
		unregister_module_notifier(&livepatch_bsc1203606_module_nb);

	module_put(mod);
	return ret;
}

void livepatch_bsc1203606_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1203606_module_nb);
}

#endif /* IS_ENABLED(CONFIG_DVB_CORE) */
