/*
 * livepatch_bsc1195949
 *
 * Fix for CVE-2022-0487, bsc#1195949
 *
 *  Upstream commit:
 *  42933c8aa14b ("memstick: rtsx_usb_ms: fix UAF")
 *
 *  SLE12-SP3 commit:
 *  9dca5583744bbaef9cb59f535fa2123442d70f87
 *
 *  SLE12-SP4, SLE12-SP5, SLE15 and SLE15-SP1 commit:
 *  9692e2521a5cd2e4186d71355ec936d4c787fd42
 *
 *  SLE15-SP2 and -SP3 commit:
 *  e5d4f2bd865f9456422beaa212b8875c1afa337b
 *
 *
 *  Copyright (c) 2022 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_MEMSTICK_REALTEK_USB)

#if !IS_MODULE(CONFIG_MEMSTICK_REALTEK_USB)
#error "Live patch supports only CONFIG_MEMSTICK_REALTEK_USB=m"
#endif

/* klp-ccp: from drivers/memstick/host/rtsx_usb_ms.c */
#include <linux/module.h>
#include <linux/highmem.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>
#include <linux/memstick.h>

/* klp-ccp: from include/linux/memstick.h */
static struct memstick_host *(*klpe_memstick_alloc_host)(unsigned int extra,
					  struct device *dev);

static int (*klpe_memstick_add_host)(struct memstick_host *host);
static void (*klpe_memstick_remove_host)(struct memstick_host *host);
static void (*klpe_memstick_free_host)(struct memstick_host *host);

static int (*klpe_memstick_next_req)(struct memstick_host *host,
		      struct memstick_request **mrq);

/* klp-ccp: from drivers/memstick/host/rtsx_usb_ms.c */
#include <linux/rtsx_usb.h>
#include <linux/pm_runtime.h>
#include <linux/mutex.h>
#include <linux/sched.h>
#include <linux/completion.h>

struct rtsx_usb_ms {
	struct platform_device	*pdev;
	struct rtsx_ucr	*ucr;
	struct memstick_host	*msh;
	struct memstick_request	*req;

	struct mutex		host_mutex;
	struct work_struct	handle_req;
	struct delayed_work	poll_card;

	u8			ssc_depth;
	unsigned int		clock;
	int			power_mode;
	unsigned char           ifmode;
	bool			eject;
	bool			system_suspending;
};

static inline struct device *ms_dev(struct rtsx_usb_ms *host)
{
	return &(host->pdev->dev);
}

static void (*klpe_rtsx_usb_ms_handle_req)(struct work_struct *work);

static void (*klpe_rtsx_usb_ms_request)(struct memstick_host *msh);

static int (*klpe_rtsx_usb_ms_set_param)(struct memstick_host *msh,
		enum memstick_param param, int value);

static void (*klpe_rtsx_usb_ms_poll_card)(struct work_struct *work);

int klpp_rtsx_usb_ms_drv_probe(struct platform_device *pdev)
{
	struct memstick_host *msh;
	struct rtsx_usb_ms *host;
	struct rtsx_ucr *ucr;
	int err;

	ucr = usb_get_intfdata(to_usb_interface(pdev->dev.parent));
	if (!ucr)
		return -ENXIO;

	dev_dbg(&(pdev->dev),
			"Realtek USB Memstick controller found\n");

	msh = (*klpe_memstick_alloc_host)(sizeof(*host), &pdev->dev);
	if (!msh)
		return -ENOMEM;

	host = memstick_priv(msh);
	host->ucr = ucr;
	host->msh = msh;
	host->pdev = pdev;
	host->power_mode = MEMSTICK_POWER_OFF;
	platform_set_drvdata(pdev, host);

	mutex_init(&host->host_mutex);
	INIT_WORK(&host->handle_req, (*klpe_rtsx_usb_ms_handle_req));

	INIT_DELAYED_WORK(&host->poll_card, (*klpe_rtsx_usb_ms_poll_card));

	msh->request = (*klpe_rtsx_usb_ms_request);
	msh->set_param = (*klpe_rtsx_usb_ms_set_param);
	msh->caps = MEMSTICK_CAP_PAR4;

	pm_runtime_get_noresume(ms_dev(host));
	pm_runtime_set_active(ms_dev(host));
	pm_runtime_enable(ms_dev(host));

	err = (*klpe_memstick_add_host)(msh);
	if (err)
		goto err_out;

	pm_runtime_put(ms_dev(host));

	return 0;
err_out:
	pm_runtime_disable(ms_dev(host));
	pm_runtime_put_noidle(ms_dev(host));
	(*klpe_memstick_free_host)(msh);
	return err;
}

int klpp_rtsx_usb_ms_drv_remove(struct platform_device *pdev)
{
	struct rtsx_usb_ms *host = platform_get_drvdata(pdev);
	struct memstick_host *msh = host->msh;
	int err;

	host->eject = true;
	cancel_work_sync(&host->handle_req);

	mutex_lock(&host->host_mutex);
	if (host->req) {
		dev_dbg(ms_dev(host),
			"%s: Controller removed during transfer\n",
			dev_name(&msh->dev));
		host->req->error = -ENOMEDIUM;
		do {
			err = (*klpe_memstick_next_req)(msh, &host->req);
			if (!err)
				host->req->error = -ENOMEDIUM;
		} while (!err);
	}
	mutex_unlock(&host->host_mutex);

	/* Balance possible unbalanced usage count
	 * e.g. unconditional module removal
	 */
	if (pm_runtime_active(ms_dev(host)))
		pm_runtime_put(ms_dev(host));

	pm_runtime_disable(ms_dev(host));

	(*klpe_memstick_remove_host)(msh);
	dev_dbg(ms_dev(host),
		": Realtek USB Memstick controller has been removed\n");
	(*klpe_memstick_free_host)(msh);
	platform_set_drvdata(pdev, NULL);

	return 0;
}



#include <linux/kernel.h>
#include <linux/module.h>
#include "livepatch_bsc1195949.h"
#include "../kallsyms_relocs.h"

#define LIVEPATCHED_MODULE "rtsx_usb_ms"

static struct klp_kallsyms_reloc klp_funcs[] = {
	{ "memstick_add_host", (void *)&klpe_memstick_add_host, "memstick" },
	{ "memstick_alloc_host", (void *)&klpe_memstick_alloc_host,
	  "memstick" },
	{ "memstick_free_host", (void *)&klpe_memstick_free_host, "memstick" },
	{ "memstick_next_req", (void *)&klpe_memstick_next_req, "memstick" },
	{ "memstick_remove_host", (void *)&klpe_memstick_remove_host,
	  "memstick" },
	{ "rtsx_usb_ms_handle_req", (void *)&klpe_rtsx_usb_ms_handle_req,
	  "rtsx_usb_ms" },
	{ "rtsx_usb_ms_poll_card", (void *)&klpe_rtsx_usb_ms_poll_card,
	  "rtsx_usb_ms" },
	{ "rtsx_usb_ms_request", (void *)&klpe_rtsx_usb_ms_request,
	  "rtsx_usb_ms" },
	{ "rtsx_usb_ms_set_param", (void *)&klpe_rtsx_usb_ms_set_param,
	  "rtsx_usb_ms" },
};

static int livepatch_bsc1195949_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, LIVEPATCHED_MODULE))
		return 0;

	ret = __klp_resolve_kallsyms_relocs(klp_funcs, ARRAY_SIZE(klp_funcs));
	WARN(ret, "livepatch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block livepatch_bsc1195949_module_nb = {
	.notifier_call = livepatch_bsc1195949_module_notify,
	.priority = INT_MIN+1,
};

int livepatch_bsc1195949_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(LIVEPATCHED_MODULE)) {
		ret = __klp_resolve_kallsyms_relocs(klp_funcs,
						    ARRAY_SIZE(klp_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&livepatch_bsc1195949_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void livepatch_bsc1195949_cleanup(void)
{
	unregister_module_notifier(&livepatch_bsc1195949_module_nb);
}

#endif /* IS_ENABLED(CONFIG_MEMSTICK_REALTEK_USB) */
