defmodule WandaWeb.Schemas.ApiSpecTest do
  use ExUnit.Case, async: false

  alias WandaWeb.Schemas.ApiSpec

  defmodule TestController do
    use Phoenix.Controller
    use OpenApiSpex.ControllerSpecs

    operation :show,
      summary: "Dummy show"

    def show(_, _), do: nil
  end

  defmodule TestRouter do
    use Phoenix.Router

    scope "/api" do
      get "/not_versioned", TestController, :show
      get "/v1/route", TestController, :show
      get "/v2/route", TestController, :show
    end

    def available_api_versions, do: ["v2", "v1"]
  end

  defmodule V1 do
    use ApiSpec,
      api_version: "v1"
  end

  defmodule V2 do
    use ApiSpec,
      api_version: "v2"
  end

  defmodule Unversioned do
    use ApiSpec,
      api_version: "unversioned"
  end

  defmodule All do
    use ApiSpec,
      api_version: "all"
  end

  describe "ApiSpec" do
    test "should render only the v1 version routes" do
      expected_version = get_app_version() <> "-v1"

      assert %OpenApiSpex.OpenApi{
               info: %{
                 version: ^expected_version
               },
               paths: %{"/api/v1/route" => _}
             } = V1.spec(TestRouter)
    end

    test "should render only the v2 version routes" do
      expected_version = get_app_version() <> "-v2"

      assert %OpenApiSpex.OpenApi{
               info: %{
                 version: ^expected_version
               },
               paths: %{"/api/v2/route" => _}
             } = V2.spec(TestRouter)
    end

    test "should render unversioned routes" do
      expected_version = get_app_version() <> "-unversioned"

      assert %OpenApiSpex.OpenApi{
               info: %{
                 version: ^expected_version
               },
               paths: %{"/api/not_versioned" => _}
             } = Unversioned.spec(TestRouter)
    end

    test "should render all the specification with all routes" do
      expected_version = get_app_version()

      assert %OpenApiSpex.OpenApi{
               info: %{
                 version: ^expected_version
               },
               paths: %{"/api/not_versioned" => _, "/api/v1/route" => _, "/api/v2/route" => _}
             } = All.spec(TestRouter)
    end

    test "should use oas_server_url if configured" do
      on_exit(fn ->
        Application.put_env(:wanda, :oas_server_url, nil)
      end)

      url = "https://my-wanda.io"
      Application.put_env(:wanda, :oas_server_url, url)

      assert %OpenApiSpex.OpenApi{
               servers: [
                 %{
                   url: ^url
                 }
               ]
             } = All.spec(TestRouter)
    end
  end

  defp get_app_version, do: to_string(Application.spec(:wanda, :vsn))
end
