package models

import (
	"testing"
	"time"

	"github.com/grafana/grafana-plugin-sdk-go/data"
	"github.com/prometheus/common/model"
	"github.com/stretchr/testify/require"

	"github.com/grafana/grafana/pkg/util"
)

func TestValidate(t *testing.T) {
	validNotificationSettings := NotificationSettingsGen(NSMuts.WithGroupBy(model.AlertNameLabel, FolderTitleLabel))

	testCases := []struct {
		name                 string
		notificationSettings NotificationSettings
		expErrorContains     string
	}{
		{
			name:                 "valid notification settings",
			notificationSettings: validNotificationSettings(),
		},
		{
			name:                 "missing receiver is invalid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithReceiver("")),
			expErrorContains:     "receiver",
		},
		{
			name:                 "group by empty is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupBy()),
		},
		{
			name:                 "group by ... is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupBy("...")),
		},
		{
			name:                 "group by with alert name and folder name labels is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupBy(model.AlertNameLabel, FolderTitleLabel)),
		},
		{
			name:                 "group by missing alert name label is invalid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupBy(FolderTitleLabel)),
			expErrorContains:     model.AlertNameLabel,
		},
		{
			name:                 "group by missing folder name label is invalid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupBy(model.AlertNameLabel)),
			expErrorContains:     FolderTitleLabel,
		},
		{
			name:                 "group wait empty is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupWait(nil)),
		},
		{
			name:                 "group wait positive is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupWait(util.Pointer(1*time.Second))),
		},
		{
			name:                 "group wait negative is invalid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupWait(util.Pointer(-1*time.Second))),
			expErrorContains:     "group wait",
		},
		{
			name:                 "group interval empty is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupInterval(nil)),
		},
		{
			name:                 "group interval positive is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupInterval(util.Pointer(1*time.Second))),
		},
		{
			name:                 "group interval negative is invalid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithGroupInterval(util.Pointer(-1*time.Second))),
			expErrorContains:     "group interval",
		},
		{
			name:                 "repeat interval empty is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithRepeatInterval(nil)),
		},
		{
			name:                 "repeat interval positive is valid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithRepeatInterval(util.Pointer(1*time.Second))),
		},
		{
			name:                 "repeat interval negative is invalid",
			notificationSettings: CopyNotificationSettings(validNotificationSettings(), NSMuts.WithRepeatInterval(util.Pointer(-1*time.Second))),
			expErrorContains:     "repeat interval",
		},
	}

	for _, tt := range testCases {
		t.Run(tt.name, func(t *testing.T) {
			err := tt.notificationSettings.Validate()
			if tt.expErrorContains != "" {
				require.Error(t, err)
				require.ErrorContains(t, err, tt.expErrorContains)
			} else {
				require.NoError(t, err)
			}
		})
	}
}

func TestNotificationSettingsLabels(t *testing.T) {
	testCases := []struct {
		name                 string
		notificationSettings NotificationSettings
		labels               data.Labels
	}{
		{
			name:                 "default notification settings",
			notificationSettings: NewDefaultNotificationSettings("receiver name"),
			labels: data.Labels{
				AutogeneratedRouteLabel:             "true",
				AutogeneratedRouteReceiverNameLabel: "receiver name",
			},
		},
		{
			name: "custom notification settings",
			notificationSettings: NotificationSettings{
				Receiver:          "receiver name",
				GroupBy:           []string{"label1", "label2"},
				GroupWait:         util.Pointer(model.Duration(1 * time.Minute)),
				GroupInterval:     util.Pointer(model.Duration(2 * time.Minute)),
				RepeatInterval:    util.Pointer(model.Duration(3 * time.Minute)),
				MuteTimeIntervals: []string{"maintenance1", "maintenance2"},
			},
			labels: data.Labels{
				AutogeneratedRouteLabel:             "true",
				AutogeneratedRouteReceiverNameLabel: "receiver name",
				AutogeneratedRouteSettingsHashLabel: "f0e23250cefc4a31",
			},
		},
	}

	for _, tt := range testCases {
		t.Run(tt.name, func(t *testing.T) {
			labels := tt.notificationSettings.ToLabels()
			require.Equal(t, tt.labels, labels)
		})
	}
}
