/* Copyright (C) 2003, 2004, 2005 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain any existing copyright
   notice, and this entire permission notice in its entirety,
   including the disclaimer of warranties.

2. Redistributions in binary form must reproduce all prior and current
   copyright notices, this list of conditions, and the following
   disclaimer in the documentation and/or other materials provided
   with the distribution.

3. The name of any author may not be used to endorse or promote
   products derived from this software without their specific prior
   written permission.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <assert.h>

#include "logindefs.h"

struct item {
  char *name;         /* name of the option.  */
  char *value;        /* value of the option.  */
  struct item *next;  /* pointer to next option.  */
  char *path;         /* name of config file for this option.  */
};

static struct item *list = NULL;

void
free_getlogindefs_data (void)
{
  struct item *ptr;

  ptr = list;
  while (ptr != NULL)
    {
      struct item *tmp;
      tmp = ptr->next;
      free (ptr->path);
      free (ptr->name);
      free (ptr->value);
      free (ptr);
      ptr = tmp;
    }

  list = NULL;
}

/* Add a new entry to the list.  */
static void
store (const char *name, const char *value, const char *path)
{
  struct item *new = malloc (sizeof (struct item));

  if (new == NULL)
    abort ();

  if (name == NULL)
    abort ();

  new->name = strdup (name);
  new->value = strdup (value?:"");
  assert (path);
  new->path = strdup (path);
  new->next = list;
  list = new;
}

/* search a special entry in the list and return the value.  */
static const char *
search (const char *name)
{
  struct item *ptr;

  ptr = list;
  while (ptr != NULL)
    {
      if (strcasecmp (name, ptr->name) == 0)
	return ptr->value;
      ptr = ptr->next;
    }

  return NULL;
}

/* search a special entry in the list and return the filename
   from where we got it.  */
static const char *
search_config (const char *name)
{
  struct item *ptr;

  ptr = list;
  while (ptr != NULL)
    {
      if (strcasecmp (name, ptr->name) == 0)
	return ptr->path;
      ptr = ptr->next;
    }

  /* Should not happen, we found the entry already.  */
  assert (NULL);
  return NULL;
}

/* Load the config file  */
static void
load_defaults_internal (const char *filename)
{
  FILE *fp;
  char *buf = NULL;
  size_t buflen = 0;

  fp = fopen (filename, "r");
  if (NULL == fp)
    return;

  while (!feof (fp))
    {
      char *tmp, *cp;
#if defined(HAVE_GETLINE)
      ssize_t n = getline (&buf, &buflen, fp);
#elif defined (HAVE_GETDELIM)
      ssize_t n = getdelim (&buf, &buflen, '\n', fp);
#else
      ssize_t n;

      if (buf == NULL)
        {
          buflen = 8096;
          buf = malloc (buflen);
        }
      buf[0] = '\0';
      fgets (buf, buflen - 1, fp);
      if (buf != NULL)
        n = strlen (buf);
      else
        n = 0;
#endif /* HAVE_GETLINE / HAVE_GETDELIM */
      cp = buf;

      if (n < 1)
        break;

      tmp = strchr (cp, '#');  /* remove comments */
      if (tmp)
        *tmp = '\0';
      while (isspace ((int)*cp))    /* remove spaces and tabs */
        ++cp;
      if (*cp == '\0')        /* ignore empty lines */
        continue;

      if (cp[strlen (cp) - 1] == '\n')
        cp[strlen (cp) - 1] = '\0';

      tmp = strsep (&cp, " \t=");
      if (cp != NULL)
	{
	  while (isspace ((int)*cp) || *cp == '=' || *cp == '"')
	    ++cp;
	  if (strlen (cp) > 1 && cp[strlen (cp) - 1] == '"')
	    cp [strlen (cp) - 1] = '\0';
	}

      store (tmp, cp, filename);
    }
  fclose (fp);

  if (buf)
    free (buf);
}

static void
load_defaults (void)
{
  load_defaults_internal ("/etc/default/login");
  load_defaults_internal ("/etc/login.defs");
  load_defaults_internal ("/etc/sysconfig/language");
}

int
getlogindefs_bool (const char *name, int dflt)
{
  const char *val;

  if (list == NULL)
    load_defaults ();

  val = search (name);

  if (val == NULL)
    return dflt;

  return (strcasecmp (val, "yes") == 0);
}

long
getlogindefs_num (const char *name, long dflt)
{
  const char *val;
  char *cp;
  long retval;

  if (list == NULL)
    load_defaults ();

  val = search (name);

  if (val == NULL)
    return dflt;

  retval = strtol (val, &cp, 0);
  if (*cp != '\0' ||
      ((retval == LONG_MAX || retval == LONG_MIN) && errno == ERANGE))
    {
      fprintf (stderr,
	       "%s: %s contains invalid numerical value: %s!\n",
	       search_config (name), name, val);
      retval = dflt;
    }
  return retval;
}

unsigned long
getlogindefs_unum (const char *name, unsigned long dflt)
{
  const char *val;
  char *cp;
  unsigned long retval;

  if (list == NULL)
    load_defaults ();

  val = search (name);

  if (val == NULL)
    return dflt;

  retval = strtoul (val, &cp, 0);
  if (*cp != '\0' || (retval == ULONG_MAX && errno == ERANGE))
    {
      fprintf (stderr,
	       "%s: %s contains invalid numerical value: %s!\n",
	       search_config (name), name, val);
      retval = dflt;
    }
  return retval;
}

const char *
getlogindefs_str (const char *name, const char *dflt)
{
  const char *retval;

  if (list == NULL)
    load_defaults ();

  retval = search (name);

  return retval ?: dflt;
}

#if defined(TEST)

int
main ()
{
  printf ("CRYPT=%s\n", getlogindefs_str ("cRypt", "no"));
  printf ("LOG_UNKFAIL_ENAB=%s\n", getlogindefs_str ("log_unkfail_enab",""));
  printf ("DOESNOTEXIST=%s\n", getlogindefs_str ("DOESNOTEXIST","yes"));
  return 0;
}

#endif
