package SUSE::Parser::ZmdConfig;
use strict;
use XML::Parser;
use SUSE::SRPrivate;

# The return value of parse is a hash reference on success
# and has the following format: 
#
# $VAR1 = {
#          'nVidia-Driver-SLE10-SP2' => {
#                                       'URL' => 'http://srv64.suse.de//repo/RPMMD/nVidia-Driver-SLE10-SP2',
#                                       'NAME' => 'nVidia-Driver-SLE10-SP2',
#                                       'TYPE' => 'zypp'
#                                     },
#          'local_nu_server' => {
#                               'URL' => 'http://srv64.suse.de/',
#                               'NAME' => 'local_nu_server',
#                               'CATALOGS' => {
#                                               'SLED10-SP2-Updates' => {
#                                                                       'URL' => 'http://srv64.suse.de//repo/$RCE/SLED10-SP2-Updates/sled-10-i586',
#                                                                       'NAME' => 'SLED10-SP2-Updates'
#                                                                     }
#                                             },
#                               'TYPE' => 'nu'
#                             },
#          'ATI-Driver-SLE10-SP2' => {
#                                    'URL' => 'http://srv64.suse.de//repo/RPMMD/ATI-Driver-SLE10-SP2',
#                                    'NAME' => 'ATI-Driver-SLE10-SP2',
#                                    'TYPE' => 'zypp'
#                                  }
#         };


# constructor
sub new
{
    my $pkgname = shift;
    my %opt   = @_;
    my $self  = {};

    $self->{CURRENT}   = undef;
    $self->{FINAL}     = undef;
    $self->{ELEMENT}   = undef;
    $self->{CATURL}    = undef;
    $self->{CATNAME}   = undef;
    $self->{CTX}       = undef;
    
    if(exists $opt{ctx} && defined $opt{ctx} && $opt{ctx})
    {
        $self->{CTX} = $opt{ctx};
    }
    bless($self);
    return $self;
}

# parses a xml resource
sub parse()
{
    my $self     = shift;
    my $xml      = shift;

    if (!defined $xml)
    {
        SUSE::SRPrivate::printLog($self->{CTX}, "error", "Invalid content");
        return 1;
    }

    my $parser = XML::Parser->new( Handlers =>
                                   {
                                    Start=> sub { handle_start_tag($self, @_) },
                                    Char => sub { handle_char_tag($self, @_) },
                                    End=> sub { handle_end_tag($self, @_) },
                                   });

    eval
    {
        $parser->parse( $xml );
    };
    if ($@) {
        # ignore the errors, but print them
        chomp($@);
        SUSE::SRPrivate::printLog($self->{CTX}, "warn", "SUSE::Parser::RepoList Invalid XML \n$xml\n: $@");
    }
    return $self->{FINAL};
}

# handles XML reader start tag events
sub handle_start_tag()
{
    my $self = shift;
    my( $expat, $element, %attrs ) = @_;

    if(lc($element) eq "service")
    {
        $self->{ELEMENT} = "service";
        
        if(exists $attrs{id} && defined $attrs{id})
        {
            $self->{CURRENT}->{NAME} = $attrs{id};
            $self->{CURRENT}->{ALIAS} = $attrs{id};
        }
        if(exists $attrs{type} && defined $attrs{type})
        {
            $self->{CURRENT}->{TYPE} = $attrs{type};
        }
        $self->{CURRENT}->{URL} = "";

    }    
    elsif(lc($element) eq "param")
    {
        if(exists $attrs{id} && defined $attrs{id})
        {
            $self->{ELEMENT} = $attrs{id};
        }
        elsif(exists $attrs{name} && defined $attrs{name})
        {
            if(exists $attrs{url} && defined $attrs{url})
            {
                $self->{CATURL} = $attrs{url};
            }
            $self->{ELEMENT} = $attrs{name};
        }
    }
}

sub handle_char_tag
{
    my $self = shift;
    my( $expat, $string) = @_;

    chomp($string);
    return if($string =~ /^\s*$/);
    $string =~ s/^\s*//;
    $string =~ s/\s*$//;
    
    return if(! defined $self->{ELEMENT});

    if($self->{ELEMENT} eq "url")
    {
        $self->{CURRENT}->{URL} .= $string;
    }
    elsif($self->{ELEMENT} eq "catalog")
    {
        $self->{CATNAME} .= $string;
    }
}

sub handle_end_tag
{
    my( $self, $expat, $element ) = @_;

    if(lc($element) eq "service")
    {
        # move CURRENT to FINAL

        if($self->{CURRENT}->{TYPE} eq "zypp" || $self->{CURRENT}->{TYPE} eq "yum")
        {
            my $name = "";
            if(exists $self->{CURRENT}->{CATALOGS} && ref($self->{CURRENT}->{CATALOGS}) eq "HASH")
            {
                $name = (keys %{$self->{CURRENT}->{CATALOGS}})[0];
            }
            else
            {
                $name = $self->{CURRENT}->{NAME};
            }
            $self->{FINAL}->{$name}->{NAME}  = $name;
            $self->{FINAL}->{$name}->{ALIAS} = $name;
            $self->{FINAL}->{$name}->{TYPE}  = $self->{CURRENT}->{TYPE};
            $self->{FINAL}->{$name}->{URL}   = $self->{CURRENT}->{URL};
        }
        elsif($self->{CURRENT}->{TYPE} eq "nu")
        {
            my $alias = $self->{CURRENT}->{ALIAS};
            $self->{FINAL}->{$alias}->{NAME}  = $self->{CURRENT}->{NAME};
            $self->{FINAL}->{$alias}->{ALIAS} = $self->{CURRENT}->{ALIAS};
            $self->{FINAL}->{$alias}->{TYPE}  = $self->{CURRENT}->{TYPE};
            $self->{FINAL}->{$alias}->{URL}   = $self->{CURRENT}->{URL};
            
            if(exists $self->{CURRENT}->{CATALOGS} && ref($self->{CURRENT}->{CATALOGS}) eq "HASH")
            {
                foreach my $catname (keys %{$self->{CURRENT}->{CATALOGS}})
                {
                    $self->{FINAL}->{$alias}->{CATALOGS}->{$catname}->{URL}   = $self->{CURRENT}->{CATALOGS}->{$catname}->{URL};
                    $self->{FINAL}->{$alias}->{CATALOGS}->{$catname}->{NAME}  = $self->{CURRENT}->{CATALOGS}->{$catname}->{NAME};
                    $self->{FINAL}->{$alias}->{CATALOGS}->{$catname}->{ALIAS} = $self->{CURRENT}->{CATALOGS}->{$catname}->{ALIAS};
                }
            }
        }
        
        $self->{ELEMENT}   = undef;
        $self->{CURRENT}   = undef;
        $self->{SERVICEID} = undef;
    }
    elsif(lc($element) eq "param")
    {
        if($self->{ELEMENT} eq "url")
        {
            $self->{ELEMENT} = undef;
        }
        elsif($self->{ELEMENT} eq "catalog")
        {
            my $catalias = "";
            if($self->{CURRENT}->{TYPE} eq "nu")
            {
                $catalias = $self->{CURRENT}->{ALIAS}.":".$self->{CATNAME};
            }
            else
            {
                $catalias = $self->{CATNAME};
            }            
            
            $self->{CURRENT}->{CATALOGS}->{$catalias}->{NAME}  = $self->{CATNAME};
            $self->{CURRENT}->{CATALOGS}->{$catalias}->{ALIAS} = $catalias;
            $self->{CURRENT}->{CATALOGS}->{$catalias}->{URL}   = $self->{CATURL};


            $self->{ELEMENT} = undef;
            $self->{CATNAME} = "";
            $self->{CATURL}  = "";
        }
    }
}

1;
