#
# Copyright (c) 2006 SUSE LINUX Products GmbH, Nuernberg, Germany.
#


package SUSE::SuseRegister;

=head1 NAME

SUSE::SuseRegister - Functions for registration at the Novell registration server

To test for an error you have to look at the context error code $ctx->{errorcode}.
If the errorcode is not equal 0 an error happens. You can get a message via $ctx->{errormsg}.

=cut


use strict;
use XML::Parser;
use XML::Writer;
use Data::Dumper;
use Getopt::Long;
use Encode;
use Sys::Syslog;
use URI;
#use URI::QueryParam;
#use Time::HiRes qw(gettimeofday tv_interval);
use SUSE::SRPrivate;

=head2 Public Functions

The set of public functions

 use SUSE::SuseRegister;

=over 2

=item *
B<$ctx = init_ctx($data)>

Initialize the context. This function must be called first, before using the 
rest of this functions.

I<$data> is a hash reference and supports the following keys:

* nooptional (bool)

* nohwdata (bool)

* forcereg (bool)

* xmlout (bool)

* logfile (filename)

* locale (string)

* noproxy (bool)

* yastcall (bool)

* norefresh (bool)

* debug (bool)

* args (hash reference with 'key' => 'value')

* extraCurlOption (array reference) *obsolete*


For registration some arguments must be provided to the registration server.
These arguments are stored in side the context $ctx. 
Here is the "$ctx->{args}" definition:

 $ctx->{args} => {
                   '<key>' => {
                                'value' => '<value>',
                                'flag'  => '<i|a|m>',
                                'kind'  => '<m|o>'
                              },
                    ...
                 };

 flag: i == initial   (still send to the server; cannot be discarded)
       a == automatic (automatic collected by the system; maybe discarded - see also "kind")
       m == manual    (must/should be set by the user - see also "kind")

 kind: m == mandatory (value must be set)
       o == optional  (value could be set)

 Every value given via init_ctx has the flag "i" and kind "m"
 After init_ctx only initial arguments are stored in this field. Every "needinfo" 
 during the registration request( I<register($ctx)> ) can modify the arguments 
 in this field.

EXAMPLE:

  my $data = {};
  $data->{nooptional} = 1;
  $data->{noproxy} = 1;

  my $ctx = SUSE::SuseRegister::init_ctx($data);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut


sub init_ctx
{
    my $data = shift;
    my $ctx = {};
    my $code = 0;
    my $msg = "";

    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";
    $ctx->{timeindent} = 0;
    $ctx->{logfile}     = undef;
    $ctx->{debug}       = 0;
    $ctx->{yastcall}    = 0; # FIXME: is this still needed?
    $ctx->{LOGDESCR}    = undef;

    # required to read informations with zypper even if another
    # instance of libzypp is running.
    $ENV{ZYPP_READONLY_HACK} = 1;

    if(exists $data->{debug} && defined $data->{debug})
    {
        $ctx->{debug} = $data->{debug};
    }

    if(exists $data->{logfile} && defined $data->{logfile})
    {
        $ctx->{logfile} = $data->{logfile};
    }

    if(exists $data->{yastcall} && defined $data->{yastcall})
    {
        $ctx->{yastcall} = $data->{yastcall};
    }

    if (defined $ctx->{logfile} && $ctx->{logfile} ne "")
    {
        open($ctx->{LOGDESCR}, ">> ".$ctx->{logfile}) or do 
        {
            if(!$ctx->{yastcall})
            {
                SUSE::SRPrivate::logPrintError($ctx, "Cannot open logfile <$ctx->{logfile}>: $!\n", 12);
                return $ctx;
            }
            else
            {
                syslog("err", "Cannot open logfile <$ctx->{logfile}>: $!(yastcall ignoring error)");
                $ctx->{LOGDESCR} = undef;
            }
        };
        # $LOGDESCR is undef if no logfile is defined
        if(defined $ctx->{LOGDESCR})
        {
            # autoflush
            $ctx->{LOGDESCR}->autoflush(1);
            print {$ctx->{LOGDESCR}} "----- ".localtime()." ---------------------------------------\n";
        }
    }


    $ctx->{version} = "1.0";

    $ctx->{configFile}      = "/etc/suseRegister.conf";
    $ctx->{sysconfigFile}   = "/etc/sysconfig/suse_register";

    # these are the old files.
    # if they still exists, we need to convert them
    # to the new format
    $ctx->{GUID_FILE}       = "/etc/zmd/deviceid";
    $ctx->{SECRET_FILE}     = "/etc/zmd/secret";

    # new credentails file
    $ctx->{CREDENTIAL_DIR}  = "/etc/";
    $ctx->{CREDENTIAL_FILE} = "NCCcredentials";

    $ctx->{SYSCONFIG_CLOCK} = "/etc/sysconfig/clock";
    $ctx->{CA_PATH}         = "/etc/ssl/certs";
    $ctx->{CA_FILE}         = ["/etc/pki/tls/cert.pem", "/usr/share/ssl/cert.pem"];

    $ctx->{URL}             = "https://secure-www.novell.com/center/regsvc/";

    $ctx->{URLlistParams}   = "command=listparams";
    $ctx->{URLregister}     = "command=register";
    $ctx->{URLlistProducts} = "command=listproducts";

    $ctx->{guid}      = undef;
    $ctx->{secret}    = undef;
    $ctx->{locale}    = undef;
    $ctx->{encoding}  = "utf-8";
    $ctx->{lang}      = "en-US";

    $ctx->{listParams}      = 0;
    $ctx->{xmlout}          = 0;  
    $ctx->{nooptional}  = 0;
    $ctx->{acceptmand}  = 0;
    $ctx->{forcereg}    = 0;
    $ctx->{nohwdata}    = 0;
    $ctx->{batch}       = 0;
    $ctx->{interactive} = 0;
    $ctx->{noproxy}     = 0;
    $ctx->{args}      = {
                         processor => { flag => "i", value => undef, kind => "mandatory" },
                         platform  => { flag => "i", value => undef, kind => "mandatory" },
                         timezone  => { flag => "i", value => undef, kind => "mandatory" },
                        };
    $ctx->{serverKnownProducts} = [];
    $ctx->{installedProducts}   = [];
    $ctx->{products} = [];

    $ctx->{installedPatterns} = [];

    $ctx->{extraCurlOption} = [];

    $ctx->{hostGUID} = undef;
    $ctx->{virtType} = "";
    $ctx->{FallbackHostGUID} = undef;
    
    $ctx->{zmdConfig} = {};
    $ctx->{ostarget} = "";

    $ctx->{norefresh} = 0;

    $ctx->{redirects} = 0;
    $ctx->{registerManuallyURL} = "";
    $ctx->{registerReadableText} = [];
    $ctx->{registerPrivPol} = "";

    $ctx->{zypper}        = "/usr/bin/zypper";
    $ctx->{lsb_release}   = "/usr/bin/lsb_release";
    $ctx->{uname}         = "/bin/uname";
    $ctx->{hwinfo}        = "/usr/sbin/hwinfo";
    $ctx->{curl}          = "/usr/bin/curl";

    $ctx->{xenstoreread}  = "/usr/bin/xenstore-read";
    $ctx->{xenstorewrite} = "/usr/bin/xenstore-write";
    $ctx->{xenstorechmod} = "/usr/bin/xenstore-chmod";
    $ctx->{lscpu}         = "/usr/bin/lscpu";

    $ctx->{createGuid}    = "/usr/bin/uuidgen";

    $ctx->{lastResponse}    = "";
    $ctx->{initialDomain}   = "";

    $ctx->{addRegSrvSrc} = 1;
    $ctx->{addAdSrc} = [];

    $ctx->{zmdcache} = "/var/cache/SuseRegister/lastzmdconfig.cache";
    $ctx->{restoreRepos} = 0;
    $ctx->{warnUnrestoredRepos} = 0;
    

    if(exists $data->{xmlout} && defined $data->{xmlout})
    {
        $ctx->{xmlout} = $data->{xmlout};
    }

    if(exists $data->{nooptional} && defined $data->{nooptional})
    {
        $ctx->{nooptional} = $data->{nooptional};
    }

    if(exists $data->{forcereg} && defined $data->{forcereg})
    {
        $ctx->{forcereg} = $data->{forcereg};
    }

    if(exists $data->{nohwdata} && defined $data->{nohwdata})
    {
        $ctx->{nohwdata} = $data->{nohwdata};
    }

    if(exists $data->{batch} && defined $data->{batch})
    {
        $ctx->{batch} = $data->{batch};
    }

    if(exists $data->{interactive} && defined $data->{interactive})
    {
        $ctx->{interactive} = $data->{interactive};
    }

    if(exists $data->{locale} && defined $data->{locale})
    {
        $ctx->{locale} = $data->{locale};
    }

    if(exists $data->{noproxy} && defined $data->{noproxy})
    {
        $ctx->{noproxy} = $data->{noproxy};
    }

    if(exists $data->{norefresh} && defined $data->{norefresh})
    {
        $ctx->{norefresh} = $data->{norefresh};
    }

    if(exists $data->{args} && ref($data->{args}) eq "HASH")
    {
        foreach my $a (keys %{$data->{args}})
        {
            $ctx->{args}->{$a} = {flag => "i", value => $data->{args}->{$a}, kind => "mandatory"};
        }
    }

    if(exists $data->{extraCurlOption} && ref($data->{extraCurlOption}) eq "ARRAY")
    {
        $ctx->{extraCurlOption} = $data->{extraCurlOption};
    }

    openlog("suse_register", "ndelay,pid", 'user');

    if(exists $data->{restoreRepos} && defined $data->{restoreRepos})
    {
        $ctx->{restoreRepos} = $data->{restoreRepos};
    }

    if(exists $ENV{LANG} && $ENV{LANG} =~ /^([\w_]+)\.?/) 
    {
        if(defined $1 && $1 ne "") 
        {
            $ctx->{lang} = $1;
            $ctx->{lang} =~ s/_/-/;
        }
    }
    elsif(exists $ENV{LANGUAGE} && $ENV{LANGUAGE} =~ /^([\w_]+)\.?/) 
    {
        if(defined $1 && $1 ne "") 
        {
            $ctx->{lang} = $1;
            $ctx->{lang} =~ s/_/-/;
        }
    }

    if (defined $ctx->{locale})
    {
        my ($l, $e) = split(/\.|@/, $ctx->{locale}, 2);
        
        if (defined $l && $l ne "")
        {
            $l =~ s/_/-/;
            $ctx->{lang} = $l;
        }
        
        if (defined $e && $e ne "") 
        {        
            $ctx->{encoding} = $e;
        }
    }
    
    # check for xen tools
    if(! -e $ctx->{xenstoreread} &&
       -e "/bin/xenstore-read")
    {
        $ctx->{xenstoreread} = "/bin/xenstore-read";
    }
    if(! -e $ctx->{xenstorewrite} &&
       -e "/bin/xenstore-write" )
    {
        $ctx->{xenstorewrite} = "/bin/xenstore-write";
    }
    if(! -e $ctx->{xenstorechmod} &&
       -e "/bin/xenstore-chmod" )
    {
        $ctx->{xenstorechmod} = "/bin/xenstore-chmod";
    }
    
    # call this as soon as possible.
    ($code, $msg) = SUSE::SRPrivate::initGUID($ctx);
    if($code != 0)
    {
        SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }

    ($code, $msg) = SUSE::SRPrivate::readSystemValues($ctx);
    if($code != 0)
    {
        SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }


    printLog($ctx, "debug1", "list-parameters:   $ctx->{listParams}");
    printLog($ctx, "debug1", "xml-output:        $ctx->{xmlout}");
    printLog($ctx, "debug1", "no-optional:       $ctx->{nooptional}");
    printLog($ctx, "debug1", "batch:             $ctx->{batch}");
    printLog($ctx, "debug1", "forcereg:          $ctx->{forcereg}");
    printLog($ctx, "debug1", "no-hw-data:        $ctx->{nohwdata}");
    printLog($ctx, "debug1", "log:               ".(($ctx->{logfile})?$ctx->{logfile}:"undef"));
    printLog($ctx, "debug1", "locale:            ".(($ctx->{locale})?$ctx->{locale}:"undef"));
    printLog($ctx, "debug1", "no-proxy:          $ctx->{noproxy}");
    printLog($ctx, "debug1", "yastcall:          $ctx->{yastcall}");
    printLog($ctx, "debug1", "arg: ".Data::Dumper->Dump([$ctx->{args}]));
    printLog($ctx, "debug1", "extra-curl-option:".Data::Dumper->Dump([$ctx->{extraCurlOption}]));
    
    printLog($ctx, "debug1", "URL:               $ctx->{URL}");
    printLog($ctx, "debug1", "listParams:        $ctx->{URLlistParams}");
    printLog($ctx, "debug1", "register:          $ctx->{URLregister}");
    printLog($ctx, "debug1", "lang:              $ctx->{lang}");
    
    
    my $iuri = URI->new($ctx->{URL});

    $ctx->{initialDomain} = $iuri->host;
    $ctx->{initialDomain} =~ s/.+(\.[^.]+\.[^.]+)$/$1/;

    printLog($ctx, "debug1", "initialDomain:     $ctx->{initialDomain}");

    ($code, $msg) = SUSE::SRPrivate::listProducts($ctx);
    if($code != 0)
    {
        #SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }
    
    $ctx->{products} = SUSE::SRPrivate::intersection($ctx, $ctx->{installedProducts}, $ctx->{serverKnownProducts});
    printLog($ctx, "debug1", "register products:".Data::Dumper->Dump([$ctx->{products}]));

    if($#{$ctx->{installedProducts}} == 0 && 
       exists $ctx->{installedProducts}->[0]->[0] &&
       $ctx->{installedProducts}->[0]->[0] =~ /FACTORY/i)
    {
        SUSE::SRPrivate::logPrintError($ctx, "FACTORY cannot be registered\n", 101);
        return $ctx;
    }

    if(@{$ctx->{products}} == 0)
    {
        SUSE::SRPrivate::logPrintError($ctx, "None of the installed products can be registered at the Novell registration server.\n", 100);
        return $ctx;
    }

    $ctx->{timeindent}--;

    return $ctx;
}

=item *
B<del_ctx($ctx)>

Delete the context

=cut

sub del_ctx
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    
    close $ctx->{LOGDESCR} if(defined $ctx->{LOGDESCR});
    closelog;

    $ctx = {};

    $ctx->{timeindent}--;
}


=item *
B<$txt = listParams($ctx)>

Return the parameter list from the server.

EXAMPLE:

  my $txt = listParams($ctx);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }
  print $txt;

=cut

sub listParams
{
    my $ctx = shift;
    my $text = "";

    $ctx->{timeindent}++;

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    my $output = '<?xml version="1.0" encoding="utf-8"?>';
    
    my $writer = new XML::Writer(OUTPUT => \$output);

    $writer->startTag("listparams", "xmlns" => "http://www.novell.com/xml/center/regsvc-1_0",
                                    "client_version" => "$SUSE::SRPrivate::SRversion");

    foreach my $PArray (@{$ctx->{products}})
    {
        if(defined $PArray->[0] && $PArray->[0] ne "" &&
           defined $PArray->[1] && $PArray->[1] ne "")
        {
            $writer->startTag("product",
                              "version" => $PArray->[1],
                              "release" => $PArray->[2],
                              "arch"    => $PArray->[3]);
            if ($PArray->[0] =~ /\s+/)
            {
                $writer->cdata($PArray->[0]);
            }
            else
            {
                $writer->characters($PArray->[0]);
            }
            $writer->endTag("product");
        }
    }
    
    $writer->endTag("listparams");

    printLog($ctx, "debug3", "XML:\n$output");

    $ctx->{redirects} = 0;                           
                                      # hard coded en-US; suse_register is in english
    my $res = SUSE::SRPrivate::sendData($ctx, $ctx->{URL}."?".$ctx->{URLlistParams}."&lang=en-US&version=$ctx->{version}", $output);
    if($ctx->{errorcode} != 0 || ! defined $res )
    {
        printLog($ctx, "info", "=>=>=> send Data failed => return");
        return;
    }
        
    if ($ctx->{xmlout})
    {
        return "$res\n";
    }
    else
    {
        my $privpol = "";
        
        if(!$ctx->{yastcall})
        {
            $text .= "Available Options:\n\n";
            $text .= "You can add these options to suse_register with the -a option.\n";
            $text .= "'-a' can be used multiple times\n\n";
        }
        
        my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
        my $tree = $p->parse($res);

        #print Data::Dumper->Dump([$tree])."\n";

        if (! defined $tree || ref($tree->[0]) ne "SR::paramlist" ||
            ! exists $tree->[0]->{Kids} || ref($tree->[0]->{Kids}) ne "ARRAY")
        {
            SUSE::SRPrivate::logPrintError($ctx, "Invalid XML format. Cannot show human readable output. Try --xml-output.\n".
                                     6);
            return;
        }

        if($ctx->{yastcall})
        {
            $text .= "<pre>";
        }
                
        foreach my $kid (@{$tree->[0]->{Kids}})
        {
            #print Data::Dumper->Dump([$tree->[1]->[$i]])."\n\n";

            if (ref($kid) eq "SR::param")
            {
                if (exists $kid->{command} && defined $kid->{command} &&
                    $ctx->{nohwdata} && $kid->{command} =~ /^hwinfo/)
                {
                    # skip; --no-hw-data was provided 
                    next;
                }
                elsif (exists $kid->{command} && defined $kid->{command} &&
                    $ctx->{nohwdata} && $kid->{command} =~ /^installed-desktops$/)
                {
                    # skip; --no-hw-data was provided 
                    next;
                }
                
                $text .= "* ".$kid->{description}.": ";
                if(!$ctx->{yastcall})
                {
                    $text .= "\n\t".$kid->{id}."=<value> ";
                }

                if (exists $kid->{command} && defined $kid->{command} && $kid->{command} ne "")
                {
                    $text .= "(command: ".$kid->{command}.")\n";
                }
                else
                {
                    $text .= "\n";
                }
                if(!$ctx->{yastcall})
                {
                    $text .= "\n";
                }
            }
            elsif (ref($kid) eq "SR::privacy" )
            {
                if (exists $kid->{description} && defined $kid->{description})
                {
                    if(!$ctx->{yastcall}) 
                    {
                        $privpol .= "\nInformation on Novell's Privacy Policy:\n";
                        $privpol .= $kid->{description}."\n";
                    }
                    else
                    {
                        $privpol .= "<p>Information on Novell's Privacy Policy:<br>\n";
                        $privpol .= $kid->{description}."</p>\n";
                    }
                }
                
                if (exists $kid->{url} && defined $kid->{url} && $kid->{url} ne "")
                {
                    if(!$ctx->{yastcall}) 
                    {
                        $privpol .= $kid->{url}."\n";
                    }
                    else
                    {
                        $privpol .= "<p><a href=\"".$kid->{url}."\">";
                        $privpol .= $kid->{url}."</a></p>\n";
                    }
                }
            }
        }
        if(!$ctx->{yastcall})
        {
            $text .= "Example:\n";
            $text .= "  suse_register -a email=\"tux\@example.com\"\n";
        }
        else
        {
            $text .= "</pre>\n";
        }

        $text .= $privpol;
    }

    $ctx->{timeindent}--;

    return "$text\n";
}

=item *
B<$ret = register($ctx)>

Perform the registration.

I<$ret> returns the status of the registration. 

 1 indicates a "needinfo" from the server. 

 0 registration completed

 2 error - but it is better to test $ctx->{errorcode}

Important values in the context are:

 $ctx->{args} the arguments which will be send to the server - see I<init_ctx()> for details.

 $ctx->{registerManuallyURL} contains a URL to a webpage where you can enter the manual values.

 $ctx->{registerReadableText} contains a human readable text about the missing arguments.


EXAMPLE:

  my $ret = register($ctx);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }

  if($ret == 1)
  {
     # you can show and modify the $ctx->{args} here.
     # Have a look for mandatory args which cannot be automaticaly
     # detected. Ask for them, or open a browser with the URL proided
     # in $ctx->{registerManuallyURL} .

  }
  if($ret == 0)
  {
     # Registration completed. You can now get a tasklist to get
     # more information on the update sources (I<getTaskList($ctx)>). 
    
  }

=cut


sub register
{
    my $ctx = shift;

    $ctx->{timeindent}++;

    my $code = 0;
    my $msg = "";
    
    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    my $output = SUSE::SRPrivate::buildXML($ctx);

    $ctx->{redirects} = 0;
    my $res = SUSE::SRPrivate::sendData($ctx, $ctx->{URL}."?".$ctx->{URLregister}."&lang=en-US&version=$ctx->{version}", $output);
    if($ctx->{errorcode} != 0)
    {
        return 2;
    }

    if($res eq $ctx->{lastResponse})
    {
        # Got the same server response as the last time
        SUSE::SRPrivate::logPrintError($ctx, "Invalid response from the registration server. Aborting.\n", 9);
        return 2;
    }
    $ctx->{lastResponse} = $res;

    my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
    my $tree = $p->parse($res);
    
    #print Data::Dumper->Dump([$tree])."\n";

    if (defined $tree && ref($tree->[0]) eq "SR::needinfo" && 
        exists $tree->[0]->{Kids} && ref($tree->[0]->{Kids}) eq "ARRAY") 
    {
        if ($ctx->{xmlout})
        {
            $ctx->{xmloutput} = "$res\n";
            return 1;
        }

        # cleanup old stuff
        $ctx->{registerReadableText} = [];
        $ctx->{registerManuallyURL} = "";
        $ctx->{registerPrivPol} = "";

        if (exists $tree->[0]->{href} &&
            defined $tree->[0]->{href} &&
            $tree->[0]->{href} ne "")
        {
            my $uri = URI->new($tree->[0]->{href});
            my %h = $uri->query_form();
            $h{lang} = "$ctx->{lang}";
            $uri->query_form(%h);
            $ctx->{registerManuallyURL} = $uri->as_string;
        }
        
        my $ret = SUSE::SRPrivate::evalNeedinfo($ctx, $tree);
        if($ctx->{errorcode} != 0)
        {
            return 2;
        }
        
        if ($#{$ctx->{registerReadableText}} > -1 && 
            $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] =~ /^\s*$/) 
        {
            pop @{$ctx->{registerReadableText}};
        }
        
        if(!$ctx->{yastcall})
        {
            unshift @{$ctx->{registerReadableText}},
            "To complete the registration, provide some additional parameters:\n\n";
            
            push @{$ctx->{registerReadableText}}, "\nYou can provide these parameters with the '-a' option.\n";
            push @{$ctx->{registerReadableText}}, "You can use the '-a' option multiple times.\n\n";
            push @{$ctx->{registerReadableText}}, "Example:\n\n";
            push @{$ctx->{registerReadableText}}, 'suse_register -a email="me@example.com"'."\n";
            push @{$ctx->{registerReadableText}}, "\nTo register your product manually, use the following URL:\n\n";
            push @{$ctx->{registerReadableText}}, "$ctx->{registerManuallyURL}\n\n";
            
        }
        else 
        {
            unshift @{$ctx->{registerReadableText}}, "<pre>";
            push @{$ctx->{registerReadableText}}, "</pre>";
            push @{$ctx->{registerReadableText}}, "<p>To register your product manually, use the following URL:</p>\n";
            push @{$ctx->{registerReadableText}}, "<pre>$ctx->{registerManuallyURL}</pre>\n\n";
        }
        
        push @{$ctx->{registerReadableText}}, $ctx->{registerPrivPol};
        
        
        # after the first needinfo, set accept=mandatory to true
        # If the application think, that this is not a good idea
        # it can reset this.
        $ctx->{acceptmand} = 1;
        
        $ctx->{timeindent}--;
        
        # return 1 == needinfo
        return 1;
    }
    elsif (defined $tree && ref($tree->[0]) eq "SR::zmdconfig" &&
           exists $tree->[0]->{Kids} && ref($tree->[0]->{Kids}) eq "ARRAY")
    {
        $ctx->{newzmdconfig} = "$res";
        
        if ($ctx->{xmlout})
        {
            $ctx->{xmloutput} = "$res\n";
        }

        # ok => parse zmdconfig and add local configured repos from suseRegister.conf
        ($code, $msg) = SUSE::SRPrivate::getZmdConfigValues($ctx);
        if($code != 0)
        {
            SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
            return 2;
        }

        return 0;
    }
    else
    {
        SUSE::SRPrivate::logPrintError($ctx, "Unknown reponse format.\n", 11);
        return 2;
    }
    $ctx->{timeindent}--;
    return 0;
}


sub manageUpdateSources
{
    my $ctx = shift;

    my $lastcode = 0;
    my $lastmsg = "";

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    if(!exists $ctx->{zmdConfig} || !defined $ctx->{zmdConfig} ||
       ref($ctx->{zmdConfig}) ne "HASH")
    {
	    # no source; return without changing sources
	    SUSE::SRPrivate::printLog($ctx, "debug2", "no source; return without changing sources");
	    return undef;
    }

    if(-d "/etc/yum.repos.d")
    {
        # RH5
        printLog($ctx, "debug2", "Found yum repo dir");
        if( -e "/etc/yum.repos.d/RH5.2.repo" )
        {
            unlink "/etc/yum.repos.d/RH5.2.repo";
        }
        
        foreach my $alias (keys %{$ctx->{zmdConfig}})
        {
            if( -e "/etc/yum.repos.d/$alias.repo" && $ctx->{restoreRepos} )
            {
                printLog($ctx, "debug2", "remove /etc/yum.repos.d/$alias.repo");
                unlink "/etc/yum.repos.d/$alias.repo"
            }
            elsif( -e "/etc/yum.repos.d/$alias.repo" && !$ctx->{restoreRepos} )
            {
                next;
            }
            
            printLog($ctx, "debug2", "create /etc/yum.repos.d/$alias.repo");
            open(REPO, "> /etc/yum.repos.d/$alias.repo") and do
            {
                print REPO "[$alias]\n";
                print REPO "name=$ctx->{zmdConfig}->{$alias}->{NAME}\n";
                print REPO "baseurl=$ctx->{zmdConfig}->{$alias}->{URL}\n";
                print REPO "enabled=1\n";
                print REPO "gpgcheck=1\n";
		#print REPO "gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-redhat-release\n";
                close REPO;
            };
        }
    }
    if( -e "/etc/sysconfig/rhn/sources" )
    {
        # RH3,4
        my @sfile = ();

        printLog($ctx, "debug2", "Found rhn sources file");
        open(SOURCES, "< /etc/sysconfig/rhn/sources") and do
        {
            while(<SOURCES>)
            {
                my $line = $_;
                chomp($line);
                
                if($line =~ /^\s*yum\s+([^\s]+)\s+/)
                {
                    if(defined $1 && ("$1" eq "RH4.7" || "$1" eq "RH3.9") )
                    {
                        next;
                    }
                    
                    if(!exists $ctx->{zmdConfig}->{$1})
                    {
                        push @sfile, "$line";
                    }
                    else
                    {
                        printLog($ctx, "debug2", "Skip \"$line\" line");
                    }
                }
                elsif($line =~ /^\s*up2date\s+/)
                {
                    printLog($ctx, "debug2", "comment up2date line");
                    push @sfile, "#$line";
                }
                else
                {
                    push @sfile, "$line";
                }
            }
            close SOURCES;
        };
        if(@sfile > 0)
        {
            foreach my $alias (keys %{$ctx->{zmdConfig}})
            {
                printLog($ctx, "debug2", "add yum $alias line");
                push @sfile, "yum $alias $ctx->{zmdConfig}->{$alias}->{URL}";
            }
            
            open(SOURCES, "> /etc/sysconfig/rhn/sources") and do
            {
                foreach my $line (@sfile)
                {
                    print SOURCES "$line\n";
                }
                close SOURCES;
            };
        }
    }
    
    SUSE::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);

    return undef;
}


#=======================================================================================================
#===== END OF PUBLIC FUNCTIONS =========================================================================
#=======================================================================================================


sub fullpathOf 
{
    my $ctx = shift;
    my $program = shift || undef;
    
    if(!defined $program || $program eq "" || $program !~ /^[\w_-]+$/)
    {
        return undef;
    }
    
    my $fullpath = `which $program 2>/dev/null`;
    chomp($fullpath);
    printLog($ctx, "debug2", "Fullpath:$fullpath");
    
    if (defined $fullpath && $fullpath ne "")
    {
        return $fullpath;
    }
    return undef;
}

1;


