
package SUSE::SRPrivate;

use strict;
use XML::Parser;
use XML::Writer;
use Data::Dumper;
use File::Temp qw(tempfile);
use File::Copy;
use Sys::Syslog;
use IPC::Open3;
use Fcntl qw(:DEFAULT);
use URI;
use SUSE::Parser::RepoList;
use SUSE::Parser::ZmdConfig;
use SUSE::Parser::Product;
use SUSE::Parser::Pattern;

use LWP::UserAgent;
use Crypt::SSLeay;
use English;


# client version number
our $SRversion = "1.3.0";

our @ISA = qw(Exporter);
our @EXPORT = qw(printLog);

sub readSystemValues
{
    my $ctx = shift;
    $ctx->{timeindent}++;
    
    my $code = 0;
    my $msg = "";
    
    ############### batch mode ########################
    
    if($ctx->{batch})
    {
        # if --no-optional or --no-hw-data are not given in batch mode
        # read the sysconfig values for the default

        my $sysconfigOptional = "false";
        my $sysconfigHWData   = "false";
        
        
        open(CNF, "< $ctx->{sysconfigFile}") and do {
        
            while(<CNF>)
            {
                if($_ =~ /^\s*#/)
                {
                    next;
                }
                elsif($_ =~ /^SUBMIT_OPTIONAL\s*=\s*"*([^"\s]*)"*\s*/ && defined $1 && $1 ne "") 
                {
                    $sysconfigOptional = $1;
                    
                }
                elsif($_ =~ /^SUBMIT_HWDATA\s*=\s*"*([^"\s]*)"*\s*/ && defined $1 && $1 ne "") 
                {
                    $sysconfigHWData = $1;
                }
            }
            close CNF;
        };

        if(!$ctx->{nooptional})
        {
            if(lc($sysconfigOptional) eq "true")
            {
                $ctx->{nooptional} = 0;
            }
            else
            {
                $ctx->{nooptional} = 1;
            }
        }
        if(!$ctx->{nohwdata})
        {
            if(lc($sysconfigHWData) eq "true")
            {
                $ctx->{nohwdata} = 0;
            }
            else
            {
                $ctx->{nohwdata} = 1;
            }
        }
   }
        
    ############### read the config ###################
    if(-e $ctx->{configFile})
    {
        open(CNF, "< $ctx->{configFile}") or do 
        {
            return logPrintReturn($ctx, "Cannot open file $ctx->{configFile}: $!\n", 12);
        };
        
        while(<CNF>)
        {
            if($_ =~ /^\s*#/)
            {
                next;
            }
            elsif($_ =~ /^url\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "") 
            {
                $ctx->{URL} = $1;
            }
            elsif($_ =~ /^listParams\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "") 
            {
                $ctx->{URLlistParams} = $1;
            }
            elsif($_ =~ /^listProducts\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "") 
            {
                $ctx->{URLlistProducts} = $1;
            }
            elsif($_ =~ /^register\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "")
            {
                $ctx->{URLregister} = $1;
            }
            elsif($_ =~ /^hostGUID\s*=\s*(\w*)\s*/ && defined $1 && $1 ne "")
            {
                $ctx->{FallbackHostGUID} = $1;
            }
            elsif($_ =~ /^addRegSrvSrc\s*=\s*(\w*)\s*/ && defined $1)
            {
                if(lc($1) eq "true")
                { 
                    $ctx->{addRegSrvSrc} = 1;
                }
                else 
                {
                    $ctx->{addRegSrvSrc} = 0;
                }
            }
            elsif($_ =~ /^addAdSrc\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "")
            {
                push @{$ctx->{addAdSrc}}, $1;
            }
        }
        close CNF;
    }
    
    ############### GUID ##############################

    ($code, $msg) = initGUID($ctx);
    if($code != 0)
    {
        return ($code, $msg);
    }
    printLog($ctx, "debug1", "GUID:$ctx->{guid}");
    
    ############### find Products #####################

    ($code, $msg) = getProducts($ctx);
    if($code != 0)
    {
        return ($code, $msg);
    }

    ########## host GUID (virtualization) #############
    ($code, $msg) = detectVirtualization($ctx);
    if($code != 0)
    {
        return ($code, $msg);
    }

    ############## some initial values ########################

    $ctx->{args}->{processor} = { flag => "i", value => `$ctx->{uname} -p`, kind => "mandatory"};
    $ctx->{args}->{platform}  = { flag => "i", value => `$ctx->{uname} -i`, kind => "mandatory"};
    $ctx->{args}->{timezone}  = { flag => "i", value => "US/Mountain", kind => "mandatory"};      # default


    open(SYSC, "< $ctx->{SYSCONFIG_CLOCK}") or do
    {
        return logPrintReturn($ctx, "Cannot open file $ctx->{SYSCONFIG_CLOCK}: $!\n", 12);
    };
    while(<SYSC>) 
    {
        if($_ =~ /^TIMEZONE\s*=\s*"?([^"]*)"?/) 
        {
            if(defined $1 && $1 ne "")
            {
                $ctx->{args}->{timezone}  = { flag => "i", value => $1, kind => "mandatory"};
            }
        }
        elsif($_ =~ /^ZONE\s*=\s*"?([^"]*)"?/) 
        {
            if(defined $1 && $1 ne "")
            {
                $ctx->{args}->{timezone}  = { flag => "i", value => $1, kind => "mandatory"};
            }
        }
    }
    close SYSC;
    
    chomp($ctx->{args}->{processor}->{value});
    chomp($ctx->{args}->{platform}->{value});

    $ctx->{timeindent}--;

    return (0, "");
}


sub evaluateCommand
{
    my $ctx = shift;
    my $command   = shift || undef;
    my $mandatory = shift || 0;
    my $cmd       = undef;
    my $out       = undef;
    my @arguments = ();

    $ctx->{timeindent}++;


    if (!defined $command || $command eq "")
    {
        logPrintError($ctx, "Missing command.\n", 14);
        return undef;
    }

    if ($command =~ /^hwinfo\s*(.*)\s*$/)
    {
        return "DISCARDED";
        #
        # we do not have hwinfo in RH
        #
        if(!$ctx->{nohwdata})
        {
            $cmd = $ctx->{hwinfo};
            if (defined $1)
            {
                @arguments = split(/\s+/, $1);
            }
        }
        elsif($ctx->{mandatory})
        {
            logPrintError($ctx, "Mandatory hardware data cannot be supplied because the option --no-hw-data was given.\n",
                          3);
            return undef;
        }
        else
        {
            return "DISCARDED";
        }
    }
    elsif ($command =~ /^lsb_release\s*(.*)\s*$/)
    {
        # maybe lsb_release is not installed
        if(-e $ctx->{lsb_release})
        {
            $cmd = $ctx->{lsb_release};
            if (defined $1) 
            {
                @arguments = split(/\s+/, $1);
            }
        }
        elsif($1 eq "-sd")
        {
            # needed for ostarget-bak; 
            # we should be able to return this even without lsb package installed.
            my $line = "";
            open(SR, "< /etc/SuSE-release") and do {
                
                $line = <SR>;
                chomp($line);
                $line =~ s/^\s*//;
                $line =~ s/\s*$//;
                close SR;
            };
            
            if($line ne "")
            {
                return '"'.$line.'"';
            }
            return "";
        }
        else
        {
            return "";
        }
    }
    elsif ($command =~ /^uname\s*(.*)\s*$/)
    {
        $cmd = $ctx->{uname};
        if (defined $1)
        {
            @arguments = split(/\s+/, $1);
        }
    }
    elsif ($command =~ /^zmd-secret$/)
    {
        $cmd = undef;
        $out = $ctx->{secret};
    }
    elsif ($command =~ /^zmd-ostarget$/)
    {
        $cmd = undef;

        return getOSTarget($ctx);
    }
    elsif ($command =~ /^cpu-count$/)
    {
        $cmd = undef;

        if(!$ctx->{nohwdata})
        {
            $out = cpuCount($ctx);
        }
        elsif($ctx->{mandatory})
        {
            logPrintError($ctx, "Mandatory hardware data cannot be supplied because the option --no-hw-data was given.\n",
                          3);
            return undef;
        }
        else
        {
            return "DISCARDED";
        }
    }
    else
    {
        $out = "DISCARDED"; # command not allowed; reply DISCARDED
        $cmd = undef;
    }

    if (defined $cmd)
    {
        my $code = 0;
        my $err = "";
        ($code, $err, $out) = executeCommand($ctx, $cmd, undef, @arguments);
        
        if(!defined $out || $out eq "") 
        {
            $out = "";
        }
        printLog($ctx, "debug1", "LENGTH: ".length($out));
    }
    
    $ctx->{timeindent}--;
    
    return $out;
}

sub cpuCount
{
    my $ctx = shift;

    $ctx->{timeindent}++;

    my $currentCPU = -1;
    my $info = {};
    
    my $haveCoreData = 0;
    my $useCoreID = 0;
    
    my $pid = -1;  # processor id
    my $cos = -1;  # cores
    my $cid = -1;  # core id
    
    my $out = "";

    my $type = `uname -m`;
    chomp($type);

    if($type =~ /ppc/i)
    {
        my $sockets = `grep cpu /proc/device-tree/cpus/*/device_type | wc -l`;
        $out = "CPUSockets: ".($sockets)."\n";
        return $out;
    }

    my $cpuinfo = `cat /proc/cpuinfo`;
    my @lines = split(/\n/, $cpuinfo);
    
    foreach my $line (@lines)
    {
        if( $line =~ /^processor\s*:\s*(\d+)\s*$/)
        {
            if($pid >= 0 )
            {
                if($cos >= 0)
                {
                    $info->{$pid} = $cos;
                    $pid = -1;
                    $cos = -1;
                    $cid = -1;
                }
                elsif($cid >= 0)
                {
                    # IA64 does have core id but not cores
                    if(! exists $info->{$pid} || $cid > $info->{$pid})
                    {
                        $useCoreID = 1;
                        $info->{$pid} = $cid;
                        $pid = -1;
                        $cos = -1;
                        $cid = -1;
                    }
                }
                else 
                {
                    $out = "Read Error";
                }
            }
            
            $currentCPU = $1;
        }
        elsif( $line =~ /^physical id\s*:\s*(\d+)\s*$/)
        {
            $haveCoreData = 1;
            $pid = $1;
        }
        elsif( $line =~ /^cpu cores\s*:\s*(\d+)\s*$/)
        {
            $haveCoreData = 1;
            $cos = $1;
        }
        elsif( $line =~ /^core id\s*:\s*(\d+)\s*$/)
        {
            $haveCoreData = 1;
            $cid = $1;
        }
        elsif( $line =~ /^processor\s+(\d+):/)
        {
            # this is used for s390
            $currentCPU = $1;
        }
    }
    
    printLog($ctx, "debug2", "       socket => cores ");
    printLog($ctx, "debug2", Data::Dumper->Dump([$info]));
    
    if(!$haveCoreData && $currentCPU >= 0)
    {
        $out = "CPUSockets: ".($currentCPU + 1)."\n";
    }
    elsif(keys %{$info} > 0)
    {
        my $cores = 0;
        foreach my $s (keys %{$info})
        {
            $cores += $info->{$s};
            if($useCoreID)
            {
                $cores += 1;
            }
        }
        $out = "CPUSockets: ".(keys %{$info})."\nCPUCores  : $cores\n"
    }
    else
    {
        $out = "Read Error";
    }
    
    printLog($ctx, "debug2", $out );
    
    $ctx->{timeindent}--;
    
    return $out;
}

sub getOSTarget
{
    my $ctx = shift;

    my $arch = `$ctx->{uname} -m`;
    chomp($arch);
    
    if($arch eq "i386" || $arch eq "i486" || $arch eq "i586" || $arch eq "i686")
    {
        return "i386";
    }
    elsif($arch eq "x86_64")
    {
     return "x86_64";
    }   

    return "DISCARDED";
}


sub evalNeedinfo
{
    my $ctx = shift;
    my $tree      = shift || undef;
    my $logic     = shift || "";
    my $indent    = shift || "";
    my $mandatory = shift || 0;
    my $modified  = shift || 0;

    $ctx->{timeindent}++;

    my $mandstr = "";

    my $nextLogic = $logic;
    if($#{$ctx->{registerReadableText}} >= 0) 
    {
        $indent = $indent."  ";
    }
        
    if (! defined $tree)
    {
        logPrintError($ctx, "Missing data.\n", 14);
        return $modified;
    }
 
    printLog($ctx, "debug3", "LOGIC: $logic");
    printLog($ctx, "debug3", Data::Dumper->Dump([$tree]));

    foreach my $kid (@$tree)
    {
        my $local_mandatory = $mandatory;
        
        if (ref($kid) eq "SR::param") 
        {
            if (@{$kid->{Kids}} > 1)
            {
                $nextLogic = "AND";
            }
            if($logic eq "") 
            {
                $logic = $nextLogic;
            }
        }
        elsif (ref($kid) eq "SR::select")
        {
            $nextLogic = "OR";
            if($logic eq "")
            {
                $logic = $nextLogic;
            }
        }
        elsif (ref($kid) eq "SR::privacy")
        { 
            if (exists $kid->{description} && defined $kid->{description})
            {
                if(!$ctx->{yastcall})
                {
                    $ctx->{registerPrivPol} .= "\nInformation on Novell's Privacy Policy:\n";
                    $ctx->{registerPrivPol} .= $kid->{description}."\n";
                }
                else
                {
                    $ctx->{registerPrivPol} .= "<p>Information on Novell's Privacy Policy:<br>\n";
                    $ctx->{registerPrivPol} .= $kid->{description}."</p>\n";
                }
            }
            
            if (exists $kid->{url} && defined $kid->{url} && $kid->{url} ne "")
            {
                if(!$ctx->{yastcall})
                {
                    $ctx->{registerPrivPol} .= $kid->{url}."\n";
                }
                else
                {
                    $ctx->{registerPrivPol} .= "<p><a href=\"".$kid->{url}."\">";
                    $ctx->{registerPrivPol} .= $kid->{url}."</a></p>\n";
                }
            }
        }
        elsif (ref($kid) eq "SR::needinfo")
        {
            # do nothing
        }
        else
        {
            # skip host, guid, product and maybe more to come later. 
            # There are no strings for the user to display.
            next;
        }

        if (exists  $kid->{class} &&
            defined $kid->{class} &&
            $kid->{class} eq "mandatory")
        {
            $local_mandatory = 1;
            $mandstr = "(mandatory)";
            printLog($ctx, "debug3", "Found mandatory");
        }
        elsif(!$local_mandatory &&
              !exists $kid->{class})
        {
            $mandstr = "(optional)";
        }
  
        if (ref($kid) ne "SR::privacy" &&
            @{$kid->{Kids}} == 0 &&
            defined $kid->{description} &&
            defined $kid->{id})
        {
            if ( ($ctx->{nooptional} && $local_mandatory) || !$ctx->{nooptional})
            {
                if(! exists $kid->{command})
                {
                    printLog($ctx, "debug3", "Add instruction");
                    
                    my $txt = $indent."* ".$kid->{description}." $mandstr";
                    $ctx->{args}->{$kid->{id}} = { flag => "m", 
                                                   value => undef, 
                                                   kind => ($local_mandatory)?"mandatory":"optional"};
                    
                    if(!$ctx->{yastcall})
                    {
                        $txt .= ":\t".$kid->{id}."=<value>\n";
                    }
                    else
                    {
                        $txt .= "\n";
                    }
                    push @{$ctx->{registerReadableText}}, $txt;
                    $modified  = 1;
                }
                else
                {
                    my $ret = evaluateCommand($ctx, $kid->{command}, $local_mandatory);
                    if($ctx->{errorcode} != 0)
                    {
                        return $modified;
                    }
                    if (defined $ret)
                    {
                        $ctx->{args}->{$kid->{id}} = { flag  => "a", 
                                                       value => $ret,
                                                       kind  => ($local_mandatory)?"mandatory":"optional"
                                                     };
                        $modified = 1;
                    }
                }
            }
        }
        elsif (ref($kid) ne "SR::privacy" && defined $kid->{description})
        {
            if ( ($ctx->{nooptional} && $local_mandatory) || !$ctx->{nooptional})
            {
                printLog($ctx, "debug3", "Add description");
                push @{$ctx->{registerReadableText}}, $indent.$kid->{description}." $mandstr with:\n";
            }
        }

        if ( exists $kid->{Kids} && @{$kid->{Kids}} > 0 )
        {
            $modified = evalNeedinfo($ctx, $kid->{Kids}, $nextLogic, $indent, $local_mandatory, $modified);
            $nextLogic = $logic;
            if (defined $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] &&
                $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] =~ /^\s*AND|OR\s*$/i)
            {
                if ($logic =~ /^\s*$/)
                {
                    pop @{$ctx->{registerReadableText}};
                }
                else
                {
                    $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] = $indent."$logic\n";
                }
            }
            else
            {
                push @{$ctx->{registerReadableText}}, $indent."$logic\n";
            }
        }
    }

    $ctx->{timeindent}--;

    return $modified;
}


#sub walkResultZmdconfig

sub buildXML
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    
    my $output = '<?xml version="1.0" encoding="utf-8"?>';
    
    my $writer = new XML::Writer(OUTPUT => \$output);

    my %a = ("xmlns" => "http://www.novell.com/xml/center/regsvc-1_0",
             "client_version" => "$SRversion");
    
    if(!$ctx->{nooptional})
    {
        $a{accept} = "optional";
    }
    if($ctx->{acceptmand} || $ctx->{nooptional}) 
    {
        $a{accept} = "mandatory";
    }
    if($ctx->{forcereg}) 
    {
        $a{force} = "registration";
    }
    if($ctx->{batch}) 
    {
        $a{force} = "batch";
    }
    
    $writer->startTag("register", %a);
    
    $writer->startTag("guid");
    $writer->characters($ctx->{guid});
    $writer->endTag("guid");

    if(defined $ctx->{virtType} && $ctx->{virtType} ne "") 
    {
        if(defined $ctx->{hostGUID} && $ctx->{hostGUID} ne "") 
        {
            $writer->startTag("host", type => $ctx->{virtType} );
            $writer->characters($ctx->{hostGUID});
            $writer->endTag("host");
        }
        else
        {
            $writer->emptyTag("host", type => $ctx->{virtType});
        }
    }
    else
    {
        $writer->emptyTag("host");
    }
    
    foreach my $PArray (@{$ctx->{products}})
    {
        if(defined $PArray->[0] && $PArray->[0] ne "" &&
           defined $PArray->[1] && $PArray->[1] ne "")
        {
            $writer->startTag("product",
                              "version" => $PArray->[1],
                              "release" => $PArray->[2],
                              "arch"    => $PArray->[3]);
            if ($PArray->[0] =~ /\s+/)
            {
                $writer->cdata($PArray->[0]);
            }
            else
            {
                $writer->characters($PArray->[0]);
            }
            $writer->endTag("product");
        }
    }
    
    foreach my $key (keys %{$ctx->{args}})
    {
        next if(!defined $ctx->{args}->{$key}->{value});

        if($ctx->{args}->{$key}->{value} eq "")
        {
            $writer->emptyTag("param", "id" => $key);
        }
        else
        {
            $writer->startTag("param",
                              "id" => $key);
            if ($ctx->{args}->{$key}->{value} =~ /\s+/)
            {
                $writer->cdata($ctx->{args}->{$key}->{value});
            }
            else
            {
                $writer->characters($ctx->{args}->{$key}->{value});
            }
            $writer->endTag("param");
        }
    }

    $writer->endTag("register");

    printLog($ctx, "debug3", "XML:\n$output");

    $ctx->{timeindent}--;
    
    return $output;
}

sub sendData
{
    my $ctx = shift;
    my $url  = shift || undef;
    my $data = shift || undef;
    
    $ctx->{timeindent}++;
    
    my $curlErr = 0;
    my $res = "";
    my $err = "";
    my %header = ();
    my $code = "";
    my $mess = "";
    
    my $content = "";

    if (! defined $url)
    {
        logPrintError($ctx, "Cannot send data to registration server. Missing URL.\n", 14);
        return;
    }
    if($url =~ /^-/)
    {
        logPrintError($ctx, "Invalid protocol($url).\n", 15);
        return;
    }

    my $uri = URI->new($url);
    
    if(!defined $uri->host || $uri->host !~ /$ctx->{initialDomain}$/)
    {
        logPrintError($ctx, "Invalid URL($url). Data could only be send to $ctx->{initialDomain} .\n", 15);
        return;
    }
    if(!defined $uri->scheme || $uri->scheme ne "https")
    {
        logPrintError($ctx, "Invalid protocol($url). https is required.\n", 15);
        return;
    }
    $url = $uri->as_string;
        
    if (! defined $data)
    {
        logPrintError($ctx, "Cannot send data. Missing data.\n", 14);
        return;
    }

    my $user = undef;
    my $pass = undef;

    my ($httpProxy, $httpsProxy, $proxyUser) = getProxySettings();

    if(defined $proxyUser)
    {
        ($user, $pass) = split(/:/, $proxyUser, 2);
    }

    if(defined $httpsProxy)
    {
        # required for Crypt::SSLeay HTTPS Proxy support
        $ENV{HTTPS_PROXY} = $httpsProxy;

        if(defined $user && defined $pass)
        {
            $ENV{HTTPS_PROXY_USERNAME} = $user;
            $ENV{HTTPS_PROXY_PASSWORD} = $pass;
        }
        elsif(exists $ENV{HTTPS_PROXY_USERNAME} && exists $ENV{HTTPS_PROXY_PASSWORD})
        {
            delete $ENV{HTTPS_PROXY_USERNAME};
            delete $ENV{HTTPS_PROXY_PASSWORD};
        }
    }

    if ( -d $ctx->{CA_PATH})
    {
        $ENV{HTTPS_CA_DIR} = $ctx->{CA_PATH};
    }
    else
    {
        foreach my $cf ( @{$ctx->{CA_FILE}} )
        {
            if( -e $cf)
            {
                $ENV{HTTPS_CA_FILE} = $cf;
                last;
            }
        }
    }
    
    
    # uncomment, if you want SSL debuging
    #$ENV{HTTPS_DEBUG} = 1;

    {
        package RequestAgent;
        @RequestAgent::ISA = qw(LWP::UserAgent);

        sub new
        {
            my($class, $puser, $ppass, %cnf) = @_;

            my $self = $class->SUPER::new(%cnf);

            bless {
                   puser => $puser,
                   ppass => $ppass
                  }, $class;
        }

        sub get_basic_credentials
        {
            my($self, $realm, $uri, $proxy) = @_;

            if($proxy)
            {
                if(defined $self->{puser} && defined $self->{ppass})
                {
                    return ($self->{puser}, $self->{ppass});
                }
            }
            return (undef, undef);
        }
    }

    my $ua = RequestAgent->new($user, $pass);

    $ua->protocols_allowed( [ 'https' ] );
    #$ua->default_headers->push_header('Content-Type' => 'text/xml');

    # required to workaround a bug in LWP::UserAgent
    $ua->no_proxy();

    #if(defined $httpProxy)
    #{
    #    $ua->proxy("http", $httpProxy);
    #}

    $ua->max_redirect(7);

    # set timeout to the same value as the iChain timeout
    $ua->timeout(130);

    my %params = ('Content' => $data);
    #$params{':content_file'} = $destfile;
    
    printLog($ctx, "debug2", "SEND DATA to URI: $url:", 1, 0);
    printLog($ctx, "debug2", "$data", 1, 0);
    printLog($ctx, "info", "SEND DATA to URI: $url:", 0, 1);
    printLog($ctx, "info", "$data", 0, 1);

    eval
    {
        $res = $ua->post( $url, {}, %params);
    };
    if($@)
    {
        printLog($ctx, "error", "Request failed: '$url'");
        if($@ =~ /SSL negotiation failed:\s+[^:]*:[^:]*:[^:]*:[^:]*:(.*) at / && 
           defined $1 && $1 ne "")
        {
            printLog($ctx, "error", "$1", 0, 1);
            $ctx->{errorcode} = 16;
            $ctx->{errormsg} = "$1";
        }
        elsif($@ =~ /SSL negotiation failed/)
        {
            printLog($ctx, "error", "SSL negotiation failed", 0, 1);
            $ctx->{errorcode} = 16;
            $ctx->{errormsg} = "SSL negotiation failed";
        }
        else
        {
            $ctx->{errorcode} = 16;
            $ctx->{errormsg} = "Cannot contact the registration server. Request failed.";
        }
        
        printLog($ctx, "debug2", $@, 1, 0);
        printLog($ctx, "error", $@, 0, 1);
        
        $ctx->{timeindent}--;
        return undef;
    }
    
    $content = $res->content();

    printLog($ctx, "debug2", "CODE: ".$res->code." MESSAGE: ".$res->message, 1, 0);
    printLog($ctx, "debug2", "RECEIVED DATA:", 1, 0);
    printLog($ctx, "debug2", $res->as_string, 1, 0);
    
    printLog($ctx, "info", "CODE: ".$res->code." MESSAGE: ".$res->message,0,1);
    printLog($ctx, "info", "RECEIVED DATA:", 0, 1);
    printLog($ctx, "info", $res->as_string(), 0, 1);
    
    $mess = $res->message;
    
    if ($res->code >= 300 && $res->code < 400)
    {
        if ($ctx->{redirects} > 5)
        {
            logPrintError($ctx, "Too many redirects. Aborting.\n", 5);
            return $res;
        }
        $ctx->{redirects}++;
        
        my $loc = $res->header("location");

        local $/ = "\r\n";
        chomp($loc);
        local $/ = "\n";

        #print STDERR "sendData(redirect): ".(tv_interval($t0))."\n" if($ctx->{time});

        $content = sendData($ctx, $loc, $data);
    }
    elsif($res->code < 200 || $res->code >= 300) 
    {
        my $b = "";
        my @c = ();

        if(-e "/usr/bin/links")
        {
            $b = "/usr/bin/links";
            push @c, "-dump" ;
        }
        elsif(-e "/usr/bin/w3m") 
        {
            $b = "/usr/bin/w3m";
            push @c, "-dump", "-T", "text/html";
        }
        
        my $out = "";
        if(-x $b)
        {
            my $code = 0;
            my $err = "";
            ($code, $err, $out) = executeCommand($ctx, $b, $content ,@c);
            
            $out .= "\n";
            if(defined $err && $err ne "")
            {
                $out .= "$err\n";
            }
        }
        $out .= "$mess\n";
        
        logPrintError($ctx, "ERROR: ".$res->code.": $out\n", 2);
        $content = undef;
    }

    $ctx->{timeindent}--;
    
    return $content;
}

sub getProxySettings
{
    my $httpProxy  = undef;
    my $httpsProxy = undef;
    my $proxyUser  = undef;

    if(exists $ENV{http_proxy} && defined $ENV{http_proxy} && $ENV{http_proxy} =~ /^http/)
    {
        $httpProxy = $ENV{http_proxy};
    }
    if(exists $ENV{https_proxy} && defined $ENV{https_proxy} && $ENV{https_proxy} =~ /^http/)
    {
        # required for Crypt::SSLeay HTTPS Proxy support
        $httpsProxy = $ENV{https_proxy};
    }

    if($UID == 0 && -e "/root/.curlrc")
    {
        # read /root/.curlrc
        open(RC, "< /root/.curlrc") or return (undef,undef);
        while(<RC>)
        {
            if($_ =~ /^\s*proxy-user\s*=\s*"(.+)"\s*$/ && defined $1 && $1 ne "")
            {
                $proxyUser = $1;
            }
        }
        close RC;
    }
    elsif($UID != 0 &&
          exists $ENV{HOME} && defined  $ENV{HOME} &&
          $ENV{HOME} ne "" && -e "$ENV{HOME}/.curlrc")
    {
        # read ~/.curlrc
        open(RC, "< $ENV{HOME}/.curlrc") or return (undef,undef);
        while(<RC>)
        {
            if($_ =~ /^\s*proxy-user\s*=\s*"(.+)"\s*$/ && defined $1 && $1 ne "")
            {
                $proxyUser = $1;
            }
            elsif($_ =~ /^\s*--proxy-user\s+"(.+)"\s*$/ && defined $1 && $1 ne "")
            {
                $proxyUser = $1;
            }
        }
        close RC;
    }

    # strip trailing /
    $httpsProxy =~ s/\/*$// if(defined $httpsProxy);
    $httpProxy  =~ s/\/*$// if(defined $httpProxy);

    return ($httpProxy, $httpsProxy, $proxyUser);
}


sub getProducts
{
    my $ctx = shift;

    $ctx->{timeindent}++;

    my $name = `$ctx->{lsb_release} -si`;
    my $version = `$ctx->{lsb_release} -sr`;
    my $release = `$ctx->{lsb_release} -sc`;
    my $arch = `$ctx->{uname} -m`;
    
    chomp $name;
    chomp $version;
    chomp $release;
    chomp $arch;

    if( $name =~ /redhat/i )
    {
        $name = "RES";
    }
    if( $name =~ /centos/i )
    {
        $name = "RES";
    }
    if( $version =~ /^(\d+)/ && defined $1 )
    {
        $version = $1;
    }
    
    push @{$ctx->{installedProducts}}, ["$name", "$version", "$release", "$arch"];
   
    printLog($ctx, "debug1", "installed products:           ".Data::Dumper->Dump([$ctx->{installedProducts}]));
    syslog("info", "Installed Products Dump: ".Data::Dumper->Dump([$ctx->{installedProducts}]));
    
    $ctx->{timeindent}--;
    
    return (0, "");
}

sub logPrintReturn
{
    my $ctx = shift;
    my $message = shift || "";
    my $code    = shift || 0;

    if($code != 0)
    {
        syslog("err", "$message($code)");
        printLog($ctx, "error", "$message($code)");
    }
    
    # cleanup errors in the context
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    return ($code, $message);
}


sub logPrintError
{
    my $ctx = shift;
    my $message = shift || "";
    my $code    = shift || 0;

    if($code != 0) 
    {
        
        if(exists $ctx->{args}->{password})
        {
            $ctx->{args}->{password}->{value} = "secret";
        }
        if(exists $ctx->{args}->{passwd})
        {
            $ctx->{args}->{passwd}->{value} = "secret";
        }
        if(exists $ctx->{args}->{secret})
        {
            $ctx->{args}->{secret}->{value} = "secret";
        }
        
        my $cmdtxt = "Commandline params: no-optional:$ctx->{nooptional}  forceregistration:$ctx->{forcereg}  ";
        $cmdtxt .= "no-hw-data:$ctx->{nohwdata} batch:$ctx->{batch} ";
        
        syslog("err", $cmdtxt);
        syslog("err", "Argument Dump: ".Data::Dumper->Dump([$ctx->{args}]));
        syslog("err", "Products Dump: ".Data::Dumper->Dump([$ctx->{products}]));
        syslog("err", "$message($code)");
        printLog($ctx, "error", "$message($code)");
    }
    
    $ctx->{errorcode} = $code;
    $ctx->{errormsg} = $message;
    
    return;
}

sub printLog
{
    my $ctx      = shift;
    my $category = shift;
    my $message  = shift;
    my $doprint  = shift;
    my $dolog    = shift;
    if (! defined $doprint) { $doprint = 1;}
    if (! defined $dolog)   { $dolog   = 1;}

    return if($ctx->{debug} == 2 && $category eq "debug3");
    return if($ctx->{debug} == 1 && ($category eq "debug2" || $category eq "debug3"));
    return if(!$ctx->{debug} && ($category eq "debug1" || $category eq "debug2" || $category eq "debug3"));
    
    if($doprint && !$ctx->{yastcall})
    {
        if(lc($category) eq "error")
        {
            print STDERR "$message\n";
        }
        else
        {
            print "$message\n";
        }
    }

    if($dolog && defined $ctx->{LOGDESCR})
    {
        my ($package, $filename, $line) = caller;
        my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
        $year += 1900;
        $mon +=1;
        my $timestamp = sprintf("%04d-%02d-%02d %02d:%02d:%02d", $year,$mon,$mday, $hour,$min,$sec);
        my $LOG = $ctx->{LOGDESCR};
        
        foreach (split(/\n/, $message))
        {
            print $LOG "$timestamp $package - [$category]  $_\n";
        }
    }
    return;
}

sub listProducts
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    
    my $output = "\n";
    
    my $writer = new XML::Writer(OUTPUT => \$output);

    $ctx->{redirects} = 0;

    my $res = sendData($ctx, $ctx->{URL}."?".$ctx->{URLlistProducts}."&lang=en-US&version=$ctx->{version}", $output);
    if($ctx->{errorcode} != 0 || ! defined $res)
    {
        return  ($ctx->{errorcode}, $ctx->{errormsg});
    }
    
    my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
    my $tree = $p->parse($res);
    
    #print Data::Dumper->Dump([$tree])."\n";
    
    if (! defined $tree || ref($tree->[0]) ne "SR::productlist")
    {
        return logPrintReturn($ctx, "Unknown XML format. Cannot show human readable output. Try --xml-output.\n",
                              6);
    }
    
    foreach my $kid (@{$tree->[0]->{Kids}})
    {
        #print Data::Dumper->Dump([$kid])."\n";
        
        if (ref($kid) eq "SR::product" &&
            exists  $kid->{Kids} &&
            exists  $kid->{Kids}->[0] &&
            ref($kid->{Kids}->[0]) eq "SR::Characters" &&
            exists  $kid->{Kids}->[0]->{Text} &&
            defined $kid->{Kids}->[0]->{Text} &&
            $kid->{Kids}->[0]->{Text} ne "")
        {
                #print "SEE:".Data::Dumper->Dump([$tree->[1]->[$i]])."\n\n";
            
            push @{$ctx->{serverKnownProducts}}, [$kid->{Kids}->[0]->{Text}, "0"];
        }
    }

    printLog($ctx, "debug2", "Server Known Products:".Data::Dumper->Dump([$ctx->{serverKnownProducts}]));

    $ctx->{timeindent}--;

    return (0, "");
}

sub getZmdConfigValues
{
    my $ctx = shift;

    if($ctx->{addRegSrvSrc})
    {
        # parse the new zmdconfig from registration service
        my $parser = SUSE::Parser::ZmdConfig->new(ctx => $ctx);
        $ctx->{zmdConfig} = $parser->parse($ctx->{newzmdconfig});
    }

    foreach my $src (@{$ctx->{addAdSrc}})
    {
        # add local configured repos from suseRegister.conf
        my $uri = URI->new($src);
        my $alias = $uri->query_param("alias");

        $ctx->{zmdConfig}->{$src}->{'URL'} = $src;
        $ctx->{zmdConfig}->{$src}->{'TYPE'} = "zypp";
        if(defined $alias && $alias ne "")
        {
            $ctx->{zmdConfig}->{$src}->{'ALIAS'} = $alias;
        }
    }

    printLog($ctx, "debug2", "zmdconfig: ".Data::Dumper->Dump([$ctx->{zmdConfig}]));

    return (0, "");
}


sub intersection
{
    my $ctx = shift;
    my $arr1 = shift || undef;
    my $arr2 = shift || undef;
    my $ret = [];
    
    if(!defined $arr1 || !defined $arr2 || 
       ref($arr1->[0]) ne "ARRAY" || ref($arr2->[0]) ne "ARRAY")
    {
        return [];
    }

    printLog($ctx, "debug3", "intersect1: ".Data::Dumper->Dump([$arr1]));
    printLog($ctx, "debug3", "intersect2: ".Data::Dumper->Dump([$arr2]));
    
    foreach my $v1 (@$arr1)
    {
        foreach my $v2 (@$arr2) 
        {
            if(lc($v1->[0]) eq lc($v2->[0]))
            {
                if($v2->[1] ne "0")
                {
                    push @$ret, $v2;
                }
                else
                {
                    push @$ret, $v1;
                }
                last;
            }
        }
    }
    
    printLog($ctx, "debug3", "intersect return : ".Data::Dumper->Dump([$ret]));
    return $ret;
}

sub initGUID
{
    my $ctx = shift;
    
    $ctx->{timeindent}++;

    my $fullpath = $ctx->{CREDENTIAL_DIR}."/".$ctx->{CREDENTIAL_FILE};
    
    if(!-d "$ctx->{CREDENTIAL_DIR}")
    {
        mkdir "$ctx->{CREDENTIAL_DIR}" or return logPrintReturn($ctx, "Cannot create directory $ctx->{CREDENTIAL_DIR}: $!\n", 12);
    }

    #
    # convert old deviceid/secret file into new format if the new file do not exist
    # We do not remove deviceid/secret because zmd is available in other products 
    # and still use these files.
    #
    if(-e $ctx->{GUID_FILE} && -e $ctx->{SECRET_FILE} && !-e "$fullpath")
    {
        printLog($ctx, "info", "Converting credentials into the new format.");
        
        # found old GUID/SECRET file. Convert them into the new format
        open(ZMD, "< $ctx->{GUID_FILE}") or do
        {
            return logPrintReturn($ctx, "Cannot open file $ctx->{GUID_FILE}: $!\n", 12);
        };
        
        $ctx->{guid} = <ZMD>;
        chomp($ctx->{guid});
        close ZMD;

        open(ZMD, "< $ctx->{SECRET_FILE}") or do
        {
            return logPrintReturn($ctx, "Cannot open file $ctx->{SECRET_FILE}: $!\n", 12);
        };
        
        $ctx->{secret} = <ZMD>;
        chomp($ctx->{secret});
        close ZMD;

        open(CRED, "> $fullpath") or do {
            return logPrintReturn($ctx, "Cannot open file fullpath for write: $!\n", 12);
        };
        print CRED "username=".$ctx->{guid}."\n";
        print CRED "password=".$ctx->{secret}."\n";
        close CRED;
        my $mode = 0600; 
        chmod $mode, "$fullpath";

        printLog($ctx, "debug1", "Credential file created: $ctx->{guid}");
        
        $ctx->{timeindent}--;

        return (0, "");
    }
    
    #
    # if NCCcredentials file do not exist, create it
    #
    if(!-e "$fullpath")
    {
        printLog($ctx, "debug1", "Generating new credentials.");
        
        my $guid = `$ctx->{createGuid} 2>/dev/null`;
        if(!defined $guid || $guid eq "")
        {
            return logPrintReturn($ctx, "Cannot create guid. Command '$ctx->{createGuid}' failed.", 13);
        }
        chomp $guid;
        $guid =~ s/-//g;  # remove the -
        $ctx->{guid} = $guid;
        
        sleep(1);

        my $secret = `$ctx->{createGuid} 2>/dev/null`;
        if(!defined $secret || $secret eq "")
        {
            return logPrintReturn($ctx, "Cannot create secret. Command '$ctx->{createGuid}' failed.", 13);
        }
        chomp $secret;
        $secret =~ s/-//g;  # remove the -
        $ctx->{secret} = $secret;

        open(CRED, "> $fullpath") or do {
            return logPrintReturn($ctx, "Cannot open file $fullpath for write: $!\n", 12);
        };
        print CRED "username=$guid\n";
        print CRED "password=$secret\n";
        close CRED;
        my $mode = 0600; 
        chmod $mode, "$fullpath";
        
        printLog($ctx, "debug1", "Credential file created: $ctx->{guid}");
        
        $ctx->{timeindent}--;

        return (0, "");
    }
    
    #
    # read credentials from NCCcredentials file
    #
    open(CRED, "< $fullpath") or do {
        return logPrintReturn($ctx, "Cannot open file $fullpath for read: $!\n", 12);
    };
    while(<CRED>)
    {
        if($_ =~ /username\s*=\s*(.*)$/ && defined $1 && $1 ne "")
        {
            $ctx->{guid} = $1;
        }
        elsif($_ =~ /password\s*=\s*(.*)$/ && defined $1 && $1 ne "")
        {
            $ctx->{secret} = $1;
        }
    }
    close CRED;
    
    $ctx->{timeindent}--;
    
    return (0, "");
}


sub indent
{
    my $ctx = shift;
    my $ind = "";
    for(my $i = 0;
        $i < $ctx->{timeindent};
        $i++)
    {
        $ind .= " ";
    }
    return $ind;
}

sub stripURL
{
    my $ctx = shift;
    my $url = shift || "";

    if($url eq "")
    {
        return "";
    }
    
    my $uri = URI->new($url);

    if($uri->scheme eq "http"  ||
       $uri->scheme eq "https" )
    {
        # delete user/password from url
        $uri->userinfo(undef);
    }
    
    # delete all query parameter from the url
    $uri->query(undef);
    
    return $uri->as_string;
}

sub urlAddCredentials
{
    my $ctx         = shift;
    my $url         = shift || "";
    
    if($url eq "")
    {
        return "";
    }
    
    my $uri = URI->new($url);
    
    if($uri->scheme eq "http"  ||
       $uri->scheme eq "https" )
    {
        my %qp = $uri->query_form();
        $qp{credentials} = $ctx->{CREDENTIAL_FILE};
        
        $uri->query_form(%qp);
    }
    return $uri->as_string;
}

sub executeCommand
{
    my $ctx = shift;
    my $command = shift;
    my $input = shift;
    my @arguments = @_;
    
    my $out = "";
    my $err = "";
    my $code = 0;
    
    my $lang     = $ENV{LANG};
    my $language = $ENV{LANGUAGE};
    

    if(!defined $command || !-x $command)
    {
        return logPrintReturn($ctx, "invalid Command '$command'", 13)
    }

    # set lang to en_US to get output in english.
    $ENV{LANG}     = "en_US";
    $ENV{LANGUAGE} = "en_US";


    printLog($ctx, "debug1", "Execute command: $command ".join(" ",@arguments));
    
    my $pid = open3(\*IN, \*OUT, \*ERR, $command, @arguments) or do {
        $ENV{LANG}     = $lang;
        $ENV{LANGUAGE} = $language;
        return logPrintReturn($ctx, "Cannot execute $command ".join(" ", @arguments).": $!\n",13);
    };
    if(defined $input)
    {
        print IN $input;
    }
    close IN;
    
    while (<OUT>)
    {
        $out .= "$_";
    }
    while (<ERR>)
    {
        $err .= "$_";
    }
    close OUT;
    close ERR;
    
    waitpid $pid, 0;
    
    chomp($out);
    chomp($err);
    
    $ENV{LANG}     = $lang     if(defined $lang);
    $ENV{LANGUAGE} = $language if(defined $language);

    $code = ($?>>8);

    printLog($ctx, "debug1", "Execute command exit($code): $err");
    printLog($ctx, "debug3", "Execute command result: $out") if(defined $out && $out ne "");
    
    return ($code, $err, $out);
}


sub detectVirtualization
{
    my $ctx  = shift;
    my $code = 1;
    my $err  = "";
    my $val  = "";
    
    if(-d "/proc/xen")
    {
        printLog($ctx, "debug2", "Found XEN");

        if(-e "/proc/xen/xsd_port")
        {
            printLog($ctx, "debug2", "We are Domain-0");

            my $xend_running=0;
            ($code, $err, $val) = executeCommand($ctx, "/etc/init.d/xend", undef, ("status"));
            if($code == 0)
            {
                $xend_running = 1;
            }
            
            # we are Domain-0 ; xenstored is required in domain 0 for xenstore-write...
            if($xend_running && -e $ctx->{xenstorewrite} && -e $ctx->{xenstorechmod}) 
            {
                printLog($ctx, "debug1", "Write /tool/SR/HostDeviceID to xenbus");
                
                executeCommand($ctx, $ctx->{xenstorewrite}, undef, ("/tool/SR/HostDeviceID", "$ctx->{guid}"));
                executeCommand($ctx, $ctx->{xenstorechmod}, undef, ("/tool/SR/HostDeviceID", "r"));
            }
        }
        else
        {
            $ctx->{virtType} = "Xen";
            $ctx->{hostGUID} = "Y";
            
            if(-e $ctx->{xenstoreread})
            {
                printLog($ctx, "debug2", "try to read /tool/SR/HostDeviceID from xenbus");
            
                ($code, $err, $val) = executeCommand($ctx, $ctx->{xenstoreread}, undef, ("/tool/SR/HostDeviceID"));
                if(defined $val && $val ne "") 
                {
                    printLog($ctx, "debug2", "Got /tool/SR/HostDeviceID: $val");
                    
                    $ctx->{hostGUID} = $val;
                }
            }
        }
    }
    elsif( -x $ctx->{lscpu})
    {
        my ($code, $err, $out) = executeCommand($ctx, $ctx->{lscpu}, undef);
        if($code != 0)
        {
            printLog($ctx, "warn", "Cannot execute lscpu: $code $err");
        }
        else
        {
            foreach my $line (split(/\n/, $out))
            {
                if($line =~ /^Hypervisor vendor:\s*(.*)\s*$/ && defined $1)
                {
                    $ctx->{virtType} = "$1";
                    $ctx->{hostGUID} = "Y";
                    last;
                }
            }
        }
        
        if(!defined $ctx->{virtType} || $ctx->{virtType} eq "")
        {
            # search for fallback.
            if(defined $ctx->{FallbackHostGUID} && $ctx->{FallbackHostGUID} ne "")
            {
                $ctx->{virtType} = "suseRegister.conf";
                $ctx->{hostGUID} = $ctx->{FallbackHostGUID};
            }
        }
    }
    
    return (0, "");
}

#
# FIXME: old curl variant. Remove this function if the new one works.
#
sub sendDataCurl
{
    my $ctx = shift;
    my $url  = shift || undef;
    my $data = shift || undef;
    
    $ctx->{timeindent}++;
    
    my $curlErr = 0;
    my $res = "";
    my $err = "";
    my %header = ();
    my $code = "";
    my $mess = "";
    
    if (! defined $url)
    {
        logPrintError($ctx, "Cannot send data to registration server. Missing URL.\n", 14);
        return;
    }
    if($url =~ /^-/)
    {
        logPrintError($ctx, "Invalid protocol($url).\n", 15);
        return;
    }

    my $uri = URI->new($url);
    
    if(!defined $uri->host || $uri->host !~ /$ctx->{initialDomain}$/)
    {
        logPrintError($ctx, "Invalid URL($url). Data could only be send to $ctx->{initialDomain} .\n", 15);
        return;
    }
    if(!defined $uri->scheme || $uri->scheme ne "https")
    {
        logPrintError($ctx, "Invalid protocol($url). https is required.\n", 15);
        return;
    }
    $url = $uri->as_string;
        
    if (! defined $data)
    {
        logPrintError($ctx, "Cannot send data. Missing data.\n", 14);
        return;
    }

    my @cmdArgs = ( "--capath", $ctx->{CA_PATH});

    my $fh = new File::Temp(TEMPLATE => 'dataXXXXX',
                            SUFFIX   => '.xml',
                            DIR      => '/tmp/');
    print $fh $data;

    push @cmdArgs, "--data", "@".$fh->filename();
    push @cmdArgs, "-i";
    push @cmdArgs, "--max-time", "130";

    foreach my $extraOpt (@{$ctx->{extraCurlOption}})
    {
        if($extraOpt =~ /^([\w-]+)[\s=]*(.*)/)
        {
            if(defined $1 && $1 ne "")
            {
                push @cmdArgs, $1;
                
                if(defined $2 && $2 ne "")
                {
                    push @cmdArgs, $2;
                }
            }
        }
    }
    
    push @cmdArgs, "$url";

    printLog($ctx, "debug2", "Call $ctx->{curl} ".join(" ", @cmdArgs));
    printLog($ctx, "debug2", "SEND DATA to URI: $url:");
    printLog($ctx, "debug2", "$data");

    printLog($ctx, "info", "\nSEND DATA to URI: $url:", 0, 1);
    printLog($ctx, "info", "$data", 0, 1);

    if($ctx->{noproxy})
    {
        delete $ENV{'http_proxy'};
        delete $ENV{'HTTP_PROXY'};
        delete $ENV{'https_proxy'};
        delete $ENV{'HTTPS_PROXY'};
        delete $ENV{'ALL_PROXY'};
        delete $ENV{'all_proxy'};
    }
    $ENV{'PATH'} = '/bin:/usr/bin:/sbin:/usr/sbin:/opt/kde3/bin/:/opt/gnome/bin/';

    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{curl}, @cmdArgs) or do {
        logPrintError($ctx, "Cannot execute $ctx->{curl} ".join(" ", @cmdArgs).": $!\n",13);
        return;
    };

    my $foundBody = 0;
    while (<OUT>)
    {
        $res = "" if(! defined $res);
        if ($foundBody)
        {
            $res .= "$_";
        }
        elsif ($_ =~ /^HTTP\/\d\.\d\s(\d+)\s(.*)$/)
        {
            if (defined $1 && $1 ne "")
            {
                $code = $1;
            }
            if (defined $2 && $2 ne "")
            {
                $mess = $2;
            }
        }
        elsif ($_ =~ /^[\w-]+:\s/)
        {
            my ($key, $val) = split(/: /, $_, 2);
            $header{$key} = $val;
        }
        elsif ($_ =~ /^\s*</)
        {
            $foundBody = 1;
            $res .= "$_";
        }
    }
    while (<ERR>)
    {
        $err .= "$_";
    }
    close OUT;
    close ERR;
    close IN;
    waitpid $pid, 0;

    $curlErr = ($?>>8);

    printLog($ctx, "debug2", "CURL RETURN WITH: $curlErr");
    printLog($ctx, "debug2", "RECEIVED DATA:");
    printLog($ctx, "debug2", "CODE: $code MESSAGE:  $mess");
    printLog($ctx, "debug2", "HEADER: ".Data::Dumper->Dump([\%header]));
    printLog($ctx, "debug2", "BODY:  $res");
    
    printLog($ctx, "info", "RECEIVED DATA:",0,1);
    printLog($ctx, "info", "CURL RETURN WITH: $curlErr",0,1);
    printLog($ctx, "info", "CODE: $code MESSAGE:  $mess",0,1);
    printLog($ctx, "info", "HEADER: ".Data::Dumper->Dump([\%header]),0,1);
    printLog($ctx, "info", "BODY:  $res",0,1);

    if ($curlErr != 0)
    {
        logPrintError($ctx, "Execute curl command failed with '$curlErr': $err", 4);
        return $res;
    }

    if ($code >= 300 && exists $header{Location} && defined $header{Location})
    {
        if ($ctx->{redirects} > 5)
        {
            logPrintError($ctx, "Too many redirects. Aborting.\n", 5);
            return $res;
        }
        $ctx->{redirects}++;
        
        my $loc = $header{Location};

        local $/ = "\r\n";
        chomp($loc);
        local $/ = "\n";

        #print STDERR "sendData(redirect): ".(tv_interval($t0))."\n" if($ctx->{time});

        $res = sendData($ctx, $loc, $data);
    }
    elsif($code < 200 || $code >= 300) 
    {
        my $b = "";
        my @c = ();

        if(-e "/usr/bin/lynx")
        {
            $b = "/usr/bin/lynx";
            push @c, "-dump", "-stdin";
        }
        elsif(-e "/usr/bin/w3m") 
        {
            $b = "/usr/bin/w3m";
            push @c, "-dump", "-T", "text/html";
        }
        
        my $out = "";
        if(-x $b)
        {
            my $pid = open3(\*IN, \*OUT, \*ERR, $b, @c) or do
            {
                logPrintError($ctx, "Cannot execute $b ".join(" ", @c).": $!\n",13);
                return undef;
            };
            
            print IN $res;
            close IN;
            
            while (<OUT>)
            {
                $out .= "$_";
            }
            #chomp($msg) if(defined $msg && $msg ne "");
            while (<ERR>)
            {
                $out .= "$_";
            }
            close OUT;
            close ERR;
            waitpid $pid, 0;
            chomp($out) if(defined $out && $out ne "");
            $out .="\n";
        }
        $out .= "$mess\n";
        
        logPrintError($ctx, "ERROR: $code: $out\n", 2);
    }

    $ctx->{timeindent}--;
    
    return $res;
}


1;
