package SUSE::Parser::Product;
use strict;
use XML::Parser;
use SUSE::SRPrivate;

# The handler is called with something like this: 
#
# $VAR1 = {
#           NAME       = 'SUSE_SLES',
#           VERSION    = '11',
#           RELEASE    = '0',
#           EPOCH      = '0',
#           ARCH       = 'i586',
#           VENDOR     = 'Novell',
#           SUMMARY    = '...',
#           FLAVOR     = 'DVD',
#           TYPE       = 'base',
#           REPO       = '@System',
#           INSTALLED  = "1",
#           DESCRIPTION = '.......
#                          .......'
#         };


# constructor
sub new
{
    my $pkgname = shift;
    my %opt   = @_;
    my $self  = {};

    $self->{CURRENT}   = {};
    $self->{ELEMENT}   = "";
    $self->{CTX}       = undef;
    $self->{HANDLER}   = undef;

    if(exists $opt{ctx} && defined $opt{ctx} && $opt{ctx})
    {
        $self->{CTX} = $opt{ctx};
    }
    bless($self);
    return $self;
}

# parses a xml resource
sub parse()
{
    my $self     = shift;
    my $xml      = shift;
    my $handler  = shift;

    $self->{HANDLER} = $handler;

    if (!defined $xml)
    {
        SUSE::SRPrivate::printLog($self->{CTX}, "error", "Invalid content");
        return 1;
    }

    my $parser = XML::Parser->new( Handlers =>
                                   {
                                    Start=> sub { handle_start_tag($self, @_) },
                                    Char => sub { handle_char_tag($self, @_) },
                                    End=> sub { handle_end_tag($self, @_) },
                                   });

    eval
    {
        $parser->parse( $xml );
    };
    if ($@) {
        # ignore the errors, but print them
        chomp($@);
        SUSE::SRPrivate::printLog($self->{CTX}, "warn", "SUSE::Parser::RepoList Invalid XML \n$xml\n: $@");
    }
    return 0;
}

# handles XML reader start tag events
sub handle_start_tag()
{
    my $self = shift;
    my( $expat, $element, %attrs ) = @_;

    if(lc($element) eq "product")
    {
        $self->{ELEMENT} = "product";
        
        foreach my $key (keys %attrs)
        {
            $self->{CURRENT}->{uc($key)} = "$attrs{$key}";
        }
        
    }
    elsif(lc($element) eq "description")
    {
        $self->{ELEMENT} = "description";
        $self->{CURRENT}->{DESCRIPTION} = "";
    }
}

sub handle_char_tag
{
    my $self = shift;
    my( $expat, $string) = @_;

    chomp($string);
    return if($string =~ /^\s*$/);
    $string =~ s/^\s*//;
    $string =~ s/\s*$//;
    
    return if(! defined $self->{ELEMENT});

    if($self->{ELEMENT} eq "description")
    {
        $self->{CURRENT}->{DESCRIPTION} .= $string;
    }
}

sub handle_end_tag
{
    my( $self, $expat, $element ) = @_;

    if(lc($element) eq "product")
    {
        # first call the callback
        $self->{HANDLER}->($self->{CURRENT});

        $self->{ELEMENT} = "";
        $self->{CURRENT} = {};
    }
}

1;
