/*
 * kgraft_patch_bsc1107832
 *
 * Fix for CVE-2018-14633, bsc#1107832
 *
 *  Upstream commit:
 *  none yet
 *
 *  SLE12(-SP1) commit:
 *  none yet
 *
 *  SLE12-SP2 commit:
 *  none yet
 *
 *  SLE12-SP3 commit:
 *  none yet
 *
 *  SLE15 commit:
 *  none yet
 *
 *
 *  Copyright (c) 2018 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <target/iscsi/iscsi_target_core.h>
#include <crypto/hash.h>
#include <linux/slab.h>
#include "kgr_patch_bsc1107832.h"
#include "kallsyms_relocs.h"

#if !IS_MODULE(CONFIG_ISCSI_TARGET)
#error "Live patch supports only CONFIG_ISCSI_TARGET=m"
#endif

#if !IS_ENABLED(CONFIG_CRYPTO_HASH2) || IS_MODULE(CONFIG_CRYPTO_HASH2)
#error "Live patch supports only CONFIG_CRYPTO_HASH2=y"
#endif


#define KGR_PATCHED_MODULE "iscsi_target_mod"


static int (*kgr_extract_param)(const char *in_buf, const char *pattern,
				unsigned int max_length, char *out_buf,
				unsigned char *type);
static void (*kgr_convert_null_to_semi)(char *buf, int len);

static struct kgr_kallsyms_reloc kgr_funcs[] = {
	{ "extract_param", (void *)&kgr_extract_param, "iscsi_target_mod" },
	{ "convert_null_to_semi", (void *)&kgr_convert_null_to_semi,
	  "iscsi_target_mod" },
};



/* from drivers/target/iscsi/iscsi_target_auth.h */
#define KGR_CHAP_DIGEST_UNKNOWN	0
#define KGR_CHAP_DIGEST_MD5		5

#define KGR_CHAP_CHALLENGE_LENGTH	16
#define KGR_CHAP_CHALLENGE_STR_LEN	4096
#define KGR_MAX_RESPONSE_LENGTH	64	/* sufficient for MD5 */
#define	KGR_MAX_CHAP_N_SIZE		512

#define KGR_MD5_SIGNATURE_SIZE	16	/* 16 bytes in a MD5 message digest */

#define KGR_CHAP_STAGE_SERVER_AIC	2
#define KGR_CHAP_STAGE_SERVER_NR	5

struct iscsi_chap {
	unsigned char	digest_type;
	unsigned char	id;
	unsigned char	challenge[KGR_CHAP_CHALLENGE_LENGTH];
	unsigned int	authenticate_target;
	unsigned int	chap_state;
} ____cacheline_aligned;


/* from drivers/target/iscsi/iscsi_target_nego.h */
#define KGR_HEX             1


/* from drivers/target/iscsi/iscsi_target_auth.c */
/* inlined */
static void kgr_chap_binaryhex_to_asciihex(char *dst, char *src, int src_len)
{
	int i;

	for (i = 0; i < src_len; i++) {
		sprintf(&dst[i*2], "%02x", (int) src[i] & 0xff);
	}
}

/* inlined */
static void kgr_chap_gen_challenge(
	struct iscsi_conn *conn,
	int caller,
	char *c_str,
	unsigned int *c_len)
{
	unsigned char challenge_asciihex[KGR_CHAP_CHALLENGE_LENGTH * 2 + 1];
	struct iscsi_chap *chap = conn->auth_protocol;

	memset(challenge_asciihex, 0, KGR_CHAP_CHALLENGE_LENGTH * 2 + 1);

	get_random_bytes(chap->challenge, KGR_CHAP_CHALLENGE_LENGTH);
	kgr_chap_binaryhex_to_asciihex(challenge_asciihex, chap->challenge,
				KGR_CHAP_CHALLENGE_LENGTH);
	/*
	 * Set CHAP_C, and copy the generated challenge into c_str.
	 */
	*c_len += sprintf(c_str + *c_len, "CHAP_C=0x%s", challenge_asciihex);
	*c_len += 1;

	pr_debug("[%s] Sending CHAP_C=0x%s\n\n", (caller) ? "server" : "client",
			challenge_asciihex);
}

/* inlined */
static int kgr_chap_check_algorithm(const char *a_str)
{
	char *tmp, *orig, *token;

	tmp = kstrdup(a_str, GFP_KERNEL);
	if (!tmp) {
		pr_err("Memory allocation failed for CHAP_A temporary buffer\n");
		return KGR_CHAP_DIGEST_UNKNOWN;
	}
	orig = tmp;

	token = strsep(&tmp, "=");
	if (!token)
		goto out;

	if (strcmp(token, "CHAP_A")) {
		pr_err("Unable to locate CHAP_A key\n");
		goto out;
	}
	while (token) {
		token = strsep(&tmp, ",");
		if (!token)
			goto out;

		if (!strncmp(token, "5", 1)) {
			pr_debug("Selected MD5 Algorithm\n");
			kfree(orig);
			return KGR_CHAP_DIGEST_MD5;
		}
	}
out:
	kfree(orig);
	return KGR_CHAP_DIGEST_UNKNOWN;
}

/* inlined */
static struct iscsi_chap *kgr_chap_server_open(
	struct iscsi_conn *conn,
	struct iscsi_node_auth *auth,
	const char *a_str,
	char *aic_str,
	unsigned int *aic_len)
{
	int ret;
	struct iscsi_chap *chap;

	if (!(auth->naf_flags & NAF_USERID_SET) ||
	    !(auth->naf_flags & NAF_PASSWORD_SET)) {
		pr_err("CHAP user or password not set for"
				" Initiator ACL\n");
		return NULL;
	}

	conn->auth_protocol = kzalloc(sizeof(struct iscsi_chap), GFP_KERNEL);
	if (!conn->auth_protocol)
		return NULL;

	chap = conn->auth_protocol;
	ret = kgr_chap_check_algorithm(a_str);
	switch (ret) {
	case KGR_CHAP_DIGEST_MD5:
		pr_debug("[server] Got CHAP_A=5\n");
		/*
		 * Send back CHAP_A set to MD5.
		*/
		*aic_len = sprintf(aic_str, "CHAP_A=5");
		*aic_len += 1;
		chap->digest_type = KGR_CHAP_DIGEST_MD5;
		pr_debug("[server] Sending CHAP_A=%d\n", chap->digest_type);
		break;
	case KGR_CHAP_DIGEST_UNKNOWN:
	default:
		pr_err("Unsupported CHAP_A value\n");
		return NULL;
	}

	/*
	 * Set Identifier.
	 */
	chap->id = conn->tpg->tpg_chap_id++;
	*aic_len += sprintf(aic_str + *aic_len, "CHAP_I=%d", chap->id);
	*aic_len += 1;
	pr_debug("[server] Sending CHAP_I=%d\n", chap->id);
	/*
	 * Generate Challenge.
	 */
	kgr_chap_gen_challenge(conn, 1, aic_str, aic_len);

	return chap;
}

/* inlined */
static void kgr_chap_close(struct iscsi_conn *conn)
{
	kfree(conn->auth_protocol);
	conn->auth_protocol = NULL;
}



/* patched, optimized */
static int kgr_chap_server_compute_md5(
	struct iscsi_conn *conn,
	struct iscsi_node_auth *auth,
	char *nr_in_ptr,
	char *nr_out_ptr,
	unsigned int *nr_out_len)
{
	unsigned long id;
	unsigned char id_as_uchar;
	unsigned char digest[KGR_MD5_SIGNATURE_SIZE];
	unsigned char type, response[KGR_MD5_SIGNATURE_SIZE * 2 + 2];
	unsigned char identifier[10], *challenge = NULL;
	unsigned char *challenge_binhex = NULL;
	unsigned char client_digest[KGR_MD5_SIGNATURE_SIZE];
	unsigned char server_digest[KGR_MD5_SIGNATURE_SIZE];
	unsigned char chap_n[KGR_MAX_CHAP_N_SIZE], chap_r[KGR_MAX_RESPONSE_LENGTH];
	size_t compare_len;
	struct iscsi_chap *chap = conn->auth_protocol;
	struct crypto_shash *tfm = NULL;
	struct shash_desc *desc = NULL;
	int auth_ret = -1, ret, challenge_len;

	memset(identifier, 0, 10);
	memset(chap_n, 0, KGR_MAX_CHAP_N_SIZE);
	memset(chap_r, 0, KGR_MAX_RESPONSE_LENGTH);
	memset(digest, 0, KGR_MD5_SIGNATURE_SIZE);
	memset(response, 0, KGR_MD5_SIGNATURE_SIZE * 2 + 2);
	memset(client_digest, 0, KGR_MD5_SIGNATURE_SIZE);
	memset(server_digest, 0, KGR_MD5_SIGNATURE_SIZE);

	challenge = kzalloc(KGR_CHAP_CHALLENGE_STR_LEN, GFP_KERNEL);
	if (!challenge) {
		pr_err("Unable to allocate challenge buffer\n");
		goto out;
	}

	challenge_binhex = kzalloc(KGR_CHAP_CHALLENGE_STR_LEN, GFP_KERNEL);
	if (!challenge_binhex) {
		pr_err("Unable to allocate challenge_binhex buffer\n");
		goto out;
	}
	/*
	 * Extract CHAP_N.
	 */
	if (kgr_extract_param(nr_in_ptr, "CHAP_N", KGR_MAX_CHAP_N_SIZE, chap_n,
				&type) < 0) {
		pr_err("Could not find CHAP_N.\n");
		goto out;
	}
	if (type == KGR_HEX) {
		pr_err("Could not find CHAP_N.\n");
		goto out;
	}

	/* Include the terminating NULL in the compare */
	compare_len = strlen(auth->userid) + 1;
	if (strncmp(chap_n, auth->userid, compare_len) != 0) {
		pr_err("CHAP_N values do not match!\n");
		goto out;
	}
	pr_debug("[server] Got CHAP_N=%s\n", chap_n);
	/*
	 * Extract CHAP_R.
	 */
	if (kgr_extract_param(nr_in_ptr, "CHAP_R", KGR_MAX_RESPONSE_LENGTH, chap_r,
				&type) < 0) {
		pr_err("Could not find CHAP_R.\n");
		goto out;
	}
	if (type != KGR_HEX) {
		pr_err("Could not find CHAP_R.\n");
		goto out;
	}
	/*
	 * Fix CVE-2018-14633
	 *  +8 lines
	 */
	if (strlen(chap_r) != KGR_MD5_SIGNATURE_SIZE * 2) {
		pr_err("Malformed CHAP_R\n");
		goto out;
	}
	if (hex2bin(client_digest, chap_r, KGR_MD5_SIGNATURE_SIZE) < 0) {
		pr_err("Malformed CHAP_R\n");
		goto out;
	}

	pr_debug("[server] Got CHAP_R=%s\n", chap_r);
	/*
	 * Fix CVE-2018-14633
	 *  -1 line
	 */

	tfm = crypto_alloc_shash("md5", 0, 0);
	if (IS_ERR(tfm)) {
		tfm = NULL;
		pr_err("Unable to allocate struct crypto_shash\n");
		goto out;
	}

	desc = kmalloc(sizeof(*desc) + crypto_shash_descsize(tfm), GFP_KERNEL);
	if (!desc) {
		pr_err("Unable to allocate struct shash_desc\n");
		goto out;
	}

	desc->tfm = tfm;
	desc->flags = 0;

	ret = crypto_shash_init(desc);
	if (ret < 0) {
		pr_err("crypto_shash_init() failed\n");
		goto out;
	}

	ret = crypto_shash_update(desc, &chap->id, 1);
	if (ret < 0) {
		pr_err("crypto_shash_update() failed for id\n");
		goto out;
	}

	ret = crypto_shash_update(desc, (char *)&auth->password,
				  strlen(auth->password));
	if (ret < 0) {
		pr_err("crypto_shash_update() failed for password\n");
		goto out;
	}

	ret = crypto_shash_finup(desc, chap->challenge,
				 KGR_CHAP_CHALLENGE_LENGTH, server_digest);
	if (ret < 0) {
		pr_err("crypto_shash_finup() failed for challenge\n");
		goto out;
	}

	kgr_chap_binaryhex_to_asciihex(response, server_digest, KGR_MD5_SIGNATURE_SIZE);
	pr_debug("[server] MD5 Server Digest: %s\n", response);

	if (memcmp(server_digest, client_digest, KGR_MD5_SIGNATURE_SIZE) != 0) {
		pr_debug("[server] MD5 Digests do not match!\n\n");
		goto out;
	} else
		pr_debug("[server] MD5 Digests match, CHAP connection"
				" successful.\n\n");
	/*
	 * One way authentication has succeeded, return now if mutual
	 * authentication is not enabled.
	 */
	if (!auth->authenticate_target) {
		auth_ret = 0;
		goto out;
	}
	/*
	 * Get CHAP_I.
	 */
	if (kgr_extract_param(nr_in_ptr, "CHAP_I", 10, identifier, &type) < 0) {
		pr_err("Could not find CHAP_I.\n");
		goto out;
	}

	if (type == KGR_HEX)
		ret = kstrtoul(&identifier[2], 0, &id);
	else
		ret = kstrtoul(identifier, 0, &id);

	if (ret < 0) {
		pr_err("kstrtoul() failed for CHAP identifier: %d\n", ret);
		goto out;
	}
	if (id > 255) {
		pr_err("chap identifier: %lu greater than 255\n", id);
		goto out;
	}
	/*
	 * RFC 1994 says Identifier is no more than octet (8 bits).
	 */
	pr_debug("[server] Got CHAP_I=%lu\n", id);
	/*
	 * Get CHAP_C.
	 */
	if (kgr_extract_param(nr_in_ptr, "CHAP_C", KGR_CHAP_CHALLENGE_STR_LEN,
			challenge, &type) < 0) {
		pr_err("Could not find CHAP_C.\n");
		goto out;
	}

	if (type != KGR_HEX) {
		pr_err("Could not find CHAP_C.\n");
		goto out;
	}
	/*
	 * Fix CVE-2018-14633
	 *  -3 lines, +1 line
	 */
	challenge_len = DIV_ROUND_UP(strlen(challenge), 2);
	if (!challenge_len) {
		pr_err("Unable to convert incoming challenge\n");
		goto out;
	}
	if (challenge_len > 1024) {
		pr_err("CHAP_C exceeds maximum binary size of 1024 bytes\n");
		goto out;
	}
	/*
	 * Fix CVE-2018-14633
	 *  +5 lines
	 */
	if (hex2bin(challenge_binhex, challenge, challenge_len) < 0) {
		pr_err("Malformed CHAP_C\n");
		goto out;
	}
	pr_debug("[server] Got CHAP_C=%s\n", challenge);
	/*
	 * During mutual authentication, the CHAP_C generated by the
	 * initiator must not match the original CHAP_C generated by
	 * the target.
	 */
	if (!memcmp(challenge_binhex, chap->challenge, KGR_CHAP_CHALLENGE_LENGTH)) {
		pr_err("initiator CHAP_C matches target CHAP_C, failing"
		       " login attempt\n");
		goto out;
	}
	/*
	 * Generate CHAP_N and CHAP_R for mutual authentication.
	 */
	ret = crypto_shash_init(desc);
	if (ret < 0) {
		pr_err("crypto_shash_init() failed\n");
		goto out;
	}

	/* To handle both endiannesses */
	id_as_uchar = id;
	ret = crypto_shash_update(desc, &id_as_uchar, 1);
	if (ret < 0) {
		pr_err("crypto_shash_update() failed for id\n");
		goto out;
	}

	ret = crypto_shash_update(desc, auth->password_mutual,
				  strlen(auth->password_mutual));
	if (ret < 0) {
		pr_err("crypto_shash_update() failed for"
				" password_mutual\n");
		goto out;
	}
	/*
	 * Convert received challenge to binary hex.
	 */
	ret = crypto_shash_finup(desc, challenge_binhex, challenge_len,
				 digest);
	if (ret < 0) {
		pr_err("crypto_shash_finup() failed for ma challenge\n");
		goto out;
	}

	/*
	 * Generate CHAP_N and CHAP_R.
	 */
	*nr_out_len = sprintf(nr_out_ptr, "CHAP_N=%s", auth->userid_mutual);
	*nr_out_len += 1;
	pr_debug("[server] Sending CHAP_N=%s\n", auth->userid_mutual);
	/*
	 * Convert response from binary hex to ascii hext.
	 */
	kgr_chap_binaryhex_to_asciihex(response, digest, KGR_MD5_SIGNATURE_SIZE);
	*nr_out_len += sprintf(nr_out_ptr + *nr_out_len, "CHAP_R=0x%s",
			response);
	*nr_out_len += 1;
	pr_debug("[server] Sending CHAP_R=0x%s\n", response);
	auth_ret = 0;
out:
	kzfree(desc);
	crypto_free_shash(tfm);
	kfree(challenge);
	kfree(challenge_binhex);
	return auth_ret;
}

/* patched, inlined, calls optimized chap_server_compute_md5() */
static int kgr_chap_got_response(
	struct iscsi_conn *conn,
	struct iscsi_node_auth *auth,
	char *nr_in_ptr,
	char *nr_out_ptr,
	unsigned int *nr_out_len)
{
	struct iscsi_chap *chap = conn->auth_protocol;

	switch (chap->digest_type) {
	case KGR_CHAP_DIGEST_MD5:
		if (kgr_chap_server_compute_md5(conn, auth, nr_in_ptr,
				nr_out_ptr, nr_out_len) < 0)
			return -1;
		return 0;
	default:
		pr_err("Unknown CHAP digest type %d!\n",
				chap->digest_type);
		return -1;
	}
}

/* patched, calls inlined chap_got_response() */
u32 kgr_chap_main_loop(
	struct iscsi_conn *conn,
	struct iscsi_node_auth *auth,
	char *in_text,
	char *out_text,
	int *in_len,
	int *out_len)
{
	struct iscsi_chap *chap = conn->auth_protocol;

	if (!chap) {
		chap = kgr_chap_server_open(conn, auth, in_text, out_text, out_len);
		if (!chap)
			return 2;
		chap->chap_state = KGR_CHAP_STAGE_SERVER_AIC;
		return 0;
	} else if (chap->chap_state == KGR_CHAP_STAGE_SERVER_AIC) {
		kgr_convert_null_to_semi(in_text, *in_len);
		if (kgr_chap_got_response(conn, auth, in_text, out_text,
				out_len) < 0) {
			kgr_chap_close(conn);
			return 2;
		}
		if (auth->authenticate_target)
			chap->chap_state = KGR_CHAP_STAGE_SERVER_NR;
		else
			*out_len = 0;
		kgr_chap_close(conn);
		return 1;
	}

	return 2;
}



static int kgr_patch_bsc1107832_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, KGR_PATCHED_MODULE))
		return 0;

	ret = __kgr_resolve_kallsyms_relocs(kgr_funcs, ARRAY_SIZE(kgr_funcs));
	WARN(ret, "kgraft-patch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block kgr_patch_bsc1107832_module_nb = {
	.notifier_call = kgr_patch_bsc1107832_module_notify,
	.priority = INT_MIN+1,
};

int kgr_patch_bsc1107832_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(KGR_PATCHED_MODULE)) {
		ret = __kgr_resolve_kallsyms_relocs(kgr_funcs,
						    ARRAY_SIZE(kgr_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&kgr_patch_bsc1107832_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void kgr_patch_bsc1107832_cleanup(void)
{
	unregister_module_notifier(&kgr_patch_bsc1107832_module_nb);
}
