/*
 * kgraft_patch_bsc1099306_irq
 *
 * Fix for CVE-2018-3646 (irq_enter() part), bsc#1099306
 *
 *  Copyright (c) 2018 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_X86_64) && IS_ENABLED(CONFIG_KVM)

#include <linux/kernel.h>
#include <linux/kallsyms.h>
#include <linux/rcupdate.h>
#include <linux/sched.h>
#include <linux/preempt.h>
#include <linux/tick.h>
#include <linux/bottom_half.h>
#include <linux/vtime.h>
#include <linux/hardirq.h>
#include "kgr_patch_bsc1099306_irq.h"
#include "bsc1099306.h"


#if !IS_ENABLED(CONFIG_TICK_ONESHOT)
#error "Live patch supports only CONFIG_TICK_ONESHOT=y."
#endif

static void (*kgr_rcu_irq_enter)(void);
static void (*kgr_tick_irq_enter)(void);

static struct {
	char *name;
	void **addr;
} kgr_funcs[] = {
	{ "rcu_irq_enter", (void *)&kgr_rcu_irq_enter },
	{ "tick_irq_enter", (void *)&kgr_tick_irq_enter },
};


DEFINE_PER_CPU(bool, kgr_kvm_cpu_l1tf_flush_l1d);


/* patched */
void kgr_irq_enter(void)
{
	kgr_rcu_irq_enter();
	if (is_idle_task(current) && !in_interrupt()) {
		/*
		 * Prevent raise_softirq from needlessly waking up ksoftirqd
		 * here, as softirq will be serviced on return from interrupt.
		 */
		local_bh_disable();
		kgr_tick_irq_enter();
		_local_bh_enable();
	}

	__irq_enter();

	/*
	 * Fix CVE-2018-3646
	 *  +1 line
	 */
	__this_cpu_write(kgr_kvm_cpu_l1tf_flush_l1d, true);
}


static int kgr_patch_bsc1099306_irq_kallsyms(void)
{
	unsigned long addr;
	int i;

	for (i = 0; i < ARRAY_SIZE(kgr_funcs); i++) {
		/* mod_find_symname would be nice, but it is not exported */
		addr = kallsyms_lookup_name(kgr_funcs[i].name);
		if (!addr) {
			pr_err("kgraft-patch: symbol %s not resolved\n",
				kgr_funcs[i].name);
			return -ENOENT;
		}

		*(kgr_funcs[i].addr) = (void *)addr;
	}

	return 0;
}

int __kgr_patch_bsc1099306_irq_init(void)
{
	return kgr_patch_bsc1099306_irq_kallsyms();
}

#endif
