/*
 * kgraft_patch_bsc1090036
 *
 * Fix for CVE-2018-1000199, bsc#1090036
 *
 *  Upstream commit:
 *  f67b15037a7a ("perf/hwbp: Simplify the perf-hwbp code, fix documentation")
 *
 *  SLE12(-SP1) commit:
 *  none yet
 *
 *  SLE12-SP2 commit:
 *  none yet
 *
 *  SLE12-SP3 commit:
 *  d5ffd3248b4feeff780e90c521527e6e6eb04c1b ("Linux 4.4.127")
 *
 *  Copyright (c) 2018 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/kallsyms.h>
#include <linux/irqflags.h>
#include <linux/perf_event.h>
#include "kgr_patch_bsc1090036.h"


static int (*kgr__perf_event_disable)(void *info);
static int (*kgr_validate_hw_breakpoint)(struct perf_event *bp);

static struct {
	char *name;
	char **addr;
} kgr_funcs[] = {
	{ "__perf_event_disable", (void *)&kgr__perf_event_disable },
	{ "validate_hw_breakpoint", (void *)&kgr_validate_hw_breakpoint },
};


/* patched */
int kgr_modify_user_hw_breakpoint(struct perf_event *bp, struct perf_event_attr *attr)
{
	/*
	 * Fix CVE-2018-1000199
	 *  -4 lines
	 */

	/*
	 * modify_user_hw_breakpoint can be invoked with IRQs disabled and hence it
	 * will not be possible to raise IPIs that invoke __perf_event_disable.
	 * So call the function directly after making sure we are targeting the
	 * current task.
	 */
	if (irqs_disabled() && bp->ctx && bp->ctx->task == current)
		kgr__perf_event_disable(bp);
	else
		perf_event_disable(bp);

	bp->attr.bp_addr = attr->bp_addr;
	bp->attr.bp_type = attr->bp_type;
	bp->attr.bp_len = attr->bp_len;
	/*
	 * Fix CVE-2018-1000199
	 *  +1 line
	 */
	bp->attr.disabled = 1;

	/*
	 * Fix CVE-2018-1000199
	 *  -19 lines, +9 lines
	 */
	if (!attr->disabled) {
		int err = kgr_validate_hw_breakpoint(bp);

		if (err)
			return err;

		perf_event_enable(bp);
		bp->attr.disabled = 0;
	}

	return 0;
}


static int kgr_patch_bsc1090036_kallsyms(void)
{
	unsigned long addr;
	int i;

	for (i = 0; i < ARRAY_SIZE(kgr_funcs); i++) {
		addr = kallsyms_lookup_name(kgr_funcs[i].name);
		if (!addr) {
			pr_err("kgraft-patch: symbol %s not resolved\n",
				kgr_funcs[i].name);
			return -ENOENT;
		}

		*(kgr_funcs[i].addr) = (void *)addr;
	}

	return 0;
}

int kgr_patch_bsc1090036_init(void)
{
	return kgr_patch_bsc1090036_kallsyms();
}
