/*
 * kgraft_patch_bsc1105323
 *
 * Fix for CVE-2018-10902, bsc#1105323
 *
 *  Upstream commit:
 *  39675f7a7c7e ("ALSA: rawmidi: Change resized buffers atomically")
 *
 *  SLE12(-SP1) commit:
 *  97b2705c10e83b61cfd8089e3afabae8c84210e4
 *
 *  SLE12-SP2 commit:
 *  57d39eb79ecfb8dbad9fbc3ed50f8d3a2c487275
 *
 *  SLE12-SP3 commit:
 *  b9f77c7a236b7acdbe73a953557bdbf731de63ee ("Linux 4.4.144")
 *
 *  SLE15 commit:
 *  70b340bfd49f4e6328df613540bfea6f6068ff2a
 *
 *
 *  Copyright (c) 2018 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#if IS_ENABLED(CONFIG_SND_RAWMIDI)

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/slab.h>
#include <sound/rawmidi.h>
#include "kgr_patch_bsc1105323.h"
#include "kallsyms_relocs.h"

#if !IS_MODULE(CONFIG_SND_RAWMIDI)
#error "Live patch supports only CONFIG_SND_RAWMIDI=m"
#endif

#define KGR_PATCHED_MODULE "snd_rawmidi"


static int
(*kgr_snd_rawmidi_drain_output)(struct snd_rawmidi_substream *substream);

static int
(*kgr_snd_rawmidi_drain_input)(struct snd_rawmidi_substream *substream);

static struct kgr_kallsyms_reloc kgr_funcs[] = {
	{ "snd_rawmidi_drain_output", (void *)&kgr_snd_rawmidi_drain_output,
	  "snd_rawmidi" },
	{ "snd_rawmidi_drain_input", (void *)&kgr_snd_rawmidi_drain_input ,
	  "snd_rawmidi" },
};



/* patched */
int kgr_snd_rawmidi_output_params(struct snd_rawmidi_substream *substream,
				  struct snd_rawmidi_params * params)
{
	/*
	 * Fix CVE-2018-10902
	 *  -1 line, +1 line
	 */
	char *newbuf, *oldbuf;
	struct snd_rawmidi_runtime *runtime = substream->runtime;

	if (substream->append && substream->use_count > 1)
		return -EBUSY;
	kgr_snd_rawmidi_drain_output(substream);
	if (params->buffer_size < 32 || params->buffer_size > 1024L * 1024L) {
		return -EINVAL;
	}
	if (params->avail_min < 1 || params->avail_min > params->buffer_size) {
		return -EINVAL;
	}
	if (params->buffer_size != runtime->buffer_size) {
		/*
		 * Fix CVE-2018-10902
		 *  -2 lines, +1 line
		 */
		newbuf = kmalloc(params->buffer_size, GFP_KERNEL);
		if (!newbuf)
			return -ENOMEM;
		/*
		 * Fix CVE-2018-10902
		 *  +2 lines
		 */
		spin_lock_irq(&runtime->lock);
		oldbuf = runtime->buffer;
		runtime->buffer = newbuf;
		runtime->buffer_size = params->buffer_size;
		runtime->avail = runtime->buffer_size;
		/*
		 * Fix CVE-2018-10902
		 *  +3 lines
		 */
		runtime->appl_ptr = runtime->hw_ptr = 0;
		spin_unlock_irq(&runtime->lock);
		kfree(oldbuf);
	}
	runtime->avail_min = params->avail_min;
	substream->active_sensing = !params->no_active_sensing;
	return 0;
}

/* patched */
int kgr_snd_rawmidi_input_params(struct snd_rawmidi_substream *substream,
				 struct snd_rawmidi_params * params)
{
	/*
	 * Fix CVE-2018-10902
	 *  -1 line, +1 line
	 */
	char *newbuf, *oldbuf;
	struct snd_rawmidi_runtime *runtime = substream->runtime;

	kgr_snd_rawmidi_drain_input(substream);
	if (params->buffer_size < 32 || params->buffer_size > 1024L * 1024L) {
		return -EINVAL;
	}
	if (params->avail_min < 1 || params->avail_min > params->buffer_size) {
		return -EINVAL;
	}
	if (params->buffer_size != runtime->buffer_size) {
		/*
		 * Fix CVE-2018-10902
		 *  -2 lines, +1 line
		 */
		newbuf = kmalloc(params->buffer_size, GFP_KERNEL);
		if (!newbuf)
			return -ENOMEM;
		/*
		 * Fix CVE-2018-10902
		 *  +2 lines
		 */
		spin_lock_irq(&runtime->lock);
		oldbuf = runtime->buffer;
		runtime->buffer = newbuf;
		runtime->buffer_size = params->buffer_size;
		/*
		 * Fix CVE-2018-10902
		 *  +3 lines
		 */
		runtime->appl_ptr = runtime->hw_ptr = 0;
		spin_unlock_irq(&runtime->lock);
		kfree(oldbuf);
	}
	runtime->avail_min = params->avail_min;
	return 0;
}



static int kgr_patch_bsc1105323_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, KGR_PATCHED_MODULE))
		return 0;

	ret = __kgr_resolve_kallsyms_relocs(kgr_funcs, ARRAY_SIZE(kgr_funcs));
	WARN(ret, "kgraft-patch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block kgr_patch_bsc1105323_module_nb = {
	.notifier_call = kgr_patch_bsc1105323_module_notify,
	.priority = INT_MIN+1,
};

int kgr_patch_bsc1105323_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(KGR_PATCHED_MODULE)) {
		ret = __kgr_resolve_kallsyms_relocs(kgr_funcs,
						    ARRAY_SIZE(kgr_funcs));
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&kgr_patch_bsc1105323_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void kgr_patch_bsc1105323_cleanup(void)
{
	unregister_module_notifier(&kgr_patch_bsc1105323_module_nb);
}

#endif /* IS_ENABLED(CONFIG_SND_RAWMIDI) */
