/*
 * kgraft_patch_bsc1062847
 *
 * Fix for stability issue bsc#1062847
 *
 *  Upstream commit:
 *  16cf72bb0856 ("team: call netdev_change_features out of team lock")
 *
 *  SLE12-SP2 commit:
 *  0de88b51b82c50c788d87415a76ec27afa163bbb
 *
 *  SLE12-SP3 commit:
 *  9c3153601a5b02d9812a5f762f474032e1a96b96
 *
 *  Copyright (c) 2017 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/kallsyms.h>
#include <linux/if_team.h>
#include <linux/netdevice.h>
#include <linux/if_vlan.h>
#include <linux/etherdevice.h>
#include <linux/netpoll.h>
#include "kgr_patch_bsc1062847.h"

#if !IS_MODULE(CONFIG_NET_TEAM)
#error "KGR patch supports only CONFIG_NET_TEAM=m."
#endif

#if !IS_ENABLED(CONFIG_NET_POLL_CONTROLLER)
#error "KGR patch supports only CONFIG_NET_POLL_CONTROLLER=y."
#endif

#define KGR_PATCHED_MODULE "team"

static const struct team_option (*kgr_team_options)[];
static const size_t kgr_team_options_array_size = 10;

static void (*kgr___team_compute_features)(struct team *team);
static int (*kgr__team_option_inst_add)(struct team *team,
					struct team_option *option,
					struct team_port *port);
static void (*kgr__team_option_inst_del_port)(struct team *team,
					      struct team_port *port);
static void (*kgr_team_adjust_ops)(struct team *team);
static void (*kgr__team_queue_override_port_add)(struct team *team,
						struct team_port *port);
static void (*kgr__team_queue_override_enabled_check)(struct team *team);
static void (*kgr_team_notify_peers)(struct team *team);
static void (*kgr_team_mcast_rejoin)(struct team *team);
static void (*kgr__team_port_change_send)(struct team_port *port, bool linkup);
static void (*kgr__team_options_change_check)(struct team *team);
static void (*kgr_team_port_disable)(struct team *team, struct team_port *port);
static int (*kgr__set_port_dev_addr)(struct net_device *port_dev,
				     const unsigned char *dev_addr);
static int (*kgr__team_change_mode)(struct team *team,
				    const struct team_mode *new_mode);
static void (*kgr__team_options_unregister)(struct team *team,
					    const struct team_option *option,
					    size_t option_count);
static rx_handler_result_t (*kgr_team_handle_frame)(struct sk_buff **pskb);

static struct {
	char *name;
	void **addr;
} kgr_funcs[] = {
	{ "team:team_options", (void *)&kgr_team_options },
	{ "team:___team_compute_features",
		(void *)&kgr___team_compute_features },
	{ "team:__team_option_inst_add", (void *)&kgr__team_option_inst_add },
	{ "team:__team_option_inst_del_port",
		(void *)&kgr__team_option_inst_del_port },
	{ "team:team_adjust_ops", (void *)&kgr_team_adjust_ops },
	{ "team:__team_queue_override_port_add",
		(void *)&kgr__team_queue_override_port_add },
	{ "team:__team_queue_override_enabled_check",
		(void *)&kgr__team_queue_override_enabled_check },
	{ "team:team_notify_peers", (void *)&kgr_team_notify_peers },
	{ "team:team_mcast_rejoin", (void *)&kgr_team_mcast_rejoin },
	{ "team:__team_port_change_send", (void *)&kgr__team_port_change_send },
	{ "team:__team_options_change_check",
		(void *)&kgr__team_options_change_check },
	{ "team:team_port_disable", (void *)&kgr_team_port_disable },
	{ "team:__set_port_dev_addr", (void *)&kgr__set_port_dev_addr },
	{ "team:__team_change_mode", (void *)&kgr__team_change_mode },
	{ "team:__team_options_unregister",
		(void *)&kgr__team_options_unregister },
	{ "team:team_handle_frame", (void *)&kgr_team_handle_frame },
};


/* from drivers/net/team/team.c */
struct team_option_inst { /* One for each option instance */
	struct list_head list;
	struct list_head tmp_list;
	struct team_option *option;
	struct team_option_inst_info info;
	bool changed;
	bool removed;
};

#define kgr_team_port_exists(dev) (dev->priv_flags & IFF_TEAM_PORT)

/* inlined */
static struct team_port *kgr_team_port_get_rtnl(const struct net_device *dev)
{
	struct team_port *port = rtnl_dereference(dev->rx_handler_data);

	return kgr_team_port_exists(dev) ? port : NULL;
}

/* inlined */
static int kgr_team_port_set_orig_dev_addr(struct team_port *port)
{
	return kgr__set_port_dev_addr(port->dev, port->orig.dev_addr);
}

/* inlined */
static inline bool kgr_team_port_enabled(struct team_port *port)
{
	return port->index != -1;
}

/* inlined */
static int kgr__team_option_inst_add_port(struct team *team,
					  struct team_port *port)
{
	struct team_option *option;
	int err;

	list_for_each_entry(option, &team->option_list, list) {
		if (!option->per_port)
			continue;
		err = kgr__team_option_inst_add(team, option, port);
		if (err)
			goto inst_del_port;
	}
	return 0;

inst_del_port:
	kgr__team_option_inst_del_port(team, port);
	return err;
}

/* inlined */
static void kgr__team_option_inst_mark_removed_port(struct team *team,
						    struct team_port *port)
{
	struct team_option_inst *opt_inst;

	list_for_each_entry(opt_inst, &team->option_inst_list, list) {
		if (opt_inst->info.port == port) {
			opt_inst->changed = true;
			opt_inst->removed = true;
		}
	}
}

/* inlined */
static void kgr_team_notify_peers_fini(struct team *team)
{
	cancel_delayed_work_sync(&team->notify_peers.dw);
}

/* inlined */
static void kgr_team_mcast_rejoin_fini(struct team *team)
{
	cancel_delayed_work_sync(&team->mcast_rejoin.dw);
}

/* inlined */
static void kgr_team_queue_override_fini(struct team *team)
{
	kfree(team->qom_lists);
}

/* inlined */
static void kgr_team_queue_override_port_add(struct team *team,
					struct team_port *port)
{
	kgr__team_queue_override_port_add(team, port);
	kgr__team_queue_override_enabled_check(team);
}

/* inlined */
static bool kgr_team_port_find(const struct team *team,
			       const struct team_port *port)
{
	struct team_port *cur;

	list_for_each_entry(cur, &team->port_list, list)
		if (cur == port)
			return true;
	return false;
}

/* optimized */
static void kgr_team_port_enable(struct team *team,
				 struct team_port *port)
{
	if (kgr_team_port_enabled(port))
		return;
	port->index = team->en_port_count++;
	hlist_add_head_rcu(&port->hlist,
			   team_port_index_hash(team, port->index));
	kgr_team_adjust_ops(team);
	kgr_team_queue_override_port_add(team, port);
	if (team->ops.port_enabled)
		team->ops.port_enabled(team, port);
	kgr_team_notify_peers(team);
	kgr_team_mcast_rejoin(team);
}

/* inlined */
static int kgr_team_port_enter(struct team *team, struct team_port *port)
{
	int err = 0;

	dev_hold(team->dev);
	if (team->ops.port_enter) {
		err = team->ops.port_enter(team, port);
		if (err) {
			netdev_err(team->dev, "Device %s failed to enter team mode\n",
				   port->dev->name);
			goto err_port_enter;
		}
	}

	return 0;

err_port_enter:
	dev_put(team->dev);

	return err;
}

/* inlined */
static void kgr_team_port_leave(struct team *team, struct team_port *port)
{
	if (team->ops.port_leave)
		team->ops.port_leave(team, port);
	dev_put(team->dev);
}

/* optimized */
static int kgr_team_port_enable_netpoll(struct team *team, struct team_port *port)
{
	struct netpoll *np;
	int err;

	if (!team->dev->npinfo)
		return 0;

	np = kzalloc(sizeof(*np), GFP_KERNEL);
	if (!np)
		return -ENOMEM;

	err = __netpoll_setup(np, port->dev);
	if (err) {
		kfree(np);
		return err;
	}
	port->np = np;
	return err;
}

/* optimized */
static void kgr_team_port_disable_netpoll(struct team_port *port)
{
	struct netpoll *np = port->np;

	if (!np)
		return;
	port->np = NULL;

	/* Wait for transmitting packets to finish before freeing. */
	synchronize_rcu_bh();
	__netpoll_cleanup(np);
	kfree(np);
}

/* inlined */
static int kgr_team_upper_dev_link(struct net_device *dev,
				   struct net_device *port_dev)
{
	int err;

	err = netdev_master_upper_dev_link(port_dev, dev);
	if (err)
		return err;
	port_dev->priv_flags |= IFF_TEAM_PORT;
	return 0;
}

/* inlined */
static void kgr_team_upper_dev_unlink(struct net_device *dev,
				      struct net_device *port_dev)
{
	netdev_upper_dev_unlink(port_dev, dev);
	port_dev->priv_flags &= ~IFF_TEAM_PORT;
}

/* inlined */
static void kgr_team_setup_by_port(struct net_device *dev,
				   struct net_device *port_dev)
{
	dev->header_ops	= port_dev->header_ops;
	dev->type = port_dev->type;
	dev->hard_header_len = port_dev->hard_header_len;
	dev->addr_len = port_dev->addr_len;
	dev->mtu = port_dev->mtu;
	memcpy(dev->broadcast, port_dev->broadcast, port_dev->addr_len);
	eth_hw_addr_inherit(dev, port_dev);
}

/* inlined */
static int kgr_team_dev_type_check_change(struct net_device *dev,
					  struct net_device *port_dev)
{
	struct team *team = netdev_priv(dev);
	char *portname = port_dev->name;
	int err;

	if (dev->type == port_dev->type)
		return 0;
	if (!list_empty(&team->port_list)) {
		netdev_err(dev, "Device %s is of different type\n", portname);
		return -EBUSY;
	}
	err = call_netdevice_notifiers(NETDEV_PRE_TYPE_CHANGE, dev);
	err = notifier_to_errno(err);
	if (err) {
		netdev_err(dev, "Refused to change device type\n");
		return err;
	}
	dev_uc_flush(dev);
	dev_mc_flush(dev);
	kgr_team_setup_by_port(dev, port_dev);
	call_netdevice_notifiers(NETDEV_POST_TYPE_CHANGE, dev);
	return 0;
}

/* optimized */
static void kgr__team_carrier_check(struct team *team)
{
	struct team_port *port;
	bool team_linkup;

	if (team->user_carrier_enabled)
		return;

	team_linkup = false;
	list_for_each_entry(port, &team->port_list, list) {
		if (port->linkup) {
			team_linkup = true;
			break;
		}
	}

	if (team_linkup)
		netif_carrier_on(team->dev);
	else
		netif_carrier_off(team->dev);
}

/* inlined */
static void kgr__team_port_change_port_added(struct team_port *port, bool linkup)
{
	kgr__team_port_change_send(port, linkup);
	kgr__team_carrier_check(port->team);
}

/* inlined */
static void kgr__team_port_change_port_removed(struct team_port *port)
{
	port->removed = true;
	kgr__team_port_change_send(port, false);
	kgr__team_carrier_check(port->team);
}



/* patched, inlined */
static void kgr__team_compute_features(struct team *team)
{
	/*
	 * Fix bsc#1062847
	 *  Former __team_compute_features() got dropped and
	 *  ___team_compute_features() renamed to
	 *  __team_compute_features().
	 *
	 *  No need to patch team_compute_features(). ___team_compute_features()
	 *  is still present in the running kernel.
	 */
	kgr___team_compute_features(team);
}

/*
 * patched: calls patched __team_compute_features(), inlined,
 * all callers also patched
 */
static int kgr_team_port_add(struct team *team, struct net_device *port_dev)
{
	struct net_device *dev = team->dev;
	struct team_port *port;
	char *portname = port_dev->name;
	int err;

	if (port_dev->flags & IFF_LOOPBACK) {
		netdev_err(dev, "Device %s is loopback device. Loopback devices can't be added as a team port\n",
			   portname);
		return -EINVAL;
	}

	if (kgr_team_port_exists(port_dev)) {
		netdev_err(dev, "Device %s is already a port "
				"of a team device\n", portname);
		return -EBUSY;
	}

	if (port_dev->features & NETIF_F_VLAN_CHALLENGED &&
	    vlan_uses_dev(dev)) {
		netdev_err(dev, "Device %s is VLAN challenged and team device has VLAN set up\n",
			   portname);
		return -EPERM;
	}

	err = kgr_team_dev_type_check_change(dev, port_dev);
	if (err)
		return err;

	if (port_dev->flags & IFF_UP) {
		netdev_err(dev, "Device %s is up. Set it down before adding it as a team port\n",
			   portname);
		return -EBUSY;
	}

	port = kzalloc(sizeof(struct team_port) + team->mode->port_priv_size,
		       GFP_KERNEL);
	if (!port)
		return -ENOMEM;

	port->dev = port_dev;
	port->team = team;
	INIT_LIST_HEAD(&port->qom_list);

	port->orig.mtu = port_dev->mtu;
	err = dev_set_mtu(port_dev, dev->mtu);
	if (err) {
		netdev_dbg(dev, "Error %d calling dev_set_mtu\n", err);
		goto err_set_mtu;
	}

	memcpy(port->orig.dev_addr, port_dev->dev_addr, port_dev->addr_len);

	err = kgr_team_port_enter(team, port);
	if (err) {
		netdev_err(dev, "Device %s failed to enter team mode\n",
			   portname);
		goto err_port_enter;
	}

	err = dev_open(port_dev);
	if (err) {
		netdev_dbg(dev, "Device %s opening failed\n",
			   portname);
		goto err_dev_open;
	}

	err = vlan_vids_add_by_dev(port_dev, dev);
	if (err) {
		netdev_err(dev, "Failed to add vlan ids to device %s\n",
				portname);
		goto err_vids_add;
	}

	err = kgr_team_port_enable_netpoll(team, port);
	if (err) {
		netdev_err(dev, "Failed to enable netpoll on device %s\n",
			   portname);
		goto err_enable_netpoll;
	}

	if (!(dev->features & NETIF_F_LRO))
		dev_disable_lro(port_dev);

	err = netdev_rx_handler_register(port_dev, kgr_team_handle_frame,
					 port);
	if (err) {
		netdev_err(dev, "Device %s failed to register rx_handler\n",
			   portname);
		goto err_handler_register;
	}

	err = kgr_team_upper_dev_link(dev, port_dev);
	if (err) {
		netdev_err(dev, "Device %s failed to set upper link\n",
			   portname);
		goto err_set_upper_link;
	}

	err = kgr__team_option_inst_add_port(team, port);
	if (err) {
		netdev_err(dev, "Device %s failed to add per-port options\n",
			   portname);
		goto err_option_port_add;
	}

	port->index = -1;
	list_add_tail_rcu(&port->list, &team->port_list);
	kgr_team_port_enable(team, port);
	kgr__team_compute_features(team);
	kgr__team_port_change_port_added(port, !!netif_carrier_ok(port_dev));
	kgr__team_options_change_check(team);

	netdev_info(dev, "Port device %s added\n", portname);

	return 0;

err_option_port_add:
	kgr_team_upper_dev_unlink(dev, port_dev);

err_set_upper_link:
	netdev_rx_handler_unregister(port_dev);

err_handler_register:
	kgr_team_port_disable_netpoll(port);

err_enable_netpoll:
	vlan_vids_del_by_dev(port_dev, dev);

err_vids_add:
	dev_close(port_dev);

err_dev_open:
	kgr_team_port_leave(team, port);
	kgr_team_port_set_orig_dev_addr(port);

err_port_enter:
	dev_set_mtu(port_dev, port->orig.mtu);

err_set_mtu:
	kfree(port);

	return err;
}

/* patched: calls patched __team_compute_features(), all callers also patched */
static int kgr_team_port_del(struct team *team, struct net_device *port_dev)
{
	struct net_device *dev = team->dev;
	struct team_port *port;
	char *portname = port_dev->name;

	port = kgr_team_port_get_rtnl(port_dev);
	if (!port || !kgr_team_port_find(team, port)) {
		netdev_err(dev, "Device %s does not act as a port of this team\n",
			   portname);
		return -ENOENT;
	}

	kgr_team_port_disable(team, port);
	list_del_rcu(&port->list);
	kgr_team_upper_dev_unlink(dev, port_dev);
	netdev_rx_handler_unregister(port_dev);
	kgr_team_port_disable_netpoll(port);
	vlan_vids_del_by_dev(port_dev, dev);
	dev_uc_unsync(port_dev, dev);
	dev_mc_unsync(port_dev, dev);
	dev_close(port_dev);
	kgr_team_port_leave(team, port);

	kgr__team_option_inst_mark_removed_port(team, port);
	kgr__team_options_change_check(team);
	kgr__team_option_inst_del_port(team, port);
	kgr__team_port_change_port_removed(port);

	kgr_team_port_set_orig_dev_addr(port);
	dev_set_mtu(port_dev, port->orig.mtu);
	kfree_rcu(port, rcu);
	netdev_info(dev, "Port device %s removed\n", portname);
	kgr__team_compute_features(team);

	return 0;
}

/* patched */
void kgr_team_uninit(struct net_device *dev)
{
	struct team *team = netdev_priv(dev);
	struct team_port *port;
	struct team_port *tmp;

	mutex_lock(&team->lock);
	list_for_each_entry_safe(port, tmp, &team->port_list, list)
		kgr_team_port_del(team, port->dev);

	kgr__team_change_mode(team, NULL); /* cleanup */
	kgr__team_options_unregister(team, *kgr_team_options,
				     kgr_team_options_array_size);
	kgr_team_mcast_rejoin_fini(team);
	kgr_team_notify_peers_fini(team);
	kgr_team_queue_override_fini(team);
	mutex_unlock(&team->lock);
	/*
	 * Fix bsc#1062847
	 *  +1 line
	 */
	netdev_change_features(dev);
}

/* patched */
int kgr_team_add_slave(struct net_device *dev, struct net_device *port_dev)
{
	struct team *team = netdev_priv(dev);
	int err;

	mutex_lock(&team->lock);
	err = kgr_team_port_add(team, port_dev);
	mutex_unlock(&team->lock);

	/*
	 * Fix bsc#1062847
	 *  +2 lines
	 */
	if (!err)
		 netdev_change_features(dev);

	return err;
}

/* patched */
int kgr_team_del_slave(struct net_device *dev, struct net_device *port_dev)
{
	struct team *team = netdev_priv(dev);
	int err;

	mutex_lock(&team->lock);
	err = kgr_team_port_del(team, port_dev);
	mutex_unlock(&team->lock);

	/*
	 * Fix bsc#1062847
	 *  +2 lines
	 */
	if (!err)
		 netdev_change_features(dev);

	return err;
}



static int kgr_patch_bsc1062847_kallsyms(void)
{
	unsigned long addr;
	int i;

	for (i = 0; i < ARRAY_SIZE(kgr_funcs); i++) {
		/* mod_find_symname would be nice, but it is not exported */
		addr = kallsyms_lookup_name(kgr_funcs[i].name);
		if (!addr) {
			pr_err("kgraft-patch: symbol %s not resolved\n",
				kgr_funcs[i].name);
			return -ENOENT;
		}

		*(kgr_funcs[i].addr) = (void *)addr;
	}

	return 0;
}

static int kgr_patch_bsc1062847_module_notify(struct notifier_block *nb,
					unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, KGR_PATCHED_MODULE))
		return 0;

	ret = kgr_patch_bsc1062847_kallsyms();
	WARN(ret, "kgraft-patch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block kgr_patch_bsc1062847_module_nb = {
	.notifier_call = kgr_patch_bsc1062847_module_notify,
	.priority = INT_MIN+1,
};

int kgr_patch_bsc1062847_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(KGR_PATCHED_MODULE)) {
		ret = kgr_patch_bsc1062847_kallsyms();
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&kgr_patch_bsc1062847_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void kgr_patch_bsc1062847_cleanup(void)
{
	unregister_module_notifier(&kgr_patch_bsc1062847_module_nb);
}
