/*
 * kgraft_patch_bsc1085114
 *
 * Fix for CVE-2018-1068, bsc#1085114
 *
 *  Upstream commits:
 *  b71812168571 ("netfilter: ebtables: CONFIG_COMPAT: don't trust userland
 *                 offsets")
 *  932909d9b28d ("netfilter: ebtables: fix erroneous reject of last rule")
 *
 *  SLE12(-SP1) commits:
 *  8d5366d651b3042a4cfa1b80e22f31b1c1268bd0
 *  44f5b40f2efe85bd17684eea272302c5e2190bf8
 *
 *  SLE12-SP2 commits:
 *  46197767dccbd3a20d0fa7b95a3e3948e7746637
 *  0db54544cc76d1678da41885f83a92a9b1f3d73d
 *
 *  SLE12-SP3 commits:
 *  523e0dcc6b20db65a8f9fb454634612e1a35ea26
 *  8797aa50c6d043033169363708d22e69b5b946c2
 *
 *  Copyright (c) 2018 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/kallsyms.h>
#include <linux/netfilter_bridge/ebtables.h>
#include <linux/netfilter/x_tables.h>
#include <asm/compat.h>
#include "kgr_patch_bsc1085114.h"

#if !IS_MODULE(CONFIG_BRIDGE_NF_EBTABLES)
#error "KGR patch supports only CONFIG_BRIDGE_NF_EBTABLES=m."
#endif

#if !IS_MODULE(CONFIG_NETFILTER_XTABLES)
#error "KGR patch supports only CONFIG_NETFILTER_XTABLES=m."
#endif

#if !IS_ENABLED(CONFIG_COMPAT)
#error "KGR patch supports only CONFIG_COMPAT=y."
#endif


#define KGR_PATCHED_MODULE "ebtables"


struct ebt_entries_buf_state;

static int (*kgr_ebt_buf_add)(struct ebt_entries_buf_state *state,
			      void *data, unsigned int sz);

static int (*kgr_xt_compat_match_offset)(const struct xt_match *match);
static struct xt_match* (*kgr_xt_request_find_match)(uint8_t nfproto,
						     const char *name,
						     uint8_t revision);
static struct xt_target* (*kgr_xt_request_find_target)(u8 af, const char *name,
						       u8 revision);
static int (*kgr_xt_compat_target_offset)(const struct xt_target *target);
static int (*kgr_xt_compat_add_offset)(u_int8_t af, unsigned int offset,
				       int delta);

static struct {
       char *name;
       void **addr;
} kgr_funcs[] = {
	{ "ebtables:ebt_buf_add", (void *)&kgr_ebt_buf_add },
	{ "x_tables:xt_compat_match_offset",
		(void *)&kgr_xt_compat_match_offset },
	{ "x_tables:xt_request_find_match",
		(void *)&kgr_xt_request_find_match },
	{ "x_tables:xt_request_find_target",
		(void *)&kgr_xt_request_find_target },
	{ "x_tables:xt_compat_target_offset",
		(void *)&kgr_xt_compat_target_offset },
	{ "x_tables:xt_compat_add_offset", (void *)&kgr_xt_compat_add_offset },
};


/* from net/bridge/netfilter/ebtables.c */
struct compat_ebt_entry_mwt {
	union {
		char name[EBT_FUNCTION_MAXNAMELEN];
		compat_uptr_t ptr;
	} u;
	compat_uint_t match_size;
	compat_uint_t data[0];
};

struct ebt_entries_buf_state {
	char *buf_kern_start;	/* kernel buffer to copy (translated) data to */
	u32 buf_kern_len;	/* total size of kernel buffer */
	u32 buf_kern_offset;	/* amount of data copied so far */
	u32 buf_user_offset;	/* read position in userspace buffer */
};

enum compat_mwt {
	EBT_COMPAT_MATCH,
	EBT_COMPAT_WATCHER,
	EBT_COMPAT_TARGET,
};

/* inlined */
static int kgr_ebt_compat_entry_padsize(void)
{
	BUILD_BUG_ON(XT_ALIGN(sizeof(struct ebt_entry_match)) <
			COMPAT_XT_ALIGN(sizeof(struct compat_ebt_entry_mwt)));
	return (int) XT_ALIGN(sizeof(struct ebt_entry_match)) -
			COMPAT_XT_ALIGN(sizeof(struct compat_ebt_entry_mwt));
}

/* inlined on ppc64le */
static int kgr_ebt_compat_match_offset(const struct xt_match *match,
				       unsigned int userlen)
{
	/*
	 * ebt_among needs special handling. The kernel .matchsize is
	 * set to -1 at registration time; at runtime an EBT_ALIGN()ed
	 * value is expected.
	 * Example: userspace sends 4500, ebt_among.c wants 4504.
	 */
	if (unlikely(match->matchsize == -1))
		return XT_ALIGN(userlen) - COMPAT_XT_ALIGN(userlen);
	return kgr_xt_compat_match_offset(match);
}

/* inlined */
static int kgr_ebt_buf_count(struct ebt_entries_buf_state *state,
			     unsigned int sz)
{
	state->buf_kern_offset += sz;
	return state->buf_kern_offset >= sz ? 0 : -EINVAL;
}

/* inlined */
static int kgr_ebt_buf_add_pad(struct ebt_entries_buf_state *state,
			       unsigned int sz)
{
	char *b = state->buf_kern_start;

	BUG_ON(b && state->buf_kern_offset > state->buf_kern_len);

	if (b != NULL && sz > 0)
		memset(b + state->buf_kern_offset, 0, sz);
	/* do not adjust ->buf_user_offset here, we added kernel-side padding */
	return kgr_ebt_buf_count(state, sz);
}

/* inlined */
static int kgr_compat_mtw_from_user(struct compat_ebt_entry_mwt *mwt,
				    enum compat_mwt compat_mwt,
				    struct ebt_entries_buf_state *state,
				    const unsigned char *base)
{
	char name[EBT_FUNCTION_MAXNAMELEN];
	struct xt_match *match;
	struct xt_target *wt;
	void *dst = NULL;
	int off, pad = 0;
	unsigned int size_kern, match_size = mwt->match_size;

	strlcpy(name, mwt->u.name, sizeof(name));

	if (state->buf_kern_start)
		dst = state->buf_kern_start + state->buf_kern_offset;

	switch (compat_mwt) {
	case EBT_COMPAT_MATCH:
		match = kgr_xt_request_find_match(NFPROTO_BRIDGE, name, 0);
		if (IS_ERR(match))
			return PTR_ERR(match);

		off = kgr_ebt_compat_match_offset(match, match_size);
		if (dst) {
			if (match->compat_from_user)
				match->compat_from_user(dst, mwt->data);
			else
				memcpy(dst, mwt->data, match_size);
		}

		size_kern = match->matchsize;
		if (unlikely(size_kern == -1))
			size_kern = match_size;
		module_put(match->me);
		break;
	case EBT_COMPAT_WATCHER: /* fallthrough */
	case EBT_COMPAT_TARGET:
		wt = kgr_xt_request_find_target(NFPROTO_BRIDGE, name, 0);
		if (IS_ERR(wt))
			return PTR_ERR(wt);
		off = kgr_xt_compat_target_offset(wt);

		if (dst) {
			if (wt->compat_from_user)
				wt->compat_from_user(dst, mwt->data);
			else
				memcpy(dst, mwt->data, match_size);
		}

		size_kern = wt->targetsize;
		module_put(wt->me);
		break;

	default:
		return -EINVAL;
	}

	state->buf_kern_offset += match_size + off;
	state->buf_user_offset += match_size;
	pad = XT_ALIGN(size_kern) - size_kern;

	if (pad > 0 && dst) {
		BUG_ON(state->buf_kern_len <= pad);
		BUG_ON(state->buf_kern_offset - (match_size + off) + size_kern > state->buf_kern_len - pad);
		memset(dst + size_kern, 0, pad);
	}
	return off + match_size;
}



/* Patched, inlined */
static int kgr_ebt_size_mwt(struct compat_ebt_entry_mwt *match32,
			    unsigned int size_left, enum compat_mwt type,
			    struct ebt_entries_buf_state *state, const void *base)
{
	int growth = 0;
	char *buf;

	if (size_left == 0)
		return 0;

	buf = (char *) match32;

	while (size_left >= sizeof(*match32)) {
		struct ebt_entry_match *match_kern;
		int ret;

		match_kern = (struct ebt_entry_match *) state->buf_kern_start;
		if (match_kern) {
			char *tmp;
			tmp = state->buf_kern_start + state->buf_kern_offset;
			match_kern = (struct ebt_entry_match *) tmp;
		}
		ret = kgr_ebt_buf_add(state, buf, sizeof(*match32));
		if (ret < 0)
			return ret;
		size_left -= sizeof(*match32);

		/* add padding before match->data (if any) */
		ret = kgr_ebt_buf_add_pad(state, kgr_ebt_compat_entry_padsize());
		if (ret < 0)
			return ret;

		if (match32->match_size > size_left)
			return -EINVAL;

		size_left -= match32->match_size;

		ret = kgr_compat_mtw_from_user(match32, type, state, base);
		if (ret < 0)
			return ret;

		BUG_ON(ret < match32->match_size);
		growth += ret - match32->match_size;
		growth += kgr_ebt_compat_entry_padsize();

		buf += sizeof(*match32);
		buf += match32->match_size;

		if (match_kern)
			match_kern->match_size = ret;

		/*
		 * Fix CVE-2018-1068
		 *  -1 line, +2 lines
		 */
		if (WARN_ON(type == EBT_COMPAT_TARGET && size_left))
			return -EINVAL;
		match32 = (struct compat_ebt_entry_mwt *) buf;
	}

	return growth;
}

/* Patched, inlined */
static int kgr_size_entry_mwt(struct ebt_entry *entry, const unsigned char *base,
			      unsigned int *total,
			      struct ebt_entries_buf_state *state)
{
	unsigned int i, j, startoff, new_offset = 0;
	/* stores match/watchers/targets & offset of next struct ebt_entry: */
	unsigned int offsets[4];
	unsigned int *offsets_update = NULL;
	int ret;
	char *buf_start;

	if (*total < sizeof(struct ebt_entries))
		return -EINVAL;

	if (!entry->bitmask) {
		*total -= sizeof(struct ebt_entries);
		return kgr_ebt_buf_add(state, entry, sizeof(struct ebt_entries));
	}
	if (*total < sizeof(*entry) || entry->next_offset < sizeof(*entry))
		return -EINVAL;

	startoff = state->buf_user_offset;
	/* pull in most part of ebt_entry, it does not need to be changed. */
	ret = kgr_ebt_buf_add(state, entry,
			offsetof(struct ebt_entry, watchers_offset));
	if (ret < 0)
		return ret;

	offsets[0] = sizeof(struct ebt_entry); /* matches come first */
	memcpy(&offsets[1], &entry->watchers_offset,
			sizeof(offsets) - sizeof(offsets[0]));

	if (state->buf_kern_start) {
		buf_start = state->buf_kern_start + state->buf_kern_offset;
		offsets_update = (unsigned int *) buf_start;
	}
	ret = kgr_ebt_buf_add(state, &offsets[1],
			sizeof(offsets) - sizeof(offsets[0]));
	if (ret < 0)
		return ret;
	buf_start = (char *) entry;
	/*
	 * 0: matches offset, always follows ebt_entry.
	 * 1: watchers offset, from ebt_entry structure
	 * 2: target offset, from ebt_entry structure
	 * 3: next ebt_entry offset, from ebt_entry structure
	 *
	 * offsets are relative to beginning of struct ebt_entry (i.e., 0).
	 */
	/*
	 * Fix CVE-2018-1068
	 *  +13 lines
	 */
	for (i = 0; i < 4 ; ++i) {
		if (offsets[i] > *total)
			return -EINVAL;

		if (i < 3 && offsets[i] == *total)
			return -EINVAL;

		if (i == 0)
			continue;
		if (offsets[i-1] > offsets[i])
			return -EINVAL;
	}

	for (i = 0, j = 1 ; j < 4 ; j++, i++) {
		struct compat_ebt_entry_mwt *match32;
		unsigned int size;
		char *buf = buf_start;

		buf = buf_start + offsets[i];
		if (offsets[i] > offsets[j])
			return -EINVAL;

		match32 = (struct compat_ebt_entry_mwt *) buf;
		size = offsets[j] - offsets[i];
		ret = kgr_ebt_size_mwt(match32, size, i, state, base);
		if (ret < 0)
			return ret;
		new_offset += ret;
		if (offsets_update && new_offset) {
			pr_debug("change offset %d to %d\n",
				offsets_update[i], offsets[j] + new_offset);
			offsets_update[i] = offsets[j] + new_offset;
		}
	}

	if (state->buf_kern_start == NULL) {
		unsigned int offset = buf_start - (char *) base;

		ret = kgr_xt_compat_add_offset(NFPROTO_BRIDGE, offset, new_offset);
		if (ret < 0)
			return ret;
	}

	startoff = state->buf_user_offset - startoff;

	BUG_ON(*total < startoff);
	*total -= startoff;
	return 0;
}

/* Patched, calls patched but inlined size_entry_mwt() */
int kgr_compat_copy_entries(unsigned char *data, unsigned int size_user,
			    struct ebt_entries_buf_state *state)
{
	unsigned int size_remaining = size_user;
	int ret;

	ret = EBT_ENTRY_ITERATE(data, size_user, kgr_size_entry_mwt, data,
					&size_remaining, state);
	if (ret < 0)
		return ret;

	WARN_ON(size_remaining);
	return state->buf_kern_offset;
}



static int kgr_patch_bsc1085114_kallsyms(void)
{
	unsigned long addr;
	int i;

	for (i = 0; i < ARRAY_SIZE(kgr_funcs); i++) {
		/* mod_find_symname would be nice, but it is not exported */
		addr = kallsyms_lookup_name(kgr_funcs[i].name);
		if (!addr) {
			pr_err("kgraft-patch: symbol %s not resolved\n",
				kgr_funcs[i].name);
			return -ENOENT;
		}

		*(kgr_funcs[i].addr) = (void *)addr;
	}

	return 0;
}

static int kgr_patch_bsc1085114_module_notify(struct notifier_block *nb,
					      unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, KGR_PATCHED_MODULE))
		return 0;

	ret = kgr_patch_bsc1085114_kallsyms();
	WARN(ret, "kgraft-patch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block kgr_patch_bsc1085114_module_nb = {
	.notifier_call = kgr_patch_bsc1085114_module_notify,
	.priority = INT_MIN+1,
};

int kgr_patch_bsc1085114_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	if (find_module(KGR_PATCHED_MODULE)) {
		ret = kgr_patch_bsc1085114_kallsyms();
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&kgr_patch_bsc1085114_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void kgr_patch_bsc1085114_cleanup(void)
{
	unregister_module_notifier(&kgr_patch_bsc1085114_module_nb);
}
