/*
 * kgraft_patch_bsc1071471_bnep
 *
 * Fix for CVE-2017-15868, bsc#1071471 -- bnep part
 *
 *  Upstream commit:
 *  71bb99a02b32 ("Bluetooth: bnep: bnep_add_connection() should verify that
 *                 it's dealing with l2cap socket")
 *
 *  SLE12-SP3 commit:
 *  not affected
 *
 *  SLE12-SP2 commit:
 *  not affected
 *
 *  SLE12(-SP1) commit:
 *  be839f6f9f2da0d6db8f637a65cde69761a08145
 *
 *  Copyright (c) 2017 SUSE
 *  Author: Nicolai Stange <nstange@suse.de>
 *
 *  Based on the original Linux kernel code. Other copyrights apply.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/kallsyms.h>
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/net.h>
#include <linux/socket.h>
#include <net/bluetooth/bluetooth.h>
#include <net/bluetooth/hci_core.h>
#include <linux/crc32.h>
#include <linux/kthread.h>

#if !IS_MODULE(CONFIG_BT_BNEP)
#error "KGR patch supports only CONFIG_BT_BNEP=m."
#endif

#if !IS_ENABLED(CONFIG_BT_BNEP_MC_FILTER)
#error "KGR patch supports only CONFIG_BT_BNEP_MC_FILTER=y."
#endif

#if !IS_ENABLED(CONFIG_BT_BNEP_PROTO_FILTER)
#error "KGR patch supports only CONFIG_BT_BNEP_PROTO_FILTER=y."
#endif

#if !IS_MODULE(CONFIG_BT)
#error "KGR patch supports only CONFIG_BT=m."
#endif

#define KGR_PATCHED_MODULE "bnep"

static struct module *kgr_bnep_module;

static struct list_head *kgr_bnep_session_list;
static struct rw_semaphore *kgr_bnep_session_sem;
static struct device_type *kgr_bnep_type;


static void (*kgr_baswap)(bdaddr_t *dst, bdaddr_t *src);
static bool (*kgr_l2cap_is_socket)(struct socket *sock);
static struct bnep_session* (*kgr__bnep_get_session)(u8 *dst);
static struct hci_dev* (*kgr_hci_get_route)(bdaddr_t *dst, bdaddr_t *src);
static int (*kgr_bnep_session)(void *arg);
static void (*kgr_bnep_net_setup)(struct net_device *dev);


static struct {
       char *name;
       void **addr;
} kgr_funcs[] = {
	{ "bnep:bnep_session_list", (void *)&kgr_bnep_session_list },
	{ "bnep:bnep_session_sem", (void *)&kgr_bnep_session_sem },
	{ "bnep:bnep_type", (void *)&kgr_bnep_type },
	{ "bluetooth:baswap", (void *)&kgr_baswap },
	{ "bluetooth:l2cap_is_socket", (void *)&kgr_l2cap_is_socket },
	{ "bnep:__bnep_get_session", (void *)&kgr__bnep_get_session },
	{ "bluetooth:hci_get_route", (void *)&kgr_hci_get_route },
	{ "bnep:bnep_session", (void *)&kgr_bnep_session },
	{ "bnep:bnep_net_setup", (void *)&kgr_bnep_net_setup },
};


/* from net/bluetooth/bnep/bnep.h */
#define BNEP_MAX_PROTO_FILTERS		5

struct bnep_connadd_req {
	int   sock;		/* Connected socket */
	__u32 flags;
	__u16 role;
	char  device[16];	/* Name of the Ethernet device */
};

struct bnep_proto_filter {
	__u16 start;
	__u16 end;
};

struct bnep_session {
	struct list_head list;

	unsigned int  role;
	unsigned long state;
	unsigned long flags;
	atomic_t      terminate;
	struct task_struct *task;

	struct ethhdr eh;
	struct msghdr msg;

	struct bnep_proto_filter proto_filter[BNEP_MAX_PROTO_FILTERS];
	unsigned long long mc_filter;

	struct socket    *sock;
	struct net_device *dev;
};

/* inlined */
static inline int kgr_bnep_mc_hash(__u8 *addr)
{
	return crc32_be(~0, addr, ETH_ALEN) >> 26;
}


/* from net/bluetooth/bnep/core.c */
/* inlined */
static void kgr__bnep_link_session(struct bnep_session *s)
{
	list_add(&s->list, kgr_bnep_session_list);
}

/* inlined */
static void kgr__bnep_unlink_session(struct bnep_session *s)
{
	list_del(&s->list);
}

/* inlined */
static inline void kgr_bnep_set_default_proto_filter(struct bnep_session *s)
{
	/* (IPv4, ARP)  */
	s->proto_filter[0].start = ETH_P_IP;
	s->proto_filter[0].end   = ETH_P_ARP;
	/* (RARP, AppleTalk) */
	s->proto_filter[1].start = ETH_P_RARP;
	s->proto_filter[1].end   = ETH_P_AARP;
	/* (IPX, IPv6) */
	s->proto_filter[2].start = ETH_P_IPX;
	s->proto_filter[2].end   = ETH_P_IPV6;
}

/* inlined */
static struct device *kgr_bnep_get_device(struct bnep_session *session)
{
	bdaddr_t *src = &bt_sk(session->sock->sk)->src;
	bdaddr_t *dst = &bt_sk(session->sock->sk)->dst;
	struct hci_dev *hdev;
	struct hci_conn *conn;

	hdev = kgr_hci_get_route(dst, src);
	if (!hdev)
		return NULL;

	conn = hci_conn_hash_lookup_ba(hdev, ACL_LINK, dst);

	hci_dev_put(hdev);

	return conn ? &conn->dev : NULL;
}


/* patched */
int kgr_bnep_add_connection(struct bnep_connadd_req *req, struct socket *sock)
{
	struct net_device *dev;
	struct bnep_session *s, *ss;
	u8 dst[ETH_ALEN], src[ETH_ALEN];
	int err;

	BT_DBG("");

	/*
	 * Fix CVE-2017-15868
	 *  +3 lines
	 */
	if (!kgr_l2cap_is_socket(sock))
		return -EBADFD;

	kgr_baswap((void *) dst, &bt_sk(sock->sk)->dst);
	kgr_baswap((void *) src, &bt_sk(sock->sk)->src);

	/* session struct allocated as private part of net_device */
	dev = alloc_netdev(sizeof(struct bnep_session),
				(*req->device) ? req->device : "bnep%d",
				kgr_bnep_net_setup);
	if (!dev)
		return -ENOMEM;

	down_write(kgr_bnep_session_sem);

	ss = kgr__bnep_get_session(dst);
	if (ss && ss->state == BT_CONNECTED) {
		err = -EEXIST;
		goto failed;
	}

	s = netdev_priv(dev);

	/* This is rx header therefore addresses are swapped.
	 * ie. eh.h_dest is our local address. */
	memcpy(s->eh.h_dest,   &src, ETH_ALEN);
	memcpy(s->eh.h_source, &dst, ETH_ALEN);
	memcpy(dev->dev_addr, s->eh.h_dest, ETH_ALEN);

	s->dev   = dev;
	s->sock  = sock;
	s->role  = req->role;
	s->state = BT_CONNECTED;

	s->msg.msg_flags = MSG_NOSIGNAL;

#ifdef CONFIG_BT_BNEP_MC_FILTER
	/* Set default mc filter */
	set_bit(kgr_bnep_mc_hash(dev->broadcast), (ulong *) &s->mc_filter);
#endif

#ifdef CONFIG_BT_BNEP_PROTO_FILTER
	/* Set default protocol filter */
	kgr_bnep_set_default_proto_filter(s);
#endif

	SET_NETDEV_DEV(dev, kgr_bnep_get_device(s));
	SET_NETDEV_DEVTYPE(dev, kgr_bnep_type);

	err = register_netdev(dev);
	if (err)
		goto failed;

	kgr__bnep_link_session(s);

	__module_get(kgr_bnep_module);
	s->task = kthread_run(kgr_bnep_session, s, "kbnepd %s", dev->name);
	if (IS_ERR(s->task)) {
		/* Session thread start failed, gotta cleanup. */
		module_put(kgr_bnep_module);
		unregister_netdev(dev);
		kgr__bnep_unlink_session(s);
		err = PTR_ERR(s->task);
		goto failed;
	}

	up_write(kgr_bnep_session_sem);
	strcpy(req->device, dev->name);
	return 0;

failed:
	up_write(kgr_bnep_session_sem);
	free_netdev(dev);
	return err;
}


static int kgr_patch_bsc1071471_bnep_kallsyms(void)
{
	unsigned long addr;
	int i;

	for (i = 0; i < ARRAY_SIZE(kgr_funcs); i++) {
		/* mod_find_symname would be nice, but it is not exported */
		addr = kallsyms_lookup_name(kgr_funcs[i].name);
		if (!addr) {
			pr_err("kgraft-patch: symbol %s not resolved\n",
				kgr_funcs[i].name);
			return -ENOENT;
		}

		*(kgr_funcs[i].addr) = (void *)addr;
	}

	return 0;
}

static int kgr_patch_bsc1071471_bnep_module_notify(struct notifier_block *nb,
					unsigned long action, void *data)
{
	struct module *mod = data;
	int ret;

	if (action != MODULE_STATE_COMING || strcmp(mod->name, KGR_PATCHED_MODULE))
		return 0;

	kgr_bnep_module = mod;

	ret = kgr_patch_bsc1071471_bnep_kallsyms();
	WARN(ret, "kgraft-patch: delayed kallsyms lookup failed. System is broken and can crash.\n");

	return ret;
}

static struct notifier_block kgr_patch_bsc1071471_bnep_module_nb = {
	.notifier_call = kgr_patch_bsc1071471_bnep_module_notify,
	.priority = INT_MIN+1,
};

int kgr_patch_bsc1071471_bnep_init(void)
{
	int ret;

	mutex_lock(&module_mutex);
	kgr_bnep_module = find_module(KGR_PATCHED_MODULE);
	if (kgr_bnep_module) {
		ret = kgr_patch_bsc1071471_bnep_kallsyms();
		if (ret)
			goto out;
	}

	ret = register_module_notifier(&kgr_patch_bsc1071471_bnep_module_nb);
out:
	mutex_unlock(&module_mutex);
	return ret;
}

void kgr_patch_bsc1071471_bnep_cleanup(void)
{
	unregister_module_notifier(&kgr_patch_bsc1071471_bnep_module_nb);
}
